/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::InitMatMonotonic(bool reserved)
{
    face_diff_.resize(cmn_face_num_);
    grad_dm_.resize(elem_num_);
    mono_mat_.malloc(cmn_face_num_, elem_num_, 0.0);
    
    gctl::point3dc tet_cen0, tet_cen1;
    gctl::point3dc fac_nor, fac_cen;
    double eff_len;
    for (size_t i = 0; i < cmn_face_num_; i++)
    {
        tet_cen0 = cmn_faces_[i].tet_ptr[0]->center();
        tet_cen1 = cmn_faces_[i].tet_ptr[1]->center();
    
        // calculate face norms
        fac_nor = gctl::cross(*cmn_faces_[i].vert_ptr[1] - *cmn_faces_[i].vert_ptr[0], 
            *cmn_faces_[i].vert_ptr[2] - *cmn_faces_[i].vert_ptr[0]).normal();

        // 计算从tet_ptr[0]指向tet_ptr[1]的公共面的单位法矢量
        // 如果小于零则反向 
        if (gctl::dot(fac_nor, tet_cen1 - tet_cen0) < 0)
        {
            fac_nor = -1.0*fac_nor;
        }

        // 计算公共面的中心位置
        fac_cen = 1.0/3.0*(*cmn_faces_[i].vert_ptr[0] + *cmn_faces_[i].vert_ptr[1] + 
            *cmn_faces_[i].vert_ptr[2]);

        // 计算fac_nor到半径方向的投影
        // 我们假设tet_ptr[0]在下 tet_ptr[1]在上 则eff_len>0
        eff_len = gctl::dot(fac_nor, fac_cen.normal());
        if (reserved) eff_len *= -1.0;

        // 注意当前的设置假设了模型数值单调向下递增时为正常情况
        // 则差分计算的结果为负 如果模型数值向下渐小则差分结果为正
        // 始终保持上减下的形式
        mono_mat_.insert(i, cmn_faces_[i].tet_ptr[0]->id, -1.0*eff_len);
        mono_mat_.insert(i, cmn_faces_[i].tet_ptr[1]->id, eff_len);
    }
    return;
}