/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::ExtractVerticProfile(const array<double> &data, const Earth1D &ref_earth, double top_dep, double btm_dep, double ddep, double srad, array<double> &pro_data)
{
    int pro_size = round((btm_dep - top_dep)/ddep) + 1;
    if (pro_size < 2)
    {
        throw std::runtime_error("The profile size must be at least two. From GIST::ModelSpace::ExtractVerticProfile(...)");
    }

    if (top_dep >= btm_dep || top_dep + ddep > btm_dep || srad <= 0.0)
    {
        throw std::runtime_error("Invalid paremeters. From GIST::ModelSpace::ExtractVerticProfile(...)");
    }

    point3ds ps;
    array<double> raw_deps, raw_wgts;
    if (data.size() == node_num_)
    {
        raw_wgts.resize(node_num_);
        raw_deps.resize(node_num_);
        for (size_t i = 0; i < node_num_; i++)
        {
            ps = nodes_[i].c2s();
            raw_deps[i] = ref_earth.get_radius(ps.lat) - ps.rad;
        }
    }
    else if (data.size() == elem_num_)
    {
        raw_wgts.resize(elem_num_);
        raw_deps.resize(elem_num_);
        for (size_t i = 0; i < elem_num_; i++)
        {
            ps = elems_[i].center().c2s();
            raw_deps[i] = ref_earth.get_radius(ps.lat) - ps.rad;
        }
    }
    else std::runtime_error("Incompatible model size. From GIST::ModelSpace::ExtractVerticProfile(...)");

    double dep, sum_wgt, sr;
    pro_data.resize(pro_size);
    for (size_t i = 0; i < pro_size; i++)
    {
        sum_wgt = 0.0;
        dep = top_dep + ddep*i;
        for (size_t j = 0; j < raw_deps.size(); j++)
        {
            sr = fabs(raw_deps[j]-dep);
            if (sr <= srad)
            {
                raw_wgts[j] = 1.0/pow(fabs(raw_deps[j]-dep) + 1e-30, 2);
            }
            else raw_wgts[j] = 0.0;

            sum_wgt += raw_wgts[j];
        }

        if (sum_wgt == 0.0)
        {
            pro_data[i] = NAN;
        }
        else
        {
            pro_data[i] = 0.0;
            for (size_t j = 0; j < raw_deps.size(); j++)
            {
                pro_data[i] += (raw_wgts[j]/sum_wgt)*data[j];
            }
        }
    }
    return;
}