/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::ExtractLocsProfile(const array<double> &data, const array<point3dc> &p_locs, array<double> &p_data, double srad)
{
    if (p_locs.empty())
    {
        std::runtime_error("Profile locations are empty. From GIST::ModelSpace::ExtractLocsProfile(...)");
    }

    p_data.resize(p_locs.size(), 0.0);
    if (data.size() == node_num_)
    {
        double sr, sum_wgt;
        _1d_array raw_wgts(node_num_);
        for (size_t i = 0; i < p_locs.size(); i++)
        {
            sum_wgt = 0.0;
            for (size_t j = 0; j < node_num_; j++)
            {
                sr = distance(p_locs[i], nodes_[j]);
                if (sr <= srad)
                {
                    raw_wgts[j] = 1.0/pow(sr + 1e-30, 2);
                }
                else raw_wgts[j] = 0.0;

                sum_wgt += raw_wgts[j];
            }

            for (size_t j = 0; j < node_num_; j++)
            {
                p_data[i] += (raw_wgts[j]/sum_wgt)*data[j];
            }
        }
    }
    else if (data.size() == elem_num_)
    {
        double sr, sum_wgt;
        _1d_array raw_wgts(elem_num_);
        for (size_t i = 0; i < p_locs.size(); i++)
        {
            sum_wgt = 0.0;
            for (size_t j = 0; j < elem_num_; j++)
            {
                sr = distance(p_locs[i], elems_[j].center());
                if (sr <= srad)
                {
                    raw_wgts[j] = 1.0/pow(sr + 1e-30, 2);
                }
                else raw_wgts[j] = 0.0;

                sum_wgt += raw_wgts[j];
            }

            for (size_t j = 0; j < elem_num_; j++)
            {
                p_data[i] += (raw_wgts[j]/sum_wgt)*data[j];
            }
        }
    }
    else std::runtime_error("Incompatible model size. From GIST::ModelSpace::ExtractLocsProfile(...)");
    return;
}