/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::ExportFaceNodeIndex(std::string fac_name, array<size_t> &node_idx, bool unpacked)
{
    if (!face_ready_ || !face_tag_ready_)
    {
        throw gctl::runtime_error("Faces (tags) are not initialized. From GIST::ModelSpace::ExportFaceNodeIndex(...)");
    }

    if (!phys_ready_)
    {
        throw gctl::runtime_error("Physics are not initialized. From GIST::ModelSpace::ExportFaceNodeIndex(...)");
    }

    int tar_tag = -9999;
    bool ok_tag = false;
    for (size_t i = 0; i < phys_.size(); i++)
    {
        if (phys_[i].name == fac_name)
        {
            tar_tag = phys_[i].phys_tag;
            ok_tag = true;
            break;
        }
    }
    
    if (!ok_tag)
    {
        throw gctl::runtime_error("Fail to find the boundary: " + fac_name + ". From GIST::ModelSpace::ExportFaceNodeIndex(...)");
    }

    size_t v_id, bn_num;
    if (unpacked)
    {
        bn_num = 0;
        for (size_t i = 0; i < face_num_; i++)
        {
            if (face_tags_[i] == tar_tag)
            {
                bn_num++;
            }
        }

        node_idx.resize(3*bn_num);

        for (size_t i = 0; i < face_num_; i++)
        {
            if (face_tags_[i] == tar_tag)
            {
                for (size_t k = 0; k < 3; k++)
                {
                    v_id = faces_[i].vert[k]->id;
                    node_idx[3*i + k] = v_id;
                }
            }
        }
    }
    else
    {
        array<bool> bn(node_num_, false);

        for (size_t i = 0; i < face_num_; i++)
        {
            if (face_tags_[i] == tar_tag)
            {
                for (size_t k = 0; k < 3; k++)
                {
                    v_id = faces_[i].vert[k]->id;
                    bn[v_id] = true;
                }
            }
        }

        bn_num = 0;
        for (size_t i = 0; i < node_num_; i++)
        {
            if (bn[i]) bn_num++;
        }
        
        node_idx.resize(bn_num);

        size_t c = 0;
        for (size_t i = 0; i < node_num_; i++)
        {
            if (bn[i])
            {
                node_idx[c] = i;
                c++;
            }
        }
    }
    return;
}

void GIST::ModelSpace::ExportElemNodeIndex(std::string ele_name, array<size_t> &node_idx)
{
    if (!elem_tag_ready_)
    {
        throw gctl::runtime_error("Elements (tags) are not initialized. From GIST::ModelSpace::ExportElemNodeIndex(...)");
    }

    if (!phys_ready_)
    {
        throw gctl::runtime_error("Physics are not initialized. From GIST::ModelSpace::ExportElemNodeIndex(...)");
    }

    int tar_tag = -9999;
    bool ok_tag = false;
    for (size_t i = 0; i < phys_.size(); i++)
    {
        if (phys_[i].name == ele_name)
        {
            tar_tag = phys_[i].phys_tag;
            ok_tag = true;
            break;
        }
    }
    
    if (!ok_tag)
    {
        throw gctl::runtime_error("Fail to find the element: " + ele_name + ". From GIST::ModelSpace::ExportElemNodeIndex(...)");
    }

    size_t v_id;
    array<bool> bn(node_num_, false);
    for (size_t i = 0; i < elem_num_; i++)
    {
        if (elem_tags_[i] == tar_tag)
        {
            for (size_t k = 0; k < 4; k++)
            {
                v_id = elems_[i].vert[k]->id;
                bn[v_id] = true;
            }
        }
    }

    size_t bn_num = 0;
    for (size_t i = 0; i < node_num_; i++)
    {
        if (bn[i]) bn_num++;
    }
    
    node_idx.resize(bn_num);

    size_t c = 0;
    for (size_t i = 0; i < node_num_; i++)
    {
        if (bn[i])
        {
            node_idx[c] = i;
            c++;
        }
    }
    return;
}