/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

/**
 * @brief Export elements' index of the given tag name(s).
 * 
 * @param ele_name Tag names of the targeting elements.
 * @param ele_idx Exported elements' index
 */
void GIST::ModelSpace::ExportElementIndex(std::string ele_name, array<size_t> &ele_idx)
{
    if (!elem_tag_ready_)
    {
        throw gctl::runtime_error("Elements' tag are not initialized. From GIST::ModelSpace::ExportElementIndex(...)");
    }

    if (!phys_ready_)
    {
        throw gctl::runtime_error("Physical groups are not initialized. From GIST::ModelSpace::ExportElementIndex(...)");
    }

    int tar_tag = -9999;
    bool ok_tag = false;
    for (size_t i = 0; i < phys_.size(); i++)
    {
        if (phys_[i].name == ele_name && phys_[i].dim_tag == 3)
        {
            tar_tag = phys_[i].phys_tag;
            ok_tag = true;
            break;
        }
    }

    if (!ok_tag)
    {
        throw gctl::runtime_error("Fail to find elements: " + ele_name + ". From GIST::ModelSpace::ExportElementIndex(...)");
    }

    array<bool> bn(elem_num_, false);
    for (size_t i = 0; i < elem_num_; i++)
    {
        if (elem_tags_[i] == tar_tag)
        {
            bn[i] = true;
        }
    }

    size_t bn_num = 0;
    for (size_t i = 0; i < elem_num_; i++)
    {
        if (bn[i]) bn_num++;
    }

    ele_idx.resize(bn_num);

    size_t c = 0;
    for (size_t i = 0; i < elem_num_; i++)
    {
        if (bn[i])
        {
            ele_idx[c] = i;
            c++;
        }
    }
    return;
}
