/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::CrtSubModelSpace(ModelSpace &out_ms, std::initializer_list<std::string> ele_names, std::initializer_list<std::string> face_names, 
    array<size_t> *sub2whole_ele_idx, array<size_t> *sub2whole_fac_idx, array<size_t> *sub2whole_node_idx)
{
    std::initializer_list<std::string>::iterator s_iter;

    // Sort out phys groups
    size_t c = 0;
    array<gmsh_physical_group> phys(face_names.size() + ele_names.size());
    for (s_iter = face_names.begin(); s_iter != face_names.end(); s_iter++)
    {
        for (size_t i = 0; i < phys_.size(); i++)
        {
            if (phys_[i].name == *s_iter)
            {
                phys[c] = phys_[i];
                c++;
                break;
            }
        }
    }

    for (s_iter = ele_names.begin(); s_iter != ele_names.end(); s_iter++)
    {
        for (size_t i = 0; i < phys_.size(); i++)
        {
            if (phys_[i].name == *s_iter)
            {
                phys[c] = phys_[i];
                c++;
                break;
            }
        }
    }
 
    array<size_t> sub_fac_idx, sub_ele_idx, tmp_idx;
    for (s_iter = face_names.begin(); s_iter != face_names.end(); s_iter++)
    {
        ExportFaceIndex(*s_iter, tmp_idx);
        sub_fac_idx.append_array(tmp_idx);
    }

    for (s_iter = ele_names.begin(); s_iter != ele_names.end(); s_iter++)
    {
        ExportElementIndex(*s_iter, tmp_idx);
        sub_ele_idx.append_array(tmp_idx);
    }

    // Sort out all nodes that need to be exported
    size_t v_id;
    array<bool> bn(node_num_, false);
    for (size_t i = 0; i < sub_fac_idx.size(); i++)
    {
        for (size_t k = 0; k < 3; k++)
        {
            v_id = faces_[sub_fac_idx[i]].vert[k]->id;
            bn[v_id] = true;
        }
    }

    for (size_t i = 0; i < sub_ele_idx.size(); i++)
    {
        for (size_t k = 0; k < 4; k++)
        {
            v_id = elems_[sub_ele_idx[i]].vert[k]->id;
            bn[v_id] = true;
        }
    }

    size_t bn_num = 0;
    for (size_t i = 0; i < node_num_; i++)
    {
        if (bn[i]) bn_num++;
    }

    c = 0;
    array<vertex3dc> nodes(bn_num);
    array<size_t> o2n_idx(node_num_, 0);
    for (size_t i = 0; i < node_num_; i++)
    {
        if (bn[i])
        {
            o2n_idx[i] = c;
            nodes[c] = nodes_[i];
            nodes[c].id = c;
            c++;
        }
    }

    if (sub2whole_node_idx != nullptr)
    {
        c = 0;
        sub2whole_node_idx->resize(bn_num);
        for (size_t i = 0; i < node_num_; i++)
        {
            if (bn[i])
            {
                sub2whole_node_idx->at(c) = i;
                c++;
            }
        }
    }

    // Sort out faces
    array<triangle> faces(sub_fac_idx.size());
    for (size_t i = 0; i < sub_fac_idx.size(); i++)
    {
        faces[i].id = i;
        for (size_t j = 0; j < 3; j++)
        {
            v_id = o2n_idx[faces_[sub_fac_idx[i]].vert[j]->id];
            faces[i].vert[j] = nodes.get(v_id);
        }
    }
  
    // Sort out elements
    array<tetrahedron> elems(sub_ele_idx.size());
    for (size_t i = 0; i < sub_ele_idx.size(); i++)
    {
        elems[i].id = i;
        for (size_t j = 0; j < 4; j++)
        {
            v_id = o2n_idx[elems_[sub_ele_idx[i]].vert[j]->id];
            elems[i].vert[j] = nodes.get(v_id);
        }
    }
   
    // Sort out element tags
    matrix<int> etags(sub_ele_idx.size(), etags_.col_size());
    for (size_t i = 0; i < etags.row_size(); i++)
    {
        for (size_t j = 0; j < etags.col_size(); j++)
        {
            etags[i][j] = etags_[sub_ele_idx[i]][j];
        }
    }

    // Sort out face tags
    matrix<int> ftags(sub_fac_idx.size(), ftags_.col_size());
    for (size_t i = 0; i < ftags.row_size(); i++)
    {
        for (size_t j = 0; j < ftags.col_size(); j++)
        {
            ftags[i][j] = ftags_[sub_fac_idx[i]][j];
        }
    }

    if (sub2whole_ele_idx != nullptr)
    {
        sub2whole_ele_idx->resize(sub_ele_idx.size());
        for (size_t i = 0; i < sub_ele_idx.size(); i++)
        {
            sub2whole_ele_idx->at(i) = sub_ele_idx[i];
        }
    }
    
    if (sub2whole_fac_idx != nullptr)
    {
        sub2whole_fac_idx->resize(sub_fac_idx.size());
        for (size_t i = 0; i < sub_fac_idx.size(); i++)
        {
            sub2whole_fac_idx->at(i) = sub_fac_idx[i];
        }
    }

    out_ms.InitMesh(nodes, faces, elems, phys, etags, ftags);
    return;
}