/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

/**
 * @brief 计算射线与三角形所在平面的交点与三角形三条边组成的三个三角形的面积之和与原三角形面积之差
 * 
 * @param origin 原点
 * @param direct 射线方向
 * @param tri_facet 直角坐标系内的一个三角形
 * @param cross_loc 以引用形式返回的射线与三角形所在平面的交点
 * @return 面积差 交点在三角形内时这个值为零 交点在三角形外这个值大于零
 */
double triangle_covered_area(const point3dc &origin, const point3dc &direct, const triangle &tri_facet, point3dc &cross_loc);

/**
 * @brief 利用线性插值计算一个层状模型
 * 
 * @param elem_tags 层状模型内单元体模型名称（可能有多层）
 * @param upfac_tag 层状模型上顶面顶点模型名称
 * @param dnfac_tag 层状模型下底面顶点模型名称
 * @param up_val 上顶面模型值
 * @param dn_val 下底面模型值
 * @param out_data 返回的层状模型的顶点数据
 * @param out_idx 返回的层状模型的顶点元素索引
 */
void GIST::ModelSpace::CrtNodeDataLayer(std::vector<std::string> elem_tags, std::string upfac_tag, std::string dnfac_tag, 
    double up_val, double dn_val, array<double> &out_data, array<size_t> &out_idx)
{
    // Find elements of the targeting layer.
    array<size_t> e_idx, tmp_idx;
    for (size_t i = 0; i < elem_tags.size(); i++)
    {
        ExportElementIndex(elem_tags[i], tmp_idx);
        e_idx.append_array(tmp_idx);
    }

    // Find faces of the upper and lower boundaries.
    array<size_t> ub_idx, lb_idx;
    ExportFaceIndex(upfac_tag, ub_idx);
    ExportFaceIndex(dnfac_tag, lb_idx);

    // Keep tracking calculated nodes.
    size_t v_id;
    array<bool> used_nodes(node_num_, false);
    array<double> foo_data(node_num_, 0.0);

    // Assign values to nodes that are belonging to the upper and lower boundies.
    for (size_t i = 0; i < ub_idx.size(); i++)
    {
        for (size_t j = 0; j < 3; j++)
        {
            v_id = faces_[ub_idx[i]].vert[j]->id;
            if (!used_nodes[v_id]) // Only operate on un-used nodes.
            {
                foo_data[v_id] = up_val;
                used_nodes[v_id] = true;
            }
        }
    }

    for (size_t i = 0; i < lb_idx.size(); i++)
    {
        for (size_t j = 0; j < 3; j++)
        {
            v_id = faces_[lb_idx[i]].vert[j]->id;
            if (!used_nodes[v_id]) // Only operate on un-used nodes.
            {
                foo_data[v_id] = dn_val;
                used_nodes[v_id] = true;
            }   
        }
    }

    // Loop elements to interpolate value for nodes that are in between boundaries.
    point3dc ori(0.0, 0.0, 0.0), tmp_cross, up_cross, dn_cross;
    double mini_area, area;
    vertex3dc *v_ptr = nullptr;
    for (size_t i = 0; i < e_idx.size(); i++)
    {
        for (size_t j = 0; j < 4; j++)
        {
            v_ptr = elems_[e_idx[i]].vert[j];
            v_id = v_ptr->id;
            if (!used_nodes[v_id]) // Only operate on un-used nodes.
            {
                // Loop boundaries to locate the instrsection point.
                mini_area = 1e+30;
                for (size_t f = 0; f < ub_idx.size(); f++)
                {
                    area = triangle_covered_area(ori, *v_ptr, faces_[ub_idx[f]], tmp_cross);
                    if (area < mini_area) // find the triangle that is closest to the intersection point.
                    {
                        mini_area = area;
                        up_cross = tmp_cross;
                    }
                }

                mini_area = 1e+30;
                for (size_t f = 0; f < lb_idx.size(); f++)
                {
                    area = triangle_covered_area(ori, *v_ptr, faces_[lb_idx[f]], tmp_cross);
                    if (area < mini_area)
                    {
                        mini_area = area;
                        dn_cross = tmp_cross;
                    }
                }

                foo_data[v_id] = (up_val - dn_val)*(v_ptr->module() - dn_cross.module())/(up_cross.module() - dn_cross.module()) + dn_val;
                used_nodes[v_id] = true;
            }
        }
    }
    
    // Copy data and index to output arraies.
    size_t used_num = 0;
    for (size_t i = 0; i < node_num_; i++)
    {
        if (used_nodes[i]) used_num++;
    }
    out_data.resize(used_num);
    out_idx.resize(used_num);

    size_t c = 0;
    for (size_t i = 0; i < node_num_; i++)
    {
        if (used_nodes[i])
        {
            out_data[c] = foo_data[i];
            out_idx[c] = i;
            c++;
        }
    }
    return;
}

double triangle_covered_area(const point3dc &origin, const point3dc &direct, const triangle &tri_facet, point3dc &cross_loc)
{
	point3dc face_nor = cross(*tri_facet.vert[1] - *tri_facet.vert[0], *tri_facet.vert[2] - *tri_facet.vert[0]).normal();
	point3dc dir_nor = direct.normal();
	cross_loc = gctl::geometry3d::line_on_plane(*tri_facet.vert[0], face_nor, origin, dir_nor);

    double area1 = gctl::geometry3d::triangle_area(*tri_facet.vert[0], *tri_facet.vert[1], cross_loc);
    double area2 = gctl::geometry3d::triangle_area(*tri_facet.vert[0], *tri_facet.vert[2], cross_loc);
    double area3 = gctl::geometry3d::triangle_area(*tri_facet.vert[1], *tri_facet.vert[2], cross_loc);
    double Area = gctl::geometry3d::triangle_area(*tri_facet.vert[0], *tri_facet.vert[1], *tri_facet.vert[2]);
    return area1 + area2 + area3 - Area;
}