/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "gravity.h"

void GIST::Gravity::SaveObs(std::string filename, bool plus_err)
{
    if (!obs_ready_)
    {
        throw gctl::runtime_error("Gravity observations not ready. From GIST::Gravity::SaveObs(...)");
    }
    
    std::ofstream outfile;
    gctl::open_outfile(outfile, filename, ".txt");

    double ele;

    outfile << "# reference system: " << std::to_string(earth_1d_->get_minor_radius()) << "/" << std::to_string(earth_1d_->get_major_radius()) << "\n";
    outfile << "# observation number\n" << obsp_num_ << "\n";

    array<double> noises;
    if (err_ready_ && plus_err)
    {
        noises.resize(obsp_num_);
        random(noises, 0.0, gravity_err_);
    }
    
    outfile << "# lon(deg) lat(deg) elev(m) gravity_for(mGal)";
    if (err_ready_ && plus_err) outfile << " STD(mGal)";
    if (!obs_gravity_.empty()) outfile << " gravity_obs(mGal) diff(mGal)";
    outfile << "\n";

    for (size_t i = 0; i < obsp_num_; i++)
    {
        ele = gravity_obsp_[i].rad - earth_1d_->get_radius(gravity_obsp_[i].lat);

        outfile << std::setprecision(12) << gravity_obsp_[i].lon << " " << gravity_obsp_[i].lat << " " << ele;

        if (err_ready_ && plus_err) outfile << std::setprecision(12) << " " << pre_gravity_[i] + noises[i] << " " << gravity_err_;
        else outfile << std::setprecision(12) << " " << pre_gravity_[i];

        if (!obs_gravity_.empty()) outfile << std::setprecision(12) << " " << obs_gravity_[i] << " " << pre_gravity_[i] - obs_gravity_[i];

        outfile << "\n";
    }

    outfile.close();
    return;
}