/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "gravity.h"
#include "omp.h"

double GIST::Gravity::GravityObjectiveFunc(const array<double> &rho, array<double> &out_grad, array<double> *ref_rho)
{
    if (!err_ready_)
    {
        throw gctl::runtime_error("The gravity uncertainty is not set. From GIST::Gravity::GravityObjectiveFunc(...)");
    }

    // Calculate the pre_gravity_ and malloc spaces
    CalGravityObs(rho);
    out_grad.resize(elem_num_);

    size_t i, j;
#pragma omp parallel for private (i, j) schedule(guided)
    for (j = 0; j < elem_num_; j++)
    {
        out_grad[j] = 0.0;
        for (i = 0; i < obsp_num_; i++)
        {
            out_grad[j] += 2.0 * gravity_kernel_[i][j] * (pre_gravity_[i] - obs_gravity_[i]) / (gravity_err_*gravity_err_*obsp_num_);
        }
    }
    /*
    // Reference/Minimal model constraint
    if (ref_rho != nullptr)
    {
#pragma omp parallel for private (j) schedule(guided)
        for (j = 0; j < elem_num_; j++)
        {
            out_grad[j] += 2.0*reg_strength_*model_wgts_[j]*(rho[j] - ref_rho->at(j))/(cnst_err_*cnst_err_*elem_num_);
        }    
    }
    else
    {
#pragma omp parallel for private (j) schedule(guided)
        for (j = 0; j < elem_num_; j++)
        {
            out_grad[j] += 2.0*reg_strength_*model_wgts_[j]*rho[j]/(cnst_err_*cnst_err_*elem_num_);
        }        
    }
    */
#pragma omp parallel for private (j) schedule(guided)
    for (j = 0; j < elem_num_; j++)
    {
        out_grad[j] *= precndt_wgts_[j];
    }

    /*
    double mini_grad = 1e+30, maxi_grad = -1e+30;
    for (size_t i = 0; i < elem_num_; i++)
    {
        mini_grad = std::min(mini_grad, out_grad[i]);
        maxi_grad = std::max(maxi_grad, out_grad[i]);
    }

    for (size_t i = 0; i < elem_num_; i++)
    {
        out_grad[i] = out_grad[i]/(maxi_grad - mini_grad);
    }
    */

    normalize(out_grad, 1.0, L2);

    double dmf = 0.0;
    for (i = 0; i < obsp_num_; i++)
    {
        dmf += power2((pre_gravity_[i] - obs_gravity_[i])/gravity_err_);
    }
    dmf /= obsp_num_;

    return dmf;
    /*
    double m_dmf = 0.0;
    if (ref_rho != nullptr)
    {
        for (j = 0; j < elem_num_; j++)
        {
            m_dmf += power2(model_wgts_[j]*(rho[j] - ref_rho->at(j))/cnst_err_);
        }
    }
    else
    {
        for (j = 0; j < elem_num_; j++)
        {
            m_dmf += power2(model_wgts_[j]*rho[j]/cnst_err_);
        }
    }
    m_dmf /= elem_num_;

    return dmf + reg_strength_*m_dmf;
    */
}