/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#ifndef _GIST_GRAVITY_H
#define _GIST_GRAVITY_H

#include "gctl/core.h"
#include "gctl/io.h"
#include "gctl/maths.h"
#include "gctl/potential.h"
#include "gctl/optimization.h"

#include "../utility/enum.h"
#include "../utility/data_type.h"
#include "../earth_1d/earth_1d.h"
#include "../model_space/model_space.h"

using namespace gctl;

namespace GIST
{
    class Gravity : public gctl::lcg_solver
    {
    public:
        Gravity();
        virtual ~Gravity();
        virtual void LCG_Mx(const array<double> &x, array<double> &ax); // LCG预优函数
        virtual void LCG_Ax(const array<double> &x, array<double> &ax); // LCG核函数
        virtual int LCG_Progress(const array<double> &m, const double converge, const gctl::lcg_para &param, size_t t); // Reload the lcg progress function.

        /**
         * @brief Import model and gravity system setups.
         * 
         * @param m_space Pointer to the model space object.
         * @param earth_1d Pointer of the reference earth object.
         * @param m_type Value type of the density model of which is working with.
         * @param o_type Value type of the observation of which is working with.
         */
        void ImportModelSetup(ModelSpace *m_space, Earth1D *earth_1d, model_val_type_e m_type, obs_val_type_e o_type);
        /**
         * @brief Initiate the geoid obervations.
         * 
         * @param obspara_str Parameters for initiating the geoid observations. Could be either a filename or a formatted string.
         */
        void InitObs(std::string obspara_str);
        /**
         * @brief Save the geoid observations.
         * 
         * @param filename Name of the target file.
         */
        void SaveObs(std::string filename, bool plus_err = true);
        /**
         * @brief Initiate the gravity kernel from a file. Otherwise calculate one.
         * 
         * @param filename The .mat binary file
         */
        void InitGravityKernel(std::string filename);
        /**
         * @brief Calculate the kernel matrix for forward modeling the predicted gravity data.
         * 
         */
        void CalGravityKernel();       
        /**
         * @brief Load the gravity kernel from a (*.ar2) binary file.
         * 
         * @param filename Name of the input file.
         */
        void LoadGravityKernel(std::string filename);
        /**
         * @brief Save the gravity kernel to a (*.ar2) binary file.
         * 
         * @param filename Name of the output file.
         */
        void SaveGravityKernel(std::string filename);
        /**
         * @brief Initiate the background density model.
         * 
         * @param bkg_den The input density model.
         */
        void InitBkgDensityModel(const array<double> &bkg_den);
        /**
         * @brief Calculate the predicted gravity data.
         * 
         * @param den density model.
         */
        void CalGravityObs(const array<double> &den);
        /**
         * @brief Calculate the predicted gravity data without assembling the gravitational kernel.
         * 
         * @param den density model.
         */
        void CalGravityObsNokernel(const array<double> &den);
        /**
         * @brief Set the model weights object
         * 
         * @param in_wgts Input elements' weight
         */
        void SetModelWeights(array<double> *in_wgts = nullptr);
        /**
         * @brief Set the model weights object
         * 
         * @param in_wgts Input elements' weight
         */
        void SetPreconditionWeights(array<double> *in_wgts = nullptr);
        /**
         * @brief Objective function of the nonlinear inversion.
         * 
         * @param rho density model.
         * @param out_grad output model gradients.
         * @return current value of the objective function.
         */
        double GravityObjectiveFunc(const array<double> &rho, array<double> &out_grad, array<double> *ref_rho);
        /**
         * @brief Solve density model with the reference model constrain.
         * 
         * @param rho Solved density model.
         */
        void Solve(array<double> &rho, const array<double> &ref_rho);
        /**
         * @brief Set the geoid uncertainty.
         * 
         * @param err The uncertainty.
         */
        void set_gravity_error(double err);
        /**
         * @brief Set the reference model uncertainty.
         * 
         * @param err The uncertainty.
         */
        void set_constrain_error(double err);
        /**
         * @brief 设置密度模型约束求解中的正则化强度
         * 
         * @param val 正则化参数
         */
        void set_regularization_strength(double val);
        /**
         * @brief Get the observation points
         * 
         * @return Locations of the observation points.
         */
        array<point3ds> &get_obsp();
        /**
         * @brief Get the gravity kernel
         * 
         * @return gravity kernel 
         */
        matrix<double> &get_kernel();

        size_t get_obs_num();

        double get_grav_err();

        array<double> &get_obs_val();

    private:
        // system ready, observation ready, error ready, gravitational kernel ready
        bool sys_ready_, obs_ready_, err_ready_, kernel_ready_, bkg_ready_;

        Earth1D *earth_1d_; // Pointer of the reference 1D earth
        ModelSpace *m_space_; // Pointer of the model space
        model_val_type_e mv_type_;
        obs_val_type_e ov_type_;

        size_t obsp_num_; // number of the gravitational observations
        array<double> obs_gravity_; // gravitational observations
        array<double> pre_gravity_; // predicted gravitational data
        array<point3ds> gravity_obsp_; // Locations of the gravitational observations

        size_t elem_num_; // number of the gravitational elements
        array<double> model_wgts_, precndt_wgts_;
        array<double> bkg_den_; // backgroud density model
        array<double> cnst_tar_;
        array<grav_tetrahedron> gravity_ele_; // gravitational elements
        array<gravtet_para> gravity_para_; // parameters of the gravitational elements
        matrix<double> gravity_kernel_; // gravitational kernel

        double reg_strength_, gravity_err_, cnst_err_, grav_rms_; // error of the gravitational observations
    };
}

#endif // _GIST_GRAVITY_H