/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "gravity.h"

GIST::Gravity::Gravity()
{
    gravity_err_ = 0.0;
    elem_num_ = obsp_num_ = 0;
    sys_ready_ = obs_ready_ = err_ready_ = kernel_ready_ = bkg_ready_ = false;
}

GIST::Gravity::~Gravity(){}

void GIST::Gravity::LCG_Mx(const array<double> &x, array<double> &ax)
{
    vecmul(ax, x, precndt_wgts_);
    return;
}

void GIST::Gravity::LCG_Ax(const array<double> &x, array<double> &ax)
{
    // Data-misfit part
    size_t i, j;
#pragma omp parallel for private (i, j) schedule(guided)
    for (i = 0; i < obsp_num_; i++)
    {
        pre_gravity_[i] = 0.0;
        for (j = 0; j < elem_num_; j++)
        {
            // Note here x is not always the density model
            pre_gravity_[i] += gravity_kernel_[i][j]*x[j];
        }
    }

#pragma omp parallel for private (i, j) schedule(guided)
    for (j = 0; j < elem_num_; j++)
    {
        ax[j] = 0.0;
        for (i = 0; i < obsp_num_; i++)
        {
            ax[j] += gravity_kernel_[i][j]*pre_gravity_[i]/(gravity_err_*gravity_err_*obsp_num_);
        }
    }

    // Reference model constraint part
#pragma omp parallel for private (j) schedule(guided)
    for (j = 0; j < elem_num_; j++)
    {
        ax[j] += reg_strength_*model_wgts_[j]*x[j]/(cnst_err_*cnst_err_*elem_num_);
    }
  
    return;
}

int GIST::Gravity::LCG_Progress(const array<double> &m, const double converge, const gctl::lcg_para &param, size_t t)
{
    CalGravityObs(m);

    // Calculate the data-rms
    grav_rms_ = 0.0;
    for (size_t i = 0; i < obsp_num_; i++)
    {
        grav_rms_ += gctl::power2((pre_gravity_[i] - obs_gravity_[i])/gravity_err_);
    }
    grav_rms_ /= obsp_num_;

    std::clog << GCTL_CLEARLINE << "\rIteration: " << t << ", Convergence: " << converge << ", Data_RMS: " << grav_rms_;

    if (grav_rms_ < 1.05) return 1;
    return 0;
}

void GIST::Gravity::set_gravity_error(double err)
{
    if (err <= 0.0)
    {
        throw gctl::runtime_error("Invalid gravity uncertainty. From GIST::Gravity::set_gravity_error(...)");
    }
    
    gravity_err_ = err;
    err_ready_ = true;
    return;
}

array<point3ds> &GIST::Gravity::get_obsp()
{
    if (!obs_ready_)
    {
        throw gctl::runtime_error("Gravity observations are not ready. From GIST::Gravity::get_obsp(...)");
    }

    return gravity_obsp_;
}

matrix<double> &GIST::Gravity::get_kernel()
{
    return gravity_kernel_;
}

size_t GIST::Gravity::get_obs_num()
{
    return obsp_num_;
}

double GIST::Gravity::get_grav_err()
{
    return gravity_err_;
}

array<double> &GIST::Gravity::get_obs_val()
{
    return obs_gravity_;
}

void GIST::Gravity::set_constrain_error(double err)
{
    cnst_err_ = err;
    return;
}

void GIST::Gravity::set_regularization_strength(double val)
{
    reg_strength_ = val;
    return;
}