/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "geophysic_model.h"

double GIST::GeophysicalModel::GeophysicalModelObjectiveFunc(const array<double> &pphys_vec, array<double> &out_grad)
{
    if (!err_ready_ || !sys_ready_ || !tar_ready_)
    {
        throw std::runtime_error("The geophysical model system is not set properly. From GIST::GeophysicalModel::GeophysicalModelObjectiveFunc(...)");
    }

    int j;
    out_grad.resize(elem_num_);

    if (mod_errs_.empty())
    {
        if (!partial_model_)
        {
#pragma omp parallel for private (j) schedule(guided)
            for (j = 0; j < elem_num_; j++)
            {
                out_grad[j] = 2.0*(pphys_vec[j] - tar_model_[j])/(mod_err_*mod_err_*elem_num_);
            }
        }
        else
        {
            out_grad.assign_all(0.0);

#pragma omp parallel for private (j) schedule(guided)
            for (j = 0; j < tar_num_; j++)
            {
                out_grad[tar_idx_[j]] = 2.0*(pphys_vec[j] - tar_model_[j])/(mod_err_*mod_err_*tar_num_);
            }
        }

        // normalize the gradients to a reasonable range
        /*
        double mini_grad = 1e+30, maxi_grad = -1e+30;
        for (size_t i = 0; i < elem_num_; i++)
        {
            mini_grad = std::min(mini_grad, out_grad[i]);
            maxi_grad = std::max(maxi_grad, out_grad[i]);
        }

        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i] = out_grad[i]/(maxi_grad - mini_grad);
        }
        */

        normalize(out_grad, 1.0, L2);

        double msf = 0.0;
        for (size_t i = 0; i < tar_num_; i++)
        {
            msf += power2((pphys_vec[i] - tar_model_[i])/mod_err_);
        }

        return msf/tar_num_;
    }
    else
    {
        if (!partial_model_)
        {
#pragma omp parallel for private (j) schedule(guided)
            for (j = 0; j < elem_num_; j++)
            {
                out_grad[j] = 2.0*(pphys_vec[j] - tar_model_[j])/(mod_errs_[j]*mod_errs_[j]*elem_num_);
            }
        }
        else
        {
            out_grad.assign_all(0.0);

#pragma omp parallel for private (j) schedule(guided)
            for (j = 0; j < tar_num_; j++)
            {
                out_grad[tar_idx_[j]] = 2.0*(pphys_vec[j] - tar_model_[j])/(mod_errs_[j]*mod_errs_[j]*tar_num_);
            }
        }

        // normalize the gradients to a reasonable range
        /*
        double mini_grad = 1e+30, maxi_grad = -1e+30;
        for (size_t i = 0; i < elem_num_; i++)
        {
            mini_grad = std::min(mini_grad, out_grad[i]);
            maxi_grad = std::max(maxi_grad, out_grad[i]);
        }

        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i] = out_grad[i]/(maxi_grad - mini_grad);
        }
        */

        normalize(out_grad, 1.0, L2);

        double msf = 0.0;
        for (size_t i = 0; i < tar_num_; i++)
        {
            msf += power2((pphys_vec[i] - tar_model_[i])/mod_errs_[j]);
        }

        return msf/tar_num_;
    }
}