/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#ifndef _GIST_GEOPHYSICAL_MODEL_H
#define _GIST_GEOPHYSICAL_MODEL_H

#include "gctl/core.h"
#include "gctl/io.h"

#include "../utility/enum.h"
#include "../utility/data_type.h"
#include "../model_space/model_space.h"

using namespace gctl;

namespace GIST
{
    class GeophysicalModel
    {
    public:
        GeophysicalModel();
        virtual ~GeophysicalModel();

        /**
         * @brief 导入模型设置
         * 
         * @param m_space 模型空间指针
         * @param earth_1d 一维参考模型指针
         */
        void ImportModelSetup(ModelSpace *m_space, Earth1D *earth_1d);

        /**
         * @brief 地球物理模型的目标函数
         * 
         * @param pphys_vec 岩石物性模型计算得到的物性值
         * @param out_grad 返回的模型梯度
         * @return 目标函数值
         */
        double GeophysicalModelObjectiveFunc(const array<double> &pphys_vec, array<double> &out_grad);

        /**
         * @brief 初始化待拟合的地球物理模型
         * 
         * @param in_mod 输入模型
         * @param in_idx 输入模型的单元索引（如果需要拟合的模型是反演模型空间的一个子集时需要提供此索引列表）
         */
        void InitTargetModel(const array<double> &in_mod, array<size_t> *in_idx = nullptr);

        /**
         * @brief 设置待拟合的地球物理模型的不确定度
         * 
         * @param err 不确定度
         */
        void set_geophysical_model_error(double err);

        /**
         * @brief 设置待拟合的地球物理模型的不确定度
         * 
         * @param errs 不确定度
         */
        void set_geophysical_model_error(const array<double> &errs);

    protected:
        bool sys_ready_, err_ready_, tar_ready_;
        size_t elem_num_;

        double mod_err_;
        array<double> mod_errs_;

        bool partial_model_;
        array<size_t> tar_idx_;
        array<double> tar_model_;
        size_t tar_num_;

        ModelSpace *m_space_;
        Earth1D *earth_1d_;
    };
}

#endif // _GIST_GEOPHYSIC_MODEL_H