/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "geoid.h"

void GIST::Geoid::Solve(array<double> &rho, const array<double> &ref_rho)
{
    // Prepare the fitting target.
    cnst_tar_.resize(elem_num_);

    size_t i, j;
    if (ov_type_ == AbsObs && mv_type_ == RelModel)
    {
#pragma omp parallel for private (i, j) schedule(guided)
        for (i = 0; i < obsp_num_; i++)
        {
            pre_geoid_[i] = 0.0;
            for (j = 0; j < elem_num_; j++)
            {
                // calculate and modify geoid data of the background model r.w.t. the observed data
                pre_geoid_[i] += geoid_kernel_[i][j]*bkg_den_[j];
            }
        }

#pragma omp parallel for private (i, j) schedule(guided)
        for (j = 0; j < elem_num_; j++)
        {
            cnst_tar_[j] = 0.0;
            for (i = 0; i < obsp_num_; i++)
            {
                cnst_tar_[j] += geoid_kernel_[i][j]*(obs_geoid_[i] - pre_geoid_[i])/(geoid_err_*geoid_err_*obsp_num_);
            }
        }

        // Reference model constraint part
#pragma omp parallel for private (j) schedule(guided)
        for (j = 0; j < elem_num_; j++)
        {
            cnst_tar_[j] += reg_strength_*model_wgts_[j]*(ref_rho[j] - bkg_den_[j])/(cnst_err_*cnst_err_*elem_num_);
        }
    }
    else if (ov_type_ == RelObs && mv_type_ == AbsModel)
    {
#pragma omp parallel for private (i, j) schedule(guided)
        for (i = 0; i < obsp_num_; i++)
        {
            pre_geoid_[i] = 0.0;
            for (j = 0; j < elem_num_; j++)
            {
                // calculate and modify geoid data of the background model r.w.t. the observed data
                pre_geoid_[i] += geoid_kernel_[i][j]*bkg_den_[j];
            }
        }

#pragma omp parallel for private (i, j) schedule(guided)
        for (j = 0; j < elem_num_; j++)
        {
            cnst_tar_[j] = 0.0;
            for (i = 0; i < obsp_num_; i++)
            {
                cnst_tar_[j] += geoid_kernel_[i][j]*(obs_geoid_[i] + pre_geoid_[i])/(geoid_err_*geoid_err_*obsp_num_);
            }
        }

        // Reference model constraint part
#pragma omp parallel for private (j) schedule(guided)
        for (j = 0; j < elem_num_; j++)
        {
            cnst_tar_[j] += reg_strength_*model_wgts_[j]*(ref_rho[j] + bkg_den_[j])/(cnst_err_*cnst_err_*elem_num_);
        }
    }
    else
    {
#pragma omp parallel for private (i, j) schedule(guided)
        for (j = 0; j < elem_num_; j++)
        {
            cnst_tar_[j] = 0.0;
            for (i = 0; i < obsp_num_; i++)
            {
                cnst_tar_[j] += geoid_kernel_[i][j]*obs_geoid_[i]/(geoid_err_*geoid_err_*obsp_num_);
            }
        }

        // Reference model constraint part
#pragma omp parallel for private (j) schedule(guided)
        for (j = 0; j < elem_num_; j++)
        {
            cnst_tar_[j] += reg_strength_*model_wgts_[j]*ref_rho[j]/(cnst_err_*cnst_err_*elem_num_);
        }
    }

    LCG_Minimize(rho, cnst_tar_, gctl::LCG_PCG);
    return;
}