/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "geoid.h"

void GIST::Geoid::InitObs(std::string obspara_str)
{
    std::string filename = gctl::patch_string(obspara_str, ".txt");
    
    if (!access(filename.c_str(), 4)) // File exists
    {
        std::clog << "Reading geoid observations from file...\n";

        _1s_vector text_lines2;
        text_descriptor t_desc(obspara_str);
        read_text_lines(t_desc, text_lines2);

        if (text_lines2.empty())
        {
            throw std::runtime_error("The input geoid observations are empty. From GIST::Geoid::InitObs(...)");
        }

        str2type(text_lines2[0], obsp_num_);

        if (obsp_num_ + 1 != text_lines2.size())
        {
            throw std::runtime_error("The input geoid observations are of the wrong size. From GIST::Geoid::InitObs(...)");
        }

        geoid_obsp_.resize(obsp_num_);
        pre_geoid_.resize(obsp_num_, 0.0);

        double ele;
        if (ov_type_ == AbsObs || ov_type_ == RelObs)
        {
            obs_geoid_.resize(obsp_num_);

            for (size_t i = 1; i < text_lines2.size(); i++)
            {
                parse_string_to_value(text_lines2[i], ' ', true, geoid_obsp_[i-1].lon, geoid_obsp_[i-1].lat, ele, obs_geoid_[i-1]);
                geoid_obsp_[i-1].rad = earth_1d_->get_radius(geoid_obsp_[i-1].lat) + ele;
            }
        }
        else
        {
            for (size_t i = 1; i < text_lines2.size(); i++)
            {
                parse_string_to_value(text_lines2[i], ' ', true, geoid_obsp_[i-1].lon, geoid_obsp_[i-1].lat, ele);
                geoid_obsp_[i-1].rad = earth_1d_->get_radius(geoid_obsp_[i-1].lat) + ele;
            }
        }
    }
    else // parse the input string to parameters. Only works for forwarding modeling
    {
        std::clog << "Initiating geoid observations from parameters...\n";

        double lon_st, lon_ed, lat_st, lat_ed, dlon, dlat, ele;
        gctl::parse_string_to_value(obspara_str, '/', true, lon_st, lon_ed, lat_st, lat_ed, dlon, dlat, ele);

        double lonmin, lonmax, latmin, latmax, dlon_f, dlat_f;
        lonmin = std::min(lon_st, lon_ed);
        lonmax = std::max(lon_st, lon_ed);
        latmin = std::min(lat_st, lat_ed);
        latmax = std::max(lat_st, lat_ed);
        dlon_f = fabs(dlon);
        dlat_f = fabs(dlat);

        if (lonmin+dlon_f > lonmax || latmin+dlat_f > latmax)
        {
            throw gctl::invalid_argument("Invalid observation parameters. From GIST::Geoid::InitObs(...)");
        }

        int lonnum = round((lonmax - lonmin)/dlon_f) + 1;
        int latnum = round((latmax - latmin)/dlat_f) + 1;

        obsp_num_ = lonnum*latnum;
        geoid_obsp_.resize(obsp_num_);
        pre_geoid_.resize(obsp_num_, 0.0);

        for (int j = 0; j < latnum; ++j)
        {
            for (int i = 0; i < lonnum; i++)
            {
                geoid_obsp_[i+j*lonnum].lon = lon_st + i*dlon;
                geoid_obsp_[i+j*lonnum].lat = lat_st + j*dlat;
                geoid_obsp_[i+j*lonnum].rad = earth_1d_->get_radius(lat_st + j*dlat) + ele;
            }
        }
    }

    obs_ready_ = true;
    return;
}