/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "earth_1d.h"

void GIST::Earth1D::InitEarth1DModel(std::string filename, std::string para_str)
{
    std::string type_str;
    double r, R, ref_g, ref_lat;
    if (3 == gctl::parse_string_to_value(para_str, '/', false, r, R, ref_g))
    {
        ref_r_ = r; ref_R_ = R; normal_grav_ = ref_g;
        rf_type_ = UserDefined;
    }
    else
    {
        int n = gctl::parse_string_to_value(para_str, '/', false, type_str, ref_lat);
        if (n == 1 && type_str == "EarthMean")
        {
            ref_r_ = ref_R_ = 6371008.8;
            rf_type_ = EarthMean;
        }
        else if (n == 2 && type_str == "WGS84")
        {
            ref_r_ = 6356752.314245179; ref_R_ = 6378137.0;
            rf_type_ = WGS84;

            if (ref_lat < -90.0 || ref_lat > 90.0)
            {
                throw std::runtime_error("Invalid reference latitude value.");
            }
        }
        else std::runtime_error("Invalid reference frame type.");
    }

    std::vector<std::string> str_vec;
    text_descriptor t_desc(filename);
    read_text_lines(t_desc, str_vec);

    name_ = str_vec[0];
    str2type(str_vec[1], layer_num_);
    if (layer_num_ < 2)
    {
        throw std::runtime_error("Insufficient layer number of the 1D reference Earth model.");
    }

    depth_.resize(layer_num_);
    den_.resize(layer_num_);
    vp_.resize(layer_num_);
    vs_.resize(layer_num_);
    t_cndt_.resize(layer_num_);
    press_.resize(layer_num_);
    tep_.resize(layer_num_);
    mg_.resize(layer_num_);

    size_t nc = 0;
    for (size_t i = 2; i < str_vec.size(); i++)
    {
        parse_string_to_value(str_vec[i], ' ', true, depth_[nc], den_[nc], vp_[nc], vs_[nc], t_cndt_[nc], tep_[nc], mg_[nc]);
        nc++;
    }
    
    if (nc != layer_num_)
    {
        throw std::runtime_error("Incorrect layers read for the 1D reference Earth model.");
    }

    // calculate pressure
    double cur_press = 0;
    if (rf_type_ == EarthMean)
    {
        for (size_t i = 0; i < layer_num_ - 1; i++)
        {
            press_[i] = cur_press;
            cur_press += 1e-6*0.5*(den_[i] + den_[i+1])*GIST_NORMAL_GRAVITY*(depth_[i+1] - depth_[i]);
        }
        press_[layer_num_ - 1] = cur_press;
    }
    else if (rf_type_ == WGS84)
    {
        for (size_t i = 0; i < layer_num_ - 1; i++)
        {
            press_[i] = cur_press;
            cur_press += 1e-6*0.5*(den_[i] + den_[i+1])*(depth_[i+1] - depth_[i])*EllipsoidalGravityWGS84(ref_lat);
        }
        press_[layer_num_ - 1] = cur_press;
    }
    else
    {
        for (size_t i = 0; i < layer_num_ - 1; i++)
        {
            press_[i] = cur_press;
            cur_press += 1e-6*0.5*(den_[i] + den_[i+1])*(depth_[i+1] - depth_[i])*ref_g;
        }
        press_[layer_num_ - 1] = cur_press;
    }

    initialized_ = true;
    return;
}