/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#ifndef _GIST_EARTH1D_H
#define _GIST_EARTH1D_H

#include "gctl/core.h"
#include "gctl/algorithm.h"
#include "gctl/io.h"

#include "../utility/enum.h"

#define GIST_NORMAL_GRAVITY 9.80665 // m/s^2

using namespace gctl;

namespace GIST
{
    enum ref_frame_type
    {
        EarthMean,
        WGS84,
        UserDefined,
    };

    class Earth1D
    {
    public:
        Earth1D();
        virtual ~Earth1D();

        /**
         * @brief Initialize the 1D reference Earth model from file.
         * 
         * @param filename Model file.
         * @param para_str Parameter string. {<ref_r>/<ref_R>}|WGS84[/<lat>]|EarthMean
         */
        void InitEarth1DModel(std::string filename, std::string para_str = "EarthMean");
        /**
         * @brief Show statistical information.
         * 
         * @param ss Output stream.
         */
        void ShowStatistics(std::ostream &ss = std::clog);
        /**
         * @brief Show current reference Earth profile.
         * 
         */
        void ShowWholeProfile(std::ostream &ss = std::clog);
        /**
         * @brief Calculate the Earth's normal gravity value on the WGS84 ellipsoid.
         * 
         * @param lat Latitude value
         * @return Gravity value on the ellipsoid in the unit m/s^2.
         */
        double EllipsoidalGravityWGS84(double lat) const;
        /**
         * @brief Get the reference model value at a given depth.
         * 
         * @param cen_rad Radius of the inquired element cell
         * @param lati latitude value
         * @param md_type Model type. Could be either Density, Vs or Vp
         * @return double Reference model value
         */
        double get_data(double cen_rad, double lati, model_type_e md_type) const;
        /**
         * @brief Get the minor radius value
         * 
         * @return radius
         */
        double get_minor_radius() const;
        /**
         * @brief Get the major radius value
         * 
         * @return radius
         */
        double get_major_radius() const;
        /**
         * @brief Get the radius of a given latitude value.
         * 
         * @param lati latitude value
         * @return radius
         */
        double get_radius(double lati) const;
        /**
         * @brief Get the normal gravity value on the reference ellipsoid at a given latitude.
         * 
         * @param lati latitude value
         * @return normal gravity value
         */
        double get_normal_gravity(double lati) const;

    protected:
        std::string name_;
        double normal_grav_;
        array<double> depth_, den_, vs_, vp_, press_, t_cndt_, tep_, mg_;
        size_t layer_num_; // layer number of the 1D reference model.
        ref_frame_type rf_type_;
        double ref_r_, ref_R_;
        bool initialized_;
    };
}

#endif // _GIST_EARTH1D_H