/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "earth_1d.h"

GIST::Earth1D::Earth1D()
{
    name_ = "UnKnown";
    normal_grav_ = GIST_NORMAL_GRAVITY;
    layer_num_ = 0;
    rf_type_ = UserDefined;
    ref_r_ = ref_R_ = 0.0;
    initialized_ = false;
}

GIST::Earth1D::~Earth1D(){}

double GIST::Earth1D::get_data(double cen_rad, double lati, model_type_e md_type) const
{
    if (!initialized_)
    {
        throw runtime_error("The reference 1D Earth model is not initialized.");
    }

    double dep = ellipse_radius_2d(ref_R_, ref_r_, lati*GCTL_Pi/180.0) - cen_rad;
    
    if (dep < depth_[0] || dep > depth_.back())
    {
        throw runtime_error("Invalid inquire depth of the 1D Earth model.");
    }

    for (size_t i = 0; i < layer_num_ - 1; i++)
    {
        if (dep >= depth_[i] && dep <= depth_[i+1])
        {
            if (md_type == Density) return line_interpolate(depth_[i], depth_[i+1], den_[i], den_[i+1], dep);
            if (md_type == Vp) return line_interpolate(depth_[i], depth_[i+1], vp_[i], vp_[i+1], dep);
            if (md_type == Vs) return line_interpolate(depth_[i], depth_[i+1], vs_[i], vs_[i+1], dep);
            if (md_type == Thermal_Conductivity) return line_interpolate(depth_[i], depth_[i+1], t_cndt_[i], t_cndt_[i+1], dep);
            if (md_type == Lithostatic_Pressure) return line_interpolate(depth_[i], depth_[i+1], press_[i], press_[i+1], dep);
            if (md_type == Temperature) return line_interpolate(depth_[i], depth_[i+1], tep_[i], tep_[i+1], dep);
            if (md_type == Magnesium) return line_interpolate(depth_[i], depth_[i+1], mg_[i], mg_[i+1], dep);
        }
    }
    
    throw runtime_error("Failed inquire to the 1D Earth model..");
}

double GIST::Earth1D::get_minor_radius() const
{
    if (!initialized_)
    {
        throw runtime_error("The reference 1D Earth model is not initialized.");
    }

    return ref_r_;
}

double GIST::Earth1D::get_major_radius() const
{
    if (!initialized_)
    {
        throw runtime_error("The reference 1D Earth model is not initialized.");
    }

    return ref_R_;
}

double GIST::Earth1D::get_radius(double lati) const
{
    if (!initialized_)
    {
        throw runtime_error("The reference 1D Earth model is not initialized.");
    }

    return ellipse_radius_2d(ref_R_, ref_r_, lati*GCTL_Pi/180.0);
}

double GIST::Earth1D::get_normal_gravity(double lati) const
{
    if (rf_type_ == EarthMean) return GIST_NORMAL_GRAVITY;
    if (rf_type_ == WGS84) return EllipsoidalGravityWGS84(lati);
    return normal_grav_;
}