/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#ifndef _GIST_ASSEMBLER_H
#define _GIST_ASSEMBLER_H

#include "../utility/enum.h"
#include "../utility/options.h"
#include "../utility/data_type.h"
#include "../earth_1d/earth_1d.h"
#include "../model_space/model_space.h"

#include "gctl/core.h"
#include "gctl/utility.h"
#include "gctl/optimization.h"

using namespace gctl;

namespace GIST
{
    /**
     * @brief 流程组装器 纯虚类 需定义接口函数 通过run函数运行
     * 
     */
    class Assembler
    {
    public:
        Assembler(){}
        virtual ~Assembler(){}
        
        /**
         * @brief 函数接口API 添加子程序所需的命令
         */
        virtual void AddOptions() = 0;

        /**
         * @brief 函数接口API 定义流程
         */
        virtual void Routine() = 0;

        /**
         * @brief 运行定义的流程
         * 
         * @param argc 运行参数个数
         * @param argv 运行参数列表
         */
        void Run(int argc, char *argv[]);

        /**
         * @brief 显示gist程序logo 已在Run函数中调用 无需再次调用
         */
        void Display_Logo();

        /**
         * @brief 显示帮助信息 已在Run函数中调用 无需再次调用
         */
        void Display_Help();

        /**
         * @brief 初始化默认运行参数 已在Run函数中调用 无需再次调用
         * 
         * @param gist_dfts gist选项文件名称
         */
        void InitDefaultSetups(std::string gist_dfts);

        /**
         * @brief 初始化lcg算法参数 需要时在AddOptions函数中调用
         * 
         * @param in_para lcg算法参数对象的引用
         */
        void InitLCGSetups(lcg_para &in_para);

        /**
         * @brief 初始化lbfgs算法参数 需要时在AddOptions函数中调用
         * 
         * @param in_para lbfgs算法参数对象的引用
         */
        void InitLBFGSSetups(lbfgs_para &in_para);

        /**
         * @brief 初始化lgd算法参数 需要时在AddOptions函数中调用
         * 
         * @param in_para lgd算法参数对象的引用
         */
        void InitLGDSetups(lgd_para &in_para);

        /**
         * @brief 添加InitModelData函数所使用的成组的选项。
         * 
         * @note 添加的命令名称为<opt_str>-value&&-tag&&-model。其中-value后缀会直接对模型赋统一值；
         * -tag后缀使用元素的标签进行赋值，格式为<tag>/<value>;-model后缀从Gmsh模型文件中读取模型数据。
         * 
         * @param opt_str 命令名称
         * @param gp_tag 组号
         */
        void AddGroupOption(std::string opt_str, int gp_tag);

        /**
         * @brief 使用ModelSpace对象初始化模型数据。
         * 
         * @note 匹配的命令名称为<opt_str>[-value|-tag|-model]。其中-value后缀会直接对模型赋统一值；
         * -tag后缀使用元素的标签进行赋值，格式为<tag>/<value>;-model后缀从Gmsh模型文件中读取模型数据。
         * 
         * @param opt_str 命令名称
         * @param data 返回的模型数据
         * @param data_idx 返回的模型数据的单元索引（注意目前只针对三维单元体数据）
         * @return true 模型数据构造成功
         * @return false 模型数据构造失败
         */
        bool InitModelData(std::string opt_str, array<double> &data, array<size_t> *data_idx = nullptr);

        /**
         * @brief 解析由格式化字符串构成的标签值对。
         * 
         * @note 字符串格式为 <tag1>/<val1>,<tag2>/<val2>,...
         * 
         * @param opt_str 格式化字符串
         * @param tag_vec 返回的标签向量
         * @param val_vec 返回的值向量
         * @param str_separator 标签值对之间的分隔符，默认为逗号
         * @param tag_separator 标签与值之间的分隔符，默认为斜杠
         */
        void ParseTagPairs(std::string opt_str, std::vector<std::string> &tag_vec, 
            std::vector<std::string> &val_vec, char str_separator = ',', char tag_separator = '/');

    protected:
        getoption gopt_; // 选项对象 通过此对象获取流程所需的命令选项 在AddOptions函数中调用

        std::string lcg_dfts_; // LCG算法参数文件名 自动获取无需操作
        std::string lbfgs_dfts_;  // LBFGS算法参数文件名 自动获取无需操作
        std::string lgd_dfts_;  // LGD算法参数文件名 自动获取无需操作

        Earth1D earth1d_obj_; // 内置1D参考模型对象 所有流程都需要这个对象 直接声名在Assembler中
        ModelSpace m_space_; // 内置模型空间对象 所有流程都需要这个对象 直接声名在Assembler中
        index_packed_e node_packed_; // 模型文件内顶点序号是否从0或1开始
    };
}

#endif // _GIST_ASSEMBLER_H