module vitalrates
        use forces
        use params

        implicit none

        contains
                function instprob(p,dt)
                        !
                        ! The probability of happening in one time step.
                        ! The base probability p is the probability of the event within a year.
                        ! dt is the length of the timestep, thus the instantaneous probability is
                        ! p1=1-(1-p)^(dt).
                        !

                        real(kind=rkd) :: p,dt
                        real(kind=rkd) :: instprob

                        !
                        ! Calculate the instantaneous probability.
                        !
                        p=min(1.0D+00,p)
                        instprob = 1.0D+00-(1.0D+00-p)**(dt)
                        !print*,"instprob",p,dt,instprob
                end function instprob

                function probdeath(n,a,rR,rA,dt)
                        !
                        ! The base probability of death depends on the age of the particle (a).
                        ! The probability of death during the year is modified by the distance
                        ! between the individual particle, the nearest resource particle, and the nearest
                        ! alcohol particle. Increasing distance between the individual and the resource
                        ! increases the probability, while decreasing distance between the individual and
                        ! the alcohol increases the probability.
                        ! The real distances are scaled to functional distances by the scale factor s.
                        !
                        ! Currently no particle specific modifiers are in use. The probability of death
                        ! depends solely on the relative distances between the particle and the resources.
                        !

                        integer :: n
                        real(kind=rkd) :: a,rR,rA,dt
                        real(kind=rkd) :: p,p1,sR,sA
                        real(kind=rkd) :: probdeath

                        !
                        ! Calculate the base probability (per year).
                        ! Here we use array vitg(type=[nx,ny,fx,fy],age,stat=[base prob, scale regular, scale alcohol]).
                        !
                        p = vitg(n,int(a),1) ! The base probability of death depends on the age of the particle.
                        if (p<1.0D+00) then
                                sR = vitg(n,int(a),2) ! The scale factor sR modifies the probability concerning
                                                      ! the individual particle and the general resource particle.
                                sA = vitg(n,int(a),3) ! The scale factor sA modifies the probability concerning
                                                      ! the individual particle and the alcohol resource particle.
                                p1 = p*(1.0D+00-sR*exp(-rR)+(sA*exp(-rA)))
                                !
                                ! Calculate the instantaneous (per dt) probability.
                                !
                                probdeath = instprob(p1,dt)
                                !print *,"rR",rR,"rA",rA,"p1",p1,"probdeath:",probdeath
                                !print *,"probdeath",p,p1,probdeath
                        else
                                probdeath = 1.1D+00 ! Certain death from old age
                        end if
                end function probdeath

                function probbirth(n,a,a0,rN,rF,rR,rA,dt)
                        !
                        ! The base probability of death depends on the age of the particle (a).
                        ! The probability of death during the year is modified by the distance
                        ! between the individual particle, the nearest resource particle, and the nearest
                        ! alcohol particle. Increasing distance between the individual and the resource
                        ! increases the probability, while decreasing distance between the individual and
                        ! the alcohol increases the probability.
                        ! The real distances are scaled to functional distances by the scale factor s.
                        !
                        ! Currently no particle specific modifiers are in use. The probability of death
                        ! depends solely on the relative distances between the particle and the resources.
                        !

                        integer :: n
                        real(kind=rkd) :: a,a0,rN,rF,rR,rA,dt
                        real(kind=rkd) :: p,p1,p2,sR,sA,pm4,pm6
                        real(kind=rkd) :: probbirth(2)

                        !
                        ! Calculate the base probability (per year).
                        ! Here we use array vitg(type=[nx,ny,fx,fy],age,stat=[base prob, scale regular, scale alcohol]).
                        !
                        if (n==3) then
                                pm4=ljpars(3,4,2)
                                pm6=ljpars(3,6,2)
                        else
                                pm4=ljpars(4,5,2)
                                pm6=ljpars(5,6,2)
                        end if
                        !
                        ! Reproduction can happen only once in a year.
                        !
                        if (a-a0<1.0D+00) then
                                p = 0.0D+00
                        else
                                p = (1.0D+00-min(rN/pm4,rF/pm6))*vitg(n,int(a),4)
                        end if

                        if (p>0.0D+00) then
                                !sR = vitg(n,int(a),5) ! The scale factor sR modifies the probability concerning
                                !                      ! Res and Ny.
                                !sA = vitg(n,int(a),6) ! The scale factor sA modifies the probability concerning
                                !                      ! Alc and Fy.
                                !p1 = p*(sR*exp(-rN)+sA*exp(-rF))
                                p2 = 2*exp(-rA) !Hardcoded
                                !!
                                !! Calculate the instantaneous (per dt) probability.
                                !!
                                !!print *,"probbirth(1)",p1,dt
                                !probbirth(1) = instprob(p1,dt)
                                !print *,"probbirth(2)",p2,dt
                                !probbirth(2) = instprob(p2,dt)
                                probbirth(1)=p
                                probbirth(2)=p2
                        else
                                probbirth(1) = 0.0D+00 ! Not possible
                                probbirth(2) = 0.0D+00 ! Not possible
                        end if
                end function probbirth

                function minimum_energy_position(i,j,pop,pn,res,rn,alc,an)
                        !
                        ! Find the minimum energy/force (x,y) position for a particle.
                        ! The position requires:
                        ! 1) L-J equilibrium distance between particles i and j.
                        ! 2) Distance to the nearest resource particles z no less than 0.9*phi_{L-J}(j,z).
                        ! If such a position can not be found, return random position with a notification.
                        !
                        ! NOTE: Because this function is rarely needed, the additional computational costs
                        !       are (probably) not very important at the moment. However, with very large populations
                        !       this might need thinking.
                        !
                        integer,intent(in) :: i,j
                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: x0,y0,x1,y1,r0(2),r10(2),r1(2),r2(2),r12,eps,phi,phi2,cut,f(5),fmin(5),posmin(2),pi8
                        real(kind=rkd) :: minimum_energy_position(3)
                        integer :: k,z,q

                        x0=pop(i,2)
                        y0=pop(i,3)
                        r0=pop(i,2:3)
                        r10=pop(j,2:3)
                        eps=ljpars(int(pop(i,1)),int(pop(j,1)),1)
                        phi=ljpars(int(pop(i,1)),int(pop(j,1)),2)
                        cut=ljpars(int(pop(i,1)),int(pop(j,1)),2)
                        pi8=4.D00*ATAN(1.D+00)/8.0D+00
                        posmin=r10
                        !
                        ! Take into account mother-child bonding.
                        ! This age dependent function.
                        !
                        if (int(pop(i,21))==-int(pop(j,10)) .OR. int(pop(j,21))==-int(pop(i,10))) then
                                phi=max(0.1D+00,phi*min(pop(i,15),pop(j,15))/18.0D+0)
                        end if


                        !
                        ! Initialize the closest resource distance to acceptable.
                        !
                        r12=ljpars(1,int(pop(j,1)),2)

                        q=0
                        do z=1,pn
                        !
                        ! We care only about the resources.
                        !
                        if (int(pop(z,1))<1 .or. int(pop(z,1))>=3) cycle
                        !
                        ! We care only about the resources that are close enough
                        ! to cause a catastropy, i.e., the absolute distance between the
                        ! resource (z) and the child (j) is shorter than 0.9*phi_{L-J}(j,z).
                        !
                        r2(1:2)=pop(z,2:3)
                        phi2=ljpars(int(pop(int(pop(z,1)),1)),int(pop(j,1)),1)
                        if (sqrt((r10(1)-r2(1))**2+(r10(2)-r2(2))**2)>(0.9D+00*phi2)) cycle
                        !
                        ! Count the close distance resources.
                        !
                        q=q+1
                        !
                        ! If the resource is close, find the position with minimum potential (L-J).
                        !
                        do k=1,8
                        x1=x0+phi*sin(pi8*real(k,kind=rkd))
                        y1=y0+phi*cos(pi8*real(k,kind=rkd))

                        r1(1)=x1
                        r1(2)=y1

                        f=flennardjones(r2,r1,eps,phi,cut)
                        if (q==0) then 
                                fmin=f
                                posmin=r1
                                r12=sqrt((posmin(1)-r2(1))**2+(posmin(2)-r2(2))**2)
                        else
                                if (sqrt(f(1)**2+f(2)**2) < sqrt(fmin(1)**2+fmin(2)**2)) then
                                        fmin=f
                                        posmin=r1
                                        r12=sqrt((posmin(1)-r2(1))**2+(posmin(2)-r2(2))**2)
                                end if
                        end if
                        end do
                        end do

                        minimum_energy_position(1:2)=posmin
                        if (r12>0.9*ljpars(1,int(pop(j,1)),2)) then
                                minimum_energy_position(3)=1.0D+00
                        else
                                minimum_energy_position(3)=-1.0D+00
                        end if
                end function minimum_energy_position

                subroutine popdeath(pop,pn,res,rn,alc,an,nt,dt)
                        !
                        ! Here we cycle through each particle in the simulation to test for the stochastic death occurrence.
                        ! Each particle has a base probability of death according to its age. The base probability is modified
                        ! by the distance to the nearest general and alcohol resource particle.
                        ! The resource particles do not suffer from mortality.
                        !

                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: nt,dt !The time step
                        real(kind=rkd) :: rminR,rminA,p,rnd1
                        integer :: i,n1

                        !
                        ! Loop through all the particles.
                        !
                        do i=1,pn-1
                        
                        !
                        ! If we have a blank space or a resource particle, then skip the process.
                        !
                        n1=int(pop(i,1))
                        if (n1<3) CYCLE
                       
                        !
                        ! Retrieve the sum of the minimum distances and calculate the average.
                        !
                        rminR = pop(i,16)/nt
                        rminA = pop(i,17)/nt

                        if (rminR<0.0D+00 .or. rminA<0.0D+00) CYCLE

                        !
                        ! Compute the probability of death.
                        !
                        p = probdeath(n1,pop(i,15),rminR,rminA,dt*nt)

                        !
                        ! If the individual dies, remove it from the array.
                        ! If the individual survives, increase the age.
                        !
                        call random_number(rnd1)
                        if (rnd1<=p) then
                                !print *, "popdeath",p, rnd1
                                !print *,"dead:",i,pop(i,15),pop(i,1)
                                pop(i,1) = 0.0D+00
                        else
                                pop(i,15) = pop(i,15)+dt
                        end if
                        end do
                end subroutine popdeath

                subroutine popbirth(pop,pn,res,rn,alc,an,nt,dt)
                        !
                        ! Here we cycle through each particle in the simulation to test for the stochastic birth occurrence.
                        ! Each particle has a base probability of reproducing according to its age and type.
                        ! The process concerns only the particles nx (3) and fx (5), though
                        ! The base probability is modified by the distance to the nearest ny (4) and fy (4) particles.
                        ! In reality, also the distance to the Res and Alc particles matter, but the parametrization needs
                        ! more background. It is on the TODO list.
                        ! The type of the new particle depends on the distance to the nearest Res and Alc particles.
                        !

                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: nt,dt !The time step
                        real(kind=rkd) :: rminR,rminA,rminN,rminF
                        real(kind=rkd) :: r1(2),p(2),rnd1,rnd2,mep(3)
                        integer :: i,j,k,n1,n2,count1,count2

                                        !count2=0
                                        !do count1=1,pn
                                        !if (pop(count1,1)>2) count2=count2+1
                                        !end do
                                        !print *,"Initially:::","disappear:",k,mep,"individuals:",count2
                        !
                        ! Loop through all particles.
                        !
                        do i=1,pn-1
                        
                        !
                        ! Reproduction process concerns only Nx and Fx.
                        !
                        n1=int(pop(i,1))
                        if (n1/=3 .and. n1/=5) CYCLE
                       
                        !
                        ! Retrieve the sum of the minimum distances and calculate the average.
                        !
                        rminR = pop(i,16)/nt
                        rminA = pop(i,17)/nt
                        rminN = pop(i,18)/nt
                        rminF = pop(i,19)/nt

                        if (rminR<0.0D+00 .or. rminA<0.0D+00 .or. rminN<0.0D+00 .or. rminF<0.0D+00) CYCLE

                        !
                        ! Compute the probability of reproduction and the type of the offspring.
                        !
                        p = probbirth(n1,pop(i,15),pop(i,20),rminN,rminF,rminR,rminA,dt*nt)

                        !
                        ! If the particle reproduces, add a new particle to the first free slot at
                        ! a close to equilibrium distance. The position is not guaranteed not to cause
                        ! a collision, however.
                        !
                        call random_number(rnd1)
                        if (rnd1<=p(1)) then
                                do k=1,pn
                                if (pop(k,1)<1.0) then
                                        EXIT
                                end if
                                end do
                                call random_number(rnd1)
                                call random_number(rnd2)
                                
                                if (rnd1>p(2) .and. rnd2<0.48) then
                                        pop(k,1) = 3.0D+00
                                else if (rnd1>p(2)) then
                                        pop(k,1) = 4.0D+00
                                else if (rnd1<=p(2) .and. rnd2<0.48) then
                                        pop(k,1) = 5.0D+00
                                else if (rnd1<=p(2)) then
                                        pop(k,1) = 6.0D+00
                                else
                                        pop(k,1) = 4.0D+00
                                end if

                                !
                                ! Add the new individual.
                                !
                                r1(1) = pop(i,2)+0.1D+00 !ljpars(int(pop(i,1)),int(pop(k,1)),2)
                                r1(2) = pop(i,3)
                                pop(k,2:3)=r1
                                pop(k,4:20)=0.0D+00
                                !
                                ! If i==Nx, find a suitable position for the child.
                                !
                                mep=minimum_energy_position(i,k,pop,pn,res,rn,alc,an)
                                if ((n1==5 .or. mep(3)>0.0D+00)) then
                                        !
                                        ! If the mother is not from FASD population,
                                        ! use the minimum energy position.
                                        !
                                        if (n1/=5) pop(k,2:3)=mep(1:2)
                                        !
                                        ! Record the age of giving birth.
                                        !
                                        pop(i,20)=pop(i,15)
                                        ! Bond the new individual to the parent.
                                        ! NOTE! Bonding mechanics work as follows:
                                        ! 1) A child bonds to the mother's ID.
                                        ! 2) A male bonds to the female's ID.
                                        ! 3) The bond between the child and the mother breaks at the age of 18.
                                        ! 4) The bond between the any two individuals breaks if the distance is
                                        !    longer than the harmonic potential (Hooke's law) cut off range.

                                        ! If the mother does not have ID, assign one.
                                        if (pop(i,10)<1.0) then
                                                idcount=maxval(pop(:,10))+1.0D+00
                                                pop(i,10)=idcount
                                        end if
                                        ! Assign ID to the child.
                                        idcount=maxval(pop(:,10))+1.0D+00
                                        pop(k,10)=idcount
                                        ! Bond the child to the mother.
                                        ! NOTE: Negative number signifies child-mother bond.
                                        pop(k,21)=-pop(i,10)
                                else
                                        !count2=0
                                        !do count1=1,pn
                                        !if (pop(count1,1)>2) count2=count2+1
                                        !end do
                                        !print *,"disappear:",k,mep,"individuals:",count2
                                        pop(k,1:21)=0.0D+00
                                end if
                        end if
                        end do
                                        !count2=0
                                        !do count1=1,pn
                                        !if (pop(count1,1)>2) count2=count2+1
                                        !end do
                                        !print *,"Finally:::","disappear:",k,mep,"individuals:",count2
                end subroutine popbirth

                subroutine distsum(pop,pn)
                        !
                        ! Here we cycle through each particle in the simulation to test for the stochastic birth occurrence.
                        ! Each particle has a base probability of reproducing according to its age and type.
                        ! The process concerns only the particles nx (3) and fx (5), though
                        ! The base probability is modified by the distance to the nearest ny (4) and fy (4) particles.
                        ! In reality, also the distance to the Res and Alc particles matter, but the parametrization needs
                        ! more background. It is on the TODO list.
                        ! The type of the new particle depends on the distance to the nearest Res and Alc particles.
                        !

                        integer,intent(in) :: pn ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21) ! The particle arrays
                        real(kind=rkd) :: r1(2),r2(2),rminR,rminA,rminN,rminF,rcur,f(5)
                        real(kind=rkd) :: p(2),rnd1,rnd2,phi
                        integer :: i,j,k,n1,n2,tmp(3)

                        !
                        ! Find the distance to the nearest resource and alcohol particle.
                        !
                        do i=1,pn-1
                        
                        !
                        ! If we have a blank space or a resource particle, then skip the process.
                        !
                        n1=int(pop(i,1))
                        if (n1<3) CYCLE
                       
                        !
                        ! Initialize the minimum distances for the current particle i.
                        rminR=-1.0D+00
                        rminA=-1.0D+00
                        rminN=-1.0D+00
                        rminF=-1.0D+00
                        
                        !
                        ! Loop through the particle array to find the minimum distances.
                        !
                        do j=i+1,pn
                        !
                        ! If we have a child parent pair, then skip the process.
                        !
                        if (int(pop(i,10))==int(abs(pop(j,21))) .and. pop(j,21)<0.0D+00) CYCLE
                        if (int(pop(j,10))==int(abs(pop(i,21))) .and. pop(i,21)<0.0D+00) CYCLE
                        
                        !
                        ! Interesting pairs are
                        ! ([Ny,Fy],[Res,Alc]),([Nx,Fx],[Ny,Fy,Res,Alc]).
                        ! Additionally, for reproduction, the particles need to fit
                        ! within the age window 15<y<51 and 15<x<60.
                        !
                        n2=int(pop(j,1))
                        if (n1==4 .or. n1==6) then
                                if (n2<1 .or. n2>2) CYCLE
                        else if (n1==3 .or. n1==5) then
                                if (n2==3 .or. n2==5) CYCLE
                                if (n2==4 .or. n2==6) then
                                        if (pop(i,15)<16.0 .or. pop(i,15)>50.0) then
                                               CYCLE
                                       else if (pop(j,15)<15.0 .or. pop(j,15)>59.0) then
                                               CYCLE
                                       end if
                               end if
                        end if

                        !
                        ! Calculate the distance between the particle pair
                        ! and memorize the minimum.
                        !
                        r1 = pop(i,2:3)
                        r2 = pop(j,2:3)
                        phi=ljpars(n1,n2,2)
                        rcur = sqrt((r1(1)-r2(1))**2+(r1(2)-r2(2))**2)/phi
                        if (pop(j,1)==1) then
                                if (rminR<0.0D00 .or. rcur<rminR) rminR=rcur
                        else if (pop(j,1)==2) then
                                if (rminA<0.0D00 .or. rcur<rminA) rminA=rcur
                        else if (pop(j,1)==4) then
                                if (rminN<0.0D00 .or. rcur<rminN) then
                                        rminN=rcur
                                        ! If j is not bonded and j is the closest individual, then bond.
                                        if (int(pop(j,21))==0) pop(j,21)=pop(i,10)
                                end if
                        else if (pop(j,1)==6) then
                                if (rminF<0.0D00 .or. rcur<rminF) then
                                        rminF=rcur
                                        ! If j is not bonded and j is the closest individual, then bond.
                                        if (int(pop(j,21))==0) pop(j,21)=pop(i,10)
                                end if
                        end if
                        
                        end do
                        !
                        ! Add the minimum distance to the running sum.
                        ! Remove the null results, i.e., r<-1.
                        !
                        if (rminR>0.0D+00) then
                                pop(i,16) = pop(i,16)+rminR
                        end if
                        if (rminA>0.0D+00) then
                                pop(i,17) = pop(i,17)+rminA
                        end if
                        if (rminN>0.0D+00) then
                                pop(i,18) = pop(i,18)+rminN
                        end if
                        if (rminF>0.0D+00) then
                                pop(i,19) = pop(i,19)+rminF
                        end if
                        end do
                end subroutine distsum

                subroutine distzero(pop,pn)
                        integer,intent(in) :: pn ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21) ! The particle arrays

                        pop(1:pn,16:19)=0.0D+00
                end subroutine distzero

end module vitalrates
