module forces
        use params

        implicit none

        contains
                function fgravity(r1,r2,m1,m2,g)
                        !
                        ! The force caused by the gravity. This is a pair potential.
                        ! Gravity is a long range potential.
                        ! The function returns the virial components as numbers 3-5.
                        !

                        real(kind=rkd) :: r1(2),r2(2),m1,m2,g
                        real(kind=rkd) :: r,a,f,runit(2)
                        real(kind=rkd) :: fgravity(5)
                        integer :: i


                        !
                        ! Calculate the unit vector and the angle.
                        !
                        a = abs(atan((r1(1)-r2(1))/(r1(2)-r2(2))))
                        if (r1(1)==r2(1)) then
                                runit(1) = 0.0D+00
                                a = 0.0D+00
                        else
                                runit(1) = (r1(1)-r2(1))/abs(r1(1)-r2(1))
                        endif
                        if (r1(2)==r2(2)) then
                                runit(2) = 0.0D+00
                                a = pi/2.0D+00
                        else
                                runit(2) = (r1(2)-r2(2))/abs(r1(2)-r2(2))
                        endif

                        !
                        ! Calculate the force relative to the particle 1.
                        !
                        r = sqrt((r1(1)-r2(1))**2 + (r1(2)-r2(2))**2)
                        f = -g*m1*m2/r**2

                        !
                        ! Calculate the components.
                        !
                        fgravity(1) = sin(a)*f*runit(1)
                        fgravity(2) = cos(a)*f*runit(2)
                        
                        !
                        ! NOTICE!
                        ! Here we make a manual adjustment, if x or y positions of
                        ! the two particles are exactly the same.
                        !
                        if (r1(1)==r2(1)) fgravity(1)=0.0D+00
                        if (r1(2)==r2(2)) fgravity(2)=0.0D+00

                        !
                        ! Compute the virial components.
                        !
                        fgravity(3) = (r1(1)-r2(1))*fgravity(1) 
                        fgravity(4) = (r1(2)-r2(2))*fgravity(2)
                        fgravity(5) = (r1(1)-r2(1))*fgravity(2)
                end function fgravity

                function flennardjones(r1,r2,eps,phi,cut)
                        !
                        ! The force caused by the Lennard-Jones potential. This is a pair potential.
                        ! Lennard-Jones potential is a short range potential.
                        ! The function returns the virial components as numbers 3-5.
                        !

                        real(kind=rkd) :: r1(2),r2(2),eps,phi,cut
                        real(kind=rkd) :: r,a,f,runit(2)
                        real(kind=rkd) :: flennardjones(5)
                        integer :: i

                        !
                        ! Apply the cut off.
                        ! NOTE: This is hard cut off, thus it causes unlinear force function.
                        ! The reason is to avoid additional free parameters that have no
                        ! known values.
                        !
                        r = sqrt((r1(1)-r2(1))**2 + (r1(2)-r2(2))**2)
                        if (r<cut) then

                        !
                        ! Particles are within the cut off range.
                        !
                        !
                        ! Calculate the unit vector and the angle.
                        !
                        a = abs(atan((r1(1)-r2(1))/(r1(2)-r2(2))))
                        if (r1(1)==r2(1)) then
                                runit(1) = 0.0D+00
                                a = 0
                        else
                                runit(1) = (r1(1)-r2(1))/abs(r1(1)-r2(1))
                        endif
                        if (r1(2)==r2(2)) then
                                runit(2) = 0.0D+00
                                a = pi/2.0D+00
                        else
                                runit(2) = (r1(2)-r2(2))/abs(r1(2)-r2(2))
                        endif

                        !
                        ! Calculate the force relative to the particle 1.
                        !
                        f = -24.0D+00*eps/r**2*(phi/r)**6*(1.0D+00-2.0D+00*(phi/r)**6)
                        f = 24.0D+00*eps/phi*(2.0D+00*(phi/r)**13-(phi/r)**7)
                        
                        !
                        ! Calculate the components.
                        !
                        flennardjones(1) = sin(a)*f*runit(1)
                        flennardjones(2) = cos(a)*f*runit(2)

                        !
                        ! Compute the virial components.
                        !
                        flennardjones(3) = (r1(1)-r2(1))*flennardjones(1)
                        flennardjones(4) = (r1(2)-r2(2))*flennardjones(2)
                        flennardjones(5) = (r1(1)-r2(1))*flennardjones(2)

                        !
                        ! Particles are beyond the cut off range.
                        !
                        else
                                flennardjones(1:5)=0.0D+00
                        end if
                end function flennardjones

                function fharmonic(r1,r2,k,r0,cut)
                        !
                        ! The force caused by the spring potential. This is a pair potential.
                        ! The potential has a cut off radius that makes this a short range potential.
                        ! At the cut off radius the potential is non-continuous, and therefore
                        ! does not follow the principle of the conservation of the energy.
                        ! The function returns the virial components as numbers 3-5.
                        !

                        real(kind=rkd) :: r1(2),r2(2),k,r0,cut
                        real(kind=rkd) :: a,r,r2adj(2),runit(2),s,f
                        real(kind=rkd) :: fharmonic(5)
                        integer :: i

                        !
                        ! Apply the cut off.
                        ! NOTE: This is hard cut off, thus it causes unlinear force function.
                        ! The reason is to avoid additional free parameters that have no
                        ! known values.
                        !
                        r = sqrt((r1(1)-r2(1))**2 + (r1(2)-r2(2))**2)
                        if (r<cut) then

                        s = 1.0D+00
 
                        !
                        ! Calculate the unit vector and the angle.
                        !
                        a = abs(atan((r1(1)-r2(1))/(r1(2)-r2(2))))
                        if (r1(1)==r2(1)) then
                                runit(1) = 0.0D+00
                                a = 0
                        else
                                runit(1) = (r1(1)-r2(1))/abs(r1(1)-r2(1))
                        endif
                        if (r1(2)==r2(2)) then
                                runit(2) = 0.0D+00
                                a = pi/2.0D+00
                        else
                                runit(2) = (r1(2)-r2(2))/abs(r1(2)-r2(2))
                        endif

                        !
                        ! Calculate the distance from the equilibrium.
                        ! The direction of the force depends now both
                        ! on the distance vector between the particles
                        ! and the distance beween the particle and the
                        ! equilibrium radius. If the distance between
                        ! the particles is less than the equilibrium
                        ! radius, then the direction of the force is
                        ! inverse compared to the situation where the
                        ! distance is greater.
                        ! 
                        r = sqrt((r1(1)-r2(1))**2+(r1(2)-r2(2))**2)-r0
                        f = -k*r*s

                        !
                        ! Calculate the x and y components.
                        !
                        fharmonic(1) = sin(a)*f*runit(1)
                        fharmonic(2) = cos(a)*f*runit(2)

                        !
                        ! Compute the virial components.
                        !
                        fharmonic(3) = (r1(1)-r2(1))*fharmonic(1)
                        fharmonic(4) = (r1(2)-r2(2))*fharmonic(2)
                        fharmonic(5) = (r1(1)-r2(1))*fharmonic(2)
                        
                        !
                        ! Particles are beyond the cut off range.
                        !
                        else
                                fharmonic(1:5)=0.0D+00
                        end if
                end function fharmonic

                subroutine reset_forces(pop,pn,res,rn,alc,an)
                        !
                        ! Set f(t0)=f(t_now) and set current forces f(t_now) to zero.
                        !

                        integer :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        integer :: i

                        !
                        ! The population
                        !
                        do i=1,pn
                        ! Move the current force to history.
                        pop(i,8:9)=pop(i,6:7)
                        ! Set the current force to 0.
                        pop(i,6:7)=0
                        ! Set the virial components to 0.
                        pop(i,11:13)=0
                        ! Set the kinetic energy to 0.
                        pop(i,11:14)=0
                        end do

                        !
                        ! The resources
                        !
                        do i=1,rn
                        ! Move the current force to history.
                        res(i,8:9)=res(i,6:7)
                        ! Set the current force to 0.
                        res(i,6:7)=0
                        ! Set the virial component to 0.
                        res(i,11)=0
                        end do

                        !
                        ! The alcohol
                        !
                        do i=1,an
                        ! Move the current force to history.
                        alc(i,8:9)=alc(i,6:7)
                        ! Set the current force to 0.
                        alc(i,6:7)=0
                        ! Set the virial component to 0.
                        alc(i,11)=0
                        end do
                end subroutine reset_forces

                subroutine comp_fgrav(pop,pn,res,rn,alc,an)
                        !
                        ! Here we compute the gravitational forces caused by all the particles in the simulation.
                        ! Each particle has a type (1...6). Each pair of different types of particles has it's
                        ! own set of parameters that are defined in the module "params".
                        !

                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: m1,m2,f(5)
                        integer :: i,j,n1,n2

                        !
                        ! Compute the forces between the individuals.
                        !
                        ! We save in computational costs as we use Newton's III law and set f2 = -f1.
                        !
                        do i=1,pn-1
                        do j=i+1,pn

                        !
                        ! If we have a blank space, then skip the computation.
                        !
                        if (pop(i,1)<1) CYCLE

                        !
                        ! Use the position from the local array.
                        ! Fetch the mass from the global arrays.
                        ! The gravitational constant gconst is a global variable.
                        !
                        n1=pop(i,1)
                        n2=pop(j,1)
                        m1=gpars(n1)
                        m2=gpars(n2)

                        !
                        ! Compute.
                        !
                        f = fgravity(pop(i,2:3),pop(j,2:3),m1,m2,gconst)
                        ! Forces
                        pop(i,6:7)=pop(i,6:7)+f(1:2)
                        pop(j,6:7)=pop(j,6:7)-f(1:2)

                        ! Virial
                        pop(i,11:13)=pop(i,11:13)+f(3:5)
                        pop(j,11:13)=pop(j,11:13)+f(3:5)
                        end do
                        end do
                end subroutine comp_fgrav

                subroutine comp_flj(pop,pn,res,rn,alc,an)
                        !
                        ! Here we compute the Lennard-Jones forces caused by all the particles
                        ! in the simulation.
                        ! Each particle has a type (1...6). Each pair of different types of particles has it's
                        ! own set of parameters that are defined in the module "params".
                        !

                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: eps,phi,cut,f(5)
                        integer :: i,j,n1,n2

                        !
                        ! Compute the forces between the individuals.
                        !
                        ! We save in computational costs as we use Newton's III law and set f2 = -f1.
                        !
                        do i=1,pn-1
                        do j=i+1,pn

                        !
                        ! If we have a blank space, then skip the computation.
                        !
                        if (pop(i,1)<1) CYCLE

                        !
                        ! Use the position from the local array.
                        ! Fetch the epsilon, phi and the cut off radius from the global arrays.
                        !
                        n1=pop(i,1)
                        n2=pop(j,1)
                        eps=ljpars(n1,n2,1)
                        phi=ljpars(n1,n2,2)
                        cut=ljpars(n1,n2,3)
                        !
                        ! Take into account mother-child bonding.
                        ! This age dependent function.
                        !
                        !if (int(pop(i,21))==-int(pop(j,10)) .OR. int(pop(j,21))==-int(pop(i,10))) then
                        if (pop(i,21)<0.0D+00 .OR. pop(j,21)<0.0D+00) then
                                phi=max(0.1D+00,phi*min(pop(i,15),pop(j,15))/18.0D+0)
                        end if
                        
                        !
                        ! Compute.
                        !
                        f = flennardjones(pop(i,2:3),pop(j,2:3),eps,phi,cut)
                        ! Forces
                        pop(i,6:7)=pop(i,6:7)+f(1:2)
                        pop(j,6:7)=pop(j,6:7)-f(1:2)
                        
                        ! Virial
                        pop(i,11:13)=pop(i,11:13)+f(3:5)
                        pop(j,11:13)=pop(j,11:13)+f(3:5)
                        end do
                        end do
                end subroutine comp_flj
 
                subroutine comp_fhl(pop,pn,res,rn,alc,an)
                        !
                        ! Here we compute the spring forces between the bonded particles
                        ! in the simulation.
                        ! Each particle has a type (1...6). Each pair of different types of particles has it's
                        ! own set of parameters that are defined in the module "params".
                        !

                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: k,r0,cut,f(5),bond
                        integer :: i,j,bp,n1,n2

                        !
                        ! Compute the forces between the individuals.
                        !
                        ! We need to go through the population array only once, as each individual
                        ! has the bonded partner(s) as a record in the population array.
                        !
                        do i=1,pn
                        !
                        ! Go through the possible number of bonds.
                        ! NOTE: This could be a variable, therefore the reservation for a loop.
                        !       Do not get confused, j /= individual ID!
                        do j=0,0
                        
                        !
                        ! If we have a blank space, then skip the computation.
                        ! NOTE: Currently bonding happens only between individuals.
                        !
                        if (pop(i,1)<1) CYCLE

                        ! 
                        ! Interpret if we have a bonded partner.
                        ! bonding individual ID(i)=popg(i,10)
                        ! bonded individual ID(i)=popg(i,21)=bp
                        !
                        bp=int(abs(pop(i,21)))
                        if (bp>0) then
                                !
                                ! Use the position from the local array.
                                ! Fetch the spring constant, the equilibrium distance
                                ! and the cut off radius from the global arrays.
                                !
                                !print*,'bond:',i,bp
                                n1=pop(i,1)
                                n2=pop(bp,1)
                                k=spars(n1,n2,1)
                                r0=spars(n1,n2,2)
                                cut=spars(n1,n2,3)
                                !
                                ! Take into account mother-child bonding.
                                ! This age dependent function.
                                !
                                !if (int(pop(i,21))==-int(pop(bp,10)) .OR. int(pop(bp,21))==-int(pop(i,10))) then
                                if (pop(i,21)<0.0D+00 .OR. pop(bp,21)<0.0D+00) then
                                        r0=max(0.1D+00,r0*min(pop(i,15),pop(bp,15))/18.0D+0)
                                end if

                                ! If the distance grows beyond the cut off range, break the bond.
                                if (sqrt((pop(i,2)-pop(bp,2))**2+(pop(i,3)-pop(bp,3))**2)>cut) then
                                        pop(bp,21)=0.0D+00
                                ! If we have child-mother bond and the child reaches adulthood, break the bond.
                                else if (pop(bp,21)<0.0D+00 .and. pop(bp,15)>18.0D+00) then
                                        pop(bp,21)=0.0D+00
                                else
                                        !
                                        ! Compute.
                                        !
                                        f = fharmonic(pop(i,2:3),pop(bp,2:3),k,r0,cut)
                                        ! Force
                                        pop(i,6:7)=pop(i,6:7)+f(1:2)
                                        pop(bp,6:7)=pop(bp,6:7)-f(1:2)
                                        ! Virial
                                        pop(i,11:13)=pop(i,11:13)+f(3:5)
                                        pop(bp,11:13)=pop(bp,11:13)+f(3:5)
                                end if

                        end if
                        end do
                        end do
                end subroutine comp_fhl

end module forces
