module energy
        use params

        implicit none

        contains
                function ekinetic(m,v)
                        !
                        ! The kinetic energy of a single particle.
                        !

                        real(kind=rkd) :: m,v(2)
                        real(kind=rkd) :: ekinetic

                        !
                        ! Calculate the kinetic energy.
                        !
                        ekinetic = 0.5D+00*m*sqrt(v(1)**2+v(2)**2)**2
                end function ekinetic

                function pgravity(r1,r2,m1,m2,g)
                        !
                        ! The gravitational potential is a pair potential.
                        ! Gravity is a long range potential.
                        !

                        real(kind=rkd) :: r1(2),r2(2),m1,m2,g
                        real(kind=rkd) :: r,a,u,runit(2)
                        real(kind=rkd) :: pgravity
                        integer :: i

                        !
                        ! Calculate the potential relative to the particle 1.
                        !
                        r = sqrt((r1(1)-r2(1))**2 + (r1(2)-r2(2))**2)
                        u = -g*m1*m2/r
                        !
                        ! NOTICE!
                        ! Here we make a manual adjustment, if x or y positions of
                        ! the two particles are exactly the same.
                        !
                        if (r == 0.0D+00) u = 0.0D+00
                        
                        pgravity = u
                end function pgravity

                function plennardjones(r1,r2,eps,phi,cut)
                        !
                        ! The Lennard-Jones potential is a pair potential.
                        ! Lennard-Jones potential is a short range potential.
                        !

                        real(kind=rkd) :: r1(2),r2(2),eps,phi,cut
                        real(kind=rkd) :: r,a,u,runit(2),alpha=4.0D+00
                        real(kind=rkd) :: plennardjones
                        integer :: i

                        !
                        ! Calculate the potential relative to the particle 1.
                        !
                        !
                        ! Here we have calculated the alpha manually to save some computing steps.
                        ! alpha = 1/(n-m)*(n**n/m**m)**(1/(n-m)) = 1/(12-6)*(12**12/6**6)**(1/(12-6))
                        ! = (1/6)*(8.9161e12/46656)**(1/6) = 4
                        !
                        r = sqrt((r1(1)-r2(1))**2 + (r1(2)-r2(2))**2)
                        u = alpha*eps*((phi/r)**12-(phi/r)**6)
                        !
                        ! NOTICE!
                        ! Here we make a manual adjustment, if x or y positions of
                        ! the two particles are exactly the same.
                        !
                        if (r == 0.0D+00) u = 0.0D+00
                        
                        plennardjones = u
                end function plennardjones

                function pharmonic(r1,r2,k,r0,cut)
                        !
                        ! The harmonic potential is a pair potential.
                        ! The potential has a cut off radius that makes this a short range potential.
                        ! At the cut off radius the potential is non-continuous, and therefore
                        ! does not follow the principle of the conservation of the energy.
                        !
                        ! NOTE: The cut off range is implemented using the bond breaking system, i.e.,
                        !       the harmonic force is working only between the bonded pairs.

                        real(kind=rkd) :: r1(2),r2(2),k,r0,cut
                        real(kind=rkd) :: r,s,u
                        real(kind=rkd) :: pharmonic
                        integer :: i

                        r = sqrt((r1(1)-r2(1))**2 + (r1(2)-r2(2))**2)-r0
                        s = 1.0D+00

                        !
                        ! Here we note that we have in effect
                        ! a serial system of two springs. Therefore we
                        ! need to use a 0.5 multiplier to calculate
                        ! the potential energy. 
                        !
                        u=0.5D+00*0.5D+00*k*r**2*s

                        !
                        ! NOTICE!
                        ! Here we make a manual adjustment, if x or y positions of
                        ! the two particles are exactly the same.
                        !
                        if (r == 0.0D+00) u = 0.0D+00

                        pharmonic = u
                end function pharmonic

                function comp_ekin(pop,pn,res,rn,alc,an)
                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: m
                        real(kind=rkd) :: comp_ekin
                        integer :: i,n

                        !
                        ! Initialize the counter/container.
                        !
                        comp_ekin = 0.0D+00
                        !
                        ! Compute the kinetic energy for each of the
                        ! particles in the whole population and sum 
                        ! the energies.
                        !
                        do i=1,pn
                        !
                        ! Fetch the parameters (the mass).
                        !
                        n=pop(i,1)
                        m=gpars(n)
                        !
                        ! Compute and add to the sum.
                        ! Here we get a positive energy always.
                        !
                        comp_ekin = comp_ekin + ekinetic(m,pop(i,4:5))
                        end do
                end function comp_ekin

                function comp_pgrav(pop,pn,res,rn,alc,an)
                        !
                        ! Here we compute the gravitational potentials caused by all the particles in the simulation.
                        ! Each particle has a type (1...6). Each pair of different types of particles has it's
                        ! own set of parameters that are defined in the module "params".
                        !

                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: m1,m2,runit(2)
                        real(kind=rkd) :: comp_pgrav
                        integer :: i,j,n1,n2

                        !
                        ! Initialize the counter/container.
                        !
                        comp_pgrav = 0.0D+00
                        !
                        ! Compute the potentials between the individuals.
                        !
                        do i=1,pn
                        do j=1,pn
                        !
                        ! There is no potential inside self.
                        !
                        if (i == j) CYCLE
                        !
                        ! Use the position from the local array.
                        ! Fetch the mass from the global arrays.
                        ! The gravitational constant gconst is a global variable.
                        !
                        n1=pop(i,1)
                        n2=pop(j,1)
                        m1=gpars(n1)
                        m2=gpars(n2)
                        !
                        ! Compute and add to the sum.
                        ! Here we use the unit vector to identify
                        ! the direction of the potential (i.e.
                        ! negative for attractive, positive for
                        ! repulsive potential).
                        ! We also have a 0.5 multiplier, as we use pair
                        ! potentials.
                        !
                        comp_pgrav = comp_pgrav + 0.5D+00*pgravity(pop(i,2:3),pop(j,2:3),m1,m2,gconst)
                        end do
                        end do
                end function comp_pgrav

                function comp_plj(pop,pn,res,rn,alc,an)
                        !
                        ! Here we compute the Lennard-Jones potentials caused by all the particles
                        ! in the simulation.
                        ! Each particle has a type (1...6). Each pair of different types of particles has it's
                        ! own set of parameters that are defined in the module "params".
                        !

                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: eps,phi,cut,runit(2)
                        real(kind=rkd) :: comp_plj
                        integer :: i,j,n1,n2

                        !
                        ! Initialize the counter/container.
                        !
                        comp_plj = 0.0D+00
                        !
                        ! Compute the potentials between the individuals.
                        !
                        do i=1,pn
                        do j=1,pn
                        !
                        ! There is no potential inside self.
                        !
                        if (i == j) CYCLE
                        !
                        ! Use the position from the local array.
                        ! Fetch the epsilon, phi and the cut off radius from the global arrays.
                        !
                        n1=pop(i,1)
                        n2=pop(j,1)
                        eps=ljpars(n1,n2,1)
                        phi=ljpars(n1,n2,2)
                        cut=ljpars(n1,n2,3)
                        !
                        ! Compute and add to the sum.
                        ! Here we use the unit vector to identify
                        ! the direction of the potential (i.e.
                        ! negative for attractive, positive for
                        ! repulsive potential).
                        ! We also have a 0.5 multiplier, as we use pair
                        ! potentials.
                        !
                        comp_plj = comp_plj + 0.5D+00*plennardjones(pop(i,2:3),pop(j,2:3),eps,phi,cut)
                        end do
                        end do
                end function comp_plj
 
                function comp_phl(pop,pn,res,rn,alc,an)
                        ! Here we compute the spring potentials between the bonded particles
                        ! in the simulation.
                        ! Each particle has a type (1...6). Each pair of different types of particles has it's
                        ! own set of parameters that are defined in the module "params".
                        !

                        integer,intent(in) :: pn,rn,an ! The sizes of the arrays
                        real(kind=rkd),intent(inout) :: pop(pn,21),res(rn,11),alc(an,11) ! The particle arrays
                        real(kind=rkd) :: k,r0,radj(2),cut
                        real(kind=rkd) :: comp_phl
                        integer :: i,j,bp,n1,n2

                        !
                        ! Compute the potentials between the individuals.
                        !
                        ! We need to go through the population array only once, as each individual
                        ! has the bonded partner(s) as a record in the population array.
                        !
                        comp_phl = 0.0D+00
                        do i=1,pn
                        !
                        ! Go through the possible number of bonds.
                        ! NOTE: This could be a variable. The loop is reservation.
                        !       Do not get confused, j /= individual ID.
                        do j=0,0
                        ! 
                        ! Interpret if we have a bonded partner.
                        !
                        bp=int(abs(pop(i,10)))
                        if (bp > 0) then
                                !
                                ! Use the position from the local array.
                                ! Fetch the spring constant, the equilibrium distance
                                ! and the cut off radius from the global arrays.
                                !
                                !print*,'bond:',i,bp
                                n1=pop(i,1)
                                n2=pop(bp,1)
                                k=spars(n1,n2,1)
                                r0=spars(n1,n2,2)
                                cut=spars(n1,n2,3)

                                !
                                ! Compute.
                                ! Because the bonds are recorded as unisided, add the energy for both
                                ! individuals.
                                !
                                comp_phl = comp_phl + 2.0D+00*pharmonic(pop(i,2:3),pop(bp,2:3),k,r0,cut)
                        end if
                        end do
                        end do
                end function comp_phl

end module energy
