function p = genParStruct(varargin)
%genParStruct       Generates a valid parameter structure for Two!Ears
%                   Auditory Front-End processor requests.
%
%USAGE:
%     p = genParStruct(par_name,par_value)
%     p = genParStruct(par_name1,par_value,...,par_nameN,par_valueN)
%
%INPUT PARAMETERS:
%  name : Name tag of parameter to be changed
% value : Value for changed parameter
%
%OUTPUT PARAMETER:
%     p : Valid structure of non-default parameters
%
%EXAMPLE:
%     p = genParStruct('nERBs',1/2,'IHCMethod','hilbert') will generate a
%     parameter structure corresponding to a request where the gammatone
%     filterbank has a resolution of 0.5 ERBs per channel and where the
%     inner hair-cell envelope is simply the Hilbert envelope.
%
%NB: The parameter structures generated by this script do not include all
%default values, use parseParameters.m subsequently if needed.
%
%SEE ALSO: parameterHelper.m parseParameters.m 

% First check on the inputs
if mod(size(varargin,2),2)==1
    % Then there is an odd number of input parameters, something is wrong
    warning(['Incorrect number of input arguments. Arguments need to be '...
        'provided in pairs of names and values. The whole parameter '...
        'request is disregarded.'])
    p = struct;
    return
else
    n_par = size(varargin,2)/2;
    p = struct;
end

% Load the parameter info file
path = fileparts(mfilename('fullpath'));
load([path filesep 'parameterInfo.mat'])

% Loop on the number of parameters
for ii = 0:n_par-1
   
    % 1-Check that the provided name is valid
    name = varargin{2*ii+1};
    
    % Loop on the parameter categories
    cats = fieldnames(pInfo);
    jj = 1;
    while jj<size(cats,1)
        if isfield(pInfo.(cats{jj}),name)
            break
        else
            jj = jj+1;
        end
    end
    
    % Go further if this parameter name is valid
    if isfield(pInfo.(cats{jj}),name)
        % Check that the provided value is valid
        value = varargin{2*ii+2};

        % Add to the parameter structure
        p.(name) = value;
        
    else
        warning(['Parameter name %s is invalid, this specific request'...
            ' is disregarded.'],name)
    end
    
    
    
end
