/* *********************************************************************** */
/*                                                                         */
/* enu - generate all 4-regular, planar, connected graphs with given       */
/* number of vertices and given number of faces                            */
/*                                                                         */
/*                                                                         */
/* author: Oliver Heidemeier                                                */
/*                                                                         */
/* *********************************************************************** */


#include<stdio.h>
#include<math.h>
#include<time.h>
#include<limits.h>
#include<malloc.h>
#include<memory.h>



struct mallinfo space_info;

/* *********************************************************************** */
/* declaration of const values                                             */
/* *********************************************************************** */

#define FALSE    0
#define TRUE     1
#define NIL      0
#define max_ecken    250 /* set this maximum because of the unsigned char varable */

#define aussen_kante max_ecken + 2 /* edge going outside  */

#define innen_kante max_ecken + 3  /* edge going inside */

#define leer max_ecken + 4 /* there is no more branch here */

#define speicher 1 /* error codes */
#define eingabe  2
#define datei    3

#define BRILLE 0 /* path type codes */

#define SANDWICH 1

#define BAUCHBINDE 2

#define Durchschnitt max_ecken + 2


typedef unsigned char KNOTENTYP; /* enough for graphs with mostly 255 vertices */


/* *********************************************************************** */
/* Globale-Variablen-Deklaration                                           */
/* *********************************************************************** */

unsigned char gerade;/* is the amount of vertices in the graph even? */ 

unsigned char berechne_symmetrie; /* switch if the symmetrie of the border sequence should be calculated or not. */

long unsigned int zum_bau_verwendet = 0; /* this is the amount of patches that had been used direct or indirect building the graphs. */

unsigned char so, no_iso, no_planar, no_spiegel, no_bb, no_br, no_sw, no_cut, patch_test, sequenz_tiefe;

unsigned char max_patches; /* this is the amount of patches needed for building the graph */

long unsigned int gleich_bb = 0, gleich_br = 0, gleich_sw = 0;

long unsigned int bauchbinden_graphen = 0, sandwich_graphen = 0, brillen_graphen = 0; /* amount of path types graphs are build from */

long unsigned int nicht_am_dreieck = 0; /* amount of paths that didn't start at a triangle */

long unsigned int graphen_geschrieben = 0; /* graphs written to HDD */

long unsigned int verworfene_graphen = 0, doppel_kanten = 0; /* amount of graphs fallen through the isomorphism check */

unsigned char pfad_typ; /* path type */

unsigned char brillen_typ; /* minor BR type */ 


unsigned char aktuelle_ecke; /* amount of vertices currently used to build the graph */


long unsigned int zaehler_visit = 0, zaehler_baum_eintrag = 0, zaehler_verklebe = 0, zaehler_memory = 0;

long unsigned int zaehler_memory_baum_knoten = 0,  zaehler_memory_liste = 0,  zaehler_memory_item = 0;

long unsigned int sandwich_pfade = 0, brillen_pfade = 0, bauchbinden_pfade = 0; /* amount of graph types build  */

FILE *test, *out_stream;

FILE *patchtest, *patchbau, *out;

unsigned char graphs_file_name[200]; /* file name */

FILE *graphs_file_ptr; /* file ptr / file handle */

unsigned char ausgabe, darstellung;

long unsigned int gute_patches = 0, anz_patches = 0; /* amount of patches generated */

long unsigned int bb_patches; /* amount of BB patches */


int anzahl_ecken; /* amount of vertices in the graph */

int flaechen[255]; /* array which includes the faces that could occur in the graphs */

long unsigned int heapzaehler = 0; /* amount of memory used */

unsigned int speicher_frei = 0; /* amount of free memory */

unsigned char *ptr_speicher; /* next free memory area */


KNOTENTYP *teiler[max_ecken + 2]; /* pointer to a list of divisors for the given number */


typedef struct K
{
  KNOTENTYP ziel;

  KNOTENTYP ursprung;
  
  KNOTENTYP bezeichnung;

  unsigned char faerbung; /* TRUE/FALSE */

  unsigned char pfad_start; /* if a path can start here: TRUE */

  unsigned char gezaehlt; /* if the vertice is already counted: TRUE */
  
  struct K *links; /* next edge to the left */
  struct K *rechts; /* to the right */
  struct K *hinten; /* backwards */
  struct K *invers; /* and invers  */
  
} KANTE;


typedef KANTE PLANMAP[max_ecken + 2][4]; /* the graphs can be found in here */

PLANMAP map; /* the graphs can be found in here */


typedef struct I
{
  struct I *next; /* pointer to the next element */
  struct I *vorfahre_0; /* first ancestor */
  struct I *vorfahre_1; /* second ancestor */
  KNOTENTYP verklebung_0; /* gluing edge of the first patch */
  KNOTENTYP verklebung_1; /* gluing edge of the second patch */
  KNOTENTYP schnitt; /* length of the cut */
  KNOTENTYP rest; /* border length of the first patch without the gluing edge */

  unsigned char wird_gebraucht; /* TRUE if the patch had been used for building a graph */

} Item; /* reconstruction data of the patch */


typedef struct IB
{
  struct IB *links; /* left branch */
  struct IB *rechts; /* right branch */
  struct Item *patch; /* pointer to the first patch */
  int nummer; /* number of the node */
  
} ItemBaum; /* tree for memorizing the plan for building the patch */


typedef struct B
{
  KNOTENTYP max_liste; /* max face structure that goes deeper in the tree starting here */
  struct B **liste; /* branches going further */
  Item *eintrag; /* entries here */
  
} Baum; /* sequence tree */


typedef struct
{
  Item *Items[3]; /* pointer to the patches */
  KNOTENTYP Rand[4]; /* border sequence of the patches */
  KNOTENTYP Flaechen[3]; /* faces of the patches */
  KANTE *Kanten[3];
  int zwischenstueck;
  KNOTENTYP pfad_laenge; /* path length */
  unsigned char test;
  Item *aktueller_patch[3];


} Patchdaten; /* this is for transfering the patch data */


typedef struct
{
  Item *ptr_item; /* address of the patch */
  KNOTENTYP seq[max_ecken + 2]; /* border sequence of the patch */
  KANTE *start_kante; /* starting edge of the patch */
  
} Lineare_Liste; /* calling patches by their priority */


Baum *wurzel; /* root of the patch tree */

Baum *bauchbinden_wurzel; /* root of the BB tree */

Patchdaten patch_daten;



/* *********************************************************************** */
/* declaration of the prototypes                                           */
/* *********************************************************************** */

void Parameter(void);
void berechne_teiler(unsigned char zahl);
unsigned char kanonisch(KNOTENTYP sequenz[max_ecken + 2]);
KNOTENTYP symmetrie(KNOTENTYP sequenz[max_ecken + 2]);
void *memory_alloc(long unsigned int anzahl_bytes);
void quicksort(int *a, int l, int r);
void bauch_binde_eintrag (Item *vorfahre[2], unsigned char typ, unsigned char verkl_0, unsigned char verkl_1, 
		     KNOTENTYP sequenz[3][max_ecken + 2], unsigned char first, unsigned char second);
unsigned char baum_eintrag(Item *vorfahre[2], KNOTENTYP sequenz[max_ecken + 2], KNOTENTYP rand_ecken, unsigned char typ, unsigned char first,
			   unsigned char second, KNOTENTYP schnitt, KNOTENTYP symm[2], KNOTENTYP verklebung[2], KNOTENTYP s_laenge[2]);
void init_liste(void);
void init_map(void);
void Create_Patches(void);
unsigned char verklebe(KNOTENTYP sequenz[3][max_ecken + 2], KNOTENTYP flaechen_patch[2], Item *vorfahre[2]);
void suche_patch(void);
void baue_auf(unsigned char Durchgang);
int plausibel(Item *ptr_item, int init);
void baue_bauchbinde(void);
unsigned char visit(unsigned char nummer, Baum *ptr_baum, Baum *wurzel_patch_2, KNOTENTYP flaechen_patch[2], KNOTENTYP sequenz[3][max_ecken + 2], Item *vorfahre[2], 
		    unsigned char init);
void fehler(unsigned char f);
void main_visit(void);
Item *test_patch(Baum *ptr33_baum, int z2);
void baue_brille(unsigned char typ);
void baue_bauchbinde(void);
void baue_sandwich(void);

/* *********************************************************************** */
/*                                                                         */
/* help page                                                               */
/*                                                                         */
/* call: Parameter()                                                       */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */

void Parameter(void)
{
  fprintf(stderr, "\n  enu generates all 4-regular, planar, connected graphs\n  with given number of vertices and given number of faces\n");
  
  fprintf(stderr,"  the program call looks as follows:\n\n");
  fprintf(stderr,"  enu v<x> f<x> [OPTIONS]\n\n");
  fprintf(stderr,"  v<x>          number <x> of vertices of the graphs.\n");
  fprintf(stderr,"  f<x>          face <x> is included in the graphs.\n");
  fprintf(stderr,"  -<a>-<b>      faces in the range <a> to <b> are included\n");
  fprintf(stderr,"                in the graphs.\n\n");
  fprintf(stderr,"  [OPTIONS] can be one or more of the following:\n\n");
  fprintf(stderr,"  no_iso        no isomorphism check.\n");
  fprintf(stderr,"  no_planar     no planar-code check in isomorphism check\n");
  fprintf(stderr,"  no_mirror     no test for mirror-images in isomorphism check.\n");
  fprintf(stderr,"  no_bb         do not generate graphs build on BB.\n");
  fprintf(stderr,"  no_br         do not generate graphs build on BR.\n");
  fprintf(stderr,"  no_sw         do not generate graphs build on SW.\n");
  fprintf(stderr,"  no_cut        no graphs with 1- or 2-cut.\n");
  fprintf(stderr,"\n  example: enu v50 f3 f4\n");
  fprintf(stderr,"  generates graphs with 50 vertices and 3- and 4-gons.\n\n");
  
}


/* *********************************************************************** */
/*                                                                         */
/* get the maximum                                                         */
/*                                                                         */
/* call: max(x, y)                                                         */
/*                                                                         */
/* return:  Maximum or '0' if x = y                                        */
/*                                                                         */
/* *********************************************************************** */
KNOTENTYP max(KNOTENTYP x, KNOTENTYP y)
{
  if (x > y)
    return(x);
  else if (y > x)
    return(y);
  else
    return(0);
}


/* *********************************************************************** */
/*                                                                         */
/* get the minimum                                                         */
/*                                                                         */
/* call: min(x, y)                                                         */
/*                                                                         */
/* return:  Minimum oder '0' if x = y                                      */
/*                                                                         */
/* *********************************************************************** */
KNOTENTYP min(KNOTENTYP x, KNOTENTYP y)
{
  if (x < y)
    return(x);
  else if (y < x)
    return(y);
  else
    return(0);
}    


/* *********************************************************************** */
/*                                                                         */
/* get the divisor                                                         */
/*                                                                         */
/* call: berechne_teiler(Maximum)                                        */
/*                                                                         */
/* return: void                                                           */
/*                                                                         */
/* *********************************************************************** */

void berechne_teiler(unsigned char zahl)
{
  unsigned char i,j,k,l,max;
  unsigned char speicher1[max_ecken + 2],speicher2[max_ecken + 2];
  unsigned char zaehler;
  
  max = zahl;
  
  for(j = 1 ; j <= zahl ; j++)
    {
      speicher1[0] = speicher2[0] = 0;
      max = zahl;
      
      for(i = 1 ; i < max ; i++)
	if (j % i == 0)
	  {
	    speicher1[0]++;
	    speicher1[speicher1[0]] = i;
	    max = j / i;
	    if (max != i)
	      {
		speicher2[0]++;
		speicher2[speicher2[0]] = max;
	      }
	  }
      
      teiler[j] = (unsigned char *) malloc((speicher1[0] + speicher2[0] + 1) * sizeof(unsigned char));
      
      *(teiler[j]) = speicher1[0] + speicher2[0];
      
      for(k = 1 ; k <= speicher1[0] ; k++)
	*(teiler[j] + k) = speicher1[k];
      
      zaehler = 0;
      for(l = speicher2[0] ; l >= 1 ; l--, zaehler++)
	*(teiler[j] + k + zaehler) = speicher2[l];
      
    }
  
}


/* *********************************************************************** */
/*                                                                         */
/* Ascertains if the sequence is canonically marked                        */
/*                                                                         */
/* call: kanonisch(Sequenz)                                                */
/*                                                                         */
/* return: TRUE if cannonically marked                                     */
/*                                                                         */
/* *********************************************************************** */

unsigned char kanonisch(KNOTENTYP sequenz[max_ecken + 2])
{
  int i, j, k, l, zaehler, bis;
  
  zaehler = 1;
  
  for (i = 3 ; i <= sequenz[0] ; i++)
    {
      if (sequenz[i] > sequenz[2])
	return(FALSE);
      else if (sequenz[i] < sequenz[2])
	zaehler++;
      else /*if ((sequenz[i] == sequenz[2]))*/
	{
	  
	  if ((sequenz[0] - 1) % zaehler == 0)
	    bis = (sequenz[0] - 1) / zaehler - 1;
	  else
	    bis = (sequenz[0] - 1) / zaehler;
	  
	  for (j = 1 ; j <= bis ; j++)
	    {
	      for (k = 2 ; k <= zaehler + 1 ; k++)
		{
		  l = j * zaehler + k;
		  
		  if (sequenz[k] < sequenz[l])
		    return(FALSE);
		  else if (sequenz[k] > sequenz[l])
		    {
		      k = zaehler + 3;
		      j = bis + 2;
		    }
		  
		} /* Ende k */
	      
	    } /* Ende j */
	  
	  if ((k == zaehler + 2) && (j == bis + 1))
	    return(TRUE);
	  
	  zaehler++;
	  
	} /* Ende else if */
      
    } /* Ende for i */
  
  return(TRUE);
}


/* *********************************************************************** */
/*                                                                         */
/* Ascertains the number of sequence entries                               */
/* >ou have to call the divisor function first                             */
/*                                                                         */
/* call: symmetrie(Nummer der Sequenz (0,1,2))                             */
/*                                                                         */
/* return: number of sequence entries                                      */
/*                                                                         */
/* *********************************************************************** */

KNOTENTYP symmetrie(KNOTENTYP sequenz[max_ecken + 2])
{
  
  int anzahl,h,i,j,bis;
  
  anzahl = sequenz[0] - 1;
  
  if (sequenz[2] == sequenz[anzahl + 1]) /* if the first and last number are the same the 
					    border structure of the patch has to be
					    symmetrical */
    return(1);
  
  if (*(teiler[anzahl]) == 2) /* only 2 divisors: prime number */
    return(anzahl);
  
  for (h = 2 ; h <= *(teiler[anzahl]) ; h++) /* symmetry can only be found in the numbers of divisors */
    {
      if (sequenz[*(teiler[anzahl] + h) + 2] == sequenz[2])
	{
	  bis = (anzahl / *(teiler[anzahl] + h)) - 1;
	  
	  for (i = 1 ; i <= bis ; i++) 
	    {
	      for (j = 2 ; j <= *(teiler[anzahl] + h) + 1 ; j++)
		{
		  if (sequenz[j] != sequenz[(i * (*(teiler[anzahl] + h))) + j])
		    {
		      j = *(teiler[anzahl] + h) + 2;
		      i = bis + 2;
		    }
		} /* end for j */
	      
	    } /* end for i */
	  
	  if ((i == bis + 1) && (j == *(teiler[anzahl] + h) + 2))
	    return(*(teiler[anzahl] + h));
	}
    }
  
  return(anzahl);
}



/* *********************************************************************** */
/*                                                                         */
/* memory allocation                                                       */
/*                                                                         */
/* call: memory_alloc(Speicher in Bytes)                                   */
/*                                                                         */
/* return: coid pointer to the beginning of the allocated memory           */
/*         if an error occurs the program is exited                        */
/*                                                                         */
/* *********************************************************************** */

void *memory_alloc(long unsigned int anzahl_bytes)
{
  long unsigned int i;
  unsigned long int size;
  unsigned char *ptr;
  static unsigned int block_groesse = 2048;
  
  zaehler_memory++;

  if (anzahl_bytes > block_groesse) /* allocating own amount of memory */
    {

      if ((ptr = (unsigned char *) malloc(anzahl_bytes)) == NULL)
	{
	  fprintf(stderr,"\n\n allocated memory: %d \n\n", heapzaehler);
	  fehler(speicher);
	}

      heapzaehler = heapzaehler + 4 + anzahl_bytes;

      for (i = 0 ; i < anzahl_bytes ; i++) /* delete the allocated memory with 0 */
	*(ptr + i) = NIL;

      return(ptr);
    }
      

  if (speicher_frei < anzahl_bytes) /* allocate new memory */
    {
      if ((ptr = (unsigned char *) malloc(block_groesse)) == NULL)
	{
	  fprintf(stderr,"\n\n allocated memory: %d \n\n", heapzaehler);
	  fehler(speicher);
	}
      
      heapzaehler = heapzaehler + 4 + block_groesse;
      
      for (i = 0 ; i < block_groesse ; i++) /* delete the allocated memory with 0 */
	*(ptr + i) = NIL;
      
      ptr_speicher = ptr + anzahl_bytes; /* delete used bytes from the allocated amount of memory ... */
      speicher_frei = block_groesse - anzahl_bytes; /* ... and get less memory :-(( */
      
      return(ptr);
    }
  else /* there is enough free memory in the allocated block*/
    {
      ptr = ptr_speicher;
      ptr_speicher = ptr_speicher + anzahl_bytes;
      speicher_frei = speicher_frei - anzahl_bytes;
      return(ptr);
    }
  
}


/* *********************************************************************** */
/*                                                                         */
/* Quicksort                                                               */
/*                                                                         */
/* call: quicksort(integer array, left, right)i                            */
/*                                                                         */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */

void quicksort(int *a, int l, int r)
{
  int v, i, j, t;
  
  if (r > l)
    {
      v = a[r];
      i = l - 1;
      j = r;
      
      for (;;)
	{
	  while(a[++i] < v);
	  while(a[--j] > v);
	  
	  if(i >= j) break;
	  
	  t = a[i]; a[i] = a[j]; a[j] = t;
	}
      
      t = a[i]; a[i] = a[r]; a[r] = t;
      
      quicksort(a, l, i - 1);
      quicksort(a, i + 1, r);
    }
}


/* *********************************************************************** */
/*                                                                         */
/* insert BB into the tree                                                 */
/*                                                                         */
/* call: bauch_binde_eintrag(...)                                          */
/* return: void                                                            */
/*                                                                         */
/* "typ" includes 0 if type is cut and 1 if type if embedding              */
/*                                                                         */
/* *********************************************************************** */
void bauch_binde_eintrag (Item *vorfahre[2], unsigned char typ, unsigned char verkl_0, unsigned char verkl_1,
			  KNOTENTYP sequenz[3][max_ecken + 2], unsigned char first, unsigned char second)
{

  Item *vor_0, *vor_1, *ptr_item, *neu_ptr_item, *vorfahre_dummy;
  Baum *ptr_baum, *neu_ptr_baum, **pptr_baum;

  int max;
  long unsigned int vorfahren[2], zusaetzliche_patches;
  unsigned char i;

  /* there is an own structure for the BB patches because there
     is no way to obtain new patches by gluing together two BB patches */
  
  max = sequenz[2][1] + 4;

  if (sequenz[2][2] > max - 2) /* die V3-Ecken des Randes einer Bauchbinde sind sicherlich eine obere Schranke fuer die Verklebbarkeit dieses Patches. */
    {
      max = sequenz[2][2];
      if (max > anzahl_ecken) /* too many vertices */
	return;
    }

  if (max > anzahl_ecken + 2) /* too many faces */
    return;


  /* a BB with a sequence not equal 4 (this one is not connectable with the smalles BB patch) is not allowed to have max faces - 4 */
  if ((sequenz[2][1] + 4 == anzahl_ecken + 2) && (sequenz[2][2] != 4))
    return;

  ptr_baum = bauchbinden_wurzel;

  ptr_baum = *(ptr_baum->liste + sequenz[2][1]); /* face type entrie are always there */

  if (*(ptr_baum->liste + sequenz[2][2]) == NIL)
    {
      zaehler_memory_baum_knoten++;
      neu_ptr_baum            = (Baum *) memory_alloc(sizeof(Baum));
      neu_ptr_baum->max_liste = leer;
      neu_ptr_baum->eintrag   = NIL;
      neu_ptr_baum->liste     = NIL;

      *(ptr_baum->liste + sequenz[2][2]) = neu_ptr_baum;

    }
  
  ptr_baum = *(ptr_baum->liste + sequenz[2][2]); /* sequence structure */

  ptr_item = ptr_baum->eintrag;
  if (ptr_item != NIL)
    while (ptr_item->next != NIL)
      ptr_item = ptr_item->next;

  vor_0 = vorfahre[first];
  vor_1 = vorfahre[second];

  zaehler_memory_item++;
  neu_ptr_item = (Item *) memory_alloc(sizeof(Item));
	  
  /* insert values ... */
  
  neu_ptr_item->next         = NIL; 
  neu_ptr_item->vorfahre_0   = vor_0;
  neu_ptr_item->vorfahre_1   = vor_1;
  neu_ptr_item->verklebung_0 = verkl_0 - 1; /* do not remember the entry of the amount of faces */
  neu_ptr_item->verklebung_1 = verkl_1 - 1;
  neu_ptr_item->schnitt      = sequenz[first][verkl_0];
  
  /* only for testing */

  neu_ptr_item->wird_gebraucht = FALSE;


  if (verkl_0 == 2) /* gluing the first part of the sequence, the second one remains */
    neu_ptr_item->rest = sequenz[first][3] + 2; /* add the two vertices that are glued */
  else
    neu_ptr_item->rest = sequenz[first][2] + 2;
  
  neu_ptr_item->verklebung_0 = neu_ptr_item->verklebung_0 << 1;
  
  /* the first bit is used for the cut type.
     0 is cut
     1 is embedding */

  if (typ == 1) /* embedding */
    {
      neu_ptr_item->verklebung_0 = neu_ptr_item->verklebung_0 + 1; /* set first bit */
      neu_ptr_item->schnitt      = sequenz[first][verkl_0]; /* value of the first gluing edge of the first patch */
      neu_ptr_item->rest         = sequenz[first][verkl_0 + 1]; /* value of the second gluing edge of the first patch */
    }

  vorfahren[0] = vorfahren[1] = 1;
  
  for (i = 0 ; i <= 1 ; i++)
    {
      vorfahre_dummy = vorfahre[i];
      
      if (vorfahre_dummy != NIL) /* count ancestors */
	while(vorfahre_dummy->next != NIL)
	  {
	    vorfahre_dummy = vorfahre_dummy->next;
	    vorfahren[i]++;
	  }
    }
  
  zusaetzliche_patches = 0;
  
  if (vorfahre[0] == vorfahre[1])
    for (i = 1 ; i <= vorfahren[0] ; i++)
      zusaetzliche_patches = zusaetzliche_patches + i;
  else
    zusaetzliche_patches = vorfahren[0] * vorfahren[1];
  
  anz_patches = anz_patches + zusaetzliche_patches;
  gute_patches = gute_patches + zusaetzliche_patches;
  bb_patches   = bb_patches + zusaetzliche_patches;

  if (ptr_baum->eintrag != NIL)
    {
      ptr_item->next = neu_ptr_item;
      ptr_item       = ptr_item->next;
    }
  else
    {
      ptr_baum->eintrag = neu_ptr_item;
      ptr_item = neu_ptr_item;
    }
  
  if (darstellung == TRUE)
    fprintf(stderr,"\r patches build allover: %d, good patches: %d", anz_patches, gute_patches);


}

/* *********************************************************************** */
/*                                                                         */
/* build all graphs out of a patch                                         */
/*                                                                         */
/* call: baue_graph(Patch P)                                               */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */
void baue_graph(Item *patch, KNOTENTYP sequenz[max_ecken + 2])
{
  KNOTENTYP rest_flaechen;
  KNOTENTYP bis, i, j, k, m, start;
  KNOTENTYP x, y, z[2];  
  KNOTENTYP flaechen_patch[2];
  
  Baum *ptr1_baum, *ptr2_baum, *ptr3_baum, *ptr33_baum;
  Item *ptr1_item, *ptr2_item, *ptr3_item;

  rest_flaechen = anzahl_ecken + 2 - sequenz[1]; /* rest_flaechen remain for the last two patches */

  i = rest_flaechen;

  if (sequenz[0] == 3) /* look for 2 more patches with one 2-valent vertice */
    {
      for (j = 4 ; j <= rest_flaechen - 4 ; j++)
	{
	  k = rest_flaechen - j; /* split rest_flaechen in j and k */
	  
	  if ((wurzel->max_liste >= max(j,k)) && (wurzel->max_liste != leer)) 
	    {
	      if ((wurzel->liste[j] != NIL) && (wurzel->liste[k] != NIL))
		{
		  ptr1_baum = wurzel->liste[j];
		  ptr2_baum = wurzel->liste[k];
		  
		  for (x = 2 ; x <= ptr1_baum->max_liste ; x = x + 2)
		    {
		      if ((*(ptr1_baum->liste + x) != NIL) && (ptr1_baum->max_liste >= x))
			ptr1_item = (*(ptr1_baum->liste + x))->eintrag; /* get entry */
		      else
			ptr1_item = NIL;
		      
		      if (ptr1_item != NIL) /* is there a patch inserted? */
			{
			  if (ptr2_baum->max_liste != leer)
			    {
			      for (y = 2 ; ((y <= ptr2_baum->max_liste) && (y <= x)) ; y = y + 2)
				{
				  if ((*(ptr2_baum->liste + y) != NIL) && (ptr2_baum->max_liste >= y))
				    ptr2_item = (*(ptr2_baum->liste + y))->eintrag;
				  else 
				    ptr2_item = NIL;
				  
				  if (ptr2_item != NIL) /* is there a patch inserted? */
				    {
				      patch_daten.Items[0] = ptr1_item;
				      patch_daten.Items[1] = ptr2_item;
				      patch_daten.Items[2] = patch;
				      
				      patch_daten.Rand[0] = x;
				      patch_daten.Rand[1] = y;
				      patch_daten.Rand[2] = sequenz[2];
				      patch_daten.Rand[3] = sequenz[3];
				      
				      patch_daten.Flaechen[0] = j;
				      patch_daten.Flaechen[1] = k;
				      patch_daten.Flaechen[2] = sequenz[1];
				      
				      if ((sequenz[2] > x) && (no_br == FALSE))
					{
					  /* BR 0 */
					  if ((sequenz[2] >= x - y) && (sequenz[3] == sequenz[2] + y - x))
					    baue_brille(0);
					  
					  /* BR 1 */
					  if ((sequenz[2] >= x + y + 2) && (sequenz[3] == sequenz[2] - x - y - 2))
					    baue_brille(1);
					}
				      else if ((sequenz[2] == x) && (sequenz[3] == y) && (no_br == FALSE))
					{
					  /* BR 2 */
					  baue_brille(2);
					}
				      else if ((sequenz[2] < x) && (no_sw == FALSE))/* SW */
					{						
					  if ((sequenz[2] >= x - y) && (sequenz[3] == sequenz[2] + y - x))
					    baue_sandwich();
					}
				    }
				}
			    }
			}
		    }
		}
	    }
	}
    }
  else /* search for one patch with one 2-valent vertices and a patch with 2 2-valent vertices */
    {
      for (j = 2 ; j <= rest_flaechen - 2 ; j++)
	{
	  k = rest_flaechen - j;
	  
	  ptr1_item = patch;
	  ptr2_baum = wurzel->liste[j];
	  ptr3_baum = wurzel->liste[k];
	  x = sequenz[2];

	  if ((wurzel->max_liste >= max(j,k)) && (wurzel->max_liste != leer)) 
	    {
	      if ((wurzel->liste[j] != NIL) && (wurzel->liste[k] != NIL))
		{
		  if (ptr2_baum->max_liste != leer)
		    {
		      for (y = 2 ; (y <= ptr2_baum->max_liste) ; y = y + 2)
			{
			  if ((*(ptr2_baum->liste + y) != NIL) && (ptr2_baum->max_liste >= y))
			    ptr2_item = (*(ptr2_baum->liste + y))->eintrag;
			  else 
			    ptr2_item = NIL;
			  
			  if (ptr2_item != NIL) /* is there a patch inserted? */
			    {
			      if (ptr3_baum->max_liste != leer)
				{
				  bis = ptr3_baum->max_liste;

				  for (z[0] = 1 ; z[0] <= bis ; z[0]++)
				    {
				      if ((*(ptr3_baum->liste + z[0]) != NIL) && (ptr3_baum->max_liste >= z[0]))
					ptr33_baum = *(ptr3_baum->liste + z[0]);
				      else 
					ptr33_baum = NIL;
				      
				      if (ptr33_baum != NIL)
					{
					  if (ptr33_baum->max_liste != leer)
					    {
					      if (y > x)
						{
						  patch_daten.Items[0] = ptr2_item;
						  patch_daten.Items[1] = ptr1_item;
						  
						  patch_daten.Rand[0] = y;
						  patch_daten.Rand[1] = x;
						  patch_daten.Rand[2] = z[0];
						  
						  patch_daten.Flaechen[0] = j;
						  patch_daten.Flaechen[1] = sequenz[1];
						  patch_daten.Flaechen[2] = k;
						}
					      else
						{
						  patch_daten.Items[0] = ptr1_item;
						  patch_daten.Items[1] = ptr2_item;
						  
						  patch_daten.Rand[0] = x;
						  patch_daten.Rand[1] = y;
						  patch_daten.Rand[2] = z[0];
						  
						  patch_daten.Flaechen[0] = sequenz[1];
						  patch_daten.Flaechen[1] = j;
						  patch_daten.Flaechen[2] = k;
						}

					      for (m = 0 ; m <= 1; m++)
						{ 
						  if ((z[0] > patch_daten.Rand[0]) && (no_br == FALSE))
						    {
						      /* BR 0 */
						      if (z[0] >= patch_daten.Rand[0] - patch_daten.Rand[1])
							{
							  z[1] =  z[0] + patch_daten.Rand[1] - patch_daten.Rand[0];
							  patch_daten.Rand[3] = z[1];
							  
							  patch_daten.Items[2] = ptr3_item = test_patch(ptr33_baum, z[1]);
							  if (ptr3_item != NIL)
							    baue_brille(0);
							}
						      
						      /* BR 1 */
						      if (z[0] >= patch_daten.Rand[0] + patch_daten.Rand[1] + 2)
							{
							  z[1] = z[0] - patch_daten.Rand[0] - patch_daten.Rand[1] - 2;
							  patch_daten.Rand[3] = z[1];
							  patch_daten.Items[2] = ptr3_item = test_patch(ptr33_baum, z[1]);
							  if (ptr3_item != NIL)
							    baue_brille(1);
							}
						    }
						  else if ((z[0] == patch_daten.Rand[0]) && (no_br == FALSE))
						    {
						      /* BR 2 */
						      z[1] = patch_daten.Rand[1];
						      patch_daten.Rand[3] = z[1];
						      patch_daten.Items[2] = ptr3_item = test_patch(ptr33_baum, z[1]);
						      if (ptr3_item != NIL)
							baue_brille(2);
						    }
						  else if ((z[0] < patch_daten.Rand[0]) && (no_sw == FALSE))/* SW */
						    {						
						      if (z[0] >= patch_daten.Rand[0] - patch_daten.Rand[1])
							{
							  z[1] = z[0] + patch_daten.Rand[1] - patch_daten.Rand[0];
							  
							  patch_daten.Rand[3] = z[1];
							  patch_daten.Items[2] = ptr3_item = test_patch(ptr33_baum, z[1]);
							  if (ptr3_item != NIL)
							    baue_sandwich();
							}
						    }
						  
						  if (x != y)
						    m = 2;
						  else
						    {
						      patch_daten.Items[0] = ptr2_item;
						      patch_daten.Items[1] = ptr1_item;
						      
						      patch_daten.Rand[0] = y;
						      patch_daten.Rand[1] = x;
						      patch_daten.Rand[2] = z[0];
						      
						      patch_daten.Flaechen[0] = j;
						      patch_daten.Flaechen[1] = sequenz[1];
						      patch_daten.Flaechen[2] = k;
						    }
						}
					    }
					}
				    }
				}
			    }
			}
		    }
		}
	    }
	}
    }
}

/* *********************************************************************** */
/*                                                                         */
/* build all necessary patches and graphs out of a given patch             */
/*                                                                         */
/* call: baue_weiter(Patch P, sequenz P)                                   */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */
unsigned char baue_weiter(Item *patch, KNOTENTYP sequenz[max_ecken + 2])
{
  KNOTENTYP i, j, bis;
  KNOTENTYP flaechen_patch[2];
  KNOTENTYP sequenz_neu[3][max_ecken + 2];
  Item *vorfahre[2]; /* pointer to the ancestors of the patches to be glued */
  
  unsigned char dummy, bb_erzeugt;

  if (sequenz[0] <= 3) /* this is a patch that can be glued to a graph */ 
    baue_graph(patch, sequenz);
  
  bis = anzahl_ecken - 3 - sequenz[1];
  
  bb_erzeugt = FALSE;

  for (i = 1 ; i <= bis ; i++)
    {
      if ((wurzel->max_liste >= i) && (wurzel->max_liste != leer)) 
	{
	  flaechen_patch[0] = sequenz[1];
	  flaechen_patch[1] = i;
	
	  for (j = 0 ; j <= (sequenz[0] * 2) - 1 ; j++)
	    sequenz_neu[0][j] = sequenz[j];
	  
	  sequenz_neu[1][0] = 1;
	  sequenz_neu[1][1] = flaechen_patch[1];
	  sequenz_neu[2][0] = 0;

	  vorfahre[0] = patch;
	  vorfahre[1] = NIL;
  
	  berechne_symmetrie = TRUE;

	  if (wurzel->liste[i] != NIL)
	    {
	      dummy = visit(1, wurzel->liste[i], wurzel->liste[i], flaechen_patch, sequenz_neu, vorfahre, FALSE);
	      if (dummy == TRUE)
		bb_erzeugt = TRUE;
	    }
	}
    }
  
  return(bb_erzeugt);

}

/* *********************************************************************** */
/*                                                                         */
/* insert a sequence into the tree                                         */
/*                                                                         */
/* call: baum_eintrag(...)                                                 */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */
unsigned char baum_eintrag (Item *vorfahre[2], KNOTENTYP sequenz[max_ecken + 2], KNOTENTYP rand_ecken, unsigned char typ, unsigned char first,
			    unsigned char second, KNOTENTYP schnitt, KNOTENTYP symm[2], KNOTENTYP verklebung[2], KNOTENTYP s_laenge[2])
{

  /* remark: "type" is 0 if type is a cut.
    if "type" is 0 then "rand_ecken" contains the amount of vertices on the margin
    of the first patch. Otherwise "rand_ecken" contains the length of the first gluing
    edge and "type" contains the length of the second gluing edge of the outer patch. */

  int i, j, k, a, gesamt, max, fd;
  long unsigned int vorfahren[2], zusaetzliche_patches;

  unsigned char verkl_0, verkl_1, bb_erzeugt;
  
  Baum *ptr_baum, *neu_ptr_baum; 
  Baum **pptr_baum;
  Item *ptr_item, *neu_ptr_item, *vor_0, *vor_1, *vorfahre_dummy;

  /* this is a patche with a V2 vertice and the margin of the patch contains more vertices than the graph contains allover.
     With a embedding the graph would not have less vertices. */
  if ((sequenz[0] == 2) && (sequenz[2] > anzahl_ecken))
    return;


  if (kanonisch(sequenz) == TRUE) /* is the sequence of the margin canonically marked? Accept the patch if it is.*/
    {
      if (ausgabe == TRUE)
	fprintf(out_stream," OK\n");
    }
  else
    {
      if (ausgabe == TRUE)
	fprintf(out_stream," wrong\n");
      return;
    }


  /* memory reducing part ;-) */

  if ((patch_test == FALSE) && (((gerade == TRUE) && (sequenz[1] >= ((anzahl_ecken + 2) / 2))) || ((gerade == FALSE) && (sequenz[1] >= ((anzahl_ecken + 2) / 2)))))
    {
      if (verklebung[first] > s_laenge[first]) /* gluing part not within the symmetrie */
	verkl_0 = 2; /* use first entry */
      else
	verkl_0 = verklebung[first]; /* ... it is ok else */
      
      if (verklebung[second] > s_laenge[second])
	verkl_1 = 2;
      else
	verkl_1 = verklebung[second];

      vor_0 = vorfahre[first];
      vor_1 = vorfahre[second];

      neu_ptr_item               = (Item *) malloc(sizeof(Item));
      neu_ptr_item->next         = NIL; 
      neu_ptr_item->vorfahre_0   = vor_0;
      neu_ptr_item->vorfahre_1   = vor_1;
      neu_ptr_item->verklebung_0 = verkl_0 - 1;
      neu_ptr_item->verklebung_1 = verkl_1 - 1;
      
      if (ausgabe == TRUE)
	fprintf(out_stream, "\nEntry with verkl_0:%d and verkl_1:%d\n", neu_ptr_item->verklebung_0,  neu_ptr_item->verklebung_1);
      
      if (typ == Durchschnitt) /* no embedding */
	{
	  neu_ptr_item->schnitt = schnitt;
	  neu_ptr_item->rest    = rand_ecken - neu_ptr_item->schnitt;
	  
	  neu_ptr_item->verklebung_0 = neu_ptr_item->verklebung_0 << 1; 
	  /* first bit is for the cut type. 0 is cut and 1 is embeding */
	}
      else /* cut */
	{
	  neu_ptr_item->schnitt      = rand_ecken;
	  neu_ptr_item->rest         = typ;
	  neu_ptr_item->verklebung_0 = (neu_ptr_item->verklebung_0 << 1) + 1;
	  /* set first bit to 1 */
	}
     
      vorfahren[0] = vorfahren[1] = 1;
      
      for (i = 0 ; i <= 1 ; i++)
	{
	  vorfahre_dummy = vorfahre[i];

	  if (vorfahre_dummy != NIL) /* count ancestors */
	    while(vorfahre_dummy->next != NIL)
	      {
		vorfahre_dummy = vorfahre_dummy->next;
		vorfahren[i]++;
	      }
	}

      zusaetzliche_patches = 0;

      if (vorfahre[0] == vorfahre[1])
	for (i = 1 ; i <= vorfahren[0] ; i++)
	  zusaetzliche_patches = zusaetzliche_patches + i;
      else
	zusaetzliche_patches = vorfahren[0] * vorfahren[1];
      
      anz_patches = anz_patches + zusaetzliche_patches;
      
      if (sequenz[0] <= 3)
	gute_patches = gute_patches + zusaetzliche_patches;
          
     
      bb_erzeugt = baue_weiter(neu_ptr_item, sequenz);

      if (bb_erzeugt == FALSE)
	free(neu_ptr_item);

    }
  else
    {
      zaehler_baum_eintrag++;
      
      ptr_baum = wurzel; /* start running at the root */
      
      
      for (i = 1 ; i <= sequenz[0] ; i++) /* run through the tree and insert branches if necessary */
	{
	  if ((ptr_baum->max_liste < sequenz[i]) || (ptr_baum->max_liste == leer) || (*(ptr_baum->liste + sequenz[i]) == NIL)) /* no branch going deeper */
	    {
	      if ((sequenz[i] > ptr_baum->max_liste) || (ptr_baum->max_liste == leer))
		{
		  /*pptr_baum = (Baum **) calloc((sequenz[i] + 1) * sizeof(*pptr_baum), sizeof(*pptr_baum));
		  heapzaehler = heapzaehler + ((sequenz[i] + 1) * sizeof(*pptr_baum)) + 4;*/
		  
		  zaehler_memory_liste++;
		  pptr_baum = (Baum **) memory_alloc((sequenz[i] + 1) * sizeof(*pptr_baum));
		  
		  if (ptr_baum->max_liste != leer)
		    {
		      for (j = 0 ; j <= ptr_baum->max_liste ; j++) /* copy old list to adjusted list */
			*(pptr_baum + j) = *(ptr_baum->liste + j);
		    }
		}
	      
	      /* new node */
	      
	      zaehler_memory_baum_knoten++;
	      neu_ptr_baum            = (Baum *) memory_alloc(sizeof(Baum));
	      neu_ptr_baum->max_liste = leer;
	      neu_ptr_baum->eintrag   = NIL;
	      neu_ptr_baum->liste     = NIL;
	      
	      if ((sequenz[i] > ptr_baum->max_liste) || (ptr_baum->max_liste == leer))
		{
		  /*free(ptr_baum->liste);*/
		  ptr_baum->liste     = pptr_baum;
		  ptr_baum->max_liste = sequenz[i];
		}
	      
	      *(ptr_baum->liste + sequenz[i]) = neu_ptr_baum;
	      
	      ptr_baum = neu_ptr_baum;
	      
	    }
	  else /* there is a branch going deeper */
	    ptr_baum = *(ptr_baum->liste + sequenz[i]);
	  
	}
      
      /* insert new patch entries */
      
      if (vorfahre[1] == NIL) /* insert base polygons */
	{
	  /* give the new patch standard values */
	  
	  zaehler_memory_item++;
	  neu_ptr_item               = (Item *) memory_alloc(sizeof(Item)); /* take memory */
	  
	  neu_ptr_item->next         = NIL; 
	  neu_ptr_item->vorfahre_0   = vorfahre[0];
	  neu_ptr_item->vorfahre_1   = vorfahre[1];
	  neu_ptr_item->verklebung_0 = 0;
	  neu_ptr_item->verklebung_1 = 0;
	  neu_ptr_item->schnitt      = 0;
	  neu_ptr_item->rest         = 0;
	  
	  /*neu_ptr_item->innere_ecken = 0;*/
	  
	  /* testing only */
	  neu_ptr_item->wird_gebraucht = FALSE;
	  anz_patches++;
	  
	  ptr_baum->eintrag = neu_ptr_item;
	  
	}/* end if */
      else /* insert generated patches */
	{
	  if (verklebung[first] > s_laenge[first]) /* gluing part out of symmetry */
	    verkl_0 = 2; /* take the first entry */
	  else
	    verkl_0 = verklebung[first]; /* else ok */
	  
	  if (verklebung[second] > s_laenge[second])
	    verkl_1 = 2;
	  else
	    verkl_1 = verklebung[second];
	  
	  
	  ptr_item = ptr_baum->eintrag;
	  
	  if (ptr_item != NIL) /* search for the last entry */
	    while(ptr_item->next != NIL)
	      ptr_item = ptr_item->next;
	  
	  
	  vor_0 = vorfahre[first];
	  vor_1 = vorfahre[second];
	  
	  zaehler_memory_item++;
	  neu_ptr_item               = (Item *) memory_alloc(sizeof(Item));
	  neu_ptr_item->next         = NIL; 
	  neu_ptr_item->vorfahre_0   = vor_0;
	  neu_ptr_item->vorfahre_1   = vor_1;
	  neu_ptr_item->verklebung_0 = verkl_0 - 1;
	  neu_ptr_item->verklebung_1 = verkl_1 - 1;
	  
	  if (ausgabe == TRUE)
	    fprintf(out_stream, "\nEntry with verkl_0:%d and verkl_1:%d\n", neu_ptr_item->verklebung_0,  neu_ptr_item->verklebung_1);
	  
	  
	  if (typ == Durchschnitt) /* no embedding/inclusion */
	    {
	      neu_ptr_item->schnitt = schnitt;
	      neu_ptr_item->rest    = rand_ecken - neu_ptr_item->schnitt;
	      
	      neu_ptr_item->verklebung_0 = neu_ptr_item->verklebung_0 << 1; 
	      
	      /*neu_ptr_item->innere_ecken = vor_0->innere_ecken + vor_1->innere_ecken + neu_ptr_item->schnitt;*/
	      
	    }
	  else /* embedding/inclusion */
	    {
	      neu_ptr_item->schnitt      = rand_ecken;
	      neu_ptr_item->rest         = typ;
	      neu_ptr_item->verklebung_0 = (neu_ptr_item->verklebung_0 << 1) + 1;
	    }
	  
	  
	  vorfahren[0] = vorfahren[1] = 1;
	  
	  for (i = 0 ; i <= 1 ; i++)
	    {
	      vorfahre_dummy = vorfahre[i];
	      
	      if (vorfahre_dummy != NIL) /* count ancestors */
		while(vorfahre_dummy->next != NIL)
		  {
		    vorfahre_dummy = vorfahre_dummy->next;
		    vorfahren[i]++;
		  }
	    }
	    
	  zusaetzliche_patches = 0;
	  
	  if (vorfahre[0] == vorfahre[1])
	    for (i = 1 ; i <= vorfahren[0] ; i++)
	      zusaetzliche_patches = zusaetzliche_patches + i;
	  else
	    zusaetzliche_patches = vorfahren[0] * vorfahren[1];
	  
	  anz_patches = anz_patches + zusaetzliche_patches;
	  
	  if (sequenz[0] <= 3)
	    gute_patches = gute_patches + zusaetzliche_patches;
	  
	  if (ptr_baum->eintrag != NIL)
	    {
	      ptr_item->next = neu_ptr_item;
	      ptr_item       = ptr_item->next;
	    }
	  else
	    {
	      ptr_baum->eintrag = neu_ptr_item;
	      ptr_item          = ptr_baum->eintrag;
	    }
	  
	}/* end else */
      
      if (darstellung == TRUE)
	fprintf(stderr,"\r patches build allover: %d, good patches: %d", anz_patches, gute_patches);
    }

  return(bb_erzeugt);

}


/* *********************************************************************** */
/*                                                                         */
/* Initialisation of the sequence tree                                     */
/*                                                                         */
/* call: init_liste()                                                      */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */
void init_liste(void)
{
  int i,j;

  unsigned char max_weiter;
  
  KNOTENTYP verklebung[2];
  KNOTENTYP sequenz[max_ecken + 2];
  Item *vorfahre[2];
  Baum **pptr_baum, *ptr_baum;
  
  zaehler_memory_baum_knoten++;
  wurzel = (Baum *) memory_alloc(sizeof(Baum)); /* get memory */
  
  wurzel->eintrag   = NIL;
  wurzel->max_liste = anzahl_ecken - 2; /* more faces are never used in a patch */
    
  zaehler_memory_liste++;
  wurzel->liste = (Baum **) memory_alloc((anzahl_ecken - 2) * sizeof(*pptr_baum));

  vorfahre[1] = vorfahre[0] = NIL; /* no ancestors for base polygons */
  verklebung[0] = verklebung[1] = 0;
  
  for (i = 1 ; i <= flaechen[0] ; i++) /* insert starting faces into the tree */
    {
      for (j = 2 ; j <= flaechen[i] + 1 ; j++)
	sequenz[j] = 0;
      
      sequenz[1] = 1;
      
      sequenz[0] = flaechen[i] + 1;
      
      baum_eintrag(vorfahre, sequenz, flaechen[i],0,0,0,0,0,0,0); /* insert polygon */
    }

  if (no_bb == FALSE) /* only if BB are to be generated */
    {

      /* init tree for BB */
      
      zaehler_memory_baum_knoten++;
      bauchbinden_wurzel = (Baum *) memory_alloc(sizeof(Baum)); /* get memory */
      zaehler_memory_liste++;
      pptr_baum = (Baum **) memory_alloc((anzahl_ecken + 2 + 1) * sizeof(*pptr_baum)); /* get memory for the list of deeper going branches */
      
      bauchbinden_wurzel->liste     = pptr_baum;
      bauchbinden_wurzel->eintrag   = NIL;
      bauchbinden_wurzel->max_liste = anzahl_ecken + 2;
      
      max_weiter = anzahl_ecken + 2 + 1;
      
      for (i = 1 ; i <= anzahl_ecken + 2 + 1 ; i++)
	{
	  zaehler_memory_baum_knoten++;
	  ptr_baum = (Baum *) memory_alloc(sizeof(Baum));
	  
	  ptr_baum->max_liste = max_weiter;
	  ptr_baum->eintrag  = NIL;
	  
	  *(bauchbinden_wurzel->liste + i) = ptr_baum;
	  
	  zaehler_memory_liste++;
	  pptr_baum = (Baum **) memory_alloc((max_weiter + 1) * sizeof(*pptr_baum));
	  
	  ptr_baum->liste = pptr_baum;
	  
	}
    }
}

/* *********************************************************************** */
/*                                                                         */
/* test for the amount of faces enclosed in a patch                        */
/*                                                                         */
/* call:test_flaechen(patch)                                               */
/* return: anzahl flaechen                                                 */
/*                                                                         */
/* *********************************************************************** */
int test_flaechen(Item *ptr_item, int init)
{
  static int summe;

  if (init == 1)
    summe = 1;

  if (ptr_item->vorfahre_0 != NIL)
    {
      summe++;
      test_flaechen(ptr_item->vorfahre_0, 0);
      test_flaechen(ptr_item->vorfahre_1, 0);
    }

  return(summe);
}
	


/* *********************************************************************** */
/*                                                                         */
/* validity patch test                                                     */
/*                                                                         */
/* call:test_patches(wurzel)                                               */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */


void test_patches(Baum *ptr_baum)
{
  int i = 0, j;
  Item *ptr_item;

  static sequenz[max_ecken];
  
  if (ptr_baum->eintrag != NIL)
    {
      if ((ptr_baum->eintrag)->wird_gebraucht == TRUE)
	fprintf(out,"** ");
      else
	fprintf(out,"   ");

      fprintf(out,"%d:",sequenz[1]);
      for (j = 2; j <= sequenz[0] - 1; j++)
	fprintf(out,"%d,",sequenz[j]);

      fprintf(out,"%d",sequenz[sequenz[0]]);
      
      for (j = sequenz[0] ; j <= sequenz_tiefe + 2 ; j++)
	fprintf(out,"  ");

      j = 1;
      ptr_item = ptr_baum->eintrag;
      
      while(ptr_item->next != NIL)
	{
	  j++;
	  ptr_item = ptr_item->next;
	}
      
      fprintf(out," amount of patches: %d\n",j);
    }
  
  while((i <= ptr_baum->max_liste) && (ptr_baum->max_liste != leer))
    {
      if (sequenz[0] < sequenz_tiefe + 1)
	{
	  sequenz[0] = sequenz[0] + 1;
	  sequenz[sequenz[0]] = i;
	  
	  if (*(ptr_baum->liste + i) != NIL)
	    test_patches(*(ptr_baum->liste + i));
	  
	  sequenz[0] = sequenz[0] - 1;
	}
      
      i++;
      
    }
}


/* *********************************************************************** */
/*                                                                         */
/* small testing routine                                                   */
/*                                                                         */
/* call: init_test()                                                       */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */

void init_test(void)
{
  patchtest = fopen("PatchTest.ENU","w+");
  
  out = patchtest;
  
  fprintf(out, "\n4-regular graphs on %d vertices (Euler formula: %d faces)\n", anzahl_ecken, anzahl_ecken + 2);
  fprintf(out, "Testing will be made up to a sequence length of %d\n", sequenz_tiefe);
  fprintf(out, "\nExplanation: \n\n");

  fprintf(out, "The first number shows the amount of faces (as shown in the tree structure).\n");
  fprintf(out, "Next will be the sequence of the patches.\n");
  fprintf(out, "The Entry ends with the amount of patches with the given structure\n\n");
  fprintf(out, "When an entry is marked with ** it is used direct or indirect to build a graph\n\n");

  test_patches(wurzel);
  
  fclose(patchtest);
  
}

/* *********************************************************************** */
/*                                                                         */
/* small test routine for the BB                                           */
/*                                                                         */
/* call: test_bauchbinde()                                                 */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */
void test_bauchbinde(void)     
{
  Baum *ptr_baum;
  Item *ptr_item, *ptr_item_2;
  
  int i, j; 
  
  
  for (i = 4 ; i <= anzahl_ecken + 2 - 4 ; i++)
    {
      ptr_baum = *(bauchbinden_wurzel->liste + i);
      
      for (j = 1 ; j <= (anzahl_ecken / 2) ; j++)
	{
	  if (ptr_baum->liste != NIL)
	    {
	      if (*(ptr_baum->liste + j) != NIL)
		{
		  ptr_baum = *(ptr_baum->liste + j);
		  ptr_item = ptr_baum->eintrag;
		  
		  while (ptr_item != NIL)
		    {
		      ptr_item_2 = ptr_baum->eintrag;
		      
		      while(ptr_item_2 != NIL)
			{
			  if (ptr_item != ptr_item_2)
			    {
			      if ((ptr_item->vorfahre_0 == ptr_item_2->vorfahre_0) && (ptr_item->vorfahre_1 == ptr_item_2->vorfahre_1) 
				  && (ptr_item->verklebung_0 == ptr_item_2->verklebung_0) && (ptr_item->verklebung_1 == ptr_item_2->verklebung_1))
				{
				  fprintf(stderr,"Error");
				  exit(0);
				}
			    }
			  
			  ptr_item_2 = ptr_item_2->next;
			}
		      
		      ptr_item = ptr_item->next;
		    }
		}
	    }
	}
    }
}
										


/* *********************************************************************** */
/*                                                                         */
/* generation of the patches                                               */
/*                                                                         */
/* call: Create_Patches()                                                  */
/* retrun: void                                                            */
/*                                                                         */
/* *********************************************************************** */

void Create_Patches(void)
{
  time_t time1, time2;
  int i, j;
  
  if (darstellung == TRUE)  
    fprintf(stderr,"\n patches build allover: %d, good patches: %d", 0, 0);

  /* insert selected base polygons */
  init_liste();
  
  /* start generating */

  time(&time1);
  
  main_visit();

  time(&time2);

  fprintf(stderr,"\r patches build allover: %d, good patches: %d, BB-patches: %d", anz_patches, gute_patches, bb_patches);
  fprintf(stderr,"\n (used time: about %.0f seconds)", difftime(time2, time1));
  
  /* is the tree generated correctly? */

  /* calculation of even pathes and gluing of the patches */

  fprintf(stderr,"\n\n generating paths and gluing in patches...\n\n");

  if (darstellung == TRUE)  
    fprintf(stderr," number of SW: %d  number of BR: %d", 0, 0);

  time(&time1);

  suche_patch(); /* search for fitting patches and generation of the graphs */

  fprintf(stderr,"\r number of SW (#:%d): %d  number of BR (#:%d): %d", sandwich_pfade, sandwich_graphen, brillen_pfade, brillen_graphen);
  fprintf(stderr,"\n");

  if (no_bb == FALSE)
    baue_bauchbinde();
  else
    fprintf(stderr,"\n\n     Warning!! No BB types generated!!\n\n");
  
  if (no_br == TRUE)
    fprintf(stderr,"\n\n     Warning!! No BR types generated!!\n\n");
  
  if (no_sw == TRUE)
    fprintf(stderr,"\n\n     Warning!! No SW types generated!!\n\n");

  time(&time2);

  if (patch_test == TRUE)
    init_test();

  if (no_bb == FALSE)
    fprintf(stderr,"\r number of BB types (#:%d): %d", bauchbinden_pfade, bauchbinden_graphen);
  
  fprintf(stderr,"\n ----------------------------------------------------");
  fprintf(stderr,"\n allover: %d graphs", 
	 brillen_graphen + sandwich_graphen + bauchbinden_graphen);
  fprintf(stderr," (%d rejected:", verworfene_graphen);
  fprintf(stderr," %d because of double edges or loops)", doppel_kanten);
  fprintf(stderr,"\n\n %d graph(s) written", graphen_geschrieben);
  if (no_cut == TRUE)
    fprintf(stderr," (graphs with  1- or 2-cut had been rejected !!!!)"); 
  fprintf(stderr,"\n (time spent: about %.0f seconds)\n", difftime(time2, time1));

  
}
/* *********************************************************************** */
/*                                                                         */
/* write the encoding                                                      */
/*                                                                         */
/* call: schreibe_graph(Adresse des Arrays)                                */
/*                                                                         */
/*                                                                         */
/* return: void                                                            */
/*                                                                         */
/* *********************************************************************** */
void schreibe_graph(KNOTENTYP code[max_ecken + 2][4])
{
   int i, j;

   fputc(anzahl_ecken, graphs_file_ptr);
   
   for (i = 1 ; i <= anzahl_ecken ; i++)
     {
       for (j = 0 ; j <= 3 ; j++)
	 fputc(code[i][j], graphs_file_ptr);
       
       fputc(0, graphs_file_ptr);
     }
   
   graphen_geschrieben++;

}



/* *********************************************************************** */
/*                                                                         */
/* insert cut edge into a BB and calculate the new sequence                */
/*                                                                         */
/* call: make_schnittkante_b(... )                                         */
/*                                                                         */
/* return: address of the static array with both marked edges              */
/*                                                                         */
/* *********************************************************************** */

void make_schnittkante_b(Item *ptr_item, KANTE *rand_kante, KNOTENTYP *seq, KNOTENTYP sequenz_neu[2][max_ecken + 2],
			 KANTE **kante_0, KANTE **kante_1) 
{

  KNOTENTYP schnitt_ecke, sequenz_dummy[max_ecken + 2];
  
  Item *ptr_item_0, *ptr_item_1;
  
  int i, j, rest[2], schnitt, zaehler, index, verklebe_kante, verklebe_ecke, innere_verklebe_ecke, verkl[2], ziel;
  
  KANTE *start_kante, *klebe_kante, *invers, *links, *rechts;


  rest[0] = seq[1]; /* with the length of the surrounding border we can calculate the length of the new borders of
		       new two new patches. */

  verkl[0] = ptr_item->verklebung_0 >> 1; /* first bit marks if embedding/inclusion or not */
  verkl[1] = ptr_item->verklebung_1;
  
  if (ptr_item->verklebung_0 % 2 == 0)
    {
      schnitt = ptr_item->schnitt;

      /* run left and reconstruct the sequence. remember the starting edge for the next generation pass */ 
      
      rest[1] = rest[0] - ptr_item->rest;

      rest[0] = ptr_item->rest - 2;

      zaehler = 0; /* count the vertices found */

      start_kante = (rand_kante->hinten)->invers;
	  
      sequenz_dummy[0] = 0;

      /* earch for the new gluing edge */

      for (i = 1 ; i <= rest[0] ; i++) /* only V3 vertices on the border */
	{
	  start_kante = (start_kante->invers)->hinten; /* run straigth */
	  zaehler++;
	}

      sequenz_neu[0][0] = 2;

      schnitt_ecke = (rand_kante->hinten)->ziel;
      klebe_kante  = (rand_kante->hinten)->rechts;
      
      invers = (rand_kante->hinten)->rechts;
      links  = (rand_kante->hinten)->invers;
      rechts = ((rand_kante->hinten)->invers)->hinten;


      if (ptr_item->schnitt > 0)
	{
	  map[aktuelle_ecke + 1][0].ziel   = schnitt_ecke;
	  klebe_kante->ziel                = aktuelle_ecke + 1;
	  
	  map[aktuelle_ecke + 1][0].invers = invers;
	  klebe_kante->invers              = map[aktuelle_ecke + 1] + 0;
	  
	  map[aktuelle_ecke + 1][0].links  = links;
	  map[aktuelle_ecke + 1][0].rechts = rechts;
	  
	  klebe_kante->links               = map[aktuelle_ecke + 1] + 2;
	  klebe_kante->rechts              = map[aktuelle_ecke + 1] + 3;
	}
      else
	{
	  (start_kante->links)->ziel   = schnitt_ecke;
	  klebe_kante->ziel            = (start_kante->links)->ursprung;
	  
	  (start_kante->links)->invers = invers;
	  klebe_kante->invers          = start_kante->links;

	  (start_kante->links)->links  = links;
	  (start_kante->links)->rechts = rechts;
	  
	  klebe_kante->links           = (start_kante->invers)->hinten;
	  klebe_kante->rechts          = start_kante->invers;
	  
	}


      aktuelle_ecke = aktuelle_ecke + schnitt;
      
      if (ptr_item->schnitt > 0)
	{
	  
	  map[aktuelle_ecke][1].ziel   = start_kante->ziel;
	  (start_kante->links)->ziel   = aktuelle_ecke;
	  
	  map[aktuelle_ecke][1].invers = start_kante->links;
	  (start_kante->links)->invers = map[aktuelle_ecke] + 1;
	  
	  map[aktuelle_ecke][1].rechts = start_kante->invers;
	  map[aktuelle_ecke][1].links  = (start_kante->invers)->hinten;
	  
	  (start_kante->links)->links  = map[aktuelle_ecke] + 3;
	  (start_kante->links)->rechts = map[aktuelle_ecke] + 2;

	}


      if (verkl[0] == 1)
	{
	  *kante_0 = (rand_kante->hinten)->invers;
	  sequenz_neu[0][1] = schnitt;
	  sequenz_neu[0][2] = zaehler;
	}
      else
	{
	  *kante_0 = start_kante->links;
	  sequenz_neu[0][2] = schnitt;
	  sequenz_neu[0][1] = zaehler;
	}
      

      start_kante = (start_kante->invers)->hinten;
      
      sequenz_neu[1][0] = 2;

       if (verkl[1] == 1)
	{
	  *kante_1 = start_kante;
	  sequenz_neu[1][1] = schnitt;
	  sequenz_neu[1][2] = rest[1];
	}
      else
	{
	  *kante_1 = (rand_kante->hinten)->rechts;
	  sequenz_neu[1][2] = schnitt;
	  sequenz_neu[1][1] = rest[1];
	}
      
    }
  else /* embedding/incusion */
    {
      sequenz_neu[1][0] = 1; /* inner patch with only 1 V3 vertice */
      
      if (ptr_item->schnitt < ptr_item->rest)
	{
	  verklebe_kante       = 3;
	  schnitt              = ptr_item->rest;
	  innere_verklebe_ecke = aktuelle_ecke + 1 + ptr_item->schnitt;
	  sequenz_neu[1][1]    = ptr_item->rest - ptr_item->schnitt - 1;
	  
	  map[aktuelle_ecke + schnitt][1].links  = map[innere_verklebe_ecke] + 0;
	  map[aktuelle_ecke + schnitt][1].rechts = map[innere_verklebe_ecke] + 1;

	  *kante_1 = map[innere_verklebe_ecke] + 3;
	  
	}
      else
	{
	  verklebe_kante       = 2;
	  schnitt              = ptr_item->schnitt;
	  innere_verklebe_ecke = aktuelle_ecke + 1 + ptr_item->rest;
	  sequenz_neu[1][1]    = ptr_item->schnitt - ptr_item->rest - 1;
	  
	  map[aktuelle_ecke + schnitt][1].rechts = map[innere_verklebe_ecke] + 0;
	  map[aktuelle_ecke + schnitt][1].links  = map[innere_verklebe_ecke] + 1;

	  *kante_1 = map[innere_verklebe_ecke] + 1;

	}
      
      klebe_kante = (rand_kante->hinten)->rechts;
      ziel = (rand_kante->hinten)->ziel;
      links = (rand_kante->hinten)->invers;
      rechts = ((rand_kante->hinten)->invers)->hinten;
      invers = (rand_kante->hinten)->rechts;


      klebe_kante->ziel                = aktuelle_ecke + 1;
      map[aktuelle_ecke + 1][0].ziel   = ziel;

      klebe_kante->invers              = map[aktuelle_ecke + 1] + 0;
      map[aktuelle_ecke + 1][0].invers = invers;

      klebe_kante->links               = map[aktuelle_ecke + 1] + 2;
      map[aktuelle_ecke + 1][0].links  = links;
      
      klebe_kante->rechts              = map[aktuelle_ecke + 1] + 3;
      map[aktuelle_ecke + 1][0].rechts = rechts;

      aktuelle_ecke = aktuelle_ecke + schnitt;
      
      map[aktuelle_ecke][1].ziel                       = innere_verklebe_ecke;
      map[innere_verklebe_ecke][verklebe_kante].ziel   = aktuelle_ecke;
      
      map[aktuelle_ecke][1].invers                     = map[innere_verklebe_ecke] + verklebe_kante;
      map[innere_verklebe_ecke][verklebe_kante].invers = map[aktuelle_ecke] + 1;
      
      map[innere_verklebe_ecke][verklebe_kante].rechts = map[aktuelle_ecke] + 2;
      map[innere_verklebe_ecke][verklebe_kante].links  = map[aktuelle_ecke] + 3;
      

      sequenz_neu[0][0] = 3;
      
      sequenz_dummy[0] = 3;
      sequenz_dummy[1] = seq[1] - 1;
      sequenz_dummy[2] = ptr_item->schnitt;
      sequenz_dummy[3] = ptr_item->rest;

      
      for (i = verkl[0] , j = 2 ; j <= sequenz_neu[0][0] ; i++ , j++)
	{
	  sequenz_neu[0][i] = sequenz_dummy[j];
	  
	  if (i == sequenz_neu[0][0])
	    i = 0;
	}

      sequenz_neu[0][i] = sequenz_dummy[1];
      
      start_kante = rand_kante;
      
      for (i  = 1 ; i <= verkl[0] - 1; i++)
	{
	  while ((start_kante->links)->ziel >= 252)
	    start_kante = (start_kante->invers)->hinten;
	  
	  start_kante = start_kante->links;
	}
      
      if (verkl[0] - 1 < 1)
	*kante_0 = (rand_kante->hinten)->invers;
      else
	*kante_0 = start_kante;
      

    }

}



/* *********************************************************************** */
/*                                                                         */
/* Fuegt die Schnittkante ein und berechnet die neuen Sequenzen            */
/*                                                                         */
/* Aufruf: make_schnittkante(... )                                         */
/*                                                                         */
/* Ausgabe: Adresse des static Array mit den beiden neuen markierten       */
/*          Kanten                                                         */
/*                                                                         */
/* *********************************************************************** */

void make_schnittkante(Item *ptr_item, KANTE *rand_kante, KNOTENTYP *seq, KNOTENTYP sequenz_neu[2][max_ecken + 2],
		       KANTE **kante_0, KANTE **kante_1) 
{
  KNOTENTYP schnitt_ecke, sequenz_dummy[max_ecken + 2];
  
  Item *ptr_item_0, *ptr_item_1;
  
  int i, j, rest[2], schnitt, zaehler, index, verklebe_kante, verklebe_ecke, innere_verklebe_ecke, verkl[2], ziel;
  
  KANTE *start_kante, *klebe_kante, *invers, *links, *rechts;


  /* Bemerkung: der Bauchbinden Fall muss besonders behandelt werden, da die seine Randstruktur nicht mit der der
     beiden anderen Faelle uebereinstimmt (keine V-2-Ecke). Deshalb finden einige if-Abfragen statt, die aber
     zeitlich nicht ins Gewicht fallen. */



  rest[0] = 0; /* Randlaenge des uebergebenen Patches ermitteln. Daraus lassen sich dann die Randlaengen
		  der beiden neu entstehenden Patches errechnen.*/
  
  for (i = 0 ; i <= seq[0] ; i++)
    rest[0] = rest[0] + seq[i];

  verkl[0] = ptr_item->verklebung_0 >> 1; /* das erste Bit kodiert die Frage ob Einschluss oder nicht */
  verkl[1] = ptr_item->verklebung_1;
  
  if (ptr_item->verklebung_0 % 2 == 0) /* Durchschnitt, da BIT 0 nicht gesetzt*/
    {
      schnitt = ptr_item->schnitt;
      
      /* links herum laufen bis rest[0] == 0 und gleichzeitig neue Sequenz rekonstruieren und 
	 Startkante fuer den naechsten Erzeugungsdurchgang merken: */
      
      rest[1] = rest[0] - ptr_item->rest;

      rest[0] = ptr_item->rest - 3; /* "0" Kante, Start-Ecke und Verklebe-Kante werden nicht gezaehlt */


      zaehler = 0; /* hier werden die Ecken gezaehlt, auf denen entlanggelaufen wird.*/
      start_kante      = rand_kante;

      sequenz_dummy[0] = 0; /* Dummy-Array fuer die neue Sequenz des ersten Patches. */
      *kante_0          = NIL; /* die neue StartKante des ersten Patches. */
      
      for (i = 1 ; i <= rest[0] ; i++)
	{
	  /* Bemerkung: die "252" steht stellvertretend fuer die Werte fuer Aussen- und Innenkante,
	     welche groesser als 251 sind; hier fuehrt der Weg also nicht weiter. Evtl. sind
	     diese Werte einfach durch "0" zu ersetzen !! */
	  
	  if ((start_kante->links)->ziel < 252 ) /* Valenz-2-Ecke */
	    {
	      start_kante = start_kante->links;
	      
	      sequenz_dummy[0]++;
	      sequenz_dummy[sequenz_dummy[0]] = zaehler;
	      zaehler = 0;
	      

	      
	      /* wenn man beim Laufen am Rand entlang auf die Ecke mit der Marke des kleineren
		 Patches trifft, wird diese in kante_0 gesichert (die neue Start - Kante
		 natuerlich) */

	      
	      if (sequenz_dummy[0] == verkl[0] - 2)
		*kante_0 = start_kante;
	      
	      /* Erlaeuterung: verkl[0] enthaelt die Nummer der Kante des ersten Patches, an welcher
		 dieser verklebt wurde (Es wird bei 1 angefangen zu zaehlen!!). Es werden von der 
		 urspruenglichen Verklebekante 2 abgezogen, da die 0 Kante und die Verklebekante des 
		 neuen Patches beim linksherum laufen ja nicht mitgezaehlt werden. */
	      
	    }
	  else /* Valenz-3-Ecke */
	    {
	      start_kante = (start_kante->invers)->hinten; /* geradeaus laufen */
	      zaehler++;
	    }
	}
      
      /* die V2-Ecke an die der bis jetzt lose Schnitt verklebt wird, wird hier kuenstlich erzeugt: */
      
      sequenz_dummy[0]++;
      sequenz_dummy[sequenz_dummy[0]] = zaehler;
      
      if (sequenz_dummy[0] == verkl[0] - 2)
	*kante_0 = start_kante->links;
      
      /* der Rest der Sequenz ist durch die bekannte Laenge der Schnitt-Kante und die 0-Kante
	 bekannt: */
      
      sequenz_dummy[0]++;
      sequenz_dummy[sequenz_dummy[0]] = schnitt;
      sequenz_dummy[0]++;
      sequenz_dummy[sequenz_dummy[0]] = 0;
      
      sequenz_neu[0][0] = sequenz_dummy[0]; /* Sequenz-Laenge uebergeben. */
      
      /* Sequenz richtig ordnen: */
      
      for (i = verkl[0] , j = sequenz_dummy[0] - 1; j >= 1 ; i++ , j--) 
	{
	  sequenz_neu[0][i] = sequenz_dummy[j];
	  if (i == sequenz_dummy[0]) /* wenn letzte Stelle, beginne Vorne */
	    i = 0;
	}
      
      sequenz_neu[0][i] = 0;
      
      schnitt_ecke = ((rand_kante->invers)->rechts)->ziel;
      klebe_kante  = ((rand_kante->invers)->rechts)->rechts;
      
      invers = ((rand_kante->invers)->rechts)->rechts;
      links  = ((rand_kante->invers)->rechts)->invers;
      rechts = (((rand_kante->invers)->rechts)->invers)->hinten;
      

      /* Schnitt-Kante nach Marke einkleben ...*/
      
      if (ptr_item->schnitt > 0)
	{
	  map[aktuelle_ecke + 1][0].ziel   = schnitt_ecke;
	  klebe_kante->ziel                = aktuelle_ecke + 1;
	  
	  map[aktuelle_ecke + 1][0].invers = invers;
	  klebe_kante->invers              = map[aktuelle_ecke + 1] + 0;
	  
	  map[aktuelle_ecke + 1][0].links  = links;
	  map[aktuelle_ecke + 1][0].rechts = rechts;
	  
	  klebe_kante->links               = map[aktuelle_ecke + 1] + 2;
	  klebe_kante->rechts              = map[aktuelle_ecke + 1] + 3;
	}
      else /* hier wird der Schnitt gleich von zwei Seiten verklebt, da keine Ecke auf dem Schnitt existiert ... */
	{
	  (start_kante->links)->ziel   = schnitt_ecke;
	  klebe_kante->ziel            = (start_kante->links)->ursprung;
	  
	  (start_kante->links)->invers = invers;
	  klebe_kante->invers          = start_kante->links;

	  (start_kante->links)->links  = links;
	  (start_kante->links)->rechts = rechts;
	  
	  klebe_kante->links           = (start_kante->invers)->hinten;
	  klebe_kante->rechts          = start_kante->invers;
	  
	}


      /* offenes Ende des Schnittes verkleben: */
      
      aktuelle_ecke = aktuelle_ecke + schnitt;
      
      /* jetzt kann verklebt werden: */

      if (ptr_item->schnitt > 0) /* wenn keine Ecken auf dem Schnitt lagen, ist er schon verklebt... (s.o.) */
	{
	  
	  map[aktuelle_ecke][1].ziel   = start_kante->ziel;
	  (start_kante->links)->ziel   = aktuelle_ecke;
	  
	  map[aktuelle_ecke][1].invers = start_kante->links;
	  (start_kante->links)->invers = map[aktuelle_ecke] + 1;
	  
	  map[aktuelle_ecke][1].rechts = start_kante->invers;
	  map[aktuelle_ecke][1].links  = (start_kante->invers)->hinten;
	  
	  (start_kante->links)->links  = map[aktuelle_ecke] + 3;
	  (start_kante->links)->rechts = map[aktuelle_ecke] + 2;

	}

      if (*kante_0 == NIL)
	*kante_0 = ((rand_kante->invers)->rechts)->invers; /* falls noch keine Ecke mit der neuen Marke
							      gefunden ist, ist es diese! */
      
      /* links herum laufen bis rest[1] == 0 und gleichzeitig neue Sequenz rekonstruieren und 
	 Startkante fuer den naechsten Erzeugungsdurchgang merken: */
      
      start_kante = (start_kante->invers)->hinten;
      
      zaehler = 0;
      sequenz_dummy[0] = 0;
      *kante_1 = NIL;
      
      if (verkl[1] == 1) /* liegt hier die Marke ? */
	*kante_1 = start_kante;
      
      for (i = 1 ; i <= rest[1] ; i++)
	{
	  if ((start_kante->links)->ziel < 252) /* Valenz-2-Ecke */
	    {
	      start_kante = start_kante->links;
	      
	      sequenz_dummy[0]++;
	      sequenz_dummy[sequenz_dummy[0]] = zaehler;
	      zaehler = 0;
	      
	      /* wenn man beim Laufen am Rand entlang auf die Ecke mit der Marke des kleineren
		 Patches trifft, wird diese in kante_1 gesichert (die neue Start - Kante
		 natuerlich) */
	      
	      if (sequenz_dummy[0] == verkl[1] - 1)
		*kante_1 = start_kante;
	    }
	  else /* Valenz-3-Ecke */
	    {
	      zaehler++;
	      start_kante = (start_kante->invers)->hinten;
	    }
	}
      
      /* der Rest der Sequenz ist wieder bekannt: */
      
      sequenz_dummy[0]++;
      sequenz_dummy[sequenz_dummy[0]] = zaehler;
      
      if (sequenz_dummy[0] == verkl[1] - 1) /* liegt hier die Marke? */
	*kante_1 = start_kante->links;
      
      sequenz_dummy[0]++;
      sequenz_dummy[sequenz_dummy[0]] = schnitt;
      
      sequenz_neu[1][0] = sequenz_dummy[0];
          
      /* Sequenz richtig ordnen: */

      for (i = verkl[1] , j = sequenz_dummy[0]; j >= 1 ; i++ , j--) 
	{
	  sequenz_neu[1][i] = sequenz_dummy[j];
	  if (i == sequenz_dummy[0]) /* wenn letzte Stelle, beginne Vorne */
	    i = 0;
	}
      
    }
  else /* Einschluss */
    {
      sequenz_neu[1][0] = 1; /* innerer Patch hat nur eine V-3-Ecke */
      
      /* In "schnitt" und "rest" werden die erste und die zweite Kante gespeichert, die nach der
	 der Verklebung im Inneren des neuen Patches liegen. Daher kann durch eine "<" ">" Abfrage
	 geklaert werden, ob das Ende des Schnittes links herum oder rechts herum verklebt werden
	 muss: */
      
      
      if (ptr_item->schnitt < ptr_item->rest) /* an welcher Kante wird der Einschluss verklebt ? */
	{
	  verklebe_kante       = 3; /* 1. Verklebe-Kante < 2. V.-Kante also links herum verkleben */
	  schnitt              = ptr_item->rest;
	  innere_verklebe_ecke = aktuelle_ecke + 1 + ptr_item->schnitt;
	  sequenz_neu[1][1]    = ptr_item->rest - ptr_item->schnitt - 1;
	  
	  map[aktuelle_ecke + schnitt][1].links  = map[innere_verklebe_ecke] + 0;
	  map[aktuelle_ecke + schnitt][1].rechts = map[innere_verklebe_ecke] + 1;

	  /* die Kante fuer den inneren Patch steht schon fest: */

	  *kante_1 = map[innere_verklebe_ecke] + 3;
	  
	}
      else
	{
	  verklebe_kante       = 2; /* ... und umgekehrt ... */
	  schnitt              = ptr_item->schnitt;
	  innere_verklebe_ecke = aktuelle_ecke + 1 + ptr_item->rest;
	  sequenz_neu[1][1]    = ptr_item->schnitt - ptr_item->rest - 1;
	  
	  map[aktuelle_ecke + schnitt][1].rechts = map[innere_verklebe_ecke] + 0;
	  map[aktuelle_ecke + schnitt][1].links  = map[innere_verklebe_ecke] + 1;

	  *kante_1 = map[innere_verklebe_ecke] + 1;

	}
      
      /* Schnittkante am Rand "befestigen": */

      klebe_kante = ((rand_kante->invers)->rechts)->rechts;
      ziel = ((rand_kante->invers)->rechts)->ziel;
      links = ((rand_kante->invers)->rechts)->invers;
      rechts = (((rand_kante->invers)->rechts)->invers)->hinten;
      invers = ((rand_kante->invers)->rechts)->rechts;
	  

      klebe_kante->ziel                = aktuelle_ecke + 1;
      map[aktuelle_ecke + 1][0].ziel   = ziel;

      klebe_kante->invers              = map[aktuelle_ecke + 1] + 0;
      map[aktuelle_ecke + 1][0].invers = invers;

      klebe_kante->links               = map[aktuelle_ecke + 1] + 2;
      map[aktuelle_ecke + 1][0].links  = links;
      
      klebe_kante->rechts              = map[aktuelle_ecke + 1] + 3;
      map[aktuelle_ecke + 1][0].rechts = rechts;

      /* offenes Ende des Schnittes verkleben: */
      
      aktuelle_ecke = aktuelle_ecke + schnitt;
      
      map[aktuelle_ecke][1].ziel                       = innere_verklebe_ecke;
      map[innere_verklebe_ecke][verklebe_kante].ziel   = aktuelle_ecke;
      
      map[aktuelle_ecke][1].invers                     = map[innere_verklebe_ecke] + verklebe_kante;
      map[innere_verklebe_ecke][verklebe_kante].invers = map[aktuelle_ecke] + 1;
      
      map[innere_verklebe_ecke][verklebe_kante].rechts = map[aktuelle_ecke] + 2;
      map[innere_verklebe_ecke][verklebe_kante].links  = map[aktuelle_ecke] + 3;
      

      /* Erstellen der neuen Sequenz fuer den 2. Vorgaenger-Patch (aeusserer Patch): */
      
      /* Sequenz rekonstruieren ... */

      sequenz_neu[0][0] = seq[0] + 3; /* drei V-2-Ecken kommen hinzu */
      
      sequenz_dummy[1] = 0;
      sequenz_dummy[2] = ptr_item->schnitt;
      sequenz_dummy[3] = ptr_item->rest;
      sequenz_dummy[4] = seq[1] - 1;
      
      for (i = 2 ; i <= seq[0] ; i++)
	sequenz_dummy[i + 3] = seq[i];
      
      
      /* Sequenz richtig ordnen: */

      for (i = verkl[0] , j = 2 ; j <= sequenz_neu[0][0] ; i++ , j++)
	{
	  sequenz_neu[0][i] = sequenz_dummy[j];
	  
	  if (i == sequenz_neu[0][0])
	    i = 0;
	}

      sequenz_neu[0][i] = sequenz_dummy[1];
      
      /* die neue zu uebergebende Kante ermitteln: */
      
      start_kante = rand_kante;

      for (i  = 1 ; i <= verkl[0] - 2; i++) /* die 0 Kante und die Kante mir der Marke werden nicht durchlaufen, also: - 2 */
	{
	  while ((start_kante->links)->ziel >= 252) /* Valenz 3 */
	    start_kante = (start_kante->invers)->hinten;
	  
	  start_kante = start_kante->links;
	}
      
      if (verkl[0] - 2 < 1)
	*kante_0 = ((rand_kante->invers)->rechts)->invers;
      else
	*kante_0 = start_kante;

    } /* Ende des Einschluss-Falles */

}

/* *********************************************************************** */
/*                                                                         */
/* Kodiert die Patches des ermittelten Graphen auf minimale planare Weise  */
/*                                                                         */
/* Aufruf: kodiere_planar_main(Nr in der MAP Struktur, Nr in der kodierten */
/*                             Struktur, 1 = init & 0 = weiter)            */
/*                                                                         */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */

unsigned char kodiere_planar_main(KANTE *start_kante, KNOTENTYP planar_code[max_ecken + 2][4], KNOTENTYP vergleichs_code[max_ecken + 2][4], 
				  unsigned char test, unsigned char spiegel)
{

  /* Bemerkung: Wenn die Variable test TRUE ist, so erwartet die Routine Daten im Array vergleichs_code. Dann wird der neue code mit dem alten code sofort
     verglichen und bei Abweichung des neuen codes nach unten wird Null, bei Abweichung nach oben 1 zurueckgegeben. */
  /* wenn die Variable spiegel TRUE ist, so wird der Graph mit den spiegelverkehrten Startkanten durchlaufen und auch gegen den Uhrzeigersinn. */

  /* returncode 0 fuehrt zu einer Verwerfung des urspruenglichen Graphen, Returncode 1 behaelt ihn. Returncode 2 bezeichnet Gleichheit. */

  KNOTENTYP i, j, k;
  KNOTENTYP code_nr;
  KNOTENTYP kleinstes_element;

  signed char korrektur = 1;

  KANTE *kanten_reihenfolge[8];
  KANTE *liste[max_ecken + 1];


  if (no_planar == TRUE)
    return(1);

  if ((no_spiegel == TRUE) && (spiegel == TRUE))
    return(1);

  code_nr = 1;

  for (i = 1 ; i <= anzahl_ecken ; i++)
    map[i][0].bezeichnung = max_ecken + 2; /* alte Kodierung loeschen */
  
  if (spiegel == TRUE)
    {
      if (pfad_typ == BAUCHBINDE)
	start_kante = start_kante->hinten;
      else
	start_kante = (start_kante->invers)->rechts;

      korrektur = -1;
    }

  map[start_kante->ziel][0].bezeichnung = 1;
  liste[1] = start_kante;

  for (i = 1 ; i <= anzahl_ecken ; i++)
    {
      start_kante = liste[i];

      kanten_reihenfolge[0] = kanten_reihenfolge[4] = (start_kante->invers)->hinten;
      kanten_reihenfolge[1] = kanten_reihenfolge[5] = start_kante->rechts;
      kanten_reihenfolge[2] = kanten_reihenfolge[6] = start_kante->invers;
      kanten_reihenfolge[3] = kanten_reihenfolge[7] = start_kante->links;

      kleinstes_element = 0;

      for (j = 1 ; j <= 3 ; j++)
	if (map[kanten_reihenfolge[j]->ziel][0].bezeichnung < map[kanten_reihenfolge[kleinstes_element]->ziel][0].bezeichnung)
	  kleinstes_element = j;
      
      if (spiegel == TRUE)
	kleinstes_element = kleinstes_element + 4;

      for (j = 0 ; j <= 3 ; j++)
	{
	  if (map[kanten_reihenfolge[kleinstes_element + (j * korrektur)]->ziel][0].bezeichnung == max_ecken + 2)
	    {
	      code_nr++;
	      map[kanten_reihenfolge[kleinstes_element + (j * korrektur)]->ziel][0].bezeichnung = code_nr;
	      
	      liste[code_nr] = kanten_reihenfolge[kleinstes_element + (j * korrektur)];
	      
	    }
	  
	  planar_code[i][j] = map[kanten_reihenfolge[kleinstes_element + (j * korrektur)]->ziel][0].bezeichnung;
	  
	  if (test == TRUE)
	    {
	      if (planar_code[i][j] < vergleichs_code[i][j])
		return(0);
	      if (planar_code[i][j] > vergleichs_code[i][j])
		return(1);
	    }
	}
    }
  
  return(2);
  
}
/* *********************************************************************** */
/*                                                                         */
/* Prueft, ob der Graph Doppelkanten oder Schlingen enthaelt               */
/*                                                                         */
/* Aufruf: doppelkanten()                                                  */
/*                                                                         */
/* Ausgabe: TRUE oder FALSE                                                */
/*                                                                         */
/* *********************************************************************** */
unsigned char doppelkanten(void)
{
  KNOTENTYP i, j, k;
  
  for (i = 1 ; i <= anzahl_ecken ; i++)
    for (j = 0 ; j <= 3 ; j++)
      if (map[i][j].ziel == map[i][j].ursprung)
	return(TRUE);

  for (i = 1 ; i <= anzahl_ecken ; i++)
    for (j = 0 ; j <= 2 ; j++)
      for (k = j + 1 ; k <= 3 ; k++)
	if (map[i][j].ziel == map[i][k].ziel)
	  return(TRUE);

  return(FALSE);
}

/* *********************************************************************** */
/*                                                                         */
/* Zaehlt die Ecken in einem gefundenen Pfad und kann so die Flaechen      */
/* berechnen                                                               */
/*                                                                         */
/* Aufruf: zaehle_ecken(Startkante)                                        */
/*                                                                         */
/* Ausgabe:                                                                */
/*                                                                         */
/* *********************************************************************** */
KNOTENTYP zaehle_ecken(KANTE *start_kante,  KNOTENTYP pfad_nummern[max_ecken + 2], unsigned char init)
{
  KANTE *lauf_kante;
  static unsigned char ecken;
  static unsigned char ende;
  static gezaehlt[max_ecken + 2];
  int i;

  lauf_kante = start_kante;

  if (init == TRUE)
    {
      ecken = 0;
      ende = start_kante->ursprung;
      for (i = 1 ; i <= anzahl_ecken ; i++)
	gezaehlt[i] = 0;
      
      while (lauf_kante->ziel != ende)
	{
	  lauf_kante = lauf_kante->links;
	  if ((pfad_nummern[lauf_kante->ziel] == 0) && (gezaehlt[lauf_kante->ziel] == 0))
	    zaehle_ecken(lauf_kante, pfad_nummern, FALSE);
	  start_kante = start_kante->invers->hinten;
	  lauf_kante = start_kante;
	}
      
      return(ecken);
      
    }
  else
    {
      ecken++;
      gezaehlt[lauf_kante->ziel] = 1;
      
      if ((pfad_nummern[lauf_kante->links->ziel] == 0) && (gezaehlt[lauf_kante->links->ziel] == 0))
	zaehle_ecken(lauf_kante->links, pfad_nummern, FALSE);
      if ((pfad_nummern[lauf_kante->invers->hinten->ziel] == 0) && (gezaehlt[lauf_kante->invers->hinten->ziel] == 0))
	zaehle_ecken(lauf_kante->invers->hinten, pfad_nummern, FALSE);
      if ((pfad_nummern[lauf_kante->rechts->ziel] == 0) && (gezaehlt[lauf_kante->rechts->ziel] == 0))
	zaehle_ecken(lauf_kante->rechts, pfad_nummern, FALSE);
    }
}

/* *********************************************************************** */
/*                                                                         */
/* Prueft, ob der Graph gespeichert werden muss oder nicht                 */
/*                                                                         */
/* Aufruf: iso_test(zaehler fuer die moeglichen Startkanten)               */
/*                                                                         */
/* Ausgabe:                                                                */
/*                                                                         */
/* *********************************************************************** */
void iso_test(void)
{

  KNOTENTYP i, j, k, l, m, n;
  KANTE *start_kante, *start_kante_2, *merke_start_kante[2], *kodier_kante_main[2];
  KANTE *kodier_kanten[2]; /* Wenn die Patch-Teile des Graphen gleich sind, werden sie anhand dieser Kanten minimal planar kodiert... */
  KNOTENTYP zaehler, merke_zaehler, grenze[2], pfad_laenge_alt[2], flaechen_alt[2];
  KNOTENTYP gesamt_pfad_laenge, ende, flaechen_pfad[2], ecken_pfad[2];
  KNOTENTYP flaechen_zaehler;

  int pfad_laenge[2], pfad_laenge_sort[2], flaechen_sort;

  KNOTENTYP pfad_nummern[max_ecken + 2]; /* dieses Array beinhaltet die Nummern des erzeugten Pfades. 
					    Bsp.: Wenn Ecke 1 als 10. Ecke durchlaufen wurde steht im Array an Stelle 1 eine 10. */

  unsigned char oben, unten, main_kodiert;

  KNOTENTYP pl_code_main[2][max_ecken + 2][4], pl_code_sub[2][max_ecken +2][4];

  /* Bemerkung: die Pfad-Typen werden gemaess folgender Liste in ihrer Prioritaet eingestuft:
     
     1.) Sandwich
     2.) Brille
     3.) Bauchbinde */
  
  /* der Isomorphietest beruht auf der Konstruktion der Graphen. Da diese aus allen denkbaren Pfaden erzeugt werden, die in ihnen verlaufen,
     werden die Graphen aus Pfaden mit hoeherer Prioritaet noch erzeugt bzw. wurden schon erzeugt. Findet man also einen Pfad in dem gerade 
     erzeugten Graphen mit hoeherer Prioritaet als der erzeugende Pfad, so ist der gerade erzeugte Graph zu verwerfen.
     Bei Gleichheit der Prioritaet muessen die Patchteile kodiert werden. Der Graph mit der minimalen Kodierung bleibt, der andere wird geloescht. */


  pfad_laenge_alt[0] = patch_daten.Rand[0];
  pfad_laenge_alt[1] = patch_daten.Rand[1];
  flaechen_alt[0] = patch_daten.Flaechen[0];
  flaechen_alt[1] = patch_daten.Flaechen[1];
  kodier_kante_main[0] = patch_daten.Kanten[0];
  kodier_kante_main[1] = patch_daten.Kanten[1];
  
  main_kodiert = FALSE;

  for (i = 1 ; i <= anzahl_ecken ; i++)
    {
      for (l = 0 ; l <= 3 ; l++)
	{
	  start_kante = map[i] + l;
	  
	  if (start_kante->pfad_start == FALSE)
	    /* hier kann der Pfad, aus dem der Graph urspruenglich erzeugt wurde, nicht gestartet werden.
	       Also kann ab hier ein neuer Pfad entstehen. */
	    {
	      for (j = 0 ; j <= anzahl_ecken + 1 ; j++) /* Felder auf 0 setzen (Pfade loeschen) */
		{
		  pfad_nummern[j] = 0;
		  for (m = 0 ; m <= 3 ; m++)
		    map[j][m].faerbung = FALSE;
		}

	      pfad_nummern[start_kante->ursprung] = 1;
	      zaehler = 1;
	      
	      while (pfad_nummern[start_kante->ziel] == 0) /* ERSTEN TEILPFAD SUCHEN... */
		{
		  zaehler++; /* Ecke zaehlen */
		  pfad_nummern[start_kante->ziel] = zaehler;
		  start_kante->faerbung = TRUE; /* Kante faerben */
		  (start_kante->invers)->faerbung = TRUE;
		  
		  start_kante = (start_kante->invers)->hinten;
		}

	      start_kante->faerbung = TRUE;
	      (start_kante->invers)->faerbung = TRUE;
	
	      merke_zaehler = zaehler; /* dient u.a. zum Berechnen des Zwischenstueckes */
	      merke_start_kante[0] = start_kante;
	      
	      /* ************************* Bauchbinden Typ gefunden ************************ */

	      if ((pfad_nummern[start_kante->ziel] == 1) && (pfad_nummern[((start_kante->invers)->hinten)->ziel] == 2))
		{
		  if (pfad_typ == BAUCHBINDE)
		    {
		      pfad_laenge[0] = pfad_laenge[1] = zaehler;

		      kodier_kanten[0] = map[i] + l;
		      kodier_kanten[1] = kodier_kanten[0]->hinten;

		      /*if ((pfad_laenge_alt[0] > pfad_laenge[0]) && (pfad_laenge[0] > 4)) 
			{
			  verworfene_graphen++;
			  return;
			}*/
		      
		      /*if ((pfad_laenge_alt[0] == pfad_laenge[0]) || (pfad_laenge_alt[0] == 4 ))
			{*/
			  ecken_pfad[0] = zaehle_ecken(kodier_kanten[0], pfad_nummern, TRUE);
			  ecken_pfad[1] = zaehle_ecken(kodier_kanten[1], pfad_nummern, TRUE);
			  
			  flaechen_pfad[0] = ((((3 * pfad_laenge[0]) + (4 * ecken_pfad[0])) / 2) - (pfad_laenge[0] + ecken_pfad[0])) + 1; /* nach Euler und ohne */
			  flaechen_pfad[1] = ((((3 * pfad_laenge[1]) + (4 * ecken_pfad[1])) / 2) - (pfad_laenge[1] + ecken_pfad[1])) + 1; /* aeussere Flaeche */
			  
			  if (flaechen_pfad[0] >= flaechen_pfad[1]) /* erster Patch hat mehr Flaechen */
			    {
			      flaechen_sort = 0;

			      if (flaechen_pfad[0] < flaechen_alt[0])
				{
				  verworfene_graphen++;
				  return;
				}
			    }
			  else if (flaechen_pfad[1] > flaechen_pfad[0]) /* zweiter Patch hat mehr Flaechen */
			    {
			      flaechen_sort = 1;

			      if (flaechen_pfad[1] < flaechen_alt[0])
				{
				  verworfene_graphen++;
				  return;
				}
			    }
			  
			  if (flaechen_alt[0] == flaechen_pfad[flaechen_sort]) /* hier folgt der Check nach PLANAR_CODE */
			    {
			      if (main_kodiert == FALSE)
				{
				  kodiere_planar_main(kodier_kante_main[0], pl_code_main[0], pl_code_sub[0], FALSE, FALSE);
				  
				  for (j = 1 ; j <= pfad_laenge_alt[0] ; j++)
				    {
				      kodier_kante_main[0] = (kodier_kante_main[0]->invers)->hinten;
				      
				      if (kodiere_planar_main(kodier_kante_main[0], pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}

				      if (kodiere_planar_main(kodier_kante_main[0], pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}

				      if (flaechen_alt[0] == flaechen_alt[1])
					{
					  if (kodiere_planar_main(kodier_kante_main[0]->invers, pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
					    {
					      verworfene_graphen++;
					      return;
					    }
					  
					  if (kodiere_planar_main(kodier_kante_main[0]->invers, pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
					    {
					      verworfene_graphen++;
					      return;
					    }
					}
				    }
				  
				  main_kodiert = TRUE;
				}
			      
			      if (flaechen_pfad[0] > flaechen_pfad[1])
				{
				  if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				  
				  if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				  
				}
			      else if (flaechen_pfad[0] < flaechen_pfad[1])
				{
  				  if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				  
				  if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			      else
				{
				  if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }

				  if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }

				  if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }

				  if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				  /*}*/
			    }
			}
		    }
		}
	      else /* entweder Sandwich oder Brille */
		{
		  if (pfad_typ == BAUCHBINDE) /* BAUCHBINDE ist schlechter als BRILLE und SANDWICH */
		    {
		      verworfene_graphen++;
		      return;
		    }

		  grenze[0] = pfad_nummern[merke_start_kante[0]->ziel]; /* Um feststellen zu koennen, ob es sich um einen Sandwich- oder Brillen-Pfad handelt, */
		  grenze[1] = pfad_nummern[merke_start_kante[0]->ursprung]; /* dienen diese Variablen. */
		  pfad_laenge[0] = grenze[1] - grenze[0]; /* die Laenge des ersten Einschlusses (muss nicht unbedingt der erste Patch sein) */
		  
		  if (pfad_nummern[(merke_start_kante[0]->links)->ziel] > pfad_nummern[(merke_start_kante[0]->rechts)->ziel])
		    {
		      if ((pfad_nummern[merke_start_kante[0]->ziel] == 1) && (pfad_nummern[(merke_start_kante[0]->rechts)->ziel] != 0))
			{
			  unten = TRUE;
			  oben = FALSE;
			  kodier_kanten[0] = merke_start_kante[0]->invers;
			}
		      else
			{
			  oben = TRUE;
			  unten = FALSE;
			  kodier_kanten[0] = merke_start_kante[0]->links;
			}
		    }
		  else
		    {
		      if ((pfad_nummern[merke_start_kante[0]->ziel] == 1) && (pfad_nummern[(merke_start_kante[0]->links)->ziel] != 0))
			{
			  oben = TRUE;
			  unten = FALSE;
			  kodier_kanten[0] = merke_start_kante[0]->links;
			}
		      else
			{
			  unten = TRUE;
			  oben = FALSE;
			  kodier_kanten[0] = merke_start_kante[0]->invers;
			}
		    }

		  start_kante = (map[i] + l)->hinten; /* in die entgegengesetzte Richtung laufen */
		  
		  while (pfad_nummern[start_kante->ziel] == 0) /* zweiten Teilpfad suchen ...*/
		    {
		      zaehler++; /* Ecke zaehlen */
		      pfad_nummern[start_kante->ziel] = zaehler;
		      start_kante->faerbung = TRUE; /* Kante faerben */
		      (start_kante->invers)->faerbung = TRUE;
		      
		      start_kante = (start_kante->invers)->hinten;
		    }
		  
		  start_kante->faerbung = TRUE;
		  (start_kante->invers)->faerbung = TRUE;
		  
		  merke_start_kante[1] = start_kante;
		  gesamt_pfad_laenge = zaehler;
		  
		  /* ************************* Sandwich Typ gefunden ************************ */
		  
		  if ((grenze[0] < pfad_nummern[start_kante->ziel]) && (pfad_nummern[start_kante->ziel] <= grenze[1])) 
		    {
		      if (pfad_typ == BRILLE) /* SANDWICH ist besser als BRILLE */
			{
			  verworfene_graphen++;
			  return;
			}
		      
		      if (oben == TRUE)
			kodier_kanten[1] = merke_start_kante[1]->links;
		      else
			kodier_kanten[1] = merke_start_kante[1]->invers;
		      
		      ende = kodier_kanten[1]->ursprung;
		      start_kante = kodier_kanten[1];
		      
		      pfad_laenge[1] = 0;
		      
		      while (start_kante->ziel != ende)
			{
			  pfad_laenge[1]++;
			  start_kante = (start_kante->invers)->hinten;
			}

		      if (pfad_laenge[0] > pfad_laenge[1])
			{
			  pfad_laenge_sort[0] = pfad_laenge[0];
			  pfad_laenge_sort[1] = pfad_laenge[1];

			  if (pfad_laenge_alt[0] > pfad_laenge[0])
			    {
			      verworfene_graphen++;
			      return;
			    }
			  else if (pfad_laenge_alt[0] == pfad_laenge[0])
			    {
			      if (pfad_laenge_alt[1] > pfad_laenge[1])
				{
				  verworfene_graphen++;
				  return;
				}
			    }    
			}
		      else if (pfad_laenge[1] > pfad_laenge[0])
			{
			  pfad_laenge_sort[0] = pfad_laenge[1];
			  pfad_laenge_sort[1] = pfad_laenge[0];

			  if (pfad_laenge_alt[0] > pfad_laenge[1])
			    {
			      verworfene_graphen++;
			      return;
			    }
			  else if (pfad_laenge_alt[0] == pfad_laenge[1])
			    {
			      if (pfad_laenge_alt[1] > pfad_laenge[0])
				{
				  verworfene_graphen++;
				  return;
				}
			    }
			}
		      else if (pfad_laenge[1] == pfad_laenge[0])
			{
			  pfad_laenge_sort[0] = pfad_laenge[0];
			  pfad_laenge_sort[1] = pfad_laenge[1];

			  if (pfad_laenge_alt[0] > pfad_laenge[0])
			    {
			      verworfene_graphen++;
			      return;
			    }
			  else if (pfad_laenge_alt[0] == pfad_laenge[0])
			    {
			      if (pfad_laenge_alt[1] > pfad_laenge[0])
				{
				  verworfene_graphen++;
				  return;
				}
			    }
			}
		      
		      
		      if ((pfad_laenge_alt[0] == pfad_laenge_sort[0]) && (pfad_laenge_alt[1] == pfad_laenge_sort[1]))
			{
			  if (main_kodiert == FALSE)
			    {
			      kodiere_planar_main(kodier_kante_main[0], pl_code_main[0], pl_code_sub[0], FALSE, FALSE);
			      
			      if (pfad_laenge_alt[0] == pfad_laenge_alt[1])
				{
				  if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			      main_kodiert = TRUE;
			    }
			  
			  if ((kodier_kanten[0]->hinten)->faerbung == FALSE) /* wenn die Kante hinter der Kodierkante nicht gefaerbt ist, so handelt es sich um einen
										bewertbaren Pfad, ansonsten um ein nicht gebautes Spiegelbild!!! */
			    {
			      
			      if (pfad_laenge[0] > pfad_laenge[1]) /* Min. und Max. existieren eindeutig. Daher kann festgelegt werden, 
								      welcher Patch der erste und welcher der zweite ist. */
				{
				  if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			      else if (pfad_laenge[0] < pfad_laenge[1])
				{
				  if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, FALSE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			      else  /* die beiden neuen Pfadlaengen sind gleich. Es laesst sich also kein Min. oder Max. bilden. 
				       Daher muessen beide Pfade als Pfad 1 bzw Pfad 2 angenommen werden. */
				{
				  if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    } 
				  
				  if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, FALSE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			    } /* if ((kodier_kanten[0]->hinten)->faerbung == FALSE) */
			  else
			    {
			      /* Spiegel des (nichtgebauten) Spiegels ueberpruefen ... */
			      
			      if (pfad_laenge[0] > pfad_laenge[1]) /* Min. und Max. existieren eindeutig. Daher kann festgelegt werden, 
								      welcher Patch der erste und welcher der zweite ist. */
				{
				  if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			      else if (pfad_laenge[0] < pfad_laenge[1])
				{
				  if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, TRUE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			      else  /* die beiden neuen Pfadlaengen sind gleich. Es laesst sich also kein Min. oder Max. bilden. 
				       Daher muessen beide Pfade als Pfad 1 bzw Pfad 2 angenommen werden. */
				{
				  if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    } 
				  
				  if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, TRUE) == 0)
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			    }
			}
		    }
		  else /* ************************* Brillen Typ gefunden ************************ */
		    {
		      if (pfad_typ == BRILLE)
			{
			  if (pfad_nummern[merke_start_kante[1]->ziel] == pfad_nummern[merke_start_kante[0]->ziel]) /* die Ziel-Ecken der beiden Durch-
														       laeufe stimmen ueberein!! */
			    {
			      if (oben == TRUE)
				kodier_kanten[1] = start_kante->links;
			      else
				kodier_kanten[1] = start_kante->invers;
			      
			      pfad_laenge[1] = zaehler - merke_zaehler + pfad_nummern[merke_start_kante[1]->ziel] - 1;
			    }
			  else if (pfad_nummern[start_kante->ziel] <= merke_zaehler) /* der Pfad trifft auf eine Ecke, die schon in Durchgang 1 durchlaufen wurde. */
			    {			    
			      if (zaehler == merke_zaehler) /* im zweiten Durchgang wurde keine weitere Ecke durchlaufen */
				pfad_laenge[1] = pfad_nummern[start_kante->ziel] - pfad_nummern[start_kante->ursprung];
			      else
				pfad_laenge[1] = pfad_nummern[start_kante->ursprung] - (merke_zaehler + 1) + pfad_nummern[start_kante->ziel];
			      
			      if (pfad_nummern[start_kante->ziel] == 1) /* der 2. Teilpfad trifft Ecke 1 aus Durchgang 1: Spezialfall!! */
				{
				  if (pfad_nummern[(start_kante->rechts)->ziel] == 2)
				    kodier_kanten[1] = start_kante->links;
				  else
				    kodier_kanten[1] = start_kante->invers;
				}
			      else
				{
				  if (pfad_nummern[(start_kante->links)->ziel] < pfad_nummern[(start_kante->rechts)->ziel])
				    kodier_kanten[1] = start_kante->links;
				  else
				    kodier_kanten[1] = start_kante->invers;
				}
			    }
			  else /* der Pfad trifft auf eine Ecke, die erst im 2. Durchgang benutzt wurde. */
			    {
			      pfad_laenge[1] = pfad_nummern[start_kante->ursprung] - pfad_nummern[start_kante->ziel];
			      
			      if (pfad_nummern[(start_kante->links)->ziel] > pfad_nummern[(start_kante->rechts)->ziel])
				kodier_kanten[1] = start_kante->links;
			      else
				kodier_kanten[1] = start_kante->invers; 
			    }
			  
			  
			  if (pfad_laenge[0] > pfad_laenge[1])
			    {
			      pfad_laenge_sort[0] = pfad_laenge[0];
			      pfad_laenge_sort[1] = pfad_laenge[1];
			      
			      if (pfad_laenge_alt[0] > pfad_laenge[0])
				{
				  verworfene_graphen++;
				  return;
				}
			      else if (pfad_laenge_alt[0] == pfad_laenge[0])
				{
				  if (pfad_laenge_alt[1] > pfad_laenge[1])
				    {
				      verworfene_graphen++;
				      return;
				    }
				}    
			    }
			  else if (pfad_laenge[1] > pfad_laenge[0])
			    {
			      pfad_laenge_sort[0] = pfad_laenge[1];
			      pfad_laenge_sort[1] = pfad_laenge[0];
			      
			      if (pfad_laenge_alt[0] > pfad_laenge[1])
				{
				  verworfene_graphen++;
				  return;
				}
			      else if (pfad_laenge_alt[0] == pfad_laenge[1])
				{
				  if (pfad_laenge_alt[1] > pfad_laenge[0])
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			    }
			  else if (pfad_laenge[1] == pfad_laenge[0])
			    {
			      pfad_laenge_sort[0] = pfad_laenge[0];
			      pfad_laenge_sort[1] = pfad_laenge[1];
			      
			      if (pfad_laenge_alt[0] > pfad_laenge[0])
				{
				  verworfene_graphen++;
				  return;
				}
			      else if (pfad_laenge_alt[0] == pfad_laenge[0])
				{
				  if (pfad_laenge_alt[1] > pfad_laenge[0])
				    {
				      verworfene_graphen++;
				      return;
				    }
				}
			    }
			  
			  
			  if ((pfad_laenge_alt[0] == pfad_laenge_sort[0]) && (pfad_laenge_alt[1] == pfad_laenge_sort[1]))
			    {
			      
			      if (main_kodiert == FALSE)
				{
				  kodiere_planar_main(kodier_kante_main[0], pl_code_main[0], pl_code_sub[0], FALSE, FALSE);
				  
				  if (pfad_laenge_alt[0] == pfad_laenge_alt[1])
				    {
				      if (brillen_typ == 1) /* beide Glaeser unten */
					{
					  if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
					    {
					      verworfene_graphen++;
					      return;
					    }

					}
				      else if (brillen_typ == 0) /* ein Brillenglas oben und ein Brillenglas unten */
					{
					  if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
					    {
					      verworfene_graphen++;
					      return;
					    }
					}
				      else if (brillen_typ == 2) /* die unendlich Brille */
					{
					  if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
					    {
					      verworfene_graphen++;
					      return;
					    }
					  
					  if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
					    {
					      verworfene_graphen++;
					      return;
					    }
					}
				    }
				  
				  if (brillen_typ == 2)
				    {
				      if (kodiere_planar_main(kodier_kante_main[0], pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  
				  main_kodiert = TRUE;
				  
				}
			      
			      
			      
			      
			      /* ************** oben / unten  Brille: Pfad mit normaler Orientierung ************** */ 
			      if (((kodier_kanten[0]->hinten)->faerbung == FALSE) && ((kodier_kanten[1]->hinten)->faerbung == FALSE))
				{
				  if (pfad_laenge[0] > pfad_laenge[1]) /* Min. und Max. existieren eindeutig. Daher kann festgelegt werden, welcher Patch der erste
									  und welcher der zweite ist. */
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else if (pfad_laenge[0] < pfad_laenge[1])
				    {
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else /* die beiden neuen Pfadlaengen sind gleich. Es laesst sich also kein Min. oder Max. bilden. 
					  Daher muessen beide Pfade als Pfad 1 bzw Pfad 2 angenommen werden. */
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				    } /* Ende gleiche Pfadlaengen */
				} /* if ((kodier_kanten[0]->hinten)->faerbung == FALSE) */
			      
			      /* ******************* unten / unten Brille ******************* */
			      else if (((kodier_kanten[0]->hinten)->faerbung == TRUE) && ((kodier_kanten[1]->hinten)->faerbung == FALSE))
				{
				  if (pfad_laenge[0] > pfad_laenge[1])
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else if (pfad_laenge[1] > pfad_laenge[0])
				    {
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				}
			      /* ******************* oben / oben Brille ******************* */
			      else if (((kodier_kanten[0]->hinten)->faerbung == FALSE) && ((kodier_kanten[1]->hinten)->faerbung == TRUE))
				{
				  if (pfad_laenge[0] > pfad_laenge[1])
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else if (pfad_laenge[1] > pfad_laenge[0])
				    {
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				}
			      /* ****************** "unendlich" Brille ****************** */
			      else if (pfad_nummern[merke_start_kante[0]->ziel] == pfad_nummern[merke_start_kante[1]->ziel])
				{
				  if (pfad_laenge[0] > pfad_laenge[1]) /* Min. und Max. existieren eindeutig. Daher kann festgelegt werden, welcher Patch der erste
									  und welcher der zweite ist. */
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else if (pfad_laenge[0] < pfad_laenge[1])
				    {
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else /* die beiden neuen Pfadlaengen sind gleich. Es laesst sich also kein Min. oder Max. bilden. 
					  Daher muessen beide Pfade als Pfad 1 bzw Pfad 2 angenommen werden. */
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, FALSE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				}
			      /* ******************* unten / oben Brille: Spiegel des eigentlichen Pfades ******************** */
			      else if (((kodier_kanten[0]->hinten)->faerbung == TRUE) && ((kodier_kanten[1]->hinten)->faerbung == TRUE))
				{
				  if (pfad_laenge[0] > pfad_laenge[1]) /* Min. und Max. existieren eindeutig. Daher kann festgelegt werden, welcher Patch der erste
									  und welcher der zweite ist. */
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else if (pfad_laenge[0] < pfad_laenge[1])
				    {
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				  else /* die beiden neuen Pfadlaengen sind gleich. Es laesst sich also kein Min. oder Max. bilden. 
					  Daher muessen beide Pfade als Pfad 1 bzw Pfad 2 angenommen werden. */
				    {
				      if (kodiere_planar_main(kodier_kanten[0], pl_code_sub[0], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				      
				      if (kodiere_planar_main(kodier_kanten[1], pl_code_sub[1], pl_code_main[0], TRUE, TRUE) == 0)
					{
					  verworfene_graphen++;
					  return;
					}
				    }
				} /* ende else */
			    }
			} /* Pfad_typ == BRILLE */
		    } /* else auf Sandwich gefunden -> Brille */
		} /* else auf Pfadtyp == BAUCHBINDE -> Sandwich oder Brille */
	    } /* pfad_start == FALSE ?? */
	} /* for l .... */
    } /* for i ... */
  


  if (main_kodiert == FALSE)
    {
      if (pfad_typ == BAUCHBINDE)
	{
	    kodiere_planar_main(kodier_kante_main[0], pl_code_main[0], pl_code_sub[0], FALSE, FALSE);

	    for (j = 1 ; j <= pfad_laenge_alt[0] ; j++)
	      {
		kodier_kante_main[0] = (kodier_kante_main[0]->invers)->hinten;
		
		if (kodiere_planar_main(kodier_kante_main[0], pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
		  {
		    verworfene_graphen++;
		    return;
		  }
		
		if (flaechen_alt[0] == flaechen_alt[1])
		  {
		    if (kodiere_planar_main(kodier_kante_main[0]->invers, pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
		      {
			verworfene_graphen++;
			return;
		      }
		  
		    if (kodiere_planar_main(kodier_kante_main[0]->invers, pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
		      {
			verworfene_graphen++;
			return;
		      }
		  }

		if (kodiere_planar_main(kodier_kante_main[0], pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
		  {
		    verworfene_graphen++;
		    return;
		  }
	      }
	    
	    main_kodiert = TRUE;
	}
      else if (pfad_typ == SANDWICH)
	{
	  kodiere_planar_main(kodier_kante_main[0], pl_code_main[0], pl_code_sub[0], FALSE, FALSE);
	  
	  if (pfad_laenge_alt[0] == pfad_laenge_alt[1])
	    {
	      if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
		{
		  verworfene_graphen++;
		  return;
		}
	    }
	  main_kodiert = TRUE;
	}
      else
	{
	  kodiere_planar_main(kodier_kante_main[0], pl_code_main[0], pl_code_sub[0], FALSE, FALSE);
	  
	  if (pfad_laenge_alt[0] == pfad_laenge_alt[1])
	    {
	      if (brillen_typ == 1) /* beide Glaeser unten */
		{
		  if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
		    {
		      verworfene_graphen++;
		      return;
		    }
		}
	      else if (brillen_typ == 0) /* ein Brillenglas oben und ein Brillenglas unten */
		{
		  if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
		    {
		      verworfene_graphen++;
		      return;
		    }
		}
	      else if (brillen_typ == 2) /* die unendlich Brille */
		{
		  if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, FALSE) == 0)
		    {
		      verworfene_graphen++;
		      return;
		    }

		  if (kodiere_planar_main(kodier_kante_main[1], pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
		    {
		      verworfene_graphen++;
		      return;
		    }
		}
	    }
	  
	  if (brillen_typ == 2)
	    {
	      if (kodiere_planar_main(kodier_kante_main[0], pl_code_main[1], pl_code_main[0], TRUE, TRUE) == 0)
		{
		  verworfene_graphen++;
		  return;
		}
	    }

	  main_kodiert = TRUE;
	}
    }

  if (no_cut == TRUE) /* Filter fuer 1- und 2-Eckencut */
    {
      for (j = 0 ; j <= anzahl_ecken + 1 ; j++) /* Felder auf 0 setzen (Pfade loeschen) */
	for (m = 0 ; m <= 3 ; m++)
	  map[j][m].faerbung = FALSE;
            
      /* Test auf 1-cut: */
      
      for (j = 1 ;  j <= anzahl_ecken ; j++)
	{
	  for (m = 0 ; m <= 3 ; m++)
	    {
	      start_kante = map[j] + m;
	      
	      if (start_kante->faerbung == FALSE)
		/* ansonsten lag diese Ecke schon auf einem bereits durchlaufenen Kreis und stellt deshalb keinen  1-cut oder 2-cut dar!!! */
		{
		  for (i = 1 ; i <= anzahl_ecken ; i++)
		    pfad_nummern[i] =0;

		  pfad_nummern[start_kante->ursprung] = 1;
		  zaehler = 1;

		  while (pfad_nummern[start_kante->ziel] == 0)
		    {
		      zaehler++;
		      start_kante->faerbung = TRUE;
		      pfad_nummern[start_kante->ziel] = zaehler;
		      
		      /* 2-cut: */

		      for (i = 0 ; i <= 3 ; i++)
			{
			  start_kante_2 = map[start_kante->ziel] + i;
			  			  
			  if (start_kante_2 != start_kante->rechts)
			    {
			      while (pfad_nummern[start_kante_2->ziel] != pfad_nummern[start_kante->ziel])
				{
				  if (pfad_nummern[start_kante_2->ziel] != 0)
				    {
				      if ((pfad_nummern[start_kante->ursprung] != pfad_nummern[start_kante_2->ziel]) 
					  && (pfad_nummern[start_kante->rechts->ziel] != pfad_nummern[start_kante_2->ziel]))
					/* Ecken sind auf der urspruenglichen Flaeche nicht benachbart, haben aber eine weitere Flaeche gemein: 2-cut */
					{
					  /*printf("\n%d",graphen_geschrieben + 1);*/
					  verworfene_graphen++;
					  return;
					}
				    }
				
				  start_kante_2 = start_kante_2->rechts;
				
				}
			    }
			}
			      
		      start_kante = start_kante->rechts;

		    }
		  
		  start_kante->faerbung = TRUE;
		  
		  if ((start_kante->rechts)->faerbung == FALSE) /* hier handelt es sich um einen 1-cut, da die Flaeche an der Startecke wieder ankommt, sie
								   aber noch nicht vollstaendig ist.  */
		    {
		      /*printf("\n%d",graphen_geschrieben + 1);*/
		      verworfene_graphen++;
		      return;
		    }
		}
	    }
	}
    }

  schreibe_graph(pl_code_main[0]);

  /* kodiere_planar(1, 1, 1); */ /* hier wird der entstandene Graph planar kodiert */

  if (pfad_typ == BAUCHBINDE)
    gleich_bb++;
  else if (pfad_typ == SANDWICH)
    gleich_sw++;
  else if (pfad_typ == BRILLE)
    gleich_br++;
      

}
      
      
/* *********************************************************************** */
/*                                                                         */
/* Loescht einen zuvor gemachten Schnitt und alle nach ihm kommenden       */
/*                                                                         */
/* Aufruf: loesche_schnitt()                                               */
/*                                                                         */
/* Ausgabe:                                                                */
/*                                                                         */
/* *********************************************************************** */
void loesche_schnitt(int merke_knoten, KANTE *rand_kante, KNOTENTYP *seq, unsigned char loesche)
{
  int i, j;
  KANTE *start_kante = rand_kante;
  KANTE *kante;

  /* die Verbindungen der bleibenden Kanten mit den zu entfernenden Kanten muessen vollstaendig
     geloescht werden. Es reicht die inneren Kanten des Patches zu loeschen, da an jeder Stelle,
     an der ein Schnitt gemacht wurde, dieser rueckgaengig gemacht wird: */

  for (i = seq[0] ; i >= 1 ; i--)
    {
      for (j = 1 ; j <= seq[i] ; j++)
	{
	  /* innere Kanten loeschen: */
	  
	  (start_kante->links)->ziel   = innen_kante;
	  (start_kante->links)->invers = NIL;
	  (start_kante->links)->links  = NIL;
	  (start_kante->links)->rechts = NIL;

	  start_kante = (start_kante->invers)->hinten;
	}
      
      start_kante = start_kante->links;
    }

  if (loesche == FALSE)
    return;
  
  for (i = merke_knoten + 1 ; i <= anzahl_ecken ; i++)
    {
      map[i][0].bezeichnung = 0;
      map[i][1].bezeichnung = 0;
      map[i][2].bezeichnung = 0;
      map[i][3].bezeichnung = 0;

      map[i][0].ursprung = i;
      map[i][1].ursprung = i;
      map[i][2].ursprung = i;
      map[i][3].ursprung = i;
      
      map[i][0].ziel = i - 1;
      map[i][1].ziel = i + 1;
      map[i][2].ziel = aussen_kante;
      map[i][3].ziel = innen_kante;
      
      map[i][0].invers = map[i - 1] + 1;
      map[i][1].invers = map[i + 1];
      map[i][2].invers = NIL;
      map[i][3].invers = NIL;
      
      map[i][0].hinten = map[i] + 1;
      map[i][1].hinten = map[i];
      map[i][2].hinten = map[i] + 3;
      map[i][3].hinten = map[i] + 2;
      
      map[i][0].links = map[i - 1] + 3;
      map[i][1].links = map[i + 1] + 2;
      map[i][2].links = NIL;
      map[i][3].links = NIL;
      
      map[i][0].rechts = map[i - 1] + 2;
      map[i][1].rechts = map[i + 1] + 3;
      map[i][2].rechts = NIL;
      map[i][3].rechts = NIL;
    }

}

/* *********************************************************************** */
/*                                                                         */
/* Laeuft die Aeste des Baumes entlang                                     */
/*                                                                         */
/* Aufruf: laufe_aeste(void)                                               */
/*                                                                         */
/* Ausgabe: 0 = Graph nicht fertig , 1 = Graph fertig                      */
/*                                                                         */
/* *********************************************************************** */
unsigned char laufe_aeste(Item *ptr_item, KNOTENTYP seq[max_ecken + 2], KANTE *kante, unsigned char init, 
			  unsigned char erster_patch, unsigned char Durchgang)

{
  unsigned char  i, j, k, dazu;
  
  KNOTENTYP sequenz_neu[2][max_ecken + 2], seq_start[3];
  
  KANTE *kante_neu_0, *kante_neu_1, *dummy_kante;
  
  static Lineare_Liste liste[max_ecken + 2]; /* diese Liste wird benutzt, um den Baum in der Reihenfolge der
					    Patch-Wechsel-Prioritaet zu durchlaufen. */

  Lineare_Liste merke_liste[max_ecken + 2]; /* Kopie der Listenstruktur von oben */
  KNOTENTYP merke_liste_zaehler;

  static KNOTENTYP liste_zaehler;
  
  static KNOTENTYP haupt_seq[3][3];

  static KNOTENTYP pfad_laenge;

  unsigned char loesche;
  unsigned char unterste_ebene;

  KNOTENTYP pl_code_main[max_ecken + 2][4], pl_code_sub[max_ecken +2][4];
  

  /**********************************************************/
  /* zur Sicherheit werden die notwendigen Werte gesichert: */

  int merke_ecke = aktuelle_ecke;
  KANTE *merke_kante;
  
  KNOTENTYP seq_kopie[max_ecken];

  /**********************************************************/

  if (init == 1)
    {
      liste_zaehler = 0;
      
      haupt_seq[0][1] = patch_daten.Rand[0];
      haupt_seq[1][1] = patch_daten.Rand[1];

      if (max_patches == 3)
	{
	  haupt_seq[2][1] = patch_daten.Rand[2];
	  haupt_seq[2][2] = patch_daten.Rand[3];
	  haupt_seq[2][0] = 2;
	}

      haupt_seq[0][0] = haupt_seq[1][0] = 1;
      
      kante = patch_daten.Kanten[0];
      
      ptr_item = patch_daten.Items[0];
      
      seq[0] = 1;
      seq[1] = haupt_seq[0][1];

      liste_zaehler = 0;

      patch_daten.aktueller_patch[0] = ptr_item;

    }

  merke_kante = kante;
  memcpy(seq_kopie, seq, seq[0] + 1);

  while (ptr_item != NIL)
    {
      if (ptr_item->wird_gebraucht == FALSE)
	{
	  ptr_item->wird_gebraucht = TRUE;
	  zum_bau_verwendet++;
	}
      
      if (ptr_item->vorfahre_0 == NIL) /* Grundpolygon erreicht */
	{
	  /*if (seq[0] >= 11)
	    merke = TRUE;*/
	        
	  if (liste_zaehler == 0) /* Patch aufgebaut */
	    {
	      if (Durchgang < max_patches - 1) /* es sind noch nicht alle Patches aufgebaut: also nimm den naechsten ... */
		{
		  patch_daten.aktueller_patch[Durchgang + 1] = patch_daten.Items[Durchgang + 1];
		  laufe_aeste(patch_daten.Items[Durchgang + 1], haupt_seq[Durchgang + 1], patch_daten.Kanten[Durchgang + 1], 0, TRUE, Durchgang + 1);
		}
	      else /* alle Patches sind aufgebaut: also teste, kodiere und schreibe den Graphen ... */
		{
		  if (pfad_typ == BRILLE)
		    brillen_graphen++;
		  else if (pfad_typ == SANDWICH)
		    sandwich_graphen++;
		  else
		    bauchbinden_graphen++;

		  if (doppelkanten() == FALSE) /* falls keine Doppelkanten vorhanden sind, kann kodiert werden */
		    {
		      if (no_iso == TRUE)
			{
			  kodiere_planar_main(patch_daten.Kanten[0], pl_code_main, pl_code_sub, FALSE, FALSE);
			  schreibe_graph(pl_code_main);
			}
		      else
			iso_test(); /* teste auf Isomorphie und schreibe - falls noetig - den Graphen. */
		    }
		  else
		    {
		      verworfene_graphen++;
		      doppel_kanten++;
		    }

		  /*merke = FALSE;*/

		}
	    }
	  else /* der Patch ist noch nicht vollstaendig; also weiter an vorgegebener Stelle ... */
	    {
	      liste_zaehler--;
	      laufe_aeste(liste[liste_zaehler + 1].ptr_item, liste[liste_zaehler + 1].seq, liste[liste_zaehler + 1].start_kante, 0, FALSE, Durchgang);
	    }
	}
      else
	{
	  if ((pfad_typ == BAUCHBINDE) && (erster_patch == TRUE))
	    make_schnittkante_b(ptr_item, kante, seq, sequenz_neu, &kante_neu_0, &kante_neu_1);
	  else
	    make_schnittkante(ptr_item, kante, seq, sequenz_neu, &kante_neu_0, &kante_neu_1);
	  
	  /* Sicherung der relevanten Daten fuer fruehere rechte Aeste: */

	  merke_liste_zaehler = liste_zaehler;
	  for (i = 1 ; i <= liste_zaehler ; i++)
	    {
	      merke_liste[i].ptr_item = liste[i].ptr_item;
	      merke_liste[i].start_kante = liste[i].start_kante;
	      
	      for (j = 0 ; j <= liste[i].seq[0] ; j++)
		merke_liste[i].seq[j] = liste[i].seq[j];
	    }

	  /* Sicherung der relevanten Daten fuer den rechten Ast: */

	  liste_zaehler++;
	  liste[liste_zaehler].ptr_item = ptr_item->vorfahre_1;

	  memcpy(liste[liste_zaehler].seq, sequenz_neu[1], sequenz_neu[1][0] + 1);

	  liste[liste_zaehler].start_kante = kante_neu_1;

	  /* linken Ast entlanglaufen: */

	  laufe_aeste(ptr_item->vorfahre_0, sequenz_neu[0], kante_neu_0, 0, FALSE, Durchgang);


	  /* gesicherte Werte zurueckholen: */
	  
	  memcpy(seq, seq_kopie, seq_kopie[0] + 1);

	  aktuelle_ecke = merke_ecke;
	  kante         = merke_kante;
	  
	  liste_zaehler = merke_liste_zaehler;
	  
	  for (i = 1 ; i <= liste_zaehler ; i++)
	    {
	      liste[i].ptr_item    = merke_liste[i].ptr_item;
	      liste[i].start_kante = merke_liste[i].start_kante;
	      
	      for (j = 0 ; j <= merke_liste[i].seq[0] ; j++)
		liste[i].seq[j] = merke_liste[i].seq[j];
	    }
	  
	  if (ptr_item->next != NIL)
	    loesche = TRUE;
	  else
	    loesche = FALSE;

	  loesche_schnitt(aktuelle_ecke, kante, seq, loesche); /* die durchgefuehrten Schnitte werden fuer den naechsten
								  Erzeugungsgang wieder rueckgaengig gemacht. */
	  
	}
      
      
      ptr_item = ptr_item->next; /* nehme den naechsten Patch in der Liste */
      
      if (erster_patch == TRUE)
	patch_daten.aktueller_patch[Durchgang] = ptr_item;


    } /* while (ptr_item !=NIL) */


}

/* *********************************************************************** */
/*                                                                         */
/* Errechnet die Laenge eines Pfades                                       */
/*                                                                         */
/* Aufruf: test_pfad(Adresse des Knotens)                                  */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */
void test_pfad(Item *ptr_item, int *anzahl)
{
  (*anzahl)++;

  if (ptr_item == NIL)
    return;

  test_pfad(ptr_item->vorfahre_0, anzahl);
  test_pfad(ptr_item->vorfahre_1, anzahl);
}


/* *********************************************************************** */
/*                                                                         */
/* Initialisiert die Struktur, in welcher der Graph aufgebaut wird         */
/*                                                                         */
/* Aufruf: init_map(void)                                                  */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */

void init_map(void)
{
  
  int i;
  
  /* Initialisierung als Kette von Valenz 4 Ecken */
  
  for (i = 1 ; i <= anzahl_ecken ; i++) /* Erzeugung aller Valenz-3-Ecken */
    { 
      map[i][0].bezeichnung = 0;
      map[i][1].bezeichnung = 0;
      map[i][2].bezeichnung = 0;
      map[i][3].bezeichnung = 0;
      
      map[i][0].ursprung = i;
      map[i][1].ursprung = i;
      map[i][2].ursprung = i;
      map[i][3].ursprung = i;
      
      map[i][0].ziel = i - 1;
      map[i][1].ziel = i + 1;
      map[i][2].ziel = aussen_kante;
      map[i][3].ziel = innen_kante;
      
      map[i][0].pfad_start = FALSE;
      map[i][1].pfad_start = FALSE;
      map[i][2].pfad_start = FALSE;
      map[i][3].pfad_start = FALSE;

      map[i][0].invers = map[i - 1] + 1;
      map[i][1].invers = map[i + 1];
      map[i][2].invers = NIL;
      map[i][3].invers = NIL;
      
      map[i][0].hinten = map[i] + 1;
      map[i][1].hinten = map[i];
      map[i][2].hinten = map[i] + 3;
      map[i][3].hinten = map[i] + 2;
      
      map[i][0].links = map[i - 1] + 3;
      map[i][1].links = map[i + 1] + 2;
      map[i][2].links = NIL;
      map[i][3].links = NIL;
      
      map[i][0].rechts = map[i - 1] + 2;
      map[i][1].rechts = map[i + 1] + 3;
      map[i][2].rechts = NIL;
      map[i][3].rechts = NIL;
    }
  
}

/* *********************************************************************** */
/*                                                                         */
/* Testet die Patches auf Eckenplausibilitaet                              */
/*                                                                         */
/*                                                                         */
/* Aufruf: plausibel(ptr_Patch)                                            */
/*                                                                         */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */
int plausibel(Item *ptr_item, int init)
{
  static int summe;

  if (init == 1)
    summe = 0;

  summe = summe + ptr_item->schnitt;


  if (ptr_item->vorfahre_0 != NIL)
    {
      plausibel(ptr_item->vorfahre_0, 0);
      plausibel(ptr_item->vorfahre_1, 0);
    }

  return(summe);

}


/* *********************************************************************** */
/*                                                                         */
/* Baut aus den vorgebenen drei Patches einen Graphen mit einem Sandwich-  */
/* Pfad                                                                    */
/*                                                                         */
/* Aufruf: baue_sandwich(void)                                             */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */

void baue_sandwich(void)
{
  
  int i, j, k;
  unsigned char wahl_kante_1, wahl_kante_2, pfad_laenge;
  
  KNOTENTYP x, y, z1, z2;
  KNOTENTYP seq_all[max_ecken + 2];


  pfad_typ = SANDWICH;
  max_patches = 3;

  x  = patch_daten.Rand[0];
  y  = patch_daten.Rand[1];
  z1 = patch_daten.Rand[2];
  z2 = patch_daten.Rand[3];
  
  for (j = 1 ; j <= 1 ; j++) /* fuer beide Richtungen des Sandwichpfades, allerdings wird nur einer gebraucht (hier also nur Testzweck) */
    {
      
      init_map();
      
      aktuelle_ecke = pfad_laenge = x + z2 + 1;
      
      if (j == 1)
	{
	  wahl_kante_1 = 2; /* Festlegung der Verbindungskante */
	  wahl_kante_2 = 3;
	}
      else
	{
	  wahl_kante_1 = 3; /* Festlegung der Verbindungskante */
	  wahl_kante_2 = 2;
	}

      /* Pfade schliessen: */
      
      map[1][0].ziel                            = x + 1;
      map[x + 1][wahl_kante_1].ziel             = 1;
      
      map[pfad_laenge][1].ziel                  = pfad_laenge - y;
      map[pfad_laenge - y][wahl_kante_2].ziel   = pfad_laenge;
      
      map[1][0].invers                          = map[x + 1] + wahl_kante_1;
      map[x + 1][wahl_kante_1].invers           = map[1] + 0; /* (+ 0) nur der Verdeutlichung halber */
      
      map[pfad_laenge][1].invers                = map[pfad_laenge - y] + wahl_kante_2;
      map[pfad_laenge - y][wahl_kante_2].invers = map[pfad_laenge] + 1;
      
      
      if (j == 1)
	{
	  map[1][0].links                           = map[x + 1] + 1;
	  map[1][0].rechts                          = map[x + 1] + 0;
	  
	  map[x + 1][wahl_kante_1].links            = map[1] + 2;
	  map[x + 1][wahl_kante_1].rechts           = map[1] + 3;

	  map[pfad_laenge][1].links                 = map[pfad_laenge - y] + 0;
	  map[pfad_laenge][1].rechts                = map[pfad_laenge - y] + 1;
	  
	  map[pfad_laenge - y][wahl_kante_2].links  = map[pfad_laenge] + 3;
	  map[pfad_laenge - y][wahl_kante_2].rechts = map[pfad_laenge] + 2;

	}
      else
	{
	  map[1][0].links                           = map[x + 1] + 0;
	  map[1][0].rechts                          = map[x + 1] + 1;

	  map[x + 1][wahl_kante_1].links            = map[1] + 2;
	  map[x + 1][wahl_kante_1].rechts           = map[1] + 3;
	  
	  map[pfad_laenge][1].links                 = map[pfad_laenge - y] + 1;
	  map[pfad_laenge][1].rechts                = map[pfad_laenge - y] + 0;

	  map[pfad_laenge - y][wahl_kante_2].links  = map[pfad_laenge] + 3;
	  map[pfad_laenge - y][wahl_kante_2].rechts = map[pfad_laenge] + 2;

	}

      if (j == 1)
	{
	  patch_daten.Kanten[0] = map[x + 1] + wahl_kante_1;
	  patch_daten.Kanten[1] = map[pfad_laenge - y] + wahl_kante_2;
	  patch_daten.Kanten[2] = map[x + 1] + 1;
	}
      else
	{
	  patch_daten.Kanten[0] = map[x + 1] + 0;
	  patch_daten.Kanten[1] = map[pfad_laenge - y] + 1;
	  patch_daten.Kanten[2] = map[pfad_laenge - y] + 2;
	}

      patch_daten.zwischenstueck = x + y - pfad_laenge;  
      
      patch_daten.pfad_laenge = pfad_laenge;
      
      /* welche Kanten duerfen/muessen nicht als Startkanten im Isotest dienen? */
      
      for (k = 1 ; k <= pfad_laenge - y ; k++)
	map[k][1].pfad_start = TRUE;
      for (k = pfad_laenge ; k >= x + 1 ; k--)
	map[k][0].pfad_start = TRUE;
      
      
      sandwich_pfade++;

      fprintf(patchbau, "sw: %d:%d, %d:%d, %d:%d,%d\n",patch_daten.Flaechen[0], x, patch_daten.Flaechen[1], y, patch_daten.Flaechen[2], z1, z2);
      
      laufe_aeste(NIL, seq_all, NIL, 1, TRUE, 0);
      
      if (darstellung == TRUE)
	fprintf(stderr,"\r SW(#:%d): %d  BR(#:%d): %d", sandwich_pfade, sandwich_graphen, brillen_pfade, brillen_graphen);
    }
}


/* *********************************************************************** */
/*                                                                         */
/* Sucht alle kombinatorisch moeglichen Sandwichpfade                      */
/*                                                                         */
/*                                                                         */
/* Aufruf: suche_sandwich()                                                */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */
void suche_sandwich(void)
{

  KNOTENTYP bis, bis2, diff;
  KNOTENTYP x, y, z1, z2, i, j, k;

  Baum *ptr1_baum, *ptr2_baum, *ptr3_baum, *ptr33_baum;
  Item *ptr1_item, *ptr2_item, *ptr3_item;

  unsigned int zaehl_pfade = 0;

  /* fuer die weiteren 2 Patches muessen mind. 2 (bzw. 4) Flaechen uebrig bleiben; deshalb bleiben fuer den 1. Patch max. 
     Gesamt-Flaechen - 6 (# Ecken + 2 - 6) Flaechen uebrig ... */

  if (anzahl_ecken + 2 - 4 - 2 >= wurzel->max_liste) 
    bis = wurzel->max_liste;
  else 
    bis = anzahl_ecken + 2 - 4 - 2;
  
  /* Bemerkung zu x, y, z1 und z2: Es soll o.B.d.A gelten: x >= y und z1 >= z2, woraus folgt, dass z1 an x liegen muss */
  
  for (i = 4 ; i <= bis ; i++) /* Flaechen 1. Patch mit einer Valenz-2-Ecke (hat mind. 4 Flaechen!!) */
    {
      ptr1_baum = *(wurzel->liste + i);

      if (ptr1_baum != NIL) /* Falls kein Eintrag gefunden, muss auch nicht weiter gesucht werden */
        {
	  if (ptr1_baum->max_liste != leer)
	    {
	      for (x = 2 ; x <= ptr1_baum->max_liste ; x = x + 2)
		{
		  if ((*(ptr1_baum->liste + x) != NIL) && (ptr1_baum->max_liste >= x))
		    ptr1_item = (*(ptr1_baum->liste + x))->eintrag; /* Eintrag holen */
		  else
		    ptr1_item = NIL;
		  
		  if (ptr1_item != NIL) /* Patch-Eintrag vorhanden ? */
		    {
		      if (anzahl_ecken + 2 - i - 2 >= i) /* fuer den letztzen Patch muessen noch 2 Flachen 
							    uebrig bleiben und die Flachen des ersten
							    Patches duerfen natuerlich nicht mitgezaehlt werden. */
			bis2 = i;
		      else
			bis2 = anzahl_ecken + 2 - i - 2;

		      for (j = 4 ; j <= bis2 ; j++) /* Flaechen 2. Patch mit einer Valenz-2-Ecke */
			{
			  ptr2_baum = *(wurzel->liste + j); /* 1. Verzweigung der Flaechen Patch 2 */
			  
			  if (ptr2_baum != NIL)
			    {
			      if (ptr2_baum->max_liste != leer)
				{
				  for (y = 2 ; ((y <= ptr2_baum->max_liste) && (y <= x)) ; y = y + 2)
				    {
				      if ((*(ptr2_baum->liste + y) != NIL) && (ptr2_baum->max_liste >= y))
					ptr2_item = (*(ptr2_baum->liste + y))->eintrag;
				      else 
					ptr2_item = NIL;
				      
				      if (ptr2_item != NIL) /* Patch-Eintrag vorhanden ? */
					{
					  diff = anzahl_ecken + 2 - i - j; /* was bleibt an Flachen noch uebrig ?? */
					  
					  ptr3_baum = *(wurzel->liste + diff);
					  
					  if (ptr3_baum != NIL)
					    {
					      if (ptr3_baum->max_liste != leer)
						{

						  for (k = 0 ; k <= y ; k++) /* ueber die gemeinsamen Ecken von Patch 1 und Patch 2 (ohne V2-Ecken) */
						    {
						      z1 = x - k - 1;
						      z2 = y - k - 1;

						      if ((*(ptr3_baum->liste + z1) != NIL) && (ptr3_baum->max_liste >= z1))
							ptr33_baum = *(ptr3_baum->liste + z1);
						      else
							ptr33_baum = NIL;

						      if (ptr33_baum != NIL)
							{
							  if (ptr33_baum->max_liste != leer)
							    {
							      if ((*(ptr33_baum->liste + z2) != NIL) && (ptr33_baum->max_liste >= z2))
								ptr3_item = (*(ptr33_baum->liste + z2))->eintrag;
							      else
								ptr3_item = NIL;
								    
							      if (ptr3_item != NIL)
								zaehl_pfade++;

							    }
							}
						    }
						}
					    }
					}
				    }
				}
			    }
			}
		    }
		}
	    }
	}
    }

  fprintf(stderr,"\nSW paths: %d\n", zaehl_pfade);

}


/* *********************************************************************** */
/*                                                                         */
/* Baut aus den vorgebenen drei Patches einen Graphen mit einem Brillen-   */
/* Pfad                                                                    */
/*                                                                         */
/* Aufruf: baue_brille(Typ)                                                */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */

void baue_brille(unsigned char typ)
{
  
  int i, k;
  unsigned char wahl_kante_1, wahl_kante_2, pfad_laenge;
  
  KNOTENTYP x, y, z1, z2;
  KNOTENTYP seq_all[max_ecken + 2];

  
  pfad_typ = BRILLE; /* die globale Variable "pfad_typ" wird zu Zaehlzwecken gesetzt. */
  brillen_typ = typ;
  max_patches = 3;
  
  x  = patch_daten.Rand[0];
  y  = patch_daten.Rand[1];
  z1 = patch_daten.Rand[2];
  z2 = patch_daten.Rand[3];
  
  init_map();
  
  if (typ == 2)
    aktuelle_ecke = pfad_laenge = x + y + 1;
  else if (typ == 1)
    aktuelle_ecke = pfad_laenge = x + y + z2 + 2;
  else if (typ == 0)
    aktuelle_ecke = pfad_laenge = x + z2 + 1;
  
  
  wahl_kante_1 = 2; /* Festlegung der Verbindungskante */
  wahl_kante_2 = 3;
  
  if (typ != 1) /* innere und aeussere Kanten wechseln */
    {
      for (i = 1 ; i <= x ; i++)
	{
	  map[i][2].ziel = innen_kante;
	  map[i][3].ziel = aussen_kante;
	}
    }
  else
    wahl_kante_1 = 3;
  
  /* Pfade schliessen: */
  
  map[1][0].ziel           = x + 1;
  map[pfad_laenge][1].ziel = pfad_laenge - y;
  
  map[x + 1][wahl_kante_1].ziel           = 1;
  map[pfad_laenge - y][wahl_kante_2].ziel = pfad_laenge;
  
  map[1][0].invers           = map[x + 1] + wahl_kante_1;
  map[pfad_laenge][1].invers = map[pfad_laenge - y] + wahl_kante_2;
  
  map[x + 1][wahl_kante_1].invers           = map[1] + 0; /* (+ 0) natuerlich nur der Verdeutlichung halber */
  map[pfad_laenge - y][wahl_kante_2].invers = map[pfad_laenge] + 1;
  
  if (wahl_kante_1 == 2)
    {
      map[1][0].links  = map[x + 1] + 1;
      map[1][0].rechts = map[x + 1] + 0;
    }
  else
    {
      map[1][0].links  = map[x + 1] + 0;
      map[1][0].rechts = map[x + 1] + 1;
    }
  
      map[x + 1][wahl_kante_1].links  = map[1] + 2;
      map[x + 1][wahl_kante_1].rechts = map[1] + 3;

  if (wahl_kante_2 == 2)
    {
      map[pfad_laenge][1].links  = map[pfad_laenge - y] + 1;
      map[pfad_laenge][1].rechts = map[pfad_laenge - y] + 0;
    }
  else
    {
      map[pfad_laenge][1].links  = map[pfad_laenge - y] + 0;
      map[pfad_laenge][1].rechts = map[pfad_laenge - y] + 1;
    }
  
  map[pfad_laenge - y][wahl_kante_2].links  = map[pfad_laenge] + 3;
  map[pfad_laenge - y][wahl_kante_2].rechts = map[pfad_laenge] + 2;
  
  patch_daten.Kanten[0] = map[x + 1] + wahl_kante_1;
  patch_daten.Kanten[1] = map[pfad_laenge - y] + wahl_kante_2;
  patch_daten.Kanten[2] = map[x + 1] + 1;
  
  if (typ == 2)
    patch_daten.zwischenstueck = -1;
  else
    patch_daten.zwischenstueck = pfad_laenge - x - y - 2;

  if (typ == 1)
    {
      patch_daten.Kanten[0] = map[x + 1] + 0;
      patch_daten.Kanten[2] = map[pfad_laenge - y] + 0; /* links um den Patch laufen, an der Ecke mit der Marke starten. */
    }

  patch_daten.pfad_laenge = pfad_laenge;
  
  /* welche Kanten duerfen nicht als Startkanten im Isotest dienen? */
  
  if (typ == 2)
    {
      for (k = 1 ; k <= pfad_laenge ; k++)
	{
	  map[k][0].pfad_start = TRUE;
	  map[k][1].pfad_start = TRUE;
	}

      map[pfad_laenge - y][2].pfad_start = TRUE;
      map[pfad_laenge - y][3].pfad_start = TRUE;
    }
  else
    {
      for (k = 1 ; k <= pfad_laenge - y ; k++)
	map[k][1].pfad_start = TRUE;
      for (k = pfad_laenge ; k >= x + 1 ; k--)
	map[k][0].pfad_start = TRUE;
    }
      

  brillen_pfade++;

  fprintf(patchbau, "br: %d:%d, %d:%d, %d:%d,%d\n",patch_daten.Flaechen[0], x, patch_daten.Flaechen[1], y, patch_daten.Flaechen[2], z1, z2);
  
  laufe_aeste(NIL, seq_all, NIL, 1, TRUE, 0);

  if (darstellung == TRUE)
    fprintf(stderr,"\r SW(#:%d): %d  BR(#:%d): %d", sandwich_pfade, sandwich_graphen, brillen_pfade, brillen_graphen);

}

/* *********************************************************************** */
/*                                                                         */
/* Testet Brillenkombinationen und ermittelt die Adresse des passenden     */
/* dritten Patches                                                         */
/*                                                                         */
/* Aufruf: test_patch()                                                    */
/* Ausgabe: Adresse des Patches                                            */
/*                                                                         */
/* *********************************************************************** */
Item *test_patch(Baum *ptr33_baum, int z2)
{
  Item *ptr3_item=NIL;
  
  if ((z2 >= 0) && (z2 <= ptr33_baum->max_liste))
    if (*(ptr33_baum->liste + z2) != NIL)
      if((*(ptr33_baum->liste + z2))->eintrag != NIL)
	ptr3_item = (*(ptr33_baum->liste + z2))->eintrag;
  
  return(ptr3_item);
}


/* *********************************************************************** */
/*                                                                         */
/* Sucht alle Bauchbinden-Kombinationen im Baum bei vorgegebener # Flaechen*/
/*                                                                         */
/* Aufruf: baue_bauchbinde()                                               */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */
void baue_bauchbinde(void)
{

  int i, j, k, diff, pfad_laenge;
  KNOTENTYP seq_all[max_ecken + 2];
  Baum *ptr_baum;
  Item *ptr_item[2];


  pfad_typ = BAUCHBINDE; /* die globale Variable "pfad_typ" wird u.a. zu Zaehlzwecken gesetzt. */
  max_patches = 2;

  for (i = (anzahl_ecken + 2) / 2 ; i <= anzahl_ecken + 2 - 4 ; i++) /* ein Bauchbinden-Patch hat mind. 4 Flaechen.
									Daraus resultiert, dass er max. #Ecken + 2 - 4 Flaechen
									haben darf, da ja der andere Patch ebenfalls mind. 4 Flaechen
									hat. */

    /*for (i = 4 ; i <= anzahl_ecken + 2 - 4 ; i++)*/
    {
      for (j = 2 ; j <= anzahl_ecken ; j = j + 2) /* Schleife ueber die Anzahl der Ecken auf dem Rand der Bauchbinde */
	{
	  ptr_baum = *(bauchbinden_wurzel->liste + i); /* Eintrag mit i Flaechen */
	  
	  if (ptr_baum->liste != NIL)
	    {
	      if (*(ptr_baum->liste + j) != NIL) /* Eintrag mit j V3-Ecken gefunden */
		{
		  ptr_baum = *(ptr_baum->liste + j);
		  ptr_item[0] = ptr_baum->eintrag; /* erster Patch */
		  
		  diff = anzahl_ecken + 2 - i; /* diff Flaechen muss der zweite Patch haben. */
		  
		  if (diff <= i) /* wenn der zweite Patch hoechstens so viele Flaechen hat wie der erste, muss das auch im Isomorphietest 
				    beruecksichtigt werden. D.h. im IsoTest muessen auch die Flaechen gezaehlt werden. */
		    {
		      
		      ptr_baum = *(bauchbinden_wurzel->liste + diff);
		      
		      if (ptr_baum->liste != NIL)
			{
			  if (*(ptr_baum->liste + j) != NIL)
			    {
			      ptr_baum = *(ptr_baum->liste + j);
			      ptr_item[1] = ptr_baum->eintrag;
			      
			      init_map();
			      
			      aktuelle_ecke = pfad_laenge = j;
			      
			      
			      /* Pfade schliessen: */
			      
			      map[1][0].ziel           = pfad_laenge;
			      map[pfad_laenge][1].ziel = 1;
			      
			      map[1][0].invers           = map[pfad_laenge] + 1;
			      map[pfad_laenge][1].invers = map[1] + 0;
			      
			      map[1][0].links           = map[pfad_laenge] + 3;
			      map[pfad_laenge][1].links = map[1] + 2;
			      
			      map[1][0].rechts           = map[pfad_laenge] + 2;
			      map[pfad_laenge][1].rechts = map[1] + 3;
			      
			      patch_daten.Rand[0]   = patch_daten.Rand[1] = pfad_laenge; /* fuer beide Patches identisch */
			      patch_daten.Kanten[0] = map[1] + 0;
			      patch_daten.Kanten[1] = map[1] + 1;
			      
			      patch_daten.Items[0] = ptr_item[0]; 
			      patch_daten.Items[1] = ptr_item[1];
			      
			      patch_daten.zwischenstueck = 0;
			      
			      patch_daten.pfad_laenge = pfad_laenge;
			      
			      patch_daten.Flaechen[0] = i;
			      patch_daten.Flaechen[1] = diff;

			      /* welche Kanten duerfen nicht als Startkanten im Isotest dienen? */
			      
			      for (k = 1 ; k <= pfad_laenge ; k++)
				{
				  map[k][0].pfad_start = TRUE;
				  map[k][1].pfad_start = TRUE;
				}
			      
			      
			      bauchbinden_pfade++;
			      
			      /*printf("\n %d ", pfad_laenge);*/
			      
			      fprintf(patchbau, "bb: %d:%d, %d:%d,\n",patch_daten.Flaechen[0], pfad_laenge, patch_daten.Flaechen[1], pfad_laenge);

			      /*if (diff == i)
				laufe_aeste_bb(NIL, seq_all, NIL, 1, TRUE, 0);
			      else*/
				laufe_aeste(NIL, seq_all, NIL, 1, TRUE, 0);
			      
			      if (darstellung == TRUE)
				fprintf(stderr,"\r BB(#:%d): %d", bauchbinden_pfade, bauchbinden_graphen);
			      
			    }
			}
		    }
		}
	    }
	}
    }
}



/* *********************************************************************** */
/*                                                                         */
/* Sucht alle Patch-Kombinationen im Baum bei vorgegebener # Flaechen      */
/*                                                                         */
/* Aufruf: suche_patch()                                                   */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */

void suche_patch(void)
{
  
  KNOTENTYP i, j, k, bis, bis2, bis3, diff, x, y, z[2];

  Baum *ptr1_baum, *ptr2_baum, *ptr3_baum, *ptr33_baum;
  Item *ptr1_item, *ptr2_item, *ptr3_item;
   
  
  /* Bemerkung: Da nach Euler "# Flaechen = # Ecken + 2" gilt wird im weiteren nicht mit der Anzahl der
     Flaechen gearbeitet, sondern mit der Anzahl der Ecken + 2, somit entfaellt lediglich die Rechnung. */
  
  /*suche_sandwich();*/

  if (anzahl_ecken + 2 - 4 - 2 >= wurzel->max_liste) /* fuer die weiteren 2 Patches muessen mind. 2 (bzw. 4) Flaechen
							uebrig bleiben; deshalb bleiben fuer den 1. Patch max. 
							Gesamt-Flaechen - 6 (# Ecken + 2 - 6) Flaechen uebrig ... */
    bis = wurzel->max_liste;
  else 
    bis = anzahl_ecken + 2 - 4 - 2;
  
  /* Bemerkung zu x, y, z1 und z2: Es soll o.B.d.A gelten: x >= y und z1 >= z2, woraus folgt, dass z1 an x liegen muss */
  
  for (i = 4 ; i <= bis ; i++) /* Flaechen 1. Patch mit einer Valenz-2-Ecke (hat mind. 4 Flaechen!!) */
    {      
      ptr1_baum = *(wurzel->liste + i); /* 1. Verzweigung der Flaechen Patch 1 */
      
      if (ptr1_baum != NIL) /* Falls kein Eintrag gefunden muss auch nicht weiter gesucht werden */
        {
	  if (ptr1_baum->max_liste != leer)
	    {
	      for (x = 2 ; x <= ptr1_baum->max_liste ; x = x + 2)
		{
		  if ((*(ptr1_baum->liste + x) != NIL) && (ptr1_baum->max_liste >= x))
		    ptr1_item = (*(ptr1_baum->liste + x))->eintrag; /* Eintrag holen */
		  else
		    ptr1_item = NIL;
		  
		  if (ptr1_item != NIL) /* Patch-Eintrag vorhanden ? */
		    {
		      if (anzahl_ecken + 2 - i - 2 >= i) /* fuer den letztzen Patch muessen noch 2 Flachen 
							    uebrig bleiben und die Flachen des ersten
							    Patches duerfen natuerlich nicht mitgezaehlt werden. */
			bis2 = i;
		      else
			bis2 = anzahl_ecken + 2 - i - 2;

		      bis2 = bis; /* solange im Iso Test noch keine Flaechenanzahl der Patches ueberprueft wird, muss der zweite Patch auch mehr Flaechen 
				     haben duerfen als der erste Patch.... */

		      for (j = 4 ; j <= bis2 ; j++) /* Flaechen 2. Patch mit einer Valenz-2-Ecke */
			{
			  ptr2_baum = *(wurzel->liste + j); /* 1. Verzweigung der Flaechen Patch 2 */
			  
			  if (ptr2_baum != NIL)
			    {
			      if (ptr2_baum->max_liste != leer)
				{
				  for (y = 2 ; ((y <= ptr2_baum->max_liste) && (y <= x)) ; y = y + 2)
				    {
				      if ((*(ptr2_baum->liste + y) != NIL) && (ptr2_baum->max_liste >= y))
					ptr2_item = (*(ptr2_baum->liste + y))->eintrag;
				      else 
					ptr2_item = NIL;
				      
				      if (ptr2_item != NIL) /* Patch-Eintrag vorhanden ? */
					{
					  diff = anzahl_ecken + 2 - i - j; /* was bleibt an Flaechen noch uebrig ?? */
					  
					  if (diff > wurzel->max_liste)
					    ptr3_baum = NIL;
					  else
					    ptr3_baum = *(wurzel->liste + diff);
					  
					  if (ptr3_baum != NIL)
					    {
					      if (ptr3_baum->max_liste != leer)
						{
						  bis3 =  ptr3_baum->max_liste;

						  for (z[0] = 1 ; z[0] <= bis3 ; z[0]++)
						    {
						      if ((*(ptr3_baum->liste + z[0]) != NIL) && (ptr3_baum->max_liste >= z[0]))
							ptr33_baum = *(ptr3_baum->liste + z[0]);
						      else 
							ptr33_baum = NIL;

						      if (ptr33_baum != NIL)
							{
							  if (ptr33_baum->max_liste != leer)
							    {
							      
							      /* Schreibe die zu uebergebenden Werte in eine Liste: */
							      

							      patch_daten.Items[0] = ptr1_item;
							      patch_daten.Items[1] = ptr2_item;
							      
							      patch_daten.Rand[0] = x;
							      patch_daten.Rand[1] = y;
							      patch_daten.Rand[2] = z[0];
							      
							      patch_daten.Flaechen[0] = i;
							      patch_daten.Flaechen[1] = j;
							      patch_daten.Flaechen[2] = diff;

							      if (ptr1_item == ptr2_item)
								patch_daten.test = TRUE;
							      else
								patch_daten.test = FALSE;

							      
							      if ((z[0] > x) && (no_br == FALSE))
								{
								  /* Brille 0 */
								  if (z[0] >= x - y)
								    {
								      z[1] =  z[0] + y - x;
								      patch_daten.Rand[3] = z[1];
								      
								      patch_daten.Items[2] = ptr3_item = test_patch(ptr33_baum, z[1]);
								      if (ptr3_item != NIL)
									baue_brille(0);
								    }
								  
								  /* Brille 1 */
								  if (z[0] >= x + y + 2)
								    {
								      z[1] = z[0] - x - y - 2;
								      patch_daten.Rand[3] = z[1];
								      patch_daten.Items[2] = ptr3_item = test_patch(ptr33_baum, z[1]);
								      if (ptr3_item != NIL)
									baue_brille(1);
								    }
								}
							      else if ((z[0] == x) && (no_br == FALSE))
								{
								  /* Brille 2 */
								  z[1] = y;
								  patch_daten.Rand[3] = z[1];
								  patch_daten.Items[2] = ptr3_item = test_patch(ptr33_baum, z[1]);
								  if (ptr3_item != NIL)
								    baue_brille(2);
								}
							      else if ((z[0] < x) && (no_sw == FALSE))/* Sandwich */
								{						
								  if (z[0] >= x - y)
								    {
								      z[1] = z[0] + y - x;

								      patch_daten.Rand[3] = z[1];
								      patch_daten.Items[2] = ptr3_item = test_patch(ptr33_baum, z[1]);
								      if (ptr3_item != NIL)
									baue_sandwich();
								    }
								}
							    }
							}
						    }
						}
					    }
					}
				    }
				}
			    }
			}
		    }
		}
            }
        }
    }
}


/* *********************************************************************** */
/*                                                                         */
/* Verklebung der Patches                                                  */
/*                                                                         */
/* Aufruf: verklebung()                                                    */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */

unsigned char verklebe(KNOTENTYP sequenz[3][max_ecken + 2], KNOTENTYP flaechen_patch[2], Item *vorfahre[2])
{
  unsigned char einschluss;
  int verkl[2],zaehler, bis;
  int bis2, i, j, k, l, m, o, s, t;
  int a, b, c, d, e, f;
  KNOTENTYP verklebung[2];
  KNOTENTYP verkl_0, verkl_1; /* die Verklebe_kanten des 1. und 2. Patches */
  KNOTENTYP first, second, rand_ecken, typ; /* welcher Patch ist der erste und welcher der zweite? -> first, second 
					       typ = 0 entspricht Durchschnitt, > 0 Einschluss */
  KNOTENTYP s_laenge[2]; /* Laenge der jeweiligen Sequenzen */
  KNOTENTYP symm[2]; /* siehe Funktion Symmetrie */
  KNOTENTYP null[2][max_ecken + 2]; /* beinhaltet die Nullen innerhalb der Patch-Sequenz */
  
  unsigned char dummy, bb_erzeugt = FALSE;

  /* die Variable rand_ecken enthaelt im Durchschnitt-Fall die Anzahl der Ecken auf dem Rand des Patches
     und im Einschluss-Fall die Anzahl der Ecken auf der 1. Kante an der verklebt wird. In diesem Fall
     enthaelt die Variable typ die die Anzahl der Ecken auf der 2. Kante an der verklebt wird. */
  
  zaehler_verklebe++;

  s_laenge[0] = sequenz[0][0];
  s_laenge[1] = sequenz[1][0];

  null[1][0] = 0;
  
  berechne_symmetrie = TRUE;

  if (berechne_symmetrie == TRUE)
    {
      berechne_symmetrie = FALSE;
      null[0][0] = 0;
      
      symm[0] = symmetrie(sequenz[0]); /* Symmetriebestimmung der Sequenzen */
      
      for (j = 2 ; j <= (symm[0] + 1) ; j++) /* sucht alle relevanten Nullen des ersten Patches */
	{
	  if (sequenz[0][j] == 0)
	    {
	      null[0][0]++;
	      null[0][null[0][0]] = j;
	    }
	}
    }

  symm[1] = symmetrie(sequenz[1]); /* Symmetriebestimmung der Sequenzen */

  for (j = 2 ; j <= sequenz[1][0] ; j++)
	sequenz[1][j + sequenz[1][0] - 1] = sequenz[1][j];

  for (j = 2 ; j <= (symm[1] + 1) ; j++) /* sucht alle relevanten Nullen des ersten Patches */
    {
      if (sequenz[1][j] == 0)
	{
	  null[1][0]++;
	  null[1][null[1][0]] = j;
	}
    }
    
  einschluss = FALSE;
  
  if ((flaechen_patch[0] == flaechen_patch[1]) && ((sequenz[0][0] == 2) && (sequenz[1][0] == 4)))
    einschluss = TRUE;
  else if (flaechen_patch[0] != flaechen_patch[1])
    einschluss = TRUE;
  
  if (((sequenz[0][0] == 2) || (sequenz[1][0] == 2)) && ((sequenz[0][0] == 4) || (sequenz[1][0] == 4)))
    /* Einschluss, aus dem eine Bauchbinde entsteht: der aeussere Patch hat genau 3 und der innere (wie immer) eine V2-Ecke. */
    {
      if (no_bb == FALSE) /* nur wenn BB gewuenscht */
	{
	  if (einschluss == TRUE)
	    {
	      if ((symm[0] == 3) || (symm[1] == 3)) /* die Laenge der Symmetrie einer Sequenz muss 3 betragen; ansonsten kann 
						       kein BB-Einschluss entstehen */
		{
		  if (sequenz[0][0] == 2) /* entweder wird der 1. Patch in den 2. Patch ... */
		    {
		      first  = 1;
		      second = 0;
		    }
		  else /* ... oder der 2. Patch in den 1. Patch geklebt. */
		    {
		      first  = 0;
		      second = 1;
		    }
		  
		  c = sequenz[second][2] + 1;
		  
		  /* diese Werte stehen fuer den aeusseren Patch schon fest */
		  sequenz[2][0] = 2;
		  sequenz[2][1] = sequenz[0][1] + sequenz[1][1];
		  
		  if (sequenz[2][1] <= anzahl_ecken - 2)
		    {
		      /* welche V-3-Kante wird verklebt? */
		      
		      for (j = 2 ; j <= sequenz[first][0] ; j++)
			{
			  if (j == 2) /* letzter Eintrag ist zustaendig fuer den Rand */
			    sequenz[2][2] = sequenz[first][4] + 1;
			  else
			    sequenz[2][2] = sequenz[first][j - 1] + 1;
			  
			  if ((sequenz[first][j] > sequenz[first][j + 1]) && (sequenz[first][j] - sequenz[first][j + 1] == c) 
			      && (sequenz[2][2] % 2 == 0))
			    {
			      bauch_binde_eintrag(vorfahre, 1, j, 2, sequenz, first, second);
			      bb_erzeugt = TRUE;

			      if (ausgabe == TRUE)
				{
				  for (i = 0 ; i <= 1 ; i++)
				    {
				      fprintf(out_stream,"%d=%d:", i, sequenz[i][1]);
				      for (k = 2 ; k <= sequenz[i][0] ; k++)
					fprintf(out_stream,"%d,", sequenz[i][k]);
				      fprintf(out_stream,"sym: %d\n", symm[i]);
				    }
				  
				  fprintf(out_stream,"\n");
				}
			      
			      if (ausgabe == TRUE)
				{
				  fprintf(out_stream,"embedding BB\n");
				  fprintf(out_stream,"___\n");
				}
			    }
			  else if ((sequenz[first][j + 1] > sequenz[first][j]) && (sequenz[first][j + 1] - sequenz[first][j] == c) 
				   && (sequenz[2][2] % 2 == 0))
			    {
			      bauch_binde_eintrag(vorfahre, 1, j, 2, sequenz, first, second);
			      bb_erzeugt = TRUE;

			      if (ausgabe == TRUE)
				{
				  for (i = 0 ; i <= 1 ; i++)
				    {
				      fprintf(out_stream,"%d=%d:", i, sequenz[i][1]);
				      for (k = 2 ; k <= sequenz[i][0] ; k++)
					fprintf(out_stream,"%d,", sequenz[i][k]);
				      fprintf(out_stream,"sym: %d\n", symm[i]);
				    }
				  
				  fprintf(out_stream,"\n");
				}
			      
			      
			      if (ausgabe == TRUE)
				{
				  fprintf(out_stream,"embedding BB\n");
				  fprintf(out_stream,"___\n");
				}
			    }
			}
		    }
		}
	    }
	}
      
    } /* Ende Bauchbinden-Einschluss */
  else if ((sequenz[0][0] == 3) && (sequenz[1][0] == 3)) /* koennte Bauchbinde werden */
    {
      if (no_bb == FALSE)
	{
	  if ((sequenz[0][2] == sequenz[1][2]) || (sequenz[0][3] == sequenz[1][3])
	      || (sequenz[0][2] == sequenz[1][3]) || (sequenz[0][3] == sequenz[1][2]))
	    {
	      sequenz[2][0] = 2; /* Laenge der neuen Sequenz immer 2 */
	      sequenz[2][1] = sequenz[1][1] + sequenz[0][1]; /* Flaechenanzahl immer gleich */
	      
	      if ((vorfahre[0] == vorfahre[1]) || (flaechen_patch[0] == flaechen_patch[1]))
		bis2 = 1;
	      else 
		bis2 = 2;
	      
	      for (k = 1 ; k <= bis2 ; k++)
		{
		  if (k == 1)
		    {
		      first = 0;
		      second = 1;
		    }
		  else
		    {
		      first = 1;
		      second = 0;
		    }
		  
		  for (l = 1 ; l <= symm[first] ; l++)
		    {
		      for (m = 1 ; m <= symm[second] ; m++)
			{
			  if (sequenz[first][l + 1] == sequenz[second][m + 1])
			    {
			      verkl[0] = l + 1;
			      verkl[1] = m + 1;
			      
			      if (verkl[0] == 2)
				sequenz[2][2] = sequenz[first][3];
			      else
				sequenz[2][2] = sequenz[first][2];
			      
			      if (verkl[1] == 2)
				sequenz[2][2] = sequenz[2][2] + sequenz[second][3] + 2;
			      else
				sequenz[2][2] = sequenz[2][2] + sequenz[second][2] + 2;
			      
			      bauch_binde_eintrag(vorfahre, 0, verkl[0], verkl[1], sequenz, first, second);
			      bb_erzeugt = TRUE;

			      if (ausgabe == TRUE)
				{
				  if (first == 1)
				    {
				      for (i = 1 ; i >= 0 ; i--)
					{
					  fprintf(out_stream,"%d=%d:", i, sequenz[i][1]);
					  for (o = 2 ; o <= sequenz[i][0] ; o++)
					    fprintf(out_stream,"%d,", sequenz[i][o]);
					  fprintf(out_stream,"sym: %d, verkl: %d\n", symm[i], verkl[i]);
					}
				    }
				  else
				    {
				      for (i = 0 ; i <= 1 ; i++)
					{
					  fprintf(out_stream,"%d=%d:", i, sequenz[i][1]);
					  for (o = 2 ; o <= sequenz[i][0] ; o++)
					    fprintf(out_stream,"%d,", sequenz[i][o]);
					  fprintf(out_stream,"sym: %d, verkl: %d\n", symm[i], verkl[i]);
					}
				    }
				  
				  fprintf(out_stream,"BB: %d:%d\n", sequenz[2][1], sequenz[2][2]);
				  
				  fprintf(out_stream,"\n");
				}
			    }
			}
		    }
		}
	    }
	}
    } /* Ende Bauchbinde */
  else if ((null[0][0] > 0) || (null[1][0] > 0)) /* nach einer Null wird verklebt (bis auf den Fall oben -> Bauchbinde) */
    {
      if (ausgabe == TRUE)
	{
	  for (i = 0 ; i <= 1 ; i++)
	    {
	      fprintf(out_stream,"%d=%d:", i, sequenz[i][1]);
	      for (j = 2 ; j <= sequenz[i][0] ; j++)
		fprintf(out_stream,"%d,", sequenz[i][j]);
	      fprintf(out_stream,"sym: %d\n", symm[i]);
	    }
	  
	  fprintf(out_stream,"\n");
	}
      
      einschluss = FALSE;
      if ((flaechen_patch[0] == flaechen_patch[1]) && ((sequenz[0][0] == 2) && (sequenz[1][0] > 4)))
	einschluss = TRUE;
      else if (flaechen_patch[0] != flaechen_patch[1])
	einschluss = TRUE;
      
      /* ********************************* EINSCHLUSS ************************************ */
      if (((sequenz[0][0] == 2) || (sequenz[1][0] == 2)) && ((sequenz[0][0] > 4) || (sequenz[1][0] > 4)))
	{
	  if (einschluss == TRUE)
	    {
	      if (sequenz[0][0] == 2) /* entweder wird der 1. Patch in den 2. Patch ... */
		{
		  first  = 1;
		  second = 0;
		}
	      else /* ... oder der 2. Patch in den 1. Patch geklebt. */
		{
		  first  = 0;
		  second = 1;
		}
	      
	      if (sequenz[first][0] >= 5) /* nach 0 Moeglichkeit zum verkleben (neuer Rand hat mind. eine V2-Ecke */
		{
		  if (null[first][0] > 0)
		    {
		      c = sequenz[second][2] + 1; /* c Ecken muessen eingeklebt werden */
		      verklebung[second] = 2;
		      
		      for (i = 1 ; i <= null[first][0] ; i++) /* laeuft alle relevanten Nullen des Patches durch */
			{
			  a = sequenz[first][null[first][i] + 1]; /* a und b Ecken stehen auf dem */
			  b = sequenz[first][null[first][i] + 2]; /* grossen Patch zur Verfuegung */
			  verklebung[first] = null[first][i] + 1;
			  
			  if ((c <= a) && (b == a - c)) /* Kante a ist groesser als Kante b (fuer die Richtung der Verklebung) */
			    {
			      sequenz[2][0] = sequenz[first][0] - 3;
			      sequenz[2][1] = sequenz[first][1] + sequenz[second][1];
			      
			      for (j = 3 ; j <= sequenz[2][0] ; j ++)
				sequenz[2][j] = sequenz[first][j + null[first][i] + 1];
			      
			      sequenz[2][2] = 1 + sequenz[first][null[first][i] + 3];
			      
			      if (ausgabe == TRUE)
				{
				  fprintf(out_stream,"%d:",sequenz[2][1]);
				  for (j = 2 ; j <= sequenz[2][0] ; j++)
				    fprintf(out_stream,"%d,",sequenz[2][j]);
				  
				  fprintf(out_stream,"\n");
				  
				  fprintf(out_stream,"embedding - ");
				}
			      
			      for (j = 2 ; j <= sequenz[2][0] ; j++)
				sequenz[2][j + sequenz[2][0] - 1] = sequenz[2][j];
			      
			      dummy = baum_eintrag(vorfahre, sequenz[2], a, b, first, second, sequenz[0][verklebung[0]], symm, verklebung, s_laenge);
			      if (dummy = TRUE)
				bb_erzeugt = TRUE;

			      if (ausgabe == TRUE)
				fprintf(out_stream,"___\n");
			    }
			  else if ((c <= b) && (a == b - c)) /* Kante b ist groesser als Kante a (fuer die Richtung der Verklebung) */
			    {
			      sequenz[2][0] = sequenz[first][0] - 3;
			      sequenz[2][1] = sequenz[first][1] + sequenz[second][1];
			      
			      for (j = 3 ; j <= sequenz[2][0] ; j ++)
				sequenz[2][j] = sequenz[first][j + null[first][i] + 1];
			      
			      sequenz[2][2] = 1 + sequenz[first][null[first][i] + 3];
			      
			      if (ausgabe == TRUE)
				{
				  fprintf(out_stream,"%d:",sequenz[2][1]);
				  for (j = 2 ; j <= sequenz[2][0] ; j++)
				    fprintf(out_stream,"%d,",sequenz[2][j]);
				  
				  fprintf(out_stream,"\n");
				  fprintf(out_stream,"embedding - ");
				}
			      
			      for (j = 2 ; j <= sequenz[2][0] ; j++)
				sequenz[2][j + sequenz[2][0] - 1] = sequenz[2][j];
			      
			      dummy = baum_eintrag(vorfahre, sequenz[2], a, b, first, second, sequenz[0][verklebung[0]], symm, verklebung, s_laenge);
			      if (dummy == TRUE)
				bb_erzeugt = TRUE;

			      if (ausgabe == TRUE)
				fprintf(out_stream,"___\n");
			      
			    }
			}
		    }
		} /* ENDE von "if (sequenz[first][0] >= 5)" */
	    }  

	  return(bb_erzeugt);
	  
	}/* if ((sequenz[0][0] == 2) || (sequenz[1][0] == 2)) */
      
      
      
      /* ************************************ Durchschnitt ************************************* */
      
      for (j = 0 ; j <= 1 ; j++) /* wechselt die zwei Patches jeweils als first und second */
	{
	  if (((vorfahre[0] != vorfahre[1]) || (j == 0)) && ((flaechen_patch[0] != flaechen_patch[1]) || (j == 0))) /* bei gleichen Vorfahren nur eine Richtung ! */
	    {
	      first = j;
	      
	      if (j == 0)
		second = 1;
	      else
		second = 0;
	      
	      if (sequenz[first][0] != 3) /* mit "... == 3" wuerde sich keine korrekte Markierung ergeben! */
		{
		  rand_ecken = 0;
		  
		  /* Zaehle die V3-Ecken auf dem Rand des ersten Patches: */
		  for (i = 2 ; i <= sequenz[first][0] ; i++)
		    rand_ecken = rand_ecken + sequenz[first][i];
		  
		  /* addiere die V2-Ecken des Randes hinzu und ziehe die Stelle fuer die Flaechenanzahl ab: */
		  rand_ecken = rand_ecken + sequenz[first][0] - 1;
		  
		  for (i = 1 ; i <= null[first][0] ; i++) /* laeuft alle relevanten Nullen der Patches durch */
		    {
		      verklebung[first] = null[first][i] + 1;
		      
		      a = 0; /* 0-Kante */
		      b = sequenz[first][verklebung[first]]; /* an dieser Kante wird verklebt */
		      c = sequenz[first][verklebung[first] + 1]; /* Kante nach der Verklebekante */
		      
		      for (k = 2 ; k <= symm[second] + 1 ; k++)
			{
			  if (sequenz[second][k] == sequenz[first][verklebung[first]])
			    {
			      verklebung[second] = k;
			      
			      if (verklebung[second] == 2)
				d = sequenz[second][sequenz[second][0]];
			      else
				d = sequenz[second][verklebung[second] - 1];
			      
			      e = sequenz[second][verklebung[second]];
			      f = sequenz[second][verklebung[second] + 1];
			      
			      if (sequenz[second][0] == 3) /* Patch mit 2 V-2-Ecken wird angeklebt,... */
				{
				  sequenz[2][0] = sequenz[second][0] + sequenz[first][0] - 5; 
				  sequenz[2][1] = sequenz[second][1] + sequenz[first][1]; 
				  sequenz[2][2] = a + f + c + 2;
				  
				  bis = verklebung[first] + sequenz[first][0] - 3;
				  
				  for (zaehler = 3 , t = verklebung[first] + 2 ; t <= bis ; zaehler++ , t++)
				    sequenz[2][zaehler] = sequenz[first][t];
				  
				  j = 1; /* umgekehrte Verklebung (falls nicht schon erfolgt) nicht	moeglich */
				  
				  if (ausgabe == TRUE)
				    {
				      fprintf(out_stream,"%d:",sequenz[2][1]);
				      for (s = 2 ; s <= sequenz[2][0] ; s++)
					fprintf(out_stream,"%d,", sequenz[2][s]);
				      fprintf(out_stream,"Verkl_0: %d, Verkl_1: %d, Schn.: %d", verklebung[0], verklebung[1], sequenz[0][verklebung[0]]);
				      
				    }
				  
				  for (s = 2 ; s <= sequenz[2][0] ; s++) /* Kopiert die Sequenz hinter sich selbst. */
				    sequenz[2][s + sequenz[2][0] - 1] = sequenz[2][s]; 
				  
				  dummy = baum_eintrag(vorfahre, sequenz[2], rand_ecken, Durchschnitt, first, second, sequenz[0][verklebung[0]], symm, verklebung, 
						       s_laenge);
				  if (dummy == TRUE)
				    bb_erzeugt = TRUE;

				  if (ausgabe == TRUE)
				    fprintf(out_stream,"___\n");
				}
			      else if (sequenz[second][0] != 3) /* 2. Patch mehr als 2 V-2-Ecken */
				{
				  sequenz[2][0] = sequenz[second][0] + sequenz[first][0] - 5; 
				  sequenz[2][1] = sequenz[second][1] + sequenz[first][1]; 
				  sequenz[2][2] = a + 1 + f;
				  
				  zaehler = verklebung[second] + 2;
				  s = 3;
				  
				  for (t = 2 ; t <= sequenz[second][0] - 3 ; t++ , zaehler++ , s++)
				    sequenz[2][s] = sequenz[second][zaehler];
				  
				  sequenz[2][s] =  d + c + 1;
				  s++;
				  
				  zaehler = verklebung[first] + 2;
				  
				  for (t = 2 ; t <= sequenz[first][0] - 3 ; t++ , zaehler++ , s++)
				    sequenz[2][s] = sequenz[first][zaehler];
				  
				  if (ausgabe == TRUE)
				    {
				      fprintf(out_stream,"%d:",sequenz[2][1]);
				      for (s = 2 ; s <= sequenz[2][0] ; s++)
					fprintf(out_stream,"%d,", sequenz[2][s]);
				      fprintf(out_stream,"Verkl_0: %d, Verkl_1: %d, Schn.: %d",verklebung[0], verklebung[1], sequenz[0][verklebung[0]]);
				    }
				  
				  for (s = 2 ; s <= sequenz[2][0] ; s++) /* Kopie hinter sich selbst (wegen Kanonizitaetstest). */
				    sequenz[2][s + sequenz[2][0] - 1] = sequenz[2][s];
				  
				  dummy = baum_eintrag(vorfahre, sequenz[2], rand_ecken, Durchschnitt, first, second, sequenz[0][verklebung[0]], symm, verklebung, 
						       s_laenge);
				  if (dummy == TRUE)
				    bb_erzeugt = TRUE;

				  if (ausgabe == TRUE)
				    fprintf(out_stream,"___\n");
				  
				} /* Ende else if */
			      
			    } /* Ende if verklebung[second] != 0*/
			  
			} /* Ende for k */
		      
		    } /* Ende for i */
		  
		} /* Ende if */
	      
	    } /* Ende gleiche Vorfahren */
	  
	} /* Ende for j */
    
    }

  return(bb_erzeugt);

}



/* *********************************************************************** */
/*                                                                         */
/* Besucht Knoten des Patch-Baumes und verklebt zu neuen Patches           */
/*                                                                         */
/*                                                                         */
/* Aufruf: visit(Nummer der Sequenz, Pointer auf Baumknoten)               */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */

unsigned char visit(unsigned char nummer, Baum *ptr_baum, Baum *wurzel_patch_2, KNOTENTYP flaechen_patch[2], KNOTENTYP sequenz[3][max_ecken + 2], Item *vorfahre[2], 
		    unsigned char init)
{
  int bis, fd, flaechen_p, flaechen_graph, sl_patch;
  int j, x; /* Zaehlvariablen */
  
  unsigned char dummy, bb_erzeugt = FALSE;

  Item *ptr_item;


  zaehler_visit++;

  if (init == TRUE)
    {
      sequenz[0][0] = sequenz[1][0] = 1;
      sequenz[2][0] = 0;
      sequenz[0][1] = flaechen_patch[0];
      sequenz[1][1] = flaechen_patch[1];
    }

  bis = 0;
  x = 0;
  
  if (nummer == 1)
    {
      if (sequenz[1][0] >= 2)
	{
	  /* Eulerkriterium: */
		
	  fd = (sequenz[0][0] + sequenz[1][0] - 2 - 4 + 4); /* Flaechendifferenz: soviele Flaechen kommen noch mind. hinzu, damit ein 4reg Graph entstehen kann */
	  flaechen_p = sequenz[0][1] + sequenz[1][1]; /* Soviele Flaechen wird der neue Patch haben */
	  flaechen_graph = anzahl_ecken + 2; /* soviele Flaechen hat ein 4reg Graph mit anzahl_ecken Ecken */

	  sl_patch = sequenz[0][0] + sequenz[1][0] - 6; /* die Sequenzlaenge des neuen Patches */
	  	  
	  if ((fd + flaechen_p > flaechen_graph) || ((anzahl_ecken > 6) && (sl_patch > anzahl_ecken - 3 - flaechen_p)))
	    {
	      x = ptr_baum->max_liste + 1; /* dieser Ast muss nicht weiter durchlaufen werden */
	      bis = anzahl_ecken + 1;
	      
	    }
	}
    }

  if (bis <= anzahl_ecken)
    {
      if (ptr_baum->eintrag != NIL)
	{
	  switch(nummer)
	    {
	    case 0:
	      {
		vorfahre[0] = ptr_baum->eintrag;
		
		/* Um Abfragen, die das Wechseln zum Anfang der Sequenz beim Erreichen des
		   Endes zu vermeiden, wird die gesamte Sequenz nochmals hinter die
		   eigentliche Sequenz kopiert: */
		
		for (j = 2 ; j <= sequenz[0][0] ; j++)
		  sequenz[0][j + sequenz[0][0] - 1] = sequenz[0][j];

		visit(1, wurzel_patch_2, wurzel_patch_2, flaechen_patch, sequenz, vorfahre, FALSE); /* Patch zum Verkleben suchen und dabei vorne anfangen */

	      } /* Ende Fall 0 */
	    
	    break;
	    
	    case 1:
	      {
		vorfahre[1] = ptr_baum->eintrag;
		for (j = 2 ; j <= sequenz[1][0] ; j++)
		  sequenz[1][j + sequenz[1][0] - 1] = sequenz[1][j];	

		dummy = verklebe(sequenz, flaechen_patch, vorfahre);
		if (dummy == TRUE)
		  bb_erzeugt = TRUE;


	      } /* Ende CASE 1 */
	    
	    } /* Ende SWITCH */
	  
	} /* Ende IF (ptr_baum->eintrag != NIL) */
      
    } /* Ende IF weiter == TRUE */
  
  
  /****************************************************************/
  /* Programmblock zum Durchlaufen der weiterfuehrenden Aeste ... */
  /****************************************************************/
  
  sequenz[nummer][0] = sequenz[nummer][0] + 1; /* die Sequenzlaenge erhoeht sich um 1 */
  
  while ((x <= ptr_baum->max_liste) && (ptr_baum->max_liste != leer)) /* solange noch ein Eintrag in der Liste
									 vorhanden ist ... */
    {
      sequenz[nummer][sequenz[nummer][0]] = x; /* Nummer des Astes an letzter Stelle der Sequenz eintragen */
      
      if (ptr_baum->liste != NIL)
	if (*((ptr_baum->liste) + x) != NIL) /* falls vorhanden, naechster Baumknoten; x entspricht Nummer des Listenelementes */
	  {
	    dummy = visit(nummer, *((ptr_baum->liste) + x), wurzel_patch_2, flaechen_patch, sequenz, vorfahre, FALSE); 
	    if (dummy == TRUE)
	      bb_erzeugt = TRUE;
	  }
      
      x++;
      
    }
  
  sequenz[nummer][0] = sequenz[nummer][0] - 1; /* wenn visit verlassen wird, Sequenzlaenge um 1 herunterzaehlen */

  return(bb_erzeugt);

}

/* *********************************************************************** */
/*                                                                         */
/* Aufrufende Funktion zu visit() verwaltet Flaechen                       */
/*                                                                         */
/*                                                                         */
/* Aufruf: visit_main(Pointer auf Baumknoten)                              */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */
void main_visit(void)
{
  KNOTENTYP i, j, k, start;
  KNOTENTYP flaechen_patch[2];
  KNOTENTYP sequenz[3][max_ecken + 2];
  Item *vorfahre[2]; /* Zeiger auf die Vorfahren der zu verklebenden Patches. */

  for (i = 2 ; i <= anzahl_ecken - 2 ; i++)
    {
      if (i % 2 == 0)   /* gerade */
	start = i / 2;
      else             /* ungerade */
	start = (i / 2) + 1; 
      
      for (j = start ; j <= i - 1 ; j++)
	{
	  k = i - j; /* i wird in j und k aufgeteilt */

	  if ((wurzel->max_liste >= j) && (wurzel->max_liste != leer)) 
	    {
	      flaechen_patch[0] = j;
	      flaechen_patch[1] = k;

	      sequenz[0][0] = sequenz[1][0] = 1;
	      sequenz[2][0] = 0;
	      sequenz[0][1] = flaechen_patch[0];
	      sequenz[1][1] = flaechen_patch[1];
    
	      if ((wurzel->liste[j] != NIL) && (wurzel->liste[k] != NIL))
		visit(0, wurzel->liste[j], wurzel->liste[k], flaechen_patch, sequenz, vorfahre, FALSE);
	    }
	}
    }
}

/* *********************************************************************** */
/*                                                                         */
/* Fehlerbehandlung                                                        */
/*                                                                         */
/* Aufruf: fehler(Art des Fehlers)                                         */
/* Ausgabe: void                                                           */
/*                                                                         */
/* *********************************************************************** */

void fehler(unsigned char f)
{
  int i;
  
  switch (f)
    {
    case speicher:
      {
	fprintf(stderr,"\n\n memory assignment error!!");
	fprintf(stderr,"\n program died\n");
	exit(0);
      }
    case eingabe:
      {
	fprintf(stderr,"\n\n input error !!");
	fprintf(stderr,"\n please review the parameters: ");
	
	fprintf(stderr,"\n\n number of vertices: %d", anzahl_ecken);
	
	for (i = 1 ; i <= flaechen[0] ; i++)
	  fprintf(stderr,"\n surface %d: %d-Eck", i, flaechen[i]);
	
	fprintf(stderr,"\n\n");

	Parameter();
	
	exit(0); /* Programm sauber beenden */
      }
    case datei:
      {
	fprintf(stderr,"\n\nerror opening the output file!!\n\n");
	
	exit(0);
      }
    default:
      {
      }
    }
}

/* *********************************************************************** */
/*                                                                         */
/* MAIN Funktion                                                           */
/*                                                                         */
/* Aufruf: enumerate #Ecken ...                                            */
/* Ausgabe: 0                                                              */
/*                                                                         */
/* *********************************************************************** */

int main(int argc, char *argv[])
{
  Item *ptr_item;
  
  time_t time1, time2;

  clock_t clock1;
  
  char dummy[100];
  int von, bis, k;
  unsigned char i, j, l, gesamt;

  char strpuffer[20];
  char filename_graphen[250];

  int fd;

  /* falls keine Parameter uebergeben werden, Hilfeseite anzeigen und abbrechen: */
  
  if (argc < 2)
	fehler(eingabe);
  
  /* Behandlung der uebergebenen Parameter und evtl. Programmabbruch bei falscher Uebergabe.
     Ausserdem werden globale Variablen gesetzt, die oft gebraucht werden: */
  
  flaechen[0] = 0;
  ausgabe     = FALSE;
  darstellung = FALSE;
  
  so = no_iso = no_planar = no_spiegel = no_bb =  no_br =  no_sw =  no_cut = patch_test = FALSE;
  sequenz_tiefe = 6; /* falls ein Test ueber die wirklich verklebten Patches stattfindet ist das die Laenge der Sequenzen die maximal dargestellt werden */

  for (i = 1 ; i < argc ; i++) /* Auswertung der uebergebenen Parameter: */
    switch (argv[i][0])
      {
      case 'n':
	{
	  strcpy(dummy, "no_iso\0");

	  if (strcmp(dummy, argv[i]) == 0) /* no_iso */
	    no_iso = TRUE;
	  
	  strcpy(dummy, "no_planar\0");
	  
	  if (strcmp(dummy, argv[i]) == 0) /* no_planar */
	    no_planar = TRUE;

	  strcpy(dummy, "no_mirror\0");
	  
	  if (strcmp(dummy, argv[i]) == 0) /* no_spiegel */
	    no_spiegel = TRUE;

	  strcpy(dummy, "no_bb\0");
	  
	  if (strcmp(dummy, argv[i]) == 0) /* no_bb */
	    no_bb = TRUE;

	  strcpy(dummy, "no_br\0");
	  
	  if (strcmp(dummy, argv[i]) == 0) /* no_br*/
	    no_br = TRUE;

	  strcpy(dummy, "no_sw\0");
	  
	  if (strcmp(dummy, argv[i]) == 0) /* no_sw */
	    no_sw = TRUE;
	  
	  strcpy(dummy, "no_cut\0");
	  
	  if (strcmp(dummy, argv[i]) == 0) /* no_1cut */
	    no_cut = TRUE;

	  if ((no_planar == FALSE) && (no_iso == FALSE) && (no_spiegel == FALSE) && (no_bb == FALSE) && (no_br == FALSE) && (no_sw == FALSE) && (no_cut == FALSE))
	    {
	      fprintf(stderr, "\n\n wrong parameter: %s\n", argv[i]);
	      exit(0);
	    } 
	}
      break;

//      case 'd':
//	{
//	  strcpy(dummy, "d\0");
//	  
//	  if (strcmp(dummy, argv[i]) == 0) /* Darstellung */
//	    darstellung = TRUE;
//	  else
//	    {
//	      fprintf(stderr, "\n\n wrong parameter: %s\n", argv[i]);
//	      exit(0);
//	    } 
//	}
 //     break;
      
      case 'v':
	{
	  anzahl_ecken = atoi(argv[i] + 1);

	  if (anzahl_ecken <= 5)
	    fehler(eingabe);
	}
      break;
      
      case '-':
	{
	  if (atoi(argv[i] + 1) == 0) /* nichts oder etwas Falsches eingegeben */
	    fehler(eingabe);
	  
	  j = 1;
	  
	  while(argv[i][j] != '-')
	    {
	      dummy[j-1] = argv[i][j];
	      dummy[j] = 0;
	      j++;
	    }
	  
	  von = atoi(dummy);
	  
	  k = 0; j++;
	  
	  while(argv[i][j] != 0)
	    {
	      dummy[k] = argv[i][j];
	      dummy[k + 1] = 0;
	      j++;
	      k++;
	    }
	  
	  bis = atoi(dummy);
	  
	  k = bis - von;
	  if (k <= 0)
	    fehler(eingabe);
	  
	  for( j = 0 ; j <= k ; j++)
	    {
	      for(l = 1 ; l <= flaechen[0] ; l++)
		if(flaechen[l] == von + j)
		  break;
	      
	      
	      if (l == flaechen[0] + 1)
		{
		  flaechen[0]++;
		  flaechen[flaechen[0]] = von + j;
		}
	    }
	}
      break;
      
      case 'f':
	{
	  for (l = 1 ; l <= flaechen[0] ; l++)
	    if (flaechen[l] == atoi(argv[i] + 1))
	      break;

	  if (flaechen[0] + 1 == l) {
	      flaechen[0]++;
	      flaechen[flaechen[0]] = atoi(argv[i] + 1);
	      if (flaechen[flaechen[0]] == 0)
			fehler(eingabe);
	    }
	}
      break;
      
//      case 'p':
//	{
//	  strcpy(dummy, "patchtest\0");
//	  
//	  if (strncmp(dummy, argv[i],9) == 0) /* patchtest */
//	    {
//	      patch_test = TRUE;
//	      sequenz_tiefe = atoi(argv[i] + 9);
//	      if (sequenz_tiefe == 0)
//		sequenz_tiefe = 5;
//	    }
//	}
//      break;
//
//      case 'a':
//	{
//	  strcpy(dummy, "a\0");
//	  
//	  if (strcmp(dummy, argv[i]) == 0) /* Ausgabe */
//	    ausgabe = TRUE;
//	  else
//	    {
//	      fprintf(stderr, "\n\n wrong parameter: %s\n", argv[i]);
//	      exit(0);
//	    } 
//	}
//      break;

      case 's':
	{
	  strcpy(dummy, "so\0");
	  
	  if (strcmp(dummy, argv[i]) == 0) /* Ausgabe auf stdout */
	    so = TRUE;
	  else
	    {
	      fprintf(stderr, "\n\n wrong parameter: %s\n", argv[i]);
	      exit(0);
	    }
	} 
      }



  if (anzahl_ecken % 2 != 0)
    {
      no_bb = TRUE; /* bei einer ungeraden Anzahl Ecken hat ein Graph immer einen besseren Pfad als einen Bauchbinden-Pfad. */
      gerade = FALSE;
    }
  else
    gerade = TRUE;

  if (anzahl_ecken > max_ecken - 3)
    {
      fprintf(stderr,"\n This program can handle up to now only %d vertices\n", max_ecken - 2);
      exit(0); /* Programm sauber beenden */
    }

  if (flaechen[0] > 20) /* mehr als 20 Flaechen eingegeben */
    {
      fprintf(stderr,"\n Only 20 faces will be recognized\n",
	     max_ecken);
      exit(0); /* Programm sauber beenden */
    }
  
  quicksort(flaechen, 1, flaechen[0]); /* Flaechen werden der Groesse nach sortiert */
  
  if (flaechen[1] < 3) {
    fprintf(stderr,"first face is lower than 3\n");
    fehler(eingabe);
  }
  
  if (flaechen[1] != 3)
    {
      fprintf(stderr,"\n\n Refering to EULER formula there is no 4-regular graph with no 3-gon. \n");
      fprintf(stderr," This face will be added automatically\n");
         
      for (i = flaechen[0] ; i >= 1 ; i--)
	flaechen[i + 1] = flaechen[i];
      
      flaechen[1] = 3;
      flaechen[0] = flaechen[0] + 1;
    }

  if (flaechen[0] == 2)
    {
      fd = 4 - flaechen[2];

      if (fd < 0)
	{
	  fd = abs(fd) + 1;
      
	  for (i = 1 ; i <= (anzahl_ecken - 6) / fd ; i++)
	    if (6 + (i * fd) == anzahl_ecken)
	      i = anzahl_ecken;

	  if (i != anzahl_ecken + 1)
	    {
	      fprintf(stderr,"\n\n Refering to EULER formula there are no graphs with this combination: \n");
	      fprintf(stderr," graphs on %d vertices with %d-gons and %d-gons\n\n", anzahl_ecken, flaechen[1], flaechen[2]);
	      exit(0);
	    }
	}
    }

  if (patch_test == TRUE)
    {
      fprintf(stderr, "\n A test will be initiated. Information about the patches and their usage in generated graphs\n");
      fprintf(stderr, "will be stored in PatchTest.ENU\n");
      fprintf(stderr, "\n The test will use sequences with a length up to %d\n", sequenz_tiefe);
    }

  /* Ausgabe der eingegebenen Parameter: */

  fprintf(stderr, "\n\n ***************************************************************************************\n");
  fprintf(stderr, "\n Generating all 4-regular, planar, connected, simple graphs with %d vertices\n\n", anzahl_ecken);
  
  for (i = 1 ; i <= flaechen[0] - 1; i++)
    fprintf(stderr," %d-gons,",flaechen[i]);
  fprintf(stderr," and %d-gons",flaechen[i]);
  fprintf(stderr, " will be used in the graphs:\n");

  /* Hier beginnt der eigentliche Programmblock */
  
  berechne_teiler(anzahl_ecken + 1); /* hier werden die Teiler einzelner Zahlen ermittelt; wird fuer den Symmetriecheck benoetigt.  */

  /* Erzeugung des Filenamens fuer das Graphen-File: */

  if (so == FALSE)
    {
      sprintf(filename_graphen,"4reg_v%d",anzahl_ecken);
      
      for (i = 1 ; i <= flaechen[0] ; i++)
	{
	  sprintf(strpuffer,"_f%d",flaechen[i]);
	  strcat(filename_graphen,strpuffer);
	}
      
      strcpy(graphs_file_name, filename_graphen); /* globale Varibale fuer den Filenamen setzen */
      
      graphs_file_ptr = fopen(graphs_file_name, "w+");
      if (graphs_file_ptr == NULL)
	fehler(datei);
    }
  else
    {
      graphs_file_ptr = stdout;
    }
  
  fprintf(graphs_file_ptr, ">>planar_code<<");
    
  if (ausgabe == TRUE)
    test = fopen("BaumTest.ENU", "w");
  
  out_stream = stderr;
  
  if (ausgabe == TRUE)
    out_stream = test;
  
  patchbau = fopen("patchbau.enu","w");
 
  if (anzahl_ecken > 5)
    {
      fprintf(stderr,"\n\n");
      fprintf(stderr,"   ***************************************************************************\n");
      fprintf(stderr,"   * If you have problems running the program, please raise                  *\n");
      fprintf(stderr,"   * the amount of memory allocated for the STACK.                           *\n");
      fprintf(stderr,"   *                                                                         *\n");
      fprintf(stderr,"   * UNIX: limit stacksize unlimited OR ulimit -s unlimited                  *\n");
      fprintf(stderr,"   *                                                                         *\n");
      fprintf(stderr,"   * If you are not permitted to change this value, please contact your      *\n");
      fprintf(stderr,"   * System Administrator.                                                   *\n");
      fprintf(stderr,"   ***************************************************************************\n\n");
    }

  fprintf(stderr,"\n Starting calculation ...");
  fprintf(stderr," Building the patches ... \n\n");

  time(&time1);
  
  Create_Patches(); 
  
  time(&time2);
  
  if (no_iso == TRUE)
    fprintf(out_stream,"\n     Warning!! No test on isomorphic graphs!!\n\n");

  if (no_planar == TRUE)
    fprintf(out_stream,"\n     Warning!! No planar-code check in isomorphism check!!\n\n");

  if (no_spiegel == TRUE)
    fprintf(out_stream,"\n     Warning!! No test on mirror-image of paths!!\n\n");

  space_info = mallinfo();

  fprintf(out_stream," graphs build on: BB:%d, SW:%d, BR:%d \n", gleich_bb, gleich_sw, gleich_br);
  fprintf(out_stream," number of patches: %d\n", anz_patches);
  fprintf(out_stream," good patches: %d\n\n", gute_patches);
  fprintf(out_stream," mediate and immediate used for building graphs: %d\n\n", zum_bau_verwendet);
  fprintf(out_stream,"\n used memory calculated by mallinfo(): %d bytes\n\n\n", space_info.usmblks + space_info.uordblks);
    /*fprintf(out_stream," used memory in byte: %d\n", heapzaehler);*/
  
  fprintf(out_stream,"\n generation time: about %.0f seconds\n", difftime(time2,time1));

  /*fprintf(out_stream,"\n counter: ");
  fprintf(out_stream,"\n visit:        %d\n", zaehler_visit);
  fprintf(out_stream," baum_eintrag: %d\n", zaehler_baum_eintrag);
  fprintf(out_stream," verklebe:     %d\n", zaehler_verklebe);
  fprintf(out_stream," memory_alloc: %d\n", zaehler_memory);*/
  /*  fprintf(out_stream," davon: %d Knoten, %d Listen und %d Patches\n", zaehler_memory_baum_knoten, zaehler_memory_liste, zaehler_memory_item); */
  
  fprintf(stderr, "\n ***************************************************************************************\n");
  
  /* Alle Files wieder schliessen: */

  fclose(patchbau);

  if (ausgabe == TRUE)
    fclose(test);
  
  fclose(graphs_file_ptr);

  /*space_info = mallinfo();*/
  /*  printf("\n used memory calculated by mallinfo(): %d bytes\n\n\n", space_info.usmblks + space_info.uordblks);*/

  return(0);
  
}/* main ENDE */
