import os
import sys
import scipy.io as spio
import numpy as np
from datetime import datetime, timedelta
import matplotlib
import matplotlib.pyplot as plt
from matplotlib.ticker import MultipleLocator, FormatStrFormatter
from matplotlib.colors import ListedColormap
from matplotlib.dates import DateFormatter, DayLocator, HourLocator, \
    MinuteLocator, date2num
plt.switch_backend('Agg')


def celltolist(xtickstr):
    """
    convert list of list to list of string.

    Examples
    --------

    [['2010-10-11'], [], ['2011-10-12]] =>
    ['2010-10-11], '', '2011-10-12']
    """

    tmp = []
    for iElement in range(0, len(xtickstr)):
        if not len(xtickstr[iElement][0]):
            tmp.append('')
        else:
            tmp.append(xtickstr[iElement][0][0])

    return tmp


def datenum_to_datetime(datenum):
    """
    Convert Matlab datenum into Python datetime.

    Parameters
    ----------
    Date: float

    Returns
    -------
    dtObj: datetime object

    """
    days = datenum % 1
    hours = days % 1 * 24
    minutes = hours % 1 * 60
    seconds = minutes % 1 * 60

    dtObj = datetime.fromordinal(int(datenum)) + \
        timedelta(days=int(days)) + \
        timedelta(hours=int(hours)) + \
        timedelta(minutes=int(minutes)) + \
        timedelta(seconds=round(seconds)) - timedelta(days=366)

    return dtObj


def rmext(filename):
    """
    remove the file extension.

    Parameters
    ----------
    filename: str
    """

    file, _ = os.path.splitext(filename)
    return file


def pollyDisplayBscRamanNR(tmpFile, saveFolder):
    """
    Description
    -----------
    Display the profiles of aerosol optical properties and meteorological data.

    Parameters
    ----------
    tmpFile: str
    the .mat file which stores the data.

    saveFolder: str

    Usage
    -----
    pollyDisplayBscRamanNR(tmpFile, saveFolder)

    History
    -------
    2019-01-10. First edition by Zhenping
    """

    if not os.path.exists(tmpFile):
        print('{filename} does not exists.'.format(filename=tmpFile))
        return

    # read matlab .mat data
    try:
        mat = spio.loadmat(tmpFile, struct_as_record=True)
        figDPI = mat['figDPI'][0][0]
        flagWatermarkOn = mat['flagWatermarkOn'][0][0]
        if mat['partnerLabel'].size:
            partnerLabel = mat['partnerLabel'][0]
        else:
            partnerLabel = ''
        startInd = mat['startInd'][:][0][0]
        endInd = mat['endInd'][:][0][0]
        height = mat['height'][:][0]
        time = mat['time'][:][0]
        refHBaseNR355 = mat['refHBaseNR355'][:][0][0]
        refHTopNR355 = mat['refHTopNR355'][:][0][0]
        refHBaseNR532 = mat['refHBaseNR532'][:][0][0]
        refHTopNR532 = mat['refHTopNR532'][:][0][0]
        aerBsc_355_raman = mat['aerBsc_355_raman'][:][0]
        aerBsc_532_raman = mat['aerBsc_532_raman'][:][0]
        aerBsc_1064_raman = mat['aerBsc_1064_raman'][:][0]
        aerBsc355_NR_raman = mat['aerBsc355_NR_raman'][:][0]
        aerBsc532_NR_raman = mat['aerBsc532_NR_raman'][:][0]
        meteorSource = mat['meteorSource'][:][0]
        temperature = mat['temperature'][:][0]
        pressure = mat['pressure'][:][0]
        pollyVersion = mat['CampaignConfig']['name'][0][0][0]
        location = mat['CampaignConfig']['location'][0][0][0]
        version = mat['PicassoConfig']['PicassoVersion'][0][0][0]
        fontname = mat['PicassoConfig']['fontname'][0][0][0]
        dataFilename = mat['PollyDataInfo']['pollyDataFile'][0][0][0]
        yLim_NR_RCS = mat['yLim_NR_RCS'][:][0]
        xLim_Profi_NR_Bsc = mat['xLim_Profi_NR_Bsc'][:][0]
        imgFormat = mat['imgFormat'][:][0]

    except Exception as e:
        print(e)
        print('Failed reading %s' % (tmpFile))
        return

    # set the default font
    matplotlib.rcParams['font.sans-serif'] = fontname
    matplotlib.rcParams['font.family'] = "sans-serif"

    # display near-range backscatter with raman method
    fig = plt.figure(figsize=[5, 8])
    ax = fig.add_axes([0.21, 0.15, 0.74, 0.75])
    p1, = ax.plot(aerBsc_355_raman * 1e6, height, color='#0000ff',
                  linestyle='-', label='355 nm (FR)', zorder=2)
    p2, = ax.plot(aerBsc_532_raman * 1e6, height, color='#00b300',
                  linestyle='-', label='532 nm (FR)', zorder=2)
    p3, = ax.plot(aerBsc_1064_raman * 1e6, height, color='#e60000',
                  linestyle='-', label='1064 nm (FR)', zorder=3)
    p4, = ax.plot(aerBsc355_NR_raman * 1e6, height, color='#9e3eff',
                  linestyle='-', label='355 nm (NR)', zorder=4)
    p5, = ax.plot(aerBsc532_NR_raman * 1e6, height, color='#42f5cb',
                  linestyle='-', label='532 nm (NR)', zorder=4)

    ax.set_xlabel('Backscatter Coefficient [$Mm^{-1}*sr^{-1}$]', fontsize=15)
    ax.set_ylabel('Height (m)', fontsize=15)
    ax.legend(handles=[p1, p2, p3, p4, p5], loc='upper right', fontsize=12)

    ax.set_ylim(yLim_NR_RCS.tolist())
    ax.yaxis.set_major_locator(MultipleLocator(1000))
    ax.yaxis.set_minor_locator(MultipleLocator(200))
    ax.set_xlim(xLim_Profi_NR_Bsc.tolist())
    ax.grid(True)
    ax.tick_params(axis='both', which='major', labelsize=15,
                   right=True, top=True, width=2, length=5)
    ax.tick_params(axis='both', which='minor', width=1.5,
                   length=3.5, right=True, top=True)

    starttime = time[startInd - 1]
    endtime = time[endInd - 1]
    ax.set_title(
        '{instrument} at {location}\n[Averaged] {starttime}-{endtime}'.format(
            instrument=pollyVersion,
            location=location,
            starttime=datenum_to_datetime(starttime).strftime('%Y%m%d %H:%M'),
            endtime=datenum_to_datetime(endtime).strftime('%H:%M')),
        fontsize=15
        )

    # add watermark
    if flagWatermarkOn:
        rootDir = os.path.dirname(
            os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
        im_license = matplotlib.image.imread(
            os.path.join(rootDir, 'img', 'by-sa.png'))

        newax_license = fig.add_axes([0.3, 0.002, 0.14, 0.07], zorder=10)
        newax_license.imshow(im_license, alpha=0.8, aspect='equal')
        newax_license.axis('off')

        fig.text(0.46, 0.012, 'Preliminary\nResults.',
                 fontweight='bold', fontsize=12, color='red',
                 ha='left', va='bottom', alpha=0.8, zorder=10)

        fig.text(
            0.69, 0.003,
            u"\u00A9 {1} {0}.\nCC BY SA 4.0 License.".format(
                datetime.now().strftime('%Y'), partnerLabel),
            fontweight='bold', fontsize=7, color='black', ha='left',
            va='bottom', alpha=1, zorder=10)

    if not np.isnan(refHBaseNR355):
        refHBase355 = refHBaseNR355/1000
        refHTop355 = refHTopNR355/1000
    else:
        refHBase355 = np.nan
        refHTop355 = np.nan
    if not np.isnan(refHBaseNR532):
        refHBase532 = refHBaseNR532/1000
        refHTop532 = refHTopNR532/1000
    else:
        refHBase532 = np.nan
        refHTop532 = np.nan
    fig.text(
        0.23, 0.80,
        'refH355: {0:4.1f}-{1:4.1f} km\n'.format(refHBase355, refHTop355) +
        'refH532: {0:4.1f}-{1:4.1f} km'.format(refHBase532, refHTop532),
        fontsize=11, backgroundcolor=[0.94, 0.95, 0.96, 0.4], alpha=1)
    fig.text(
        0.02, 0.01, 'Version: {0}\n'.format(version) +
        'Method: {0}'.format('Raman'), fontsize=11)

    fig.savefig(
        os.path.join(
            saveFolder,
            '{dataFile}_{starttime}_{endtime}_Bsc_Raman_NR.{imgFmt}'.format(
                dataFile=rmext(os.path.basename(dataFilename)),
                starttime=datenum_to_datetime(starttime).strftime('%H%M'),
                endtime=datenum_to_datetime(endtime).strftime('%H%M'),
                imgFmt=imgFormat)
                ),
        dpi=figDPI
        )
    plt.close()


def main():
    pollyDisplayBscRamanNR(
        'D:\\coding\\matlab\\pollynet_Processing_Chain\\tmp\\',
        'C:\\Users\\zpyin\\Desktop')


if __name__ == '__main__':
    # main()
    pollyDisplayBscRamanNR(sys.argv[1], sys.argv[2])
