function [reportStr] = pollyWriteDonelist(data)
% POLLYWRITEDONELIST Write processing information to done list file and generate report
% for the current task. These report can be used for further examination.
%
% USAGE:
%    [reportStr] = pollyWriteDonelist(data)
%
% INPUTS:
%    data: struct
%
% OUTPUTS:
%    reportStr: char
%
% HISTORY:
%    - 2021-06-21: first edition by Zhenping
%
% .. Authors: - zhenping@tropos.de

global CampaignConfig PicassoConfig PollyConfig PollyDataInfo

reportStr = cell(0);

reportStr{end + 1} = sprintf('Task: %s', PollyDataInfo.pollyDataFile);
reportStr{end + 1} = sprintf('Start time: %s', datestr(PollyDataInfo.dataTime, 'yyyy-mm-dd HH:MM:SS'));
reportStr{end + 1} = sprintf('Instrument: %s', CampaignConfig.name);
reportStr{end + 1} = sprintf('Location: %s', CampaignConfig.location);

if isempty(data.rawSignal)
    reportStr{end + 1} = sprintf('comment: %s', 'no measurement');
    return;
end

%% general reports of processing results
reportStr{end + 1} = sprintf('Measruement time: %s - %s', datestr(data.mTime(1), 'yyyy-mm-dd HH:MM:SS'), datestr(data.mTime(end), 'HH:MM:SS'));
reportStr{end + 1} = sprintf('Continuous cloud free profiles: %d', size(data.clFreGrps, 1));
cloudFreeStr = '';
meteorStr = '';
ref355Str = '';
ref532Str = '';
ref1064Str = '';
flagSNR387 = '';
flagSNR607 = '';
for iGrp = 1:size(data.clFreGrps, 1)
    cloudFreeStr = [cloudFreeStr, sprintf('%s - %s; ', datestr(data.mTime(data.clFreGrps(iGrp, 1)), 'HH:MM'), datestr(data.mTime(data.clFreGrps(iGrp, 2)), 'HH:MM'))];
    meteorStr = [meteorStr, sprintf('%s; ', data.meteorAttri.dataSource{iGrp})];

    if isnan(data.refHInd355(iGrp, 1))
        ref355Str = [ref355Str, sprintf('NaN - NaN m; ')];
    else
        ref355Str = [ref355Str, sprintf('%7.1f - %7.1f m; ', data.height(data.refHInd355(iGrp, 1)), data.height(data.refHInd355(iGrp, 2)))];
    end

    if isnan(data.refHInd532(iGrp, 1))
        ref532Str = [ref532Str, sprintf('NaN - NaN m; ')];
    else
        ref532Str = [ref532Str, sprintf('%7.1f - %7.1f m; ', data.height(data.refHInd532(iGrp, 1)), data.height(data.refHInd532(iGrp, 2)))];
    end

    if isnan(data.refHInd1064(iGrp, 1))
        ref1064Str = [ref1064Str, sprintf('NaN - NaN m; ')];
    else
        ref1064Str = [ref1064Str, sprintf('%7.1f - %7.1f m; ', data.height(data.refHInd1064(iGrp, 1)), data.height(data.refHInd1064(iGrp, 2)))];
    end

    if isnan(data.aerBsc355_raman(iGrp, 100))
        flagSNR387 = [flagSNR387, 'low; '];
    else
        flagSNR387 = [flagSNR387, 'high; '];
    end

    if isnan(data.aerBsc532_raman(iGrp, 100))
        flagSNR607 = [flagSNR607, 'low; '];
    else
        flagSNR607 = [flagSNR607, 'high; '];
    end

end
reportStr{end + 1} = sprintf('Cloud-free regions: %s', cloudFreeStr);
reportStr{end + 1} = sprintf('Meteorological data from: %s', meteorStr);
reportStr{end + 1} = sprintf('Reference height for 355 nm: %s', ref355Str);
reportStr{end + 1} = sprintf('Reference height for 532 nm: %s', ref532Str);
reportStr{end + 1} = sprintf('Reference height for 1064 nm: %s', ref1064Str);
reportStr{end + 1} = sprintf('SNR of 387 nm at reference height: %s', flagSNR387);
reportStr{end + 1} = sprintf('SNR of 607 nm at reference height: %s', flagSNR607);
reportStr{end + 1} = sprintf('Depol constant for 355 nm: %f', data.polCaliFac355);
reportStr{end + 1} = sprintf('Depol constant for 532 nm: %f', data.polCaliFac532);
reportStr{end + 1} = sprintf('Depol constant for 1064 nm: %f', data.polCaliFac1064);
flagWVCali = logical2str(data.wvconstUsedInfo.flagCalibrated, 'yes');
reportStr{end + 1} = sprintf('WV calibrated?: %s', flagWVCali{1});
reportStr{end + 1} = sprintf('WV calibration constant: %f g*kg^{-1}', data.wvconstUsed);
reportStr{end + 1} = sprintf('Lidar constant at 355 nm: %3.1e', data.LCUsed.LCUsed355);
reportStr{end + 1} = sprintf('Lidar calibration status at 355 nm: %s', PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag355 + 1});
reportStr{end + 1} = sprintf('Lidar constant at 532 nm: %3.1e', data.LCUsed.LCUsed532);
reportStr{end + 1} = sprintf('Lidar calibration status at 532 nm: %s', PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag532 + 1});
reportStr{end + 1} = sprintf('Lidar constant at 1064 nm: %3.1e', data.LCUsed.LCUsed355);
reportStr{end + 1} = sprintf('Lidar calibration status at 1064 nm: %s', PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag1064 + 1});

%% Write done_filelist
% TODO:
% the 'active' status should be obtained from the database
active = 1;

flag355FR = data.flagFarRangeChannel & data.flag355nmChannel & data.flagTotalChannel;
flag532FR = data.flagFarRangeChannel & data.flag532nmChannel & data.flagTotalChannel;
flag1064FR = data.flagFarRangeChannel & data.flag1064nmChannel & data.flagTotalChannel;
flag355NR = data.flagNearRangeChannel & data.flag355nmChannel & data.flagTotalChannel;
flag532NR = data.flagNearRangeChannel & data.flag532nmChannel & data.flagTotalChannel;
flag407FR = data.flagFarRangeChannel & data.flag407nmChannel;
flag387FR = data.flagFarRangeChannel & data.flag387nmChannel;
flag607FR = data.flagFarRangeChannel & data.flag607nmChannel;
flag387NR = data.flagNearRangeChannel & data.flag387nmChannel;
flag607NR = data.flagNearRangeChannel & data.flag607nmChannel;
flag355FRS = data.flagFarRangeChannel & data.flag355nmChannel & data.flagCrossChannel;
flag532FRS = data.flagFarRangeChannel & data.flag532nmChannel & data.flagCrossChannel;

% figure of housekeeping data
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_monitor.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', 'data based on laserlogbook.', PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'monitor', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 355 nm far-field range-corrected signal
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_RCS_FR_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', '355 nm Far-Range', PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'RCS_FR_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 532 nm far-field range-corrected signal
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_RCS_FR_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', '532 nm Far-Range', PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'RCS_FR_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 1064 nm far-field range-corrected signal
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '1064', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_RCS_FR_1064.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', '1064 nm Far-Range', PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'RCS_FR_1064', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 355 nm near-field range-corrected signal
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_RCS_NR_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', '355 nm Near-Range', PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'RCS_NR_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 532 nm near-field range-corrected signal
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_RCS_NR_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', '532 nm Near-Range', PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'RCS_NR_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 355 nm far-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_FR_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 355 nm. SNR threshold is %d', PollyConfig.mask_SNRmin(flag355FR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_FR_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 532 nm far-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_FR_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 532 nm. SNR threshold is %d', PollyConfig.mask_SNRmin(flag532FR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_FR_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 1064 nm far-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '1064', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_FR_1064.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 1064 nm. SNR threshold is %d', PollyConfig.mask_SNRmin(flag1064FR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_FR_1064', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 532 nm near-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_NR_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 532 nm NR. SNR threshold is %d', PollyConfig.mask_SNRmin(flag532NR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_NR_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 355 nm near-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_NR_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 355 nm NR. SNR threshold is %d', PollyConfig.mask_SNRmin(flag355NR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_NR_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 407 nm far-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '407', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_FR_407.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 407 nm. SNR threshold is %d', PollyConfig.mask_SNRmin(flag407FR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_FR_407', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 387 nm far-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '387', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_FR_387.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 387 nm. SNR threshold is %d', PollyConfig.mask_SNRmin(flag387FR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_FR_387', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 607 nm far-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '607', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_FR_607.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 607 nm. SNR threshold is %d', PollyConfig.mask_SNRmin(flag607FR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_FR_607', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 387 nm near-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '387', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_NR_387.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 387 nm NR. SNR threshold is %d', PollyConfig.mask_SNRmin(flag387NR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_NR_387', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 607 nm near-field signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '607', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_NR_607.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 607 nm NR. SNR threshold is %d', PollyConfig.mask_SNRmin(flag607NR)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_NR_607', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 355 nm cross signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_FR_355s.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 355s nm. SNR threshold is %d', PollyConfig.mask_SNRmin(flag355FRS)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_FR_355s', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 532 nm cross signal status
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_FR_532s.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 532s nm. SNR threshold is %d', PollyConfig.mask_SNRmin(flag532FRS)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_FR_532s', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of 1064 nm cross signal status
%write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '1064', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_SAT_FR_1064s.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('signal status at 1064s nm. SNR threshold is %d', PollyConfig.mask_SNRmin(flag1064FRS)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SAT_FR_1064s', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of overlap function
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_overlap.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', 'overlap function. Preliminary results only for Internal use. The overlap was calculated by comparing the signal between far-range and near-range channel cloud-free signal.', PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'overlap', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% averaged profiles
for iGrp = 1:size(data.clFreGrps, 1)
    startInd = data.clFreGrps(iGrp, 1);
    endInd = data.clFreGrps(iGrp, 2);

    % GDAS1 timestamp or standard atmosphere
    meteorStr = '';
    if strcmpi(data.meteorAttri.dataSource{iGrp}, 'gdas1')
        meteorStr = sprintf('Meteorological data from %s at %s on %s UTC', upper(PollyConfig.meteorDataSource), PollyConfig.gdas1Site, datestr(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH'));
    elseif strcmpi(data.meteorAttri.dataSource{iGrp}, 'radiosonde')
        meteorStr = sprintf('Meteorological data from %s at %s on %s UTC', upper(PollyConfig.meteorDataSource), CampaignConfig.location, datestr(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH'));
    else
        meteorStr = sprintf('Meteorological data from %s', data.meteorAttri.dataSource{iGrp});
    end

    % averaged profile of range-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_SIG.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SIG', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % averaged profile of overlap-corrected range-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_SIG.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SIG_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % reference height
    if isnan(data.refHInd355(iGrp, 1))
        refH355 = [NaN, NaN];
    else
        refH355 = data.height(data.refHInd355(iGrp, :));
    end
    if isnan(data.refHInd532(iGrp, 1))
        refH532 = [NaN, NaN];
    else
        refH532 = data.height(data.refHInd532(iGrp, :));
    end
    if isnan(data.refHInd1064(iGrp, 1))
        refH1064 = [NaN, NaN];
    else
        refH1064 = data.height(data.refHInd1064(iGrp, :));
    end

    % figure of averaged backscatter coefficient with Klett/Fernald method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Bsc_Klett.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Lidar ratio is %5.1fsr at 355nm, %5.1f at 532nm and %5.1f at 1064nm. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm and %5.1fm. No overlap correction.', meteorStr, PollyConfig.LR355, PollyConfig.LR532, PollyConfig.LR1064, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Bsc_Klett', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged backscatter coefficient with Raman method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Bsc_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. Angstroem exponent is %3.1f. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes, PollyConfig.angstrexp), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Bsc_Raman', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged backscatter coefficient with rotation Raman method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Bsc_RR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. Angstroem exponent is %3.1f. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes, PollyConfig.angstrexp), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Bsc_RR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged backscatter coefficient with constrained Klett/Fernald method
    % write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Bsc_Aeronet.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). The lidar ratio is tuned to make the AOD from AERONET and lidar converged. Lidar ratio is %5.1fsr at 355nm, %5.1fsr at 532nm and %5.1fsr at 1064nm. Smoothing window is %5.1fm, %5.1fm, %5.1fm. Only for internal use.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), data.LR355_aeronet(iGrp), data.LR532_aeronet(iGrp), data.LR1064_aeronet(iGrp), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Bsc_AERONET', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged backscatter coefficient by Klett/Fernald method with overlap-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_Bsc_Klett.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Lidar ratio is %5.1fsr at 355nm, %5.1f at 532nm and %5.1f at 1064nm. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm and %5.1fm. No overlap correction.', meteorStr, PollyConfig.LR355, PollyConfig.LR532, PollyConfig.LR1064, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Bsc_Klett_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged backscatter coefficient by Raman method with overlap-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_Bsc_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. Angstroem exponent is %3.1f. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes, PollyConfig.angstrexp), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Bsc_Raman_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged backscatter coefficient by Raman method with near-field signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Bsc_Raman_NR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. Angstroem exponent is %3.1f. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes, PollyConfig.angstrexp), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Bsc_Raman_NR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged backscatter coefficient by Klett/Fernald method with near-field signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Bsc_Klett_NR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. Angstroem exponent is %3.1f. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes, PollyConfig.angstrexp), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Bsc_Klett_NR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged extinction coefficient by Klett/Fernald method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Ext_Klett.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Lidar ratio is %5.1fsr at 355nm, %5.1f at 532nm and %5.1f at 1064nm. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. No overlap correction.', meteorStr, PollyConfig.LR355, PollyConfig.LR532, PollyConfig.LR1064, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Ext_Klett', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged extinction coefficient by Raman method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Ext_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Ext_Raman', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged extinction coefficient by rotation Raman method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Ext_RR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Ext_RR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    %figure of averaged extinction coefficient by constrained Klett/Fernald method
    % write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Ext_Aeronet.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). The lidar ratio is tuned to make the AOD from AERONET and lidar converged. Lidar ratio is %5.1fsr at 355nm, %5.1fsr at 532nm and %5.1fsr at 1064nm. Smoothing window is %5.1fm, %5.1fm, %5.1fm. Only for internal use.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), data.LR355_aeronet(iGrp), data.LR532_aeronet(iGrp), data.LR1064_aeronet(iGrp), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Ext_AERONET', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged extinction coefficient by Klett/Fernald method with overlap-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_Ext_Klett.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Lidar ratio is %5.1fsr at 355nm, %5.1f at 532nm and %5.1f at 1064nm. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. No overlap correction.', meteorStr, PollyConfig.LR355, PollyConfig.LR532, PollyConfig.LR1064, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Ext_Klett_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged extinction coefficient by Raman method with overlap-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_Ext_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Ext_Raman_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged extinction coefficient by Klett/Fernald method with near-field signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Ext_Klett_NR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Lidar ratio is %5.1fsr at 355nm, %5.1f at 532nm and %5.1f at 1064nm. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. No overlap correction.', meteorStr, PollyConfig.LR355, PollyConfig.LR532, PollyConfig.LR1064, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Ext_Klett_NR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged extinction coefficient by Raman method with near-field signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Ext_Raman_NR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Ext_Raman_NR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged lidar ratio by Raman method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_LR_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'LR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged lidar ratio by rotation Raman method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_LR_RR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'LR_RR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged lidar ratio by Raman method with overlap-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_LR_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'LR_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged lidar ratio by Raman method with near-field signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_LR_Raman_NR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'LR_NR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged Angstroem exponent by Klett/Fernald method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_ANGEXP_Klett.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Lidar ratio is %5.1fsr at 355nm, %5.1f at 532nm and %5.1f at 1064nm. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. No overlap correction.', meteorStr, PollyConfig.LR355, PollyConfig.LR532, PollyConfig.LR1064, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ANGEXP_Klett', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged Angstroem exponent by Raman method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_ANGEXP_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ANGEXP_Raman', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged Angstroem exponent by Klett/Fernald method with overlap-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_ANGEXP_Klett.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Lidar ratio is %5.1fsr at 355nm, %5.1f at 532nm and %5.1f at 1064nm. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm.', meteorStr, PollyConfig.LR355, PollyConfig.LR532, PollyConfig.LR1064, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_klett_355*data.hRes, PollyConfig.smoothWin_klett_532*data.hRes, PollyConfig.smoothWin_klett_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ANGEXP_Klett_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged Angstroem exponent by Raman method with overlap-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_ANGEXP_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. If SNR for Raman signal at reference height is low, the Raman method will not be applied.', meteorStr, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ANGEXP_Raman_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged Angstroem exponent by Raman method with near-field signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_ANGEXP_Raman_NR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s. Lidar ratio is %5.1fsr at 355nm, %5.1f at 532nm and %5.1f at 1064nm. Reference height is [%7.1f - %7.1fm] (355nm), [%7.1f - %7.1fm] (532nm) and [%7.1f - %7.1fm] (1064nm). Smoothing window is %5.1fm, %5.1fm, %5.1fm. No overlap correction.', meteorStr, PollyConfig.LR355, PollyConfig.LR532, PollyConfig.LR1064, refH355(1), refH355(2), refH532(1), refH532(2), refH1064(1), refH1064(2), PollyConfig.smoothWin_raman_355*data.hRes, PollyConfig.smoothWin_raman_532*data.hRes, PollyConfig.smoothWin_raman_1064*data.hRes), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ANGEXP_Raman_NR', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged volume/particle depolarization ratio by Klett/Fernald method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_DepRatio_Klett.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('Depolarization factor is %6.4f at 355nm, %6.4f at 532nm and %6.4f at 1064nm. Molecule volume depolarization ratio is %6.4f at 355nm and %6.4f at 532nm and %6.4f at 1064nm.', data.polCaliFac355, data.polCaliFac532, data.polCaliFac1064, data.mdr355(iGrp), data.mdr532(iGrp), data.mdr1064(iGrp)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'DepRatio_Klett', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged volume/particle depolarization ratio by Raman method
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_DepRatio_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('Depolarization factor is %6.4f at 355nm, %6.4f at 532nm and %6.4f at 1064nm. Molecule volume depolarization ratio is %6.4f at 355nm and %6.4f at 532nm and %6.4f at 1064nm.', data.polCaliFac355, data.polCaliFac532, data.polCaliFac1064, data.mdr355(iGrp), data.mdr532(iGrp), data.mdr1064(iGrp)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'DepRatio_Raman', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged volume/particle depolarization ratio by Klett/Fernald method with overlap-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_DepRatio_Klett.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('Depolarization factor is %6.4f at 355nm, %6.4f at 532nm and %6.4f at 1064nm. Molecule volume depolarization ratio is %6.4f at 355nm and %6.4f at 532nm and %6.4f at 1064nm.', data.polCaliFac355, data.polCaliFac532, data.polCaliFac1064, data.mdr355(iGrp), data.mdr532(iGrp), data.mdr1064(iGrp)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'DepRatio_Klett_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged volume/particle depolarization ratio by Raman method with overlap-corrected signal
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_OC_DepRatio_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('Depolarization factor is %6.4f at 355nm, %6.4f at 532nm and %6.4f at 1064nm. Molecule volume depolarization ratio is %6.4f at 355nm and %6.4f at 532nm and %6.4f at 1064nm.', data.polCaliFac355, data.polCaliFac532, data.polCaliFac1064, data.mdr355(iGrp), data.mdr532(iGrp), data.mdr1064(iGrp)), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'DepRatio_Raman_OC', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged water vapor mixing ratio
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '407', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_WVMR.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('Water vapor calibration constant is %5.1fg*kg^{-1}. Calibration status %s. %s.', data.wvconstUsed, flagWVCali{1}, meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'WVMR_av', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of averaged relative humidity
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '407', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_RH.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('Water vapor calibration constant is %5.1fg*kg^{-1}. Calibration status %s. %s.', data.wvconstUsed, flagWVCali{1}, meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'RH_av', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of temperature
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Meteor_T.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Meteor_T', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));

    % figure of pressure
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_Meteor_P.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Meteor_P', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));


    % figure of POLIPHON
    write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_%s_POLIPHON_step1_Raman.png', rmext(basename(PollyDataInfo.pollyDataFile)), datestr(data.mTime(startInd), 'HHMM'), datestr(data.mTime(endInd), 'HHMM'))), '0', sprintf('%s.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.meteorAttri.datetime(iGrp), 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'POLIPHON_step1_Raman', datestr(data.mTime(startInd), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(endInd), 'yyyymmdd HH:MM:SS'));


end

% height-time figure of attenuated backscatter at 355 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at 355nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed355, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag355 + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ATT_BETA_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of attenuated backscatter at 532 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at 532nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed532, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag532 + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ATT_BETA_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of attenuated backscatter at 1064 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '1064', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_1064.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at 1064nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed1064, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag1064 + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ATT_BETA_1064', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of overlap-corrected attenuated backscatter at 355 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_OC_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at 355nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed355, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag355 + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ATT_BETA_355_OC', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of overlap-corrected attenuated backscatter at 532 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_OC_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at 532nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed532, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag532 + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ATT_BETA_532_OC', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of overlap-corrected attenuated backscatter at 1064 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '1064', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_OC_1064.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at 1064nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed1064, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag1064 + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ATT_BETA_1064_OC', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of near-field attenuated backscatter at 355 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_NR_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at near-field 355nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed355NR, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag355NR + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ATT_BETA_355_NR', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of near-field attenuated backscatter at 532 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_NR_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at near-field 532nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed532NR, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag532NR + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'ATT_BETA_532_NR', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of water vapor mixing ratio
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '407', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_WVMR.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Water vapor calibration constant is %5.1fg*kg^{-1}. Calibration status %s. %s.', data.wvconstUsed, flagWVCali{1}, meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'WVMR', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of relative humidity
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '407', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_RH.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Water vapor calibration constant is %5.1fg*kg^{-1}. Calibration status %s. %s.', data.wvconstUsed, flagWVCali{1}, meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'RH', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of volume depolarization ratio at 355 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_VDR_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Depolarization calibration factor is %f+-%f', data.polCaliFac355, data.polCaliEtaStd355), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'VDR_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of volume depolarization ratio at 532 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_VDR_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Depolarization calibration factor is %f+-%f', data.polCaliFac532, data.polCaliEtaStd532), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'VDR_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of volume depolarization ratio at 1064 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '1064', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_VDR_1064.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Depolarization calibration factor is %f+-%f', data.polCaliFac1064, data.polCaliEtaStd1064), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'VDR_1064', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of quasi-retrieved backscatter at 355 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_Quasi_Bsc_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at 355nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed355, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag355 + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Quasi_Bsc_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of quasi-retrieved backscatter at 532 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_Quasi_Bsc_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at 532nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed532, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag532 + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Quasi_Bsc_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of quasi-retrieved backscatter at 1064 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '1064', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_Quasi_Bsc_1064.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar constant at 1064nm is %3.1e. Lidar constant calibration status: %s', data.LCUsed.LCUsed1064, PollyConfig.LCCalibrationStatus{data.LCUsed.LCUsedTag1064 + 1}), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Quasi_Bsc_1064', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of quasi-retrieved particle depolarization ratio at 532 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_Quasi_PDR_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Depolarization factor is %6.4f at 532nm. Molecule volume depolarization ratio is %6.4f at 532nm. %s.', data.polCaliFac532, data.mdr532(iGrp), meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Quasi_PDR_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of quasi-retrieved Angstroem exponent
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_Quasi_ANGEXP_532_1064.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('%s.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'Quasi_ANGEXP_532_1064', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of lidar constant at 355 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_LC_355.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('%s. Lidar constant is sensible to the system condition, like temperature, humidity.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'LC_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of lidar constant at 532 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_LC_532.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('%s. Lidar constant is sensible to the system condition, like temperature, humidity.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'LC_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of lidar constant at 1064 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '1064', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_LC_1064.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('%s. Lidar constant is sensible to the system condition, like temperature, humidity.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'LC_1064', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of lidar constant at 387 nm
% write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '387', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_LC_387.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('%s. Lidar constant is sensible to the system condition, like temperature, humidity.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'LC_387', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% lidar constant at 607 nm
% write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '607', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_LC_607.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('%s. Lidar constant is sensible to the system condition, like temperature, humidity.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'LC_607', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of aerosol target classification
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_TC.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar Target Categorization'), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'TC', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% figure of aerosol target classification (version 2)
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_TC_V2.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Lidar Target Categorization'), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'TC_v2', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% long-term lidar calibration results
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_%s_long_term_cali_results.png', CampaignConfig.name, datestr(data.mTime(1), 'yyyymmdd'))), '0', sprintf('%s. Lidar constant is sensible to the system condition, like temperature, humidity.', meteorStr), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'long_term_monitor', datestr(CampaignConfig.startTime, 'yyyymmdd 00:00:00'), datestr(data.mTime(end), 'yyyymmdd 23:59:59'));

% height-time figure of near-field SNR at 355 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_NR_355_SNR.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Signal to Noise ratio at near-field 355nm'), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SNR_NR_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of near-field SNR at 532 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_NR_532_SNR.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Signal to Noise ratio at near-field 532nm'), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SNR_NR_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of far-field SNR at 355 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '355', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_355_SNR.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Signal to Noise ratio at far-field 355nm'), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SNR_FR_355', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of far-field SNR at 532 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '532', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_532_SNR.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Signal to Noise ratio at far-field 532nm'), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SNR_FR_532', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

% height-time figure of far-field SNR at 1064 nm
write_2_donelist(PicassoConfig.doneListFile, 'a', CampaignConfig.name, CampaignConfig.location, datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'), datestr(PollyDataInfo.dataTime, 'yyyymmdd HH:MM:SS'), '1064', fullfile(basedir(PicassoConfig.pic_folder), CampaignConfig.name, datestr(data.mTime(1), 'yyyy'), datestr(data.mTime(1), 'mm'), datestr(data.mTime(1), 'dd'), sprintf('%s_ATT_BETA_1064_SNR.png', rmext(basename(PollyDataInfo.pollyDataFile)))), '0', sprintf('Signal to Noise ratio at far-field 1064nm'), PollyDataInfo.zipFile, num2str(PollyDataInfo.dataSize), num2str(active), num2str(data.quasiAttri.flagGDAS1), datestr_convert_0(data.quasiAttri.timestamp, 'yyyymmdd HH:MM:SS'), '50', PicassoConfig.PicassoVersion, 'SNR_FR_1064', datestr(data.mTime(1), 'yyyymmdd HH:MM:SS'), datestr(data.mTime(end), 'yyyymmdd HH:MM:SS'));

end
