
package services.outputProcessors.soot

import interfaces.OutputProcessor

import static app.MiningFramework.arguments

/**
 * @requires: that soot-analysis.jar is in the dependencies folder and that
 * FetchBuildsOutputProcessor and GenerateSootInputFilesOutputProcessor were ran
 * @provides: a [outputPath]/data/soot-results.csv file with the results for the soot algorithms ran
 */
class RunSootAnalysisOutputProcessor implements OutputProcessor {

    private final String RESULTS_FILE_PATH = "/data/results-with-build-information.csv"

    private final SootAnalysisWrapper sootWrapper = new SootAnalysisWrapper("0.2.1-SNAPSHOT")

    private final ConflictDetectionAlgorithm[] detectionAlgorithms = [
            new NonCommutativeConflictDetectionAlgorithm("DF Intra", "svfa-intraprocedural", sootWrapper, 120),
            new NonCommutativeConflictDetectionAlgorithm("DF Inter", "svfa-interprocedural", sootWrapper, 120),
            new ConflictDetectionAlgorithm("Confluence Intra", "svfa-confluence-intraprocedural", sootWrapper, 240),
            new ConflictDetectionAlgorithm("Confluence Inter", "svfa-confluence-interprocedural", sootWrapper, 240),
            new ConflictDetectionAlgorithm("OA Intra", "overriding-intraprocedural", sootWrapper, 120),
            new ConflictDetectionAlgorithm("OA Inter", "overriding-interprocedural", sootWrapper, 120)
    ]

    void processOutput () {
        // check if file generated by FetchBuildsOutputProcessor exists
        println "Executing RunSootAnalysisOutputProcessor"
        executeAllAnalyses(arguments.getOutputPath())
    }

    void executeAllAnalyses(String outputPath) {
        File sootResultsFile = createOutputFile(outputPath)

        File resultsWithBuildsFile = new File(outputPath + RESULTS_FILE_PATH)
        if (resultsWithBuildsFile.exists()) {

            List<Scenario> sootScenarios = ScenarioReader.read(outputPath, RESULTS_FILE_PATH);

            for (scenario in sootScenarios) {
                if (scenario.getHasBuild()) {
                    println "Running soot scenario ${scenario.toString()}"
                    List<String> results = [];

                    for (ConflictDetectionAlgorithm algorithm : detectionAlgorithms) {
                        String algorithmResult = algorithm.run(scenario);

                        results.add(algorithmResult)
                    }
                    sootResultsFile << "${scenario.toString()};${results.join(";")}\n"
                }
            }
        }
    }

    private File createOutputFile(String outputPath) {
        File sootResultsFile = new File(outputPath + "/data/soot-results.csv")

        if (sootResultsFile.exists()) {
            sootResultsFile.delete()
        }

        sootResultsFile << sootWrapper.getSootAnalysisVersionDisclaimer();
        sootResultsFile << buildCsvHeader();
    
        return sootResultsFile
    }

    private String buildCsvHeader () {
        StringBuilder resultStringBuilder = new StringBuilder("project;class;method;merge commit");

        for (ConflictDetectionAlgorithm algorithm : detectionAlgorithms) {
            resultStringBuilder.append(";${algorithm.generateHeaderName()}");
        }
        resultStringBuilder.append("\n");

        return resultStringBuilder.toString();
    }

}