import arrivaltimes
import pandas as pd
import glob
from datetime import datetime
import numpy as np
import matplotlib.pyplot as plt # v3.8.3
from collections import namedtuple
import os

import warnings
warnings.filterwarnings("ignore")

DM_FRB20121102A = 560.105 # DM-of-Shortest. apply DM of B30 of Snelders+2023, then apply eq. B4 of Jahns to find deltaDM
# DM_FRB20121102A = 560.0927 # Bootstrapped DM

DM_FRB20220912A = 219.356 # DM-of-Shortest. apply DM of B1 of Hewitt+2023, then apply eq. B4 of Jahns to find deltaDM
print(f"Info: {DM_FRB20121102A = } pc/cm3")
print(f"Info: {DM_FRB20220912A = } pc/cm3")

## Script configuration. Use these to choose what is run, what is saved, and to specify data paths
# 'datasetname': [run_measurements_bool, save_spreadsheet_bool, prefix_str]
Dataset = namedtuple('Dataset', 'run save path')
datasets = {
	'gajjar'               : Dataset(True, True, 'data/gajjar2018/'),
	'michilli'             : Dataset(True, True, 'data/michilli2018/'),
	'oostrum'              : Dataset(True, True, 'data/oostrum2020/R1_frb121102/'),
	'li'                   : Dataset(True, True, 'data/li2021/npzs/'),
	'scholz'               : Dataset(True, True, 'data/scholz2016/'),
	'aggarwal'             : Dataset(True, True, 'data/aggarwal2021/'),
	'snelders'             : Dataset(True, True, 'data/snelders2023/figure_1/'),
	'hewitt'               : Dataset(True, True, 'data/FRB20220912A/hewitt2023/npzs/'),
	'sheikh'               : Dataset(True, True, 'data/FRB20220912A/sheikh2023/npzs_highres/'),
	'zhang'                : Dataset(True, True, 'data/FRB20220912A/zhang2023/Figure-5+9/'),
	'nimmo'                : Dataset(True, True, 'data/FRB20200120E/waterfalls/'),
	'20180916B_Effelsberg' : Dataset(True, True, 'data/20180916B_Effelsberg/'),
	'201124_Effelsberg'    : Dataset(True, True, 'data/201124_Effelsberg/'),
	'20180916B_LOFAR'      : Dataset(True, True, 'data/20180916B_LOFAR/'),
	'20180916B_CHIME'      : Dataset(True, True, 'data/20180916B_CHIME/'),
	'20180916B_uGMRT'      : Dataset(True, True, 'data/20180916B_uGMRT/'),
	'20180814A_CHIME'      : Dataset(True, True, 'data/20180814_CHIME/'),
	'20200929C_CHIME'      : Dataset(True, True, 'data/20200929C_CHIME/'),
	'20180301_FAST'        : Dataset(True, True, 'data/20180301_FAST/'),
	'20200223B_CHIME'      : Dataset(True, True, 'data/20200223B_CHIME/'),
	'20190804E_CHIME'      : Dataset(True, True, 'data/20190804E_CHIME/'),
	'20121102_Arecibo_1'   : Dataset(True, True, 'data/20121102_Arecibo_1/'),
	'20201130A_CHIME_'     : Dataset(True, True, 'data/20201130A_CHIME/'),
	'20190915D_CHIME'      : Dataset(True, True, 'data/20190915D_CHIME/'),
}

def listnpzs(path):
	files = glob.glob(path+'*.npz')
	[print(f) for f in sorted(files)]
	exit()

plt.rcParams.update({'font.size': 14}) # For paper

########################################
########################################
if datasets['gajjar'].run: ## 'gajjar__main__'
	prefix = datasets['gajjar'].path

	# listnpzs(prefix)
	arrivaltimes.logdebug = False

	# Deleted 11L and 11P, lots of RFI, saw no burst
	files = {
		'11A.npz': { # scintillation/scalloping, note use of `bandmask_thres` option
			'crop': (800, 900+400),
			'downfactors': (5,4),
			'xos': [1.96, 2.27, 3.04, 3.58],
			'cuts': [2.13, 2.67, 3.37],
			'fix_xos' : True,
			'submasks': ([1409,1363,1438],[],[],[range(692,838)]), # 11a 1143
			'legendloc' : 3,
			'hide_legend': False,
			'bandmask_thres': 0.11,
			'pslopeidx': 0,
			# 'show_arrtime': [7048]
		},
		'11B.npz': {
			'crop': (1484, 1940),
			'downfactors': (5,4),
			't_filter_factor':1,
		},
		'11C.npz': {
			'crop': (1316, 1316+256),
			'downfactors': (10,4),
			'bandmask_thres': 0.035
		},
		'11D.npz': {
			'crop': (777, 1243),
			'downfactors': (5,1),
			'bandmask_thres': 0.07,
		},
		'11E.npz': {
			'crop': (1110, 1800),
			'xos' : [2.82, 4.44],
			'fix_xos': True,
			'cuts':[3.64],
			'downfactors': (10,2),
			'bandmask_thres':0.03
		},
		'11F.npz': {
			'crop': (730,1236),
			'downfactors': (10,2),
			'bandmask_thres': 0.036
		},
		'11G.npz': {
			'crop':(300, 300+512),
			'xos': [1.64],
			'masks': [range(512,654)],
			'downfactors': (10,4)
		},
		'11H.npz': {'crop': (820, 1240)},
		'11I.npz': { # Dubious
			'crop':(0,256),
			'downfactors': (5,8),
			'masks': [range(365, 422)]
		},
		'11J.npz': {
			'crop':(780, 780+512),#1280),
			'downfactors': (5,8),
			'bw_width_factor': 4,
			'masks': [range(1137, 1187), range(83, 217), range(928, 1038)]
		},
		'11K.npz': {
			'crop': (842, 842+1024),#1670),
			'downfactors': (5,8),
			'masks': [527]
		},
		'11M.npz': {
			# Microshots! in 2018. Only visible at this lower DM, and from the same dataset analysed in Snelders+2023.
			# I don't believe these were mentioned in Snelders+2023, and they are a bit longer than those ones
			# About 150 and 70 μs
			'crop':(920,920+256),#1230),
			'downfactors':(5,1),
			'xos' : [1.09, 1.37],
			'cuts': [1.24],
			'submasks': [[range(460,463)],[]], # very flat noise in that channel by chance
			'hide_legend': False
		},
		'11N.npz': {
			'crop':(530, 790),
			'masks':[236, 325, 326, 324, 284, range(355,392), range(331,344), 192],
			'downfactors': (5, 2)
		},
		'11O.npz': { #'okay'
			'crop':(1792,2040),
			'bw_width_factor': 15, # high bw_width_factor means the bandwidth is underestimated
			'downfactors': (5,4)
		},
		'11Q.npz': {'crop':(906, 1130)},
		'12A.npz': {
			'crop': (904, 904+256),#1166),
			'xos' : [1.17, 0.67], # 1st is too low snr
			'cuts': [0.88],
			'downfactors': (10,4)
		},
		'12B.npz': {
			'crop': (700, 1240),
			'xos':[2.37, 4.08],
			'submasks': [[],[range(849,1129)]],
			'downfactors':(10,1),
			'legendloc': 3,
			'bandmask_thres': [0.07, None]
		},
		'12C.npz': {
			'crop':(860,1240),
			'downfactors': (10,5)
		},
	}

	save = datasets['gajjar'].save
	results = []
	targetDM = DM_FRB20121102A
	for filename, kwargs in files.items():
		downfactors = (2,2) if 'downfactors' not in kwargs else kwargs['downfactors']
		kwargs['downfactors'] = downfactors

		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/frb20121102A/gajjar2018/',
			show=not save,
			save=save,
			targetDM=targetDM,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	fileout = f"measurements/frb20121102A/gajjar2018/results_{datetime.now().strftime('%b-%d-%Y')}-{targetDM:.3f}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}")

########################################
########################################

if datasets['michilli'].run: ## 'michilli__main__'
	prefix = datasets['michilli'].path

	# listnpzs(prefix)
	arrivaltimes.logdebug = False
	files = {
		'M001.npz': {'masks':[453]},
		'M002.npz': {
			'masks':[341,411,412, 253, 208, 194, 189, 161],
			'downfactors': (4,4),
			'snr_cutoff': 2
		},
		'M003.npz': {'downfactors':(2,1)}, # Downfactored fit looks better to me visually
		'M004.npz': {'downfactors':(16,8)},
		'M005.npz': {
			'downfactors':(16,4),
			'masks':[100,124,415,416],
		},
		'M006.npz': {
			'crop':(200, 320),
			'targetDM':559.7
		}, # overdedispersed, and blended
		'M007.npz': {
			'xos':[2.54, 1.94],
			'cuts': [2.32],
			'fix_xos': True,
			'downfactors': (8,1)
		},
		'M008.npz': {
			'downfactors': (4,4),
		},
		'M009.npz': {'downfactors':(4,4)}, # extremely low snr
		'M010.npz': {
			'crop': (105, 105+256),
			'xos' : [1.7, 1.01],
			'downfactors': (8,4),
			'cuts': [1.41]
		},
		'M011.npz': {
			'downfactors': (16,4),
		},
		'M012.npz': {
			'masks':[287, 289],
			'downfactors': (4,2),
			'bandmask_thres': 0.14#0.18
		},
		'M013.npz': { # Perhaps exclude component d
			'xos':[1.98, 2.52, 2.82, 3.13],
			'cuts': [2.32, 2.68, 2.96],
			'downfactors': (4,2),
			'legendloc':3
		},
		'M014.npz': {},
		'M015.npz': {
			'downfactors':(4,1),
			'bw_width_factor':5,
			'bandmask_thres': 0.4
		}, # loosen bandpass filter
		'M016.npz': {'downfactors':(2,1), 'masks':[164,405,414,415,416]},
	}

	save = datasets['michilli'].save
	# for targetDM in [556., 557., 558., 559., 559.7, 560., 560.105, 561.]:
	for targetDM in [DM_FRB20121102A]:
		results = []
		compare_results = []
		for filename, kwargs in files.items():
			filename = f'{prefix}{filename}'

			## Testing band restriction on widths
			# width = int(512//5)
			# testmask = [range(0,width)] # 1
			# testmask = [range(width,2*width)] # 2
			# testmask = [range(2*width,3*width)] # 3
			# testmask = [range(3*width,4*width)] # 4
			# testmask = [range(4*width,5*width)] # 5

			# width = int(512*4//6) # 2/3 snip
			# testmask = [range(0,width)] # top
			# if 'masks' in kwargs:
			# 	kwargs['masks'] += list(testmask)
			# else:
			# 	kwargs['masks'] = list(testmask)

			# targetDM = DM_FRB20121102A
			if 'targetDM' in kwargs:
				# targetDM = kwargs['targetDM'] # Overwrite global DM
				del kwargs['targetDM']

			burst_results = arrivaltimes.measureburst(
				filename,
				outdir='measurements/frb20121102A/michilli2018/',
				show=not save,
				save=save,
				targetDM=targetDM,
				correctTimes=True,
				**kwargs
			)
			for row in burst_results:
				results.append(row)

			# arrtimesdf.to_csv('M001.csv')
			# exit()

			# compare_results.append(
			# 	arrivaltimes.measure_allmethods(
			# 		filename,
			# 		show=False,
			# 		targetDM=targetDM,
			# 		**kwargs
			# 	)
			# )

		resultsdf = pd.DataFrame(
			data=results,
			columns=arrivaltimes.results_columns
		).set_index('name')

		# comparedf = pd.DataFrame(
		# 	data=compare_results,
		# 	columns=arrivaltimes.results_columns+arrivaltimes.allmethods_columns
		# ).set_index('name')
		# comparedf.to_csv(f"measurements/collected/allmethods_results_michilli_{datetime.now().strftime('%b-%d-%Y')}.csv")

		fileout = f"measurements/frb20121102A/michilli2018/results_DM_{datetime.now().strftime('%b-%d-%Y')}-{targetDM:.3f}.csv"
		if save:
			resultsdf.to_csv(fileout)
			print(f"Saved {fileout}.")

########################################
########################################

if datasets['oostrum'].run: ## 'oostrum__main__'
	prefix = datasets['oostrum'].path

	# listnpzs(prefix)

	# Deleted B11, can't see burst
	files = {
		'R1_B01.npz': {'downfactors': (4,2)},
		'R1_B02.npz': {'downfactors': (8,2)},
		'R1_B03.npz': {'downfactors': (4,2)},
		'R1_B04.npz': {'downfactors': (8,1)},
		'R1_B05.npz': {'xos':[15.85], 'downfactors': (4,1)}, # doesn't seem to match paper waterfall?
		'R1_B06.npz': {'downfactors': (8,4)},
		'R1_B07.npz': {
			'xos':([32.32, 39.69], [35.54]),
			'downfactors': (8,1)
		},
		'R1_B08.npz': {'downfactors': (16,2)},
		'R1_B09.npz': {'downfactors':(8,4)},
		'R1_B10.npz': {'downfactors':(8,4)},
		'R1_B12.npz': {
			'xos':[35.41],
			'downfactors': (4,4)
		}, # looks different from paper
		'R1_B13.npz': {'downfactors':(8,2)},
		'R1_B14.npz': {'downfactors': (8,4)},
		'R1_B15.npz': {'downfactors': (8,2)},
		'R1_B16.npz': {'downfactors': (8,1)}, # 8,2 works but its iffy
		'R1_B17.npz': {},
		'R1_B18.npz': {'downfactors': (8,4)},
		'R1_B19.npz': {'downfactors': (8,2)},
		'R1_B20.npz': {'downfactors':(8,2)},
		'R1_B21.npz': {
			'masks':[range(7,11)],
			'downfactors': (8,4)
		}, # innaccurate bandpass fit? cutoff at least
		'R1_B22.npz': {'downfactors':(4,1)},
		'R1_B23.npz': {'downfactors':(8,4)},
		'R1_B24.npz': {'downfactors':(8,8)},
		'R1_B25.npz': {'downfactors': (8,2)},
		'R1_B26.npz': {'downfactors': (8,2)},
		'R1_B27.npz': {'downfactors': (8,4)},
		'R1_B28.npz': {'downfactors': (8,4)},
		'R1_B29.npz': {'masks':[66], 'downfactors':(8,4)},
		'R1_B30.npz': {'downfactors': (8,4)},
	}

	save = datasets['oostrum'].save
	results = []
	compare_results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/frb20121102A/oostrum2020/',
			show=not save,
			targetDM=DM_FRB20121102A,
			correctTimes=True,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

		# compare_results.append(
		# 	arrivaltimes.measure_allmethods(
		# 		filename,
		# 		show=False,
		# 		targetDM=DM_FRB20121102A,
		# 		p0tw=5,
		# 		p0bw=200,
		# 		**kwargs
		# 	)
		# )

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	# comparedf = pd.DataFrame(
	# 	data=compare_results,
	# 	columns=arrivaltimes.results_columns+arrivaltimes.allmethods_columns
	# ).set_index('name')
	# comparedf.to_csv(f"measurements/collected/allmethods_results_oostrum_{datetime.now().strftime('%b-%d-%Y')}.csv")

	fileout = f"measurements/frb20121102A/oostrum2020/results_{datetime.now().strftime('%b-%d-%Y')}-{DM_FRB20121102A:.3f}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['li'].run: ## 'li__main__'
	# prefix = '/Volumes/moonhd/Li2021/'
	prefix = datasets['li'].path

	# listnpzs(prefix)

	# Deleted:
	# 'FRB121102_tracking-M01_0017.npz': {'downfactors':(8,4)},
	# 'FRB121102_tracking-M01_0042.npz': {},
	# 'FRB121102_tracking-M01_0065.npz': {},
	# 'FRB121102_tracking-M01_0072.npz': {},
	# 'FRB121102_tracking-M01_0121.npz': {},
	# 'FRB121102_tracking-M01_0124.npz': {},
	# 'FRB121102_tracking-M01_0137.npz': {},
	# 'FRB121102_tracking-M01_0182.npz': {},
	# 'FRB121102_tracking-M01_0923.npz': {'downfactors':(4,2)},
	# 'FRB121102_tracking-M01_0015.npz': {'downfactors':(4,4)},
	# 'FRB121102_tracking-M01_0024.npz': {'downfactors':(4,4)},
	# 'FRB121102_tracking-M01_0151.npz': {},
	# 'FRB121102_tracking-M01_0153.npz': {},
	# 'FRB121102_tracking-M01_0164.npz': {},
	# 'FRB121102_tracking-M01_0184.npz': {},
	# 'FRB121102_tracking_3-M01_0203.npz': {},
	# 'FRB121102_tracking-M01_0014.npz': {'downfactors':(4,2)},
	# 'FRB121102_tracking-M01_0239.npz': {'downfactors':(4,8)},
	# 'FRB121102_tracking_3-M01_0178.npz': {'downfactors':(4,4)},
	# 'FRB121102_tracking-M01_0041.npz': {'downfactors':(4,4)},
	# 'FRB121102_tracking-M01_0085.npz': {'downfactors':(4,4)},
	# 'FRB121102_tracking-M01_0140.npz': {'downfactors':(4,4)},
	# 'FRB121102_tracking-M01_0179.npz': {'downfactors':(4,4)},
	# 'FRB121102_tracking-M01_0977.npz': {'downfactors':(4,2)},


	files = {
		'FRB121102_tracking-M01_0136.npz': {'downfactors': (8,8)},
		'FRB121102_tracking-M01_0071.npz': {'downfactors':(8,8)},#'targetDM': 562.},
		'FRB121102_tracking-M01_0097.npz': {'xos':[23.36], 'downfactors': (8,8)},
		'FRB121102_tracking-M01_0536.npz': {'downfactors': (8,8)},
		'FRB121102_tracking-M01_0538.npz': {'downfactors': (8,8)},#'targetDM': 562.},
		'FRB121102_tracking_3-M01_0111.npz': {'downfactors': (8,8)},
		'FRB121102_tracking_3-M01_0171.npz': {'downfactors': (8,8)},
		'FRB121102_tracking-M01_0112.npz': {'downfactors': (8,8)},#'targetDM':564.}, # a little overdedispersed
		'FRB121102_tracking-M01_0118.npz': {
			'masks':[173, 174, 133, 125, 132,222,223],
			'downfactors': (16,10)
		},
		'FRB121102_tracking-M01_0148.npz': {
			'downfactors':(16,10), 'xos':[51.77],
		},
		'FRB121102_tracking-M01_0448.npz': {},#'targetDM':562.}, # arbitrary DM
		'FRB121102_tracking-M01_0533.npz': {
			'downfactors': (8,4)
		},
		'FRB121102_tracking-M01_0754.npz': {
			'downfactors': (8,8)
		},
		'FRB121102_tracking_3-M01_0034.npz': {
			'downfactors': (8,8)
		},
		'FRB121102_tracking_3-M01_0167.npz': {
			'downfactors': (8,2)
		},
		'FRB121102_tracking-M01_0133.npz': {
			'downfactors':(8,8),
			'masks':[54, 247, 246, range(15, 20), 64, 243, 242, 244, 210],
			'xos': [49.14, 60.12],
			'cuts':[53.98],
			# 'fix_xos': True,
			# 'sigmas':[1]
		},
		'FRB121102_tracking-M01_0245.npz': {
			'downfactors':(8,4),
			'xos' : [45.3, 54.21, 65.61],
			'cuts' : [60.0, 49.81],
			'masks': [range(5, 56)],
			'submasks': [[],[],[range(194,253)]]
			# 'fix_xos': True
		},
		'FRB121102_tracking-M01_0437.npz': {
			'downfactors': (8,4)
		},
		'FRB121102_tracking-M01_0055.npz': {
			'xos':[50.6, 57.17],
			'cuts': [53.93],
			'downfactors': (8,8)
		},
		'FRB121102_tracking-M01_0095.npz': {
			'xos': [48.35, 56.19],
			'cuts': [53.14],
			'downfactors': (8,4)
		},
		'FRB121102_tracking-M01_0103.npz': {
			'downfactors': (8,4)
		},
		'FRB121102_tracking-M01_0163.npz': {
			'crop':(340, 700),
			'xos':[10.8, 16.63, 22.7],
			'cuts':[13.25, 19.43],
			'downfactors': (8,4),
			'fix_xos': True
		},
		'FRB121102_tracking-M01_0221.npz': {
			'downfactors': (8,8)
		},
		'FRB121102_tracking-M01_0238.npz': {
			'downfactors': (8,2)
		},
		'FRB121102_tracking-M01_0264.npz': {
			'targetDM': 562.7, # Burst DM
			'crop':(390, 720),
			'xos': [10.63, 11.61, 12.85],
			'cuts': [12.42, 11.07],
			'fix_xos':True
		}, # 3 blended bursts, very clean drift rate though. 3rd component is inaccurate
		'FRB121102_tracking-M01_0388.npz': {
			'xos': [50.26, 56.63],
			'cuts': [56.76],
			'downfactors': (8,8)
		},
		'FRB121102_tracking-M01_0415.npz': {
			'crop': (390, 390+256),
			'xos': [11.92, 17.1, 21.83],
			'cuts' : [13.85, 19.48],
			'downfactors': (8,8)
		},
		'FRB121102_tracking-M01_0487.npz': {
			'xos':[48.52, 43.98],
			'cuts': [47.09],
			'downfactors': (8,2)
		},
		'FRB121102_tracking-M01_0576.npz': {
			# 'xos': [49.2, 52.16, 57.93],
			# 'cuts': [50.78, 54.76],
			# 'fix_xos':True
			'downfactors': (8,2)
		}, # 3-blended
		'FRB121102_tracking-M01_0630.npz': {
			'downfactors':(8,4),
			'xos': [49.32, 56.16],
			'cuts': [52.81],
		},
		'FRB121102_tracking-M01_0634.npz': {'xos': ([50.21, 45.67],[47.89])}, # maybe 3?
		'FRB121102_tracking-M01_0774.npz': {},
		'FRB121102_tracking-M01_0826.npz': {
			'downfactors': (8,8)
		},
		'FRB121102_tracking-M01_1180.npz': {'downfactors': (8,10)},
		'FRB121102_tracking_3-M01_0031.npz': {},
		'FRB121102_tracking_3-M01_0153.npz': {
			'downfactors': (8,8)
		},
	}

	save = datasets['li'].save
	results = []
	compare_results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'

		downfactors = (4,1) if 'downfactors' not in kwargs else kwargs['downfactors']
		kwargs['downfactors'] = downfactors

		targetDM = DM_FRB20121102A if 'targetDM' not in kwargs else kwargs['targetDM']
		kwargs['targetDM'] = targetDM

		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/frb20121102A/li2021/',
			show=not save,
			save=save,
			correctTimes=False,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

		# if 'xos' not in kwargs: # one burst waterfalls
		# 	compare_results.append(
		# 		arrivaltimes.measure_allmethods(
		# 			filename,
		# 			show=False,
		# 			# targetDM=DM_FRB20121102A,
		# 			p0tw=5,
		# 			# p0bw=200,
		# 			**kwargs
		# 		)
		# 	)

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	# comparedf = pd.DataFrame(
	# 	data=compare_results,
	# 	columns=arrivaltimes.results_columns+arrivaltimes.allmethods_columns
	# ).set_index('name')
	# comparedf.to_csv(f"measurements/collected/allmethods_results_li_{datetime.now().strftime('%b-%d-%Y')}.csv")

	fileout = f"measurements/frb20121102A/li2021/results_{datetime.now().strftime('%b-%d-%Y')}-{targetDM:.3f}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['scholz'].run: ## 'scholz__main__'
	prefix = datasets['scholz'].path

	# listnpzs(prefix)

	files = {
		'guppi_57339_FRB121102_0004_fullsplit_0004_wfall_bpass_masked_full_498.frbgui.npz' : {
			'downfactors': (16,10)
		},
		'guppi_57345_FRB121102_0002_fullsplit_0002_wfall_bpass_masked_full_596.frbgui.npz' : {
			'crop': (265, 265+600),
			'downfactors': (8,10)
		},
		'guppi_57345_FRB121102_0002_fullsplit_0003_wfall_bpass_masked_full_324.frbgui.npz' : {
			'masks':[204],
			'crop': (310,310+500),
			'downfactors': (8,10)
		},
		'guppi_57345_FRB121102_0002_fullsplit_0004_wfall_bpass_masked_full_494.frbgui.npz' : {
			'downfactors': (16,4)
		}, # Bulbuous bursts
		'guppi_57345_FRB121102_0002_fullsplit_0004_wfall_bpass_masked_full_78.frbgui.npz' : {
			'downfactors': (8,4)
		},
		'puppi_57364_C0531+33_4998_ds_fullsplit_0001_wfall_bpass_masked_full_129.frbgui.npz' : {
			'downfactors':(2, 5),
			'crop':(260, 730),
			'bw_width_factor':1
		},
	}

	save = datasets['scholz'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'

		downfactors = (2,4) if 'downfactors' not in kwargs else kwargs['downfactors']
		kwargs['downfactors'] = downfactors

		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/frb20121102A/scholz2016/',
			# downfactors=downfactors,
			targetDM=DM_FRB20121102A,
			show=not save,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	fileout = f"measurements/frb20121102A/scholz2016/results_{datetime.now().strftime('%b-%d-%Y')}-{DM_FRB20121102A:.3f}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['aggarwal'].run: ## 'aggarwal__main__'
	prefix = datasets['aggarwal'].path

	# listnpzs(prefix)

	# Cross check with paper repo: https://thepetabyteproject.github.io/FRB121102/
	# Deleted:
	# 'B002.npz': {}, # Removed from paper repo
	# 'B009.npz': {}, # Removed from paper repo
	# 'B018.npz': {}, # Removed from paper repo
	# 'B019.npz': {}, # Removed from paper repo
	# 'B041.npz': {}, # Removed from paper repo
	# 'B061.npz': {'crop':None, 'downfactors':(1,4), 'xos' : [47.32]}, # no burst not even in paper repo
	# 'B085.npz': {'crop':None, 'downfactors':(1,4)}, # Very invisible
	# 'B099.npz': {'crop':None, 'xos' : [49.13]}, # Invisible
	# 'B096.npz': {'crop':None}, # Invisible
	# 'B101.npz': {'crop':None}, # Removed from paper repo
	# 'B110.npz': {}, # Removed from paper repo
	# 'B123.npz': {}, # Removed from paper repo

	# Positively sloped:
	# B015_a                                              560.105  ...   3.399393
	# M006                                                560.105  ...   0.123702
	# M007_b                                              560.105  ...   0.344826
	files = {
		'B001.npz': {
			'crop':None,
			'xos':[52.36],
			'downfactors': (4, 4)
		},
		'B003.1.npz': {
			'crop':None,
			'downfactors':(1,4),
			'xos' : [52.75, 64.98],
			'submasks': [[],[range(11,32)]]
		},
		'B004.npz': {'downfactors': (1,1)},
		'B005.npz': {'crop':None, 'downfactors':(2,10), 'xos' : [50.96]},
		'B006.1.npz': {
			'subbandslice': slice(-10,-1)
		}, # May be a severe blend
		'B007.1.npz': {
			'crop':None,
			'downfactors':(2,4),
			'xos' : [60.3, 81.96],
			# 'fix_xos': True
		},
		'B008.npz': {'crop':None, 'downfactors': (2, 10)},
		'B010.npz': {'crop':None, 'xos':[52.19], 'downfactors': (1, 4)},
		'B011.npz': {'downfactors': (2,10)},
		'B012.npz': {'crop':None, 'downfactors':(1,4), 'xos' : [48.55], 'masks':[50]},
		'B013.npz': {'crop':None, 'xos' : [51.97], 'downfactors': (2,10)},
		'B014.npz': {'crop':None, 'xos' : [64.59], 'downfactors':(1,4)},
		'B015.1.npz': {
			'crop':None,
			'xos' : [50.71, 58.38],
			'downfactors':(2,10),
			'legendloc':3
		}, # not great
		'B016.npz': {'crop':None, 'xos' : [54.78], 'downfactors': (2,10)},
		'B017.npz': {'crop':None, 'downfactors':(1,4), 'xos' : [50.79]},
		'B020.npz': {'crop':None, 'downfactors':(2,10)},
		'B021.npz': {
			'masks':[39,40,41,42, range(11,20)],
			'crop':None,
			'downfactors':(1,4),
			'xos' : [76.41]
		},
		'B022.npz': {
			'crop':(370,890),
			'downfactors':(1,4)
		},
		'B023.npz': {'crop':None, 'downfactors':(2,5)}, # Shows second unmarked burst in paper repo
		'B024.npz': {'crop':None, 'downfactors':(2,4)}, # Shows two more unmarked.. echoes?
		'B025.npz': {'crop':(420, 890), 'masks':[49], 'downfactors':(2,5), 'xos':[17.39]},
		'B026.npz': {'crop':(400,950), 'downfactors':(2,5)},
		'B027.npz': {'crop':None, 'downfactors':(1,4)},
		'B028.npz': {'downfactors': (2,4)},
		'B029.npz': {'downfactors': (2,4)},
		'B030.npz': {'crop':None, 'downfactors':(4,4), 'xos' : [55.37]},
		'B031.npz': {'crop':None, 'xos' : [56.08], 'downfactors': (4,4)},
		'B032.npz': {'masks':[57], 'xos' : [53.3], 'downfactors':(1,1)}, # Difficult
		'B033.npz': {'crop':None, 'masks':[62], 'downfactors':(2,10)},
		'B034.npz': {'crop':None, 'downfactors':(2,4), 'xos' : [49.54]},
		'B035.npz': {'crop':None, 'masks':[54], 'downfactors': (4,4)},
		'B036.npz': {'crop':None, 'xos' : [56.14], 'downfactors':(4,4), 'masks':[46,47,49]},
		'B037.npz': {'downfactors': (4,10)}, # decimated
		'B038.npz': {},
		'B039.npz': {}, # noise bands
		'B040.npz': {'crop':None, 'downfactors':(1,4)},
		'B042.npz': {'downfactors': (1,1)},
		'B043.npz': {'crop':(385,385+512), 'downfactors': (2,8)}, # curved slope
		'B044.npz': {'crop':None, 'masks':[13], 'xos' : [46.73]},
		'B045.npz': {},
		'B046.npz': {'crop':None, 'downfactors': (2,4)},
		'B047.npz': {'downfactors': (4,2)},
		'B048.1.npz': {
			'crop': None,
			'xos' : [49.26, 58.33],
			'downfactors':(1,1)
		}, # Zoom in for better view
		'B049.1.npz': {
			'crop':None,
			'downfactors':(4,4),
			'masks': [43,44,51,60,61,62]
		}, # repo shows 2nd burst 70ms away from center, need to recrop from filterbank to see it
		'B050.npz': {'crop':None, 'masks':[61], 'xos' : [53.56]},
		'B051.npz': {'crop':None, 'masks':[61], 'xos' : [49.8]},
		'B052.npz': {'crop':None, 'downfactors': (2,10)},
		'B053.npz': {'crop':None, 'downfactors': (4,10)},
		'B054.npz': {'crop':None, 'masks':[62, 11], 'xos' : [47.77], 'downfactors':(1,2)},
		'B055.npz': {'crop':None, 'masks':[49], 'downfactors': (4,10)},
		'B056.npz': {'crop':None, 'masks':[15,16,17,18], 'downfactors': (2,10)},
		'B057.npz': {'downfactors': (1,2)},
		'B058.npz': {'downfactors': (2,2)},
		'B059.npz': {'crop':None, 'downfactors':(2,5), 'masks':[30,31,32,36,38]},
		'B060.1.npz': {'crop':None, 'xos' : [38.61, 50.29], 'downfactors': (2,10)},
		'B062.npz': {'crop':None, 'downfactors': (2,10)}, # only two points but lines up well
		'B063.npz': {},
		'B064.npz': {
			'crop':None,
			'downfactors': (2,10),
			'xos':[52.56]
		},
		'B065.npz': {'crop':None, 'xos' : [49.23], 'downfactors':(2,10)},
		'B066.npz': {'crop':None, 'xos' : [56.08], 'fix_xos': True, 'downfactors': (2,5)},
		'B067.npz': {'downfactors': (4,10)}, # decimated
		'B068.npz': {'crop':None},
		'B069.npz': {'crop':None, 'downfactors':(2,4)},
		'B070.npz': {},
		'B071.npz': { # Blended? That's why the multiple masks
			'crop':None,
			'xos':[48.23, 56.69],
			'downfactors':(2,10),
			'masks':[11,13, 40, 41, 42, 43, 48, 49, 50],
			'cuts' : [50.58],
			'fix_xos': True
		},
		'B072.npz': {
			'crop':None,
			'xos' : [52.1],
			'downfactors':(2,4),
			'masks':[29]
		}, # band cut
		'B073.npz': {'crop':None, 'downfactors':(1,4)},
		'B074.npz': {
			'crop':None,
			'downfactors':(1,4),
			'masks':[52, range(54,56)]
		},
		'B075.npz': {
			'crop':None, 'downfactors':(4,10)
		},
		'B076.npz': {'downfactors': (2,4)},
		'B077.npz': {'crop':None, 'xos': [62.83], 'downfactors': (4,2)},
		'B078.npz': {
			'crop':None,
			'downfactors':(1,4),
			'xos' : [64.05],
			'fix_xos': True,
			'masks':[range(29,64)],
			'downfactors': (4,10),
			'sigmas': [1]
		},
		'B079.npz': {
			'crop':None,
			'downfactors': (2,4)
		},
		'B080.npz': {'downfactors': (2,2)},
		'B081.npz': {'crop':None, 'masks':[56], 'downfactors': (2,2)},
		'B082.npz': {
			'crop':None,
			'masks': [59],
			'downfactors': (4,4)
		},
		'B083.npz': {'crop':None},
		'B084.npz': {'crop':None, 'xos' : [49.25], 'downfactors':(2,4)},
		'B086.npz': {'crop':None, 'xos' : [48.98], 'downfactors': (4,4)},
		'B087.npz': {'crop':None, 'downfactors': (2,10)},
		'B088.1.npz': {'crop':None}, # Can't get first component
		'B089.npz': {'crop':None, 'downfactors': (4,10)},
		'B090.npz': {'crop':None, 'downfactors':(1,4),'xos':[52.75], 'masks':[59]},
		'B091.npz': {'crop':None, 'downfactors': (4,2)},
		'B092.npz': {'crop':None, 'xos' : [52.53], 'downfactors':(1,4)}, # Can't get
		'B093.1.npz': { # A True outlier
			'crop': None,
			'xos' : [37.61, 47.36],
			'cuts':[42.69], # 41.23
			'downfactors':(1,2),
			'masks':[62,63,24]
		},
		'B094.npz': {},
		'B095.npz': {},
		'B097.npz': {'crop':None, 'xos' : [54.24], 'downfactors':(1,4)},
		'B098.npz': {},
		'B100.npz': {'downfactors': (2,2)},
		'B102.npz': {'crop':None, 'xos' : [52.47], 'downfactors':(1,2)},
		'B103.npz': {'crop':None, 'downfactors':(1,4)},
		'B104.npz': {'crop':None, 'downfactors':(1,4), 'xos' : [57.65], 'masks':[36]},
		'B105.npz': {'downfactors': (4,2)},
		'B106.npz': {
			'crop':None,
			'xos':[58.33],
			# 'masks':[58,55,56,57],
			'downfactors':(4,2)#1,4
		}, # Enh
		'B107.npz': {'crop':None, 'downfactors':(1,4), 'masks':[50]},
		'B108.npz': {'downfactors': (2,2)},
		'B109.npz': {
			'crop':None,
			'xos' : [81.54, 55.42],
			'downfactors': (2,4)
		},
		'B111.npz': {
			'crop':None,
			'downfactors': (4,4)
		},
		'B112.npz': {'downfactors': (4,1)}, # over-dedispersed. In general: fit seems to have the totally wrong sign 'targetDM':560.
		'B113.npz': {'crop':None, 'masks':[53], 'downfactors': (4,4)},
		'B114.npz': {'crop':None, 'masks':[53]},
		'B115.npz': {'masks':[53]},
		'B116.npz': {'crop':None, 'downfactors':(1,4)},
		'B117.npz': { # Hard to see
			'crop':None,
			'xos' : [50.68],
			'downfactors':(1,4), #(1,2) as well
			'masks':[33,34,35, 40]
		},
		'B118.npz': {'crop':None, 'downfactors':(1,4), 'masks':[33]},
		'B119.npz': {
			'downfactors': (1,4),
			'crop': None,
			'xos' : [51.64],
			'masks': [50, 51,53, 54,60, 61, 23, 24],
		},
		'B120.npz': {'crop':None, 'xos' : [55.44], 'downfactors': (1,1)},
		'B121.1.npz': {
			'crop':(1220//2-200, 1220//2+300),
			'xos' : [15.1, 21.88, 29.25],
			'cuts' : [17.74, 26.16],
			'submasks': [[],[],[28,15]]
		},
		'B122.npz': {'crop':None, 'downfactors': (2,10)}, # not great
		'B124.npz': {
			'crop':None,
			'xos' : [51.23],
			'downfactors':(4,10)
		},
		'B125.npz': {'downfactors': (4,10)}, # decimated
		'B126.npz': {},
		'B127.npz': {'crop':None, 'downfactors':(1,4), 'masks':[range(14,37)]},
		'B128.npz': {'crop':None, 'downfactors':(1,4), 'xos' : [51.06], 'masks':[55]},
		'B129.npz': {'crop':None, 'downfactors': (1,1)}, # cutoff
		'B130.npz': {'crop':None, 'downfactors':(1,4)},
		'B131.npz': {'crop':None, 'downfactors':(1,4), 'masks':[61]},
		'B132.npz': {'downfactors': (4,1)},
		'B133.npz': {'crop':None},
	}

	save = datasets['aggarwal'].save
	results = []
	compare_results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		masks = [12]
		if 'masks' in kwargs:
			kwargs['masks'] = kwargs['masks']+masks
		else:
			kwargs['masks'] = masks

		# Burst-basis cropping
		crop = (1220//2-150, 1220//2+150) if 'crop' not in kwargs else kwargs['crop']
		kwargs['crop'] = crop

		# Burst-basis DM
		targetDM = DM_FRB20121102A if 'targetDM' not in kwargs else kwargs['targetDM']
		kwargs['targetDM'] = targetDM

		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/frb20121102A/aggarwal2021/',
			show=not save,
			save=save,
			correctTimes=False,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

		# compare_results.append(
		# 	arrivaltimes.measure_allmethods(
		# 		filename,
		# 		show=True,
		# 		targetDM=targetDM,
		# 		crop=crop,
		# 		**kwargs
		# 	)
		# )

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	# comparedf = pd.DataFrame(
	# 	data=compare_results,
	# 	columns=arrivaltimes.results_columns+arrivaltimes.allmethods_columns
	# ).set_index('name')
	# comparedf.to_csv(f"measurements/collected/allmethods_results_aggarwal_{datetime.now().strftime('%b-%d-%Y')}.csv")

	fileout = f"measurements/frb20121102A/aggarwal2021/results_{datetime.now().strftime('%b-%d-%Y')}-{targetDM:.3f}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['snelders'].run: ## 'snelders__main__'
	prefix = datasets['snelders'].path

	# This will need to be combined in order to get a drift rate
	# uncombined:
	# 'burst_B06_a.npz' : {},#'targetDM': 560.105},
	# 'burst_B06_b.npz' : {'crop': (0,200)},
	# 'burst_B31_a.npz' : {},
	# 'burst_B31_b.npz' : {},

	arrivaltimes.logdebug = False
	files = {
		'burst_B06.npz'   : {
			'crop': (197, 312),
			'xos' : [0.02, 0.06],
			'masks': [61,62,63, 0,1,2,3,50,51],
			'cuts': [0.04]
		},
		'burst_B07.npz'   : {
			'crop': (90,170),
			'bw_width_factor': 20,
			'masks':[0, 1, 18],
		},
		'burst_B10.npz'   : { # ??? why is chisq 0?
			'crop': (100,275),
			'downfactors': (2,1)
		}, # high uncertainty
		'burst_B30.npz'   : {
			'crop': (158, 303),
			'masks': [145,165],
		},
		'burst_B31.npz'   : {
			'crop': (188,290),
			'xos' : [0.23, 0.18],
			'cuts': [0.2],
			'submasks': [[],[15]],
			'downfactors': (2,2)
		},
		'burst_B38.npz'   : {
			'crop': (96, 195)
		},
		'burst_B43.npz'   : {
			'masks': [98,17,130,185,189,205,210],
			'band_int_width_fac' : 2
		},
		'burst_B44.npz'   : {
			'downfactors': (1,1),
			'crop': (280,490)
		},
	}

	save = datasets['snelders'].save
	results = []
	compare_results = []

	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/frb20121102A/snelders2023/',
			show=not save,
			targetDM=DM_FRB20121102A,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

		# compare_results.append(
		# 	arrivaltimes.measure_allmethods(
		# 		filename,
		# 		show=False,
		# 		targetDM=DM_FRB20121102A,
		# 		**kwargs
		# 	)
		# )

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	# comparedf = pd.DataFrame(
	# 	data=compare_results,
	# 	columns=arrivaltimes.results_columns+arrivaltimes.allmethods_columns
	# ).set_index('name')
	# comparedf.to_csv(f"measurements/collected/allmethods_results_snelders_{datetime.now().strftime('%b-%d-%Y')}.csv")
	# print(f"Saved measurements/collected/allmethods_results_snelders_{datetime.now().strftime('%b-%d-%Y')}.csv")

	fileout = f"measurements/frb20121102A/snelders2023/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}")

########################################
########################################

if datasets['hewitt'].run: ## 'hewitt__main__'
	prefix = datasets['hewitt'].path

	files = {
		'B1.npz': {
			'xos': [ # B1
				3.799324596774193,
				11.130263359395325,
				12.026910462950072,
				13.043694163004853,
				13.23,
				13.55710055414921,
				13.619936523018335,
				13.969846624171904,
				14.322220880967398,
				14.435572040496018,
				14.546459044382713,
				14.747522576106348,
				14.810192278418322,
				14.884202212577222,
				15.164126882904032,
				15.268576386757319,
				15.540407417928401,
				15.812582793419582,
				15.844893927693217,
				15.972919176701959,
				16.260671164950182,
				17.072343190089658,
				17.288272495869172,
				17.746806314871293,
				18.398159483402505,
				18.8763356207015,
				22.762659868555932,
				24.66600838026668,
				25.557983513061014,
				26.07771705500417,
				30.599137495217533,
				30.890346692548704,
				31.43356384910877,
				31.668771277722406,
				32.239989318641236,
				32.77760629832955,
				37.31434494055073,
				38.63140753213523,
				39.338360246735725,
				40.78510863851335,
				41.22997223516458,
				41.69601981260873,
				43.94861535949389,
				51.000348705216325
			],
			'cuts' : [
				7.71,
				11.57,
				12.64,
				13.14,
				13.34,
				13.59,
				13.86,
				14.16,
				14.41,
				14.49,
				14.66,
				14.79,
				14.86,
				15.06,
				15.22,
				15.45,
				15.66,
				15.83,
				15.92,
				16.13,
				16.96,
				17.2,
				17.4,
				18.2,
				18.77,
				21.23,
				24.0,
				25.11,
				25.85,
				29.8,
				30.75,
				31.23,
				31.56,
				31.95,
				32.62,
				35.03,
				38.15,
				39.08,
				40.55,
				40.95,
				41.52,
				42.91,
				47.75
			],
			'sigmas': [ # B1
				1.0531866359446997,
				0.803747695852536,
				0.9700403225806475,
				0.01,
				0.01,
				0.05446849850814317,
				0.07122803651065013,
				0.02,
				0.05, # h
				0.01,
				0.02,
				0.04,
				0.02,
				0.05027861400751732, # m
				0.0586583830087708,
				0.05,
				0.05,
				0.02,
				0.02,
				0.01, #s
				0.025,
				0.02,
				0.08798757451315353,
				0.42736821906390077,
				0.38965925855826455,
				0.255582954538216,
				1.2276361586835556, # z
				0.4985962555745509,
				0.62848267509397,
				0.3,
				0.3184312220476144,
				0.36033006705388004,
				0.22625376303382794,
				0.2639627235394677,
				0.43574798806515247,
				0.331000875549492,
				1.3742821162054852, # jj
				0.4566974105682817,
				0.44831764156703713,
				0.16759538002505536,
				0.29748179954447806,
				1.0390913561553674,
				1.0390913561553674,
				0.4315581035645266
			],
			'fix_xos': True,
			'tolms': 0.001,
			'load_solutions': 'measurements/frb20220912A/hewitt2023/B1-Mar-19-2025.sols.npz',
			# 'save_solutions': True
		},
		'B2.npz': {
			'xos' : [ # B2
				5.75,
				8.75,
				8.96,
				9.07,
				9.12,
				9.6,
				9.64,
				9.72,
				9.55,
				10.87,
				11.14,
				11.23,
				11.4,
				11.47,
				12.34,
				12.52,
				12.78,
				13.05,
				13.56,
				18.92,
				19.87
			],
			'sigmas': [0.05]*21,
			'cuts' : [
				8.3,
				8.86,
				9.03,
				9.11,
				9.44,
				9.58,
				9.62,
				9.69,
				10.73,
				11.05,
				11.19,
				11.33,
				11.44,
				11.69,
				12.48,
				12.59,
				12.91,
				13.29,
				18.03,
				19.52
			],
			'submasks':[[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[range(87,89), range(80,82)],[],[]],
			'fix_xos': True,
			'tolms': 0.001,
			'load_solutions' : 'measurements/frb20220912A/hewitt2023/B2-Mar-19-2025.sols.npz'
			# 'save_solutions': True
		},
		'B3.npz': {
			'xos' : [6.82, 8.0, 8.55, 8.91, 13.42, 15.03],
			'cuts' : [7.38, 8.15, 8.72, 11.96, 14.88],
			'fix_xos': True,
		}
	}

	# arrivaltimes.logdebug = True
	save = datasets['hewitt'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/frb20220912A/hewitt2023/',
			show=not save,
			subtractbg=(False,False),
			save=save,
			# save_solutions=True,
			hide_legend=True,
			label_components=True,
			cmap_norm='log',
			cmap='inferno',
			measure_drift=False,
			# show_components=True,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	fileout = f"measurements/frb20220912A/hewitt2023/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['sheikh'].run: ## 'sheikh__main__'
	prefix = datasets['sheikh'].path
	# listnpzs(prefix)

	## After 35 initial bursts, reanalysis found 67 additional bursts in Sheikh+2024
	## Refer to the rename scripts to map the original filenames to these names
	arrivaltimes.logdebug = False
	files = {
		'B01.npz' : {
			'xos' : [5.28, 5.8, 6.56, 7.12, 8.56, 9.46],
			'cuts' : [5.57, 6.2, 6.83, 7.93, 9.0],
			'sigmas': [0.2]*6,
			'fix_xos': True,
			'legendloc': 4
		},
		'B02.npz' : {
			'xos' : [17.54, 22.76],
			'downfactors': (12,4)
		}, # B01
		'B03.npz' : {
			'crop': (250,430),
			'xos': [6.24],
			'sigmas':[0.1],
			'downfactors': (12,2)
		}, # large uncertainty
		'B04.npz' : {
			'crop': (280, 280+512),
			'xos' : [13.6, 17.1, 22.17],
			'cuts' : [16.43, 19.85],
			'fix_xos': True,
			'downfactors': (8,2)
		},
		'B05.npz' : {
			'legendloc': 4,
			'xos' : [10.0, 11.76],
			'cuts' : [10.8],
			'submasks': [[],[range(712,740), range(1193,1224)]]
		}, # B02
		'B06.npz' : {
			'xos' : [21.78, 6.27, 17.28, 19.51],
			'cuts' : [20.51, 18.59, 10.41],
			'sigmas': [0.5,1.5,0.5,0.5],
			'fix_xos': True,
			'downfactors': (4,4)
		}, # cutoff
		'B07.npz' : {
			'crop': (45,-1),
			'xos' : [3.49, 3.98],
			'cuts' : [3.79]
		}, # B03
		'B08.npz' : {
			'xos' : [8.14, 8.81, 14.46],
			'cuts' : [8.53, 11.57],
			'fix_xos': True,
			'legendloc': 3,
			'downfactors': (2,1)
		},
		'B09.npz' : {
			'crop': (130,660),
			'downfactors': (16,2),
			'masks': [range(1072,1184), range(54,315)]
		},
		'B10.npz' : {
			'measure_drift' : False, # up and down drifting
			'xos'  : [9.77, 22.84, 27.41, 29.08, 32.59, 38.22, 40.17],
			'cuts' : [39.5, 35.85, 30.71, 28.51, 24.83, 16.29],
			'downfactors': (2,4), # 2,2
			'submasks': [[],[range(144,317)],[],[range(35,310)],[],[],[range(25,67)]]
		}, # Drop comp a and f
		'B11.npz' : {
			'measure_drift': False, # Messy
			'crop' : (464,464+1024),
			'downfactors': (4,2),
			'xos'  : [22.25, 24.44, 25.13, 26.71, 31.17, 32.34, 36.82, 38.0, 43.92, 48.17, 49.79],
			'sigmas': [3,0.5,0.5,0.5,1,2,0.5,1,0.1,0.1,3],
			'cuts' : [31.67, 23.74, 24.87, 26.3, 29.11, 34.94, 37.37, 41.25, 46.26, 49.03],
			'fix_xos': True,
			'submasks': [[],[],[],[],[],[],[],[],[range(320,740)],[range(320,740)],[range(320,740)]]
		}, # dropping i,j,k, via submask
		'B12.npz' : {
			'crop': (240,-1),
			'downfactors': (6,1),
			'masks'  : [range(2088, 2507), range(1748,1867)],
			'xos'    : [6.3, 7.8, 8.37, 12.87, 47.35, 48.93, 56.47],
			'sigmas' : [0.5, 1, 0.1, 0.1, 0.5, 1, 0.5],
			'cuts'   : [6.8, 8.12, 11.38, 32.03, 48.2, 53.23],
			'fix_xos': True
		}, # letting b and c get filtered
		'B13.npz' : {
			'downfactors': (6,1),
			'xos' : [8.43, 30.04, 32.68, 35.45, 36.61, 42.57],
			'cuts' : [22.86, 31.34, 34.41, 36.04, 40.06],
			'sigmas': [1,1,1,1,1,1],
			'submasks': [range(1150,1193),[],[],[],[],[]],
			'fix_xos': True
		}, # B04
		'B14.npz' : {
			'xos' : [2.43, 7.49, 9.05, 8.53],
			'cuts' : [5.29, 8.05, 8.72],
			'fix_xos': True,
			'masks': [range(1216,1338)]
		}, # B05
		'B15.npz' : {
			'xos' : [23.11, 16.08],
			'cuts' : [20.37],
			'downfactors': (4,1)
		}, # B06
		'B16.npz' : {
			'downfactors': (2,1)
		}, # B07 cutoff
		'B17.npz' : {
			'downfactors': (4,1),
			'xos' : [5.39, 2.34],
			'cuts' : [4.1]
		}, # B08
		'B18.npz' : {
			'crop': (20,294),
			'xos' : [5.61, 10.29, 11.09, 4.77],
			'sigmas': [0.1,0.1,0.1,0.1],
			'cuts' : [8.22, 10.62, 5.11],
			'fix_xos': True,
			'downfactors': (8,2)
		}, # Letting c get filtered out. Blended.
		'B19.npz' : {
			'crop': (211,-1),
			'xos' : [20.79, 25.42],
			'cuts' : [23.28],
			'downfactors': (16,2)
		},
		'B20.npz' : {
			'crop': (310,1000),
			'downfactors': (8,2),
			'xos' : [15.41, 22.4],
			'cuts' : [19.56]
		}, # B09
		'B21.npz' : {
			'crop': (130, 508),
			'xos' : [ 8.35, 13.67, 15.43],
			'cuts' : [11.5, 14.52],
			'downfactors': (8,2)
		},
		'B22.npz' : {
			'crop'   : (240,240+512),
			'downfactors': (8,2),
			'xos'    : [9.45, 16.84, 20.3],
			'cuts'   : [13.9, 18.98],
			'sigmas' : [1,1,1],
			'fix_xos': True,
		},
		'B23.npz' : {
			'crop': (260, 880),
			'downfactors': (16,4),
			'xos' : [10.85, 17.72, 24.34, 29.92],
			'cuts' : [14.44, 20.82, 27.41],
			'masks': [range(1,40), range(1208,1130)],
			'submasks': [[range(1139,1192), range(820,920)], [range(425,541)], [],[]]
		}, # B10
		'B24.npz' : {
			'xos' : [12.94, 3.25, 7.67],
			'cuts' : [3.68, 11.14],
			'submasks': [[range(0,1549)],[range(1517,1768)],[]] # zero out comp b
		},
		'B25.npz' : {
			'crop': (484,893),
			'xos' : [10.31, 15.65],
			'cuts' : [13.1],
			'legendloc': 3,
		},
		'B26.npz' : {
			'crop': (421,-1),
			'xos' : [11.9, 20.07, 21.94, 28.72],
			'cuts' : [17.18, 21.08, 27.05],
			'sigmas': [0.5,0.1,0.1,0.1],
			'fix_xos': True,
			'downfactors': (4,2)
		}, # B11
		'B27.npz' : {
			'xos' : [22.48, 30.0],
			'cuts' : [27.02]
		}, # B12
		'B28.npz' : {
			'xos' : [21.38, 14.8, 16.68],
			'cuts' : [16.0, 18.28],
			'downfactors': (6,4),
			'snr_cutoff': 4,
			'fix_xos': True,
		}, # B13
		'B29.npz' : {
			'xos' : [16.63, 24.09],
			'cuts' : [21.8]
		}, # B14
		'B30.npz' : { # comp b has a long tail
			'cuts' : [19.01, 21.76, 11.02],
			'xos' : [14.53, 20.36, 22.41, 7.75],
			'downfactors': (4,2),
			'fix_xos': True
		}, # B15
		'B31.npz' : { # Cutoff
			'crop': (495,840),
			'xos' : [10.98, 8.76],
			'cuts' : [10.27],
		}, # B16
		'B32.npz' : {
			'downfactors': (4,4),
			'xos': [3.91],
			'fix_xos': True
		},
		'B33.npz' : {
			'xos' : [6.81, 8.55],
			'cuts' : [7.76],
			'downfactors': (6,2),
		}, # B17
		'B34.npz' : {
			'crop': (445,-1),
			'xos' : [22.82, 8.4, 14.89, 17.0, 20.92, 27.65],
			'cuts' : [15.84, 19.28, 21.61, 25.87, 12.28],
			'fix_xos': True,
			'sigmas': [0.5]*6,
			'downfactors': (2,1),
		}, # B18
		'B35.npz' : { # low snr
			'crop': (5110,6045),
			# 'downfactors': (4,5),
			'masks': [range(332,385)]
		},
		'B36.npz' : {
			'crop': (108,-1),
			'xos' : [6.55, 9.57, 11.26, 17.79, 12.67],
			'cuts' : [8.8, 10.46, 15.8, 12.12],
			'downfactors': (2,1),
			'fix_xos': True
		},
		'B37.npz' : {},
		'B38.npz' : {
			'xos' : [14.33, 18.12, 21.63, 26.77, 16.6],
			'cuts' : [15.49, 19.9, 25.46, 17.44],
			'fix_xos': True,
			'masks': [0,1,2,3,range(350,396)]
		}, # B19
		'B39.npz' : {'crop': (955,1180)},
		'B40.npz' : {
			'crop': (1160,1600),
			'downfactors': (2,2)
		}, # B20
		'B41.npz' : {
			'crop': (545,1190),
			'xos' : [20.25, 23.19, 15.68],
			'cuts' : [22.11, 17.94],
			'downfactors': (4,1),
			'masks': [range(0,8)]
		},
		'B42.npz' : {
			'crop': (267,-1),
			'xos' : [10.66, 11.57, 9.43, 7.99, 17.87, 19.12, 15.53, 22.41, 26.02, 28.71, 29.96, 36.47],
			'sigmas': [0.5]*12,
			'cuts' : [11.13, 9.98, 8.83, 18.38, 13.19, 17.15, 21.19, 23.72, 28.07, 29.42, 35.11],
			'fix_xos': True,
		}, # Probably a few that need to be filtered later
		'B43.npz' : {},
		'B44.npz' : { # ok
			'crop': (97,437),
			'xos' : [4.84, 6.4, 12.9],
			'cuts' : [5.39, 7.37],
			'fix_xos': True,
			'sigmas': [1,0.1,1],
			'masks': [range(329,406)],
			'downfactors': (4,1)
		},
		'B45.npz' : {
			'crop': (553,1310),
			'xos' : [13.66, 16.45, 21.04, 23.99, 27.16, 29.16, 19.21],
			'sigmas': [0.1]*7,
			'cuts' : [14.77, 22.79, 25.08, 27.87, 17.82, 19.95],
			'fix_xos': True,
			'masks': [range(356,392)],
			'submasks': [[],[],[range(450,459)],[range(403,415)],[],[],[]],
		}, # B21
		'B46.npz' : {
			'xos' : [13.24, 14.4, 18.55],
			'cuts' : [13.85, 16.39]
		},
		'B47.npz' : {'downfactors': (4,2)}, # B22
		'B48.npz' : {
			'crop': (208, 1120),
			'downfactors': (8,4)
		},
		'B49.npz' : {
			'xos' : [27.36, 20.69, 39.46, 43.11],
			'cuts' : [41.58, 34.15, 24.03],
			'fix_xos': True,
			'downfactors': (4,2),
			'legendloc': 3
		},
		'B50.npz' : {
			'crop': (483,991),
			'downfactors': (1,2),
			'masks': [range(760,1168)]
		},
		'B51.npz' : {
			'xos' : [7.75, 12.18, 16.25, 21.99],
			'cuts' : [19.43, 14.0, 9.38],
			'fix_xos': True,
			'masks': [range(345,405)],
			'downfactors': (4,2)
		},
		'B52.npz' : {
			'crop': (281,-1),
			'masks': [range(0,36)],
			'downfactors': (4,2),
			'xos' : [24.25, 10.68, 12.28, 18.32, 31.51, 37.25],
			'cuts' : [11.52, 14.26, 20.94, 29.3, 35.46],
			'fix_xos': True
		}, # B23
		'B53.npz' : {
			'xos' : [7.54, 6.73, 10.09, 12.2, 13.4, 18.49, 19.6],
			'cuts' : [7.01, 8.95, 11.2, 12.79, 16.68, 19.01],
			'fix_xos': True,
			'masks': [range(373,391)]
		},
		'B54.npz' : {
			'crop': (262,430),
			'downfactors': (4,1)
		},
		'B55.npz' : {
			'crop' : (575,1500),
			'masks'  : [range(2406,2500), range(2058,2174),range(1208,1258),range(314,369),range(0,49), range(1446,1459), range(1249,1271)],
			'xos'    : [12.02, 19.38, 20.88, 21.47, 22.36, 27.47, 35.68, 51.1, 36.88],
			'sigmas' : [1,1,0.5,0.5,0.5,1,1,1,1],
			'cuts'   : [15.8, 19.92, 21.14, 21.95, 24.78, 31.29, 41.22, 36.43],
			'submasks': [[],[],[],[range(456,477)],[],[],[],[],[]],
			'fix_xos': True,
			'downfactors': (4,1),
			'snr_cutoff': 5,
			'hide_legend': True
		},
		'B56.npz' : {'downfactors': (1,1), 'masks': [range(396,458)]}, # low snr
		'B57.npz' : { # only b has a sharp enough edge
			'xos' : [13.48, 17.72, 20.8, 22.6],
			'cuts' : [15.67, 19.58, 21.87],
			'crop': (300,890),
			'downfactors': (8,5),
			'masks': [range(0,40)],
			'submasks': [[range(0,600)],[],[range(0,600)],[range(0,600)]]
		}, # B24
		'B58.npz' : {'downfactors': (4,4)},
		'B59.npz' : {
			'xos' : [17.97, 20.83, 24.4, 27.67, 31.45, 33.85],
			'cuts' : [18.59, 23.11, 26.14, 29.33, 32.49],
			'crop': (866,1674),
			'fix_xos': True,
			'snr_cutoff': 4
		}, # B25
		'B60.npz' : {
			'xos' : [21.68, 7.81, 15.82, 27.54, 29.82],
			'cuts' : [13.72, 18.51, 25.37, 28.93],
			'fix_xos': True
		},
		'B61.npz' : {
			'crop': (155,655),
			'xos' : [6.35, 8.3, 9.43, 14.08, 16.22, 22.69],
			'cuts' : [8.74, 6.97, 11.8, 15.06, 19.43],
			'fix_xos': True,
			'snr_cutoff' : 4
		}, # B26
		'B62.npz' : {
			'xos'  : [14.1, 20.04, 22.41],
			'cuts' : [18.47, 21.01],
			'downfactors': (4,2)
		},
		'B63.npz' : {
			'crop': (2350,2928),
			'downfactors': (8,2)
		}, # B27
		'B64.npz' : {},
		'B65.npz' : {
			'xos' : [15.63, 8.76, 22.25],
			'cuts' : [12.27, 19.83],
			'crop': (300,780),
			'snr_cutoff': 3,
			'masks': [range(367,396)]
		},
		'B66.npz' : {
			'cuts' : [15.21],
			'xos' : [12.03, 18.43],
			'masks': [range(33,70)],
			'downfactors': (4,2)
		},
		'B67.npz' : {
			'crop': (500,800),
			'downfactors': (2,1)
		},
		'B68.npz' : {
			'masks': [range(930,950)],
			'crop': (522,1168),
			'cuts' : [20.3, 25.08],
			'xos' : [26.91, 12.65, 22.02],
			'downfactors': (4,1)
		},
		'B69.npz' : {
			'crop': (566,1216),
			'cuts' : [15.29, 12.36, 9.3, 19.82, 23.55, 27.54],
			'xos' : [7.8, 16.85, 10.56, 13.96, 22.85, 24.28, 29.94],
			'fix_xos': True,
			'masks': [range(372,383)],
		},
		'B70.npz' : {'snr_cutoff':5}, # B28
		'B71.npz' : {
			'xos' : [13.14, 15.48, 20.83, 29.06, 33.43, 35.25, 41.07, 42.99, 49.36, 22.57],
			'cuts' : [13.9, 25.28, 32.08, 34.07, 38.8, 41.96, 46.76, 19.14, 21.76],
			'crop': (884,1880),
			'sigmas': [1]*10,
			'fix_xos': True,
			'masks': [range(750,840)],
			'downfactors': (2,2),
			'snr_cutoff': 5,
			# Comp b has a long tail so is not well modelled by a gaussian, c has a bulbous shape, and the rest are blended. I only trust e
			'submasks': [[range(0,860)],[range(0,860)],[range(0,860)],[range(0,860)],[],[range(0,860)],[range(0,860)],[range(0,860)],[range(0,860)],[range(0,860)]]
		},
		'B72.npz' : {
			'crop': (213,-1),
			'cuts' : [7.83, 13.5],
			'xos' : [7.0, 8.99, 14.72],
			'downfactors': (4,1)
		},
		'B73.npz' : {
			'xos' : [33.84, 40.13, 47.47],
			'cuts' : [37.03, 44.64],
			'masks': [range(0,10)],
			'downfactors': (2,1)
		},
		'B74.npz' : {}, # B29
		'B75.npz' : {
			'cuts' : [24.1, 30.62],
			'xos' : [20.78, 26.34, 33.85],
			'fix_xos': True,
			'measure_drift': False
		}, # B30
		'B76.npz' : {'downfactors': (4,1)},
		'B77.npz' : {
			'downfactors': (4,2)
		}, # B31
		'B78.npz' : {
			'crop': (667,-1),
			# 'cuts' : [51.36],
			# 'xos' : [50.57, 52.38]
		},
		'B79.npz' : {},
		'B80.npz' : {'snr_cutoff': 4},
		'B81.npz' : {},
		'B82.npz' : {'downfactors': (8,1)},
		'B83.npz' : {
			'downfactors': (8,1),
			'cuts' : [5.66],
			'xos' : [5.21, 6.21],
			'masks': [range(1247,1310)]
		}, # B32
		'B84.npz' : {
			'crop': (270,-1),
			'cuts' : [12.46, 15.48, 18.23, 22.17, 23.96, 26.14, 29.05, 31.15, 34.73],
			'xos' : [11.19, 13.5, 16.88, 20.38, 23.04, 24.95, 27.57, 32.63, 30.2, 41.45],
			'fix_xos': True,
			'masks': [range(367,389)],
			'submasks': [[],[],[],[],[],[],[],[],[],[range(0,123)]]
		},
		'B85.npz' : {'crop': (1000,1660)},
		'B86.npz' : { # blend
			'downfactors': (8,4),
		},
		'B87.npz' : {
			'crop': (680, 1230),
			'masks': [range(13,64)],
			'downfactors': (8,2),
			'cuts' : [15.4],
			'xos' : [14.53, 16.63],
			'fix_xos': True
		}, # B33
		'B88.npz' : {}, # B34
		'B89.npz' : {
			'crop': (760,1700),
			'cuts' : [20.79, 22.8, 24.69, 28.25, 35.11],
			'xos' : [18.93, 23.71, 21.87, 25.74, 30.26, 37.86],
			'masks': [range(366,391)]
		}, # B35
	}

	save = datasets['sheikh'].save
	results = []
	compare_results = []
	targetDM = DM_FRB20220912A
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'

		downfactors = (1,1)
		if 'downfactors' in kwargs:
			downfactors = kwargs['downfactors']
			del kwargs['downfactors']

		burst_results = arrivaltimes.measureburst(
			filename,
			downfactors=downfactors,
			subtractbg=True,
			outdir='measurements/frb20220912A/sheikh2024/',
			show=not save,
			show_components=False,
			targetDM=targetDM, # Hewitt B1 reported DM
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

		# compare_results.append(
		# 	arrivaltimes.measure_allmethods(
		# 		filename,
		# 		show=True,
		# 		targetDM=targetDM,
		# 		**kwargs
		# 	)
		# )

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	fileout = f"measurements/frb20220912A/sheikh2024/results_{datetime.now().strftime('%b-%d-%Y')}-{targetDM:.3f}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['zhang'].run: ## 'zhang__main__'
	prefix = datasets['zhang'].path
	# listnpzs(prefix)

	arrivaltimes.logdebug = False
	files = {
		'5-01.npz': {
			'bw_width_factor':2,
			'masks':[11,15,47,48,137,206,57,76],
			'downfactors': (2,2),
		},
		'5-02.npz': {
			'xos' : [13.22, 14.67],
			'cuts': [14.04],
			'sigmas': [0.5,0.5],
			'downfactors': (2,2)
			# 'fix_xos': True,
		},
		'5-03.npz': {'downfactors': (4,1)},
		'5-04.npz': {
			'xos' : [11.18, 12.87],
			'cuts': [12.27],
			'masks': [range(41,47)],
			'legendloc': 4,
			'measure_drift': True,
			# 'recalculateDM': True
			'targetDM': 220.556,
			'cuts': [11.97]

		},
		'5-05.npz': {}, # 2? cutoff
		'5-06.npz': {},
		'5-07.npz': {'masks':[44,45,46]},
		'5-08.npz': {},
		'5-09.npz': {'masks':[158]},
		'9-01.npz': {
			'crop': (512,1024+512),
			'xos': [17.64, 25.71, 33.69, 36.11],
			'cuts' : [35.29, 30.23, 20.78]
		}, # Multiple DMs? May need special attention
		'9-02.npz': { # Ignoring middle component due to blend
			'masks':[0,1,7,6,8,137,104, 112, 126, 149],
			'xos' : [12.6, 18.05],
			'cuts' : [16.36],
			'fix_xos': True
		},
		'9-03.npz': { # Condiser dropping component d due to blend
			'masks': [135,136,132,130,133,125,103,104,105,106,range(121,146)],
			# 'xos' : [19.9, 51.39, 85.3],
			'xos' : [19.9, 85.3, 48.77, 51.56, 54.4],
			'cuts' : [39.56, 49.77, 53.72, 71.11],
			'fix_xos': True,
			'correctTimes': True,
			'submasks': [[],[],[],[range(61,125)],[]]
			# 'cuts' : [53.34, 49.95, 38.28, 72.39]
		}, # Extremely blended
		'9-04.npz': {'xos':[27.9, 64.11]}, # Clean. Use in paper?
		'9-05.npz': {
			'xos':[7.49, 12.4, 15.98],
			'cuts' : [10.21, 14.54],
			'masks':[range(126,190)]
		}, #DM seems to differ burst to burst
		'9-06.npz': { # 16 Microshots, exported at higher resolution
			# 'targetDM':219.425,
			'crop':(2048-256,2048+256),
			'xos' : [7.59, 11.87, 12.23, 12.55, 12.93, 13.41, 13.67, 16.06, 16.43, 17.54, 18.05, 19.02, 21.56, 22.45, 18.29],
			'cuts' : [12.06, 12.46, 12.66, 13.18, 13.55, 14.26, 16.24, 17.07, 17.76, 18.68, 20.4, 22.09, 10.08, 18.22],
			'fix_xos':True,
			# 'sigmas': [0.1]*15,
			'sigmas': [0.4, 0.09, 0.08, 0.01, 0.13, 0.1, 0.06, 0.87, 0.16, 0.1, 0.11, 0.12, 2.64, 0.2, 0.05],
			'bw_width_factor':1,
			'hide_legend': True
		},
		'9-07.npz': {},
		'9-08.npz': {
			'xos' : [9.21, 12.32],
			'cuts' : [11.05],
			'fix_xos': True,
			'correctTimes': True
		}, # component b truly positive at this DM
		'9-09.npz': {
			# 'targetDM':224,
			'xos' : [39.83, 44.89],
			'cuts': [41.37],
			'masks':[range(221,256)],
			'submasks':[[range(167,188), range(4,41)],[]]
		}, # DM seems off, burst seems blended
		'9-10.npz': { # This could be a long burst coincident with a brighter short burst
			# 'targetDM': 219.75,
			'crop': (512+460, -512-420),
			'xos': [3.72, 4.41],
			'cuts': [4.13],
			## Alternate
			# 'xos':[13.24, 13.94],
			# 'cuts': [13.62],
			# 'fix_xos': True,
			# 'correctTimes': True,
			# 't_filter_factor': 5
		},
		'9-11.npz': {
			'xos' : [40.86, 49.15],
			'cuts' : [47.91],
			'fix_xos': True
		}, # Very blended
		'9-12.npz': {
			'xos' : [15.83, 17.65],
			'cuts': [17.46],
			't_filter_factor': 2,
			'submasks': [[],[range(61,85), range(141,165)]]
		}, # Blended and two DMs. Over dedispersed on component B?
		'9-13.npz': {'xos' : [10.93, 18.95]},
		'9-14.npz': { # Busy bursts. Component d is very low SNR
			'masks': [range(42, 50), 92, 106, 137, 140],
			'xos' : [46.76, 50.17, 54.28, 60.59, 66.85, 69.15, 73.26, 75.95, 80.15, 82.79],
			'cuts' : [48.85, 52.62, 58.63, 64.16, 67.83, 71.55, 74.92, 77.66, 81.47],
			'fix_xos': True,
			'correctTimes': True,
			'hide_legend': True
		},
		'9-15.npz': {
			'xos' : [12.66, 19.56],
			'cuts' : [17.76, 14.54],
			'fix_xos': True,
			'correctTimes': True,
			't_filter_factor': 2,
			'bw_width_factor': 2
		},
		'9-16.npz': {
			'xos' : [5.0, 49.12],
			'fix_xos': True
		}, # Probably two bursts
	}

	save = datasets['zhang'].save
	results = []
	compare_results = []
	targetDM = DM_FRB20220912A
	for filename, kwargs in files.items():
		# bypass cropping for some bursts
		if filename not in [
			'9-01.npz',
			'9-03.npz',
			'9-04.npz',
			'9-06.npz',
			'9-09.npz',
			'9-11.npz',
			'9-14.npz',
			'9-16.npz'
		]:
			crop = (512+256, -512-256) if 'crop' not in kwargs else kwargs['crop']
		else:
			crop = None if 'crop' not in kwargs else kwargs['crop']
		kwargs['crop'] = crop

		if 'targetDM' in kwargs:
			targetDM = kwargs['targetDM']
			del kwargs['targetDM']

		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/frb20220912A/zhang2023/',
			show=not save,
			show_components=False,
			subtractbg=(False,False),
			targetDM=targetDM,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

		# compare_results.append(
		# 	arrivaltimes.measure_allmethods(
		# 		filename,
		# 		show=True,
		# 		targetDM=targetDM,
		# 		**kwargs
		# 	)
		# )

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	fileout = f"measurements/frb20220912A/zhang2023/results_{datetime.now().strftime('%b-%d-%Y')}-{targetDM:.3f}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['nimmo'].run: ## 'nimmo__main__'
	prefix = datasets['nimmo'].path
	# listnpzs(prefix)
	# exit()

	arrivaltimes.logdebug = False
	files = {
		'20220114_B1.npz': {
			'masks':[250,251,243,245,232, range(1,26), range(213,216)],
			'snr_cutoff': 3,
			'downfactors': (4,2)
		},
		'20220114_B10.npz': {
			'downfactors':(4,2)
		},
		'20220114_B11.npz': {
			'crop': (209,383),
			'downfactors': (4,2),
			'masks': [200,184]
		},
		'20220114_B12.npz': {
			'downfactors': (4,1)
		},
		'20220114_B13.npz': {
			'crop': (180, 360),
			'xos' : [1.13, 0.72],
			'submasks': [[range(236,250), range(1,10)],[]],
			'downfactors': (4,2)
		},
		'20220114_B14.npz': {
			'downfactors': (8,2)
		},
		'20220114_B15.npz': {
			'crop': (600, 880),
			'downfactors': (8, 2),
			'masks': range(430, 494)
		},
		'20220114_B16.npz': {
			'downfactors': (8,2)
		},
		'20220114_B17.npz': {
			'downfactors': (4,2)
		},
		'20220114_B18.npz': {
			'xos' : [0.47, 0.61],
			'cuts': [0.54],
			'downfactors':(4,1)
		},
		'20220114_B19.npz': {
			'downfactors': (2,1)
		},
		'20220114_B2.npz': {
			'crop': (100, 250),
			'masks': [range(191,202), range(181,185), 84, 97],
			'bandmask_thres': 1.1
		},
		'20220114_B20.npz': {
			'downfactors': (4,1),
			'masks': [153,143,125,127,55]
		},
		'20220114_B21.npz': {
			'downfactors': (8,2)
		},
		'20220114_B22.npz': {
			'downfactors': (8,2),
			'snr_cutoff': 2
			# 'bw_filter': 'model_width'
		},
		'20220114_B23.npz': {
			'downfactors': (8,2),
			'snr_cutoff': 2
		},
		'20220114_B24.npz': {
			'downfactors': (8,2),
			'snr_cutoff': 1
		},
		'20220114_B25.npz': { # No bandwidth fit
			'crop': (100, 500),
			'downfactors': (4,4),
			'xos': [2.06],
			'fix_xos': True,
			'masks': [93, 190, range(53, 121)],
			'sigmas':[0.01]
		},
		'20220114_B26.npz': {
			'downfactors': (4,1)
		},
		'20220114_B27.npz': {
			'crop': (170, 430),
			'downfactors': (8,2)
		},
		'20220114_B28.npz': {
			'downfactors': (4,2)
		},
		'20220114_B29.npz': {
			'downfactors': (4,2)
		},
		'20220114_B3.npz': {
			'downfactors': (8,5),
			'bandmask_thres': 0.17
		},
		'20220114_B30.npz': {
			'crop': (170,430),
			'downfactors': (8,2)
		},
		'20220114_B31.npz': {
			'masks': [range(208,218), 127],
			'downfactors': (4,1),
			'bandmask_thres': 0.68
		},
		'20220114_B32.npz': {
			'downfactors': (4,2),
			'masks': [0]
		},
		'20220114_B33.npz': {
			'masks': [range(228,246)],
			'downfactors': (4,1)
		},
		'20220114_B34.npz': {
			'masks': [range(67,71)],
			'bandmask_thres': 0.7
		},
		'20220114_B35.npz': {
			'masks': [range(226,256)],
			'downfactors': (8,2)
		},
		'20220114_B36.npz': {
			'downfactors': (8,2), 'masks': [range(176, 190)]
		},
		'20220114_B37.npz': {
			'masks': [range(230, 256)],
			'bandmask_thres': 1.46
		},
		'20220114_B38.npz': {
			'masks': [range(233, 256)],
			'downfactors': (4,1)
		},
		'20220114_B39.npz': {
			'crop': (180, 400),
			'downfactors': (8,2),
			# 'masks': [range(127,226), 12, 106, 105]
		},
		'20220114_B4.npz': {
			'crop': (268,406),
			'downfactors': (8,1),
			# 'snr_cutoff': 1
		},
		'20220114_B40.npz': {
			'masks': [range(132,138)],
			'downfactors': (2,2)
		},
		'20220114_B41.npz': {
			'downfactors': (8,2)
		},
		'20220114_B42.npz': {
			'masks': [range(30, 48)],
			'downfactors': (8,1)
		},
		'20220114_B43.npz': {
			'downfactors': (4,1)
		},
		'20220114_B44.npz': {
			'masks': [range(229, 253), range(187, 193)],
			'downfactors': (4,2)
		},
		'20220114_B45.npz': {
			'downfactors': (8,2)
		},
		'20220114_B46.npz': {
			'downfactors': (8,2)
		},
		'20220114_B47.npz': {'bandmask_thres':0.845},
		'20220114_B48.npz': {
			'downfactors': (8,2)
		},
		'20220114_B49.npz': {},
		'20220114_B5.npz': {
			'crop': (185, 416),
			'xos' : [1.05, 1.21],
			'cuts': [1.12],
			'submasks': [[],[range(197, 225)]],
			'downfactors': (4,1)
		},
		'20220114_B50.npz': {
			'downfactors': (8,1)
		},
		'20220114_B51.npz': {
			'downfactors': (2,2)
		},
		'20220114_B52.npz': {
			'downfactors': (2,2)
		},
		'20220114_B53.npz': {
			# 'masks': [range(78,99), range(227,245)],
			'downfactors': (4,2)
		},
		'20220114_B6.npz': {
			'downfactors': (4,2),
			'masks': [range(119,146)]
		},
		'20220114_B7.npz': {
			'crop': (198,385),
			'downfactors': (8,1)
		},
		'20220114_B8.npz': {
			'downfactors': (8,1)
		},
		'20220114_B9.npz': {
			'crop': (168, 374),
			'downfactors': (8,2)
		},
		'20220221_B1.npz': {
			'crop': (30, 120),
			'downfactors': (2,1)
		},
		'20220221_B2.npz': {
			'crop': (80, 200),
			'masks':[range(199,220)],
			'downfactors': (2,1)
		},
		'20220223_B1.npz': {
			'crop':(163, 227),
			'downfactors': (2,1)
		},
		'B2.npz': {
			'crop':(2350,2600),
			'xos' : [0.79, 1.03],
			'cuts': [0.9],
			'downfactors': (2,1),
		},
		'B3.npz': {
			'crop':(1450,1550),
			'xos' : [0.49, 0.51], # second component improves first
			'cuts':[0.5],
			'legendloc': 2,
			'bandmask_thres': 0.47
		},
		'B4.npz': {
			'crop':(1100,1300),
			'masks': [range(196,202)],
		},
		'B5.npz': {
			'crop':(1900,2050),
			'xos':[13.9],
			'masks':[37],
			'downfactors': (1,1)
		},
		'B1.npz': {
			'crop': (0,2080),
			'downfactors': (8,2)
		} # invisible
	}

	save = datasets['nimmo'].save
	results = []
	outdir = 'measurements/frb20200120E/nimmo2023'
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'

		crop = (230,360) if 'crop' not in kwargs else kwargs['crop']
		kwargs['crop'] = crop

		burst_results = arrivaltimes.measureburst(
			filename,
			outdir=outdir,
			show=not save,
			save=save,
			# bw_filter='model',
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(
		data=results,
		columns=arrivaltimes.results_columns
	).set_index('name')

	fileout = f"{outdir}/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

if datasets['20180916B_Effelsberg'].run: ## '20180916B_Effelsberg'
	prefix = datasets['20180916B_Effelsberg'].path
	# prefix = 'E:/Downloads/FRB/20180916B_Effelsberg/'

	# listnpzs(prefix)
	#Bad data set
	# transpose and dedisperse

	# Removed: low res
		# 'Bc.npz' : {
		# 	'masks': [range(756,824), range(1733,1992)],
		# 	'crop': (600, 710)
		# },
		# 'Bd.npz' : {'crop': (320, 370)},
		# 'Be.npz' : {'crop': (400,450)},
		# 'Bg.npz' : {'crop':[630,680]},


	files = {
		'Ba.npz' : {
			'crop': (360, 410),
			'downfactors': (16,1),
			'masks': [range(1755,1859)]
		},
		'Bb.npz'   : {
			'crop': (0,200),
			'masks': [range(230,380)],
			'xos' : [12.74, 13.73, 14.86],
			'cuts' : [13.29, 14.44],
			'downfactors': (2,1),
			'fix_xos': True
		},
		'Bf.npz' : {'crop': (100,180)},
		'Bh.npz' : {'crop': (330,430), 'downfactors': (8, 2)},
	}

	save = datasets['20180916B_Effelsberg'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20180916B_Effelsberg/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20180916B_Effelsberg/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['201124_Effelsberg'].run: ## '201124_Effelsberg'
	prefix = datasets['201124_Effelsberg'].path
	# prefix = 'E:/Downloads/FRB/201124_Effelsberg/'

	#listnpzs(prefix)

	files = {
		'B01_0002.RM.npz' : {'downfactors': (1,32), 'masks' : [29]},
		'B02_0002.RM.npz' : {
			'xos' : [30.96, 36.72, 43.95],
			'cuts' : [34.18, 39.1],
			'downfactors': (1,2)
		},
		'B03_0002.RM.npz' : {
			'downfactors': (1,4),
			'xos' : [16.89, 20.67, 24.28],
			'cuts' : [19.5, 22.7]
		}, # blended
		'B04_0002.RM.npz' : {'downfactors': (1,8), 'xos' : [25.34]}, # could be nothing
		'B04_0006.RM.npz' : {
			'downfactors': (2,1),
			'xos' : [26.27],
			'masks' : [55,57,24,27]
		},
		'B05_0002.RM.npz' : {
			'xos' : [17.07, 23.59, 30.02],
			'cuts' : [19.33, 26.34],
			'submasks':[[],[49,50,51,53,54,55,56,57,61],[49,50,51]]
		},
		'B06_0003.RM.npz' : {'xos' : [25.83], 'downfactors': (1,8)},
		'B07_0003.RM.npz' : {'downfactors': (2,8), 'xos' : [24.12]}, #
		'B08_0002.RM.npz' : {}, #good
		'B09_0002.RM.npz' : {'downfactors': (1,32)}, #low SNR or no burst
		'B09_0003.RM.npz' : {
			'xos' : [16.95, 36.22],
			'downfactors': (1,4)
		}, #Noise?
		'B10_0002.RM.npz' : {
			'xos' : [22.34, 26.09, 29.3],
			'cuts' : [24.33, 27.48],
			'masks' : [40,32],
			'downfactors': (1,8)
		},
		'B11_0002.RM.npz' : {'xos' : [26.09], 'downfactors': (1,32)},
		'B12_0003.RM.npz' : {'downfactors': (1,8), 'xos' : [26.36]}, #low SNR
		'B13_0003.RM.npz' : {'xos' : [12.73]}, #good
		'B14_0003.RM.npz' : {'xos' : [11.61], 'downfactors': (1,8)}, #good
		'B15_0002.RM.npz' : {'xos' : [25.88], 'downfactors': (1,32)},
		'B16_0002.RM.npz' : {'xos' : [25.18], 'masks' : [0,1,27], 'downfactors': (1,32)},
		'B17_0002.RM.npz' : {'xos' : [38.33], 'downfactors': (1,16)},
		'B18_0003.RM.npz' : {'xos' : [26.27], 'downfactors': (1,8)}, #good
	}

	save = datasets['201124_Effelsberg'].save
	results = []
	compare_results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'

		targetDM = 411.6
		if 'targetDM' in kwargs:
			targetDM = kwargs['targetDM']
			del kwargs['targetDM']

		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/201124_Effelsberg/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)

		for row in burst_results:
			results.append(row)

		# compare_results.append(
		# 	measure_allmethods(
		# 		filename,
		# 		show=True,
		# 		targetDM=targetDM,
		# 		**kwargs
		# 	)
		# )

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	# comparedf = pd.DataFrame(
	# 	data=compare_results,
	# 	columns=results_columns+allmethods_columns
	# ).set_index('name')
	# comparedf.to_csv(f"201124_Effelsberg/allmethods_results_{datetime.now().strftime('%b-%d-%Y')}.csv")

	fileout = f"measurements/victor/201124_Effelsberg/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20180916B_LOFAR'].run: ## '20180916B_LOFAR__main__'
	prefix = datasets['20180916B_LOFAR'].path

	# very good data set overall

	# listnpzs(prefix)

	# Discarded:
	## 'L734093_b.npz' : {'xos' : [ 92.1], 'masks' : [59]}, #not a burst
	## 'L781499_b2.npz' : {'xos' : [ 94.8]}, #positive but low SNR, discard
	files = {
		'L734091_b.npz'  : {
			'xos' : [84.48],
			'downfactors': (2,1)
		},
		'L741963_b.npz'  : {
			'xos' : [99.66],
			'crop': (0,200),
			'masks': [range(113,121)],
			'downfactors': (2,4)
		},
		'L741965_b.npz'  : {'xos' : [84.35], 'downfactors': (4,1)},
		'L775801_b.npz'  : {'xos' : [57.23], 'masks' : [61,62,50,82,182]},
		'L775953_b1.npz' : {'xos' : [87.52], 'masks': [range(38,74)], 'downfactors': (4,1)},
		'L775953_b2.npz' : {'xos' : [79.93], 'downfactors': (4,1)}, # biased towards the bright edge
		'L775955_b1.npz' : {'xos' : [97.12], 'masks':[38,40,159,184], 'crop': (0,200), 'downfactors': (4,2)},
		'L775955_b2.npz' : {'xos' : [97.68], 'masks':[82], 'crop': (0,200), 'downfactors': (4,4)},
		'L775977_b1.npz' : {'xos' : [87.97], 'masks':[8,9], 'crop': (0,200), 'downfactors': (2,1)},
		'L775977_b2.npz' : {'xos' : [100.87],'masks':[140], 'crop': (0,200), 'downfactors': (4,2)},
		'L775979_b.npz'  : {
			'xos' : [78.66],
			'masks' : [191,183,113,115,118,194, range(106,111)],
			'crop': (0,200),
			'downfactors':(4,2)
		},
		'L781499_b1.npz' : {
			'xos' : [92.15],
			'crop': (0,200),
			'downfactors': (4,4)
		},
		'L781503_b1.npz' : {
			'xos' : [83.78],
			'masks':[199],
			'downfactors': (2,1)
		},
		'L781503_b2.npz' : {
			'xos' : [115.77],
			'crop': (0,200),
			'downfactors': (4,4)
		},
		'L781503_b3.npz' : {'xos' : [99.77]},
		'L781519_b.npz'  : {'xos' : [92.5], 'downfactors': (4,4), 'crop': (0,200)},
	}

	save = datasets['20180916B_LOFAR'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20180916B_LOFAR/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20180916B_LOFAR/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20180916B_CHIME'].run: ## '20180916B_CHIME__main__'
	prefix = datasets['20180916B_CHIME'].path
	# prefix = 'E:/Downloads/FRB/20180916B_CHIME/'

	#errors are large on a lot of these measurements bc resolution is bad

	#listnpzs(prefix)

	# Removed:
		## Very low time resolution
		# 'FRB20181222A.npz' : {
		# 	'xos' : [47.42, 55.14, 59.5, 65],
		# 	'cuts' : [52.23, 58.22, 60.04]
		# },
		# 'FRB20181226A.npz' : {
		# 	'xos' : [20.76, 24.6],
		# 	'cuts' : [22.98],
		# 	'masks':[191,196,197,214, 215]
		# },
		# 'FRB20190518A.npz' : {'xos' : [25.46], 'masks':[range(54,62)]},
		# 'FRB20190519A.npz' : {'xos' : [18.7, 24.21], 'cuts' : [22.27], 'masks':[219], 'legendloc': 4},
		## RFI
		# 'FRB20190519C.npz' : {'xos' : [23.91]},
		# 'FRB20190605A.npz' : {'xos' : [23.69] , 'masks' : [168,195,201,203,207]}, #large error
	files = {
		'FRB20180916B.npz' : {'xos' : [23.89]}, #good
		'FRB20181019A.npz' : {
			'xos' : [21.53, 86.58],
			'cuts': [57.8],
			'crop': (0,128),
			'downfactors': (8,1),
			'masks' : [125,130,132, 136,129,118,113,131, 150,154]
		},
		'FRB20181104A.npz' : {'xos' : [23.89]}, #good
		'FRB20181104B.npz' : {'xos' : [50.56], 'downfactors': (2,1)}, #very noisy, might not be a burst
		'FRB20181120B.npz' : {'xos' : [24.96], 'masks': [60,93,133]},
		'FRB20181223A.npz' : {'xos' : [48.94], 'masks' : [80,138,160], 'fix_xos' : [True]}, #good
		'FRB20181225A.npz' : {
			'xos' : [37.11],
			'masks' : [88,92,94, 187,184,181,176,255],
			'downfactors': (2,1),
			'legendloc' : 4
		},
		'FRB20190126A.npz' : {'xos' : [25.16], 'masks':[162, range(188,199)]},
		'FRB20190518E.npz' : {'xos' : [24.41],'masks':[94,201, range(177,186)]}, #okay
		'FRB20190519B.npz' : {
			'xos' : [22.78, 49.67],
			'fix_xos': True,
			'downfactors': (2,2),
			'masks' : [113,119, 135,142,154,156,167,173,238, range(38,50)],
			'cuts' : [37.47]
		},
		'FRB20190603A.npz' : {'xos' : [25.58], 'masks' : [26,91]}, #large uncert
		'FRB20190604F.npz' : {
			'xos' : [31.87 , 36.4],
			'fix_xos' : True,
			'cuts' : [34.51]
		},
		'FRB20190605B.npz' : {
			'xos' : [26.86, 37.11],
			'fix_xos':[True],
			'masks' : [79,143 ,145,154,167,151,152,196,215,226,228,254,244,241,240,236],
			'cuts' : [31.7],
			'legendloc': 3
		},
	}

	save = datasets['20180916B_CHIME'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20180916B_CHIME/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20180916B_CHIME/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20180916B_uGMRT'].run: ## '20180916B_uGMRT__main__'
	prefix = prefix = datasets['20180916B_uGMRT'].path
	# prefix = 'E:/Downloads/FRB/20180916B_uGMRT/'


	# listnpzs(prefix)
	## Removed:
		# 'burst_10.npz' : {'xos' : [37.16], 'crop':[81,135]}, # RFI and cutoff
		# 'burst_8.npz'  : {'xos' : [36.23]}, # noise
	files = {
		'burst_1.npz'  : {'xos' : [36.38], 'downfactors': (2,4)},
		'burst_2.npz'  : {'xos' : [38.17], 'masks':[153,181,185], 'downfactors': (1,2)}, #good!
		'burst_3.npz'  : {'xos' : [35.22], 'masks' : [24,146,154,150,166,137,139,121,120,115,145,116], 'downfactors': (1,4)}, #good
		'burst_4.npz'  : {
			'xos' : [31.32, 36.84, 39.19, 41.47],
			'fix_xos' : True,
			'cuts': [34.14, 38.36, 40.47],
		},
		'burst_5.npz'  : {'xos' : [37.28], 'masks':[79]}, #good
		'burst_6.npz'  : {'downfactors': (1,2), 'xos' : [36.61], 'masks':[9,205,210,224]}, #good
		'burst_7.npz'  : {'xos' : [37.19]}, #good
		'burst_11.npz' : {
			'xos' : [29.65, 33.83, 37, 46.09],
			'fix_xos': True,
			'cuts' : [32.16, 35.55, 41.34],
			'masks' : [28,16,17,14,25,18,20,15,254]
		},
		'burst_12.npz' : {'xos' : [36.93], 'masks' : [60,62,70,89,86,91,102,74,76,64,68], 'downfactors': (1,2)}, #good
		'burst_13.npz' : {
			'xos' : [36.61, 49.76],
			'masks':[178,181],
			'submasks': [[],[range(67,91)]],
			'downfactors': (1,1),
			'snr_cutoff': 3
		},
		'burst_14.npz' : {'xos' : [56.89], 'masks':[range(21,92)]},   # just noise
	}

	save = datasets['20180916B_uGMRT'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20180916B_uGMRT/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20180916B_uGMRT/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20180814A_CHIME'].run: ## '20180814A_CHIME'
	# prefix = 'E:/Downloads/FRB/20180814A_CHIME/'
	prefix = datasets['20180814A_CHIME'].path
	# a lot of positive slopes and large uncertainties

	# Removed:
		# 'FRB20180814A.npz' : { # Very low res and hard to visually confirm. Is one of 6-7 outliers
		# 	'xos' : [24.57],
		# 	'legendloc': 4,
		# 	# 'snr_cutoff':
		# },
	files = {
		'FRB20180911C.npz' : {'xos' : [37.84], 'fix_xos' : [True], 'masks':[154]},
		'FRB20180917A.npz' : {
			'xos' : [103.59, 131.93, 146.6,],
			# 'fix_xos' : True,
			'cuts' : [120.34, 139.54],
			'snr_cutoff': 5,
			'masks':[7]
		},
		'FRB20180919A.npz' : {},
		'FRB20181028A.npz' : {
			'xos' : [ 29.97, 41.42, 54.9, 72.44],
			'fix_xos' : True,
			'cuts':[35.23, 47.43, 63.3],
			'masks' :[242, 241],
			'legendloc' : 4
		},
		'FRB20181118C.npz' : {'xos' : []}, #large uncert but burst partially came masked out, might be writeoff
		'FRB20181120C.npz' : {'xos' : [60.26], 'masks' : [129,131, 137]}, #pos slope and large uncert
		'FRB20190517B.npz' : {'xos' : [48.64], 'fix_xos' : True, 'masks' : [61,47,51,45,79,53,109,55]}, #good
		'FRB20190611A.npz' : {'xos' : [37.17], 'masks' : [112, 116,124,126, 153,111,106]}, #good
		'FRB20190625E.npz' : {
			'xos' : [40.48, 48.4] ,
			'cuts' : [44.38] ,
			'fix_xos' : [True],
			'masks' : [72,66,75,83,137,165,84,120]
		}, #data set is bad
		'FRB20190626A.npz' : {'xos' : [76.7], 'fix_xos' : True, 'masks' : [213,250,244]}, #noisy, burst bandwidth is too small to make a good measurement
	}

	save = datasets['20180814A_CHIME'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20180814A_CHIME',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20180814A_CHIME/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20200929C_CHIME'].run: ## 20200929C_CHIME
	prefix = datasets['20200929C_CHIME'].path
	# prefix = 'E:/Downloads/FRB/20200929C_CHIME/'
	#listnpzs(prefix)

	## Too much RFI
		# '20200929C.npz' : {
		# 	'xos' : [39.21, 45.32, 52.62],
		# 	'cuts' : [42.01, 49.87],
		# 	'masks':[223],
		# 	'submasks':[[],[197],[]],
		# 	'legendloc': 4
		# },
		# '20201203C.npz' : {'masks':[244]}, #good
		# '20201229C.npz' : {
		# 	'xos' : [23.81, 35.52],
		# 	'fix_xos' : True,
		# 	'cuts':[31.25],
		# },
		# '20210314A.npz' : {'xos' : [25], 'masks' : [243,253,251,244]}, # good

	files = {
		'20201125B.npz' : {'masks' : [161,163,165,167,181]},
		'20210313B.npz' : {'xos' : [23], 'masks':[179,200]},#good
		'20210326B.npz' : {'xos' : [24.64], 'masks':[151,196,220]}, # good
	}

	save = datasets['20200929C_CHIME'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20200929C_CHIME/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20200929C_CHIME/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20180301_FAST'].run: ## '20180301_FAST'
	prefix = datasets['20180301_FAST'].path

	# email mohammed about this set of data.
	# need to uncomment all and process all bursts as a whole after
	# bad data set

	# listnpzs(prefix)

	# Removed:
		# 'M01_0098a.npz' : { # same waterfall as M01_0098b.npz but at a higher DM
		# 	'xos' :  [30.58, 33.96, 44.2, 45.69],
		# 	'fix_xos' :  True,
		# 	'cuts' : [32.21, 40.1, 44.92],
		# 	'masks' : [0,44,45,46,124,125,119,120],
		# 	'crop':[3428,3771],
		# 	'submasks':[[3,17,47,49],[],[],[]]
		# },
	files = {
		'FRB180301_20190716-M07_0254_sub.npz' : {
			'crop': (3850, 4700),
			'masks':[0,1,2,range(40,50), range(118,126)],
			'downfactors': (1, 10),
		},
		'FRB180301_20190716-M07_0260_sub.npz' : {
			'masks':[0,1,2,range(40,50), range(118,126)],
			'downfactors': (1,32)
		}, # can't find
		'FRB180301_20190716-M07_0409_sub.npz' : {
			'masks':[0,1,2,range(40,50), range(118,126), range(20,25)],
			'downfactors': (1,32)
		}, # can't find
		'FRB180301_20191007-M01_0520_sub.npz' : {
			'xos' : [712],
			'masks':[0,1,2, range(40,47), range(116,121), range(81,85), range(60,63)],
			'crop': (3400,3750),
			'downfactors': (1,10)
		}, # RFI on burst
		'FRB180301_20191007-M01_1003_sub.npz' : {
			'masks': [range(117, 122), range(66, 72), range(51, 55), range(43,47), range(21,24)],
			'crop': (3900, 4050)
		}, # cutoff
		'FRB180301_20191008-M01_0850_sub.npz' : {
			'crop': [2900, 2900+512],
			'masks': [range(117, 122), range(61,75), range(20,23), range(43,47), range(50,55)],
			'downfactors': (1,1), # outlies aggressively at (1,8) and 3σ
		},
		'FRB180301_20191008-M01_1509_sub.npz' : {
			'crop': (200,1200),
			'masks': [range(118,121), range(65,71), range(42,46), range(50,55)],
			'xos' : [53.77, 122.64],
			'downfactors': (1,10)
		},
		'M01_0030.npz'  : {
			'masks': [0, range(41,47), range(118,126)],
			'crop': (3475, 3475+256),
			'cuts' : [23.09, 20.79, 24.49],
			'xos' : [20.33, 21.36, 24.02, 24.94]
			# alternate
			# 'crop': (3525, 3525+128),
			# 'cuts' : [10.97, 13.26, 14.63],
			# 'xos' : [10.49, 11.68, 14.19, 15.05]
		},
		'M01_0098a.npz' : { # same waterfall as M01_0098b.npz but at a higher DM
			'xos' :  [30.58, 33.96, 44.2, 45.69],
			'fix_xos' :  True,
			'cuts' : [32.21, 40.1, 44.92],
			'masks' : [0,44,45,46,124,125,119,120],
			'crop':[3428,3771],
			'submasks':[[3,17,47,49],[],[],[]]
		},
		'M01_0098b.npz' : { # same as M01_0098a but at a lower DM
			'masks' : [0,1,44,45,46,120, range(122,126)],
			'sigmas':[3.4, 3.4 ,0.5,0.5],
			'crop': (3480,3775),
			'xos' : [20.45, 23.87, 34.45, 35.75],
			'fix_xos': True,
			'cuts' : [22.18, 31.22, 35.05]
		},
		'M01_0334.npz'  : {
			'masks' : [0,range(38,44), 1,43,44,45,46,119,120,121,122,123,124,125],
			'crop' : (3525,3525+128),
			'xos' : [12.26, 11.7],
			'cuts': [12.28],
			'legendloc' : 4
		},
		'M01_0621.npz'  : {
			'crop' : (3525,3525+128),
			'masks': [range(40,46),0,1,2,45,46,47,48,50,51,52,53,54,58,62,63,64,68,73,119,120,121,122,123,124,125],
			'sigmas':[4.35]
		},
		'M01_1463.npz'  : {
			'crop' : (3525,3525+128),
			'xos' : [11.74, 12.58, 14.61, 16.41],
			'cuts' : [12.26, 13.55, 15.62],
			'fix_xos':[True],
			'masks': [22,23,45,46,47,48,50,51,52,53,54,62,63,68,69,70,71,119,120,121,122,123,124,125]
		},
		'M01_1481.npz'  : {
			'crop' : (3525,3525+128),
			'fix_xos' : True,
			'xos'     : [11.8, 14.34],
			'cuts'    : [13.55],
			'masks'   : [119,120,123,124,125],
			'downfactors': (2,1)
		},
		'M07_0086.npz'  : {
			'crop' : (3525,3525+150),
			'downfactors': (2,2),
			'masks':[73,72,71,43,44,45,46,82,123,124,125]
		},
	}

	save = datasets['20180301_FAST'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20180301_FAST/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20180301_FAST/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20200223B_CHIME'].run: ## '20200223B_CHIME'
	prefix = datasets['20200223B_CHIME'].path
	# prefix = 'E:/Downloads/FRB/20200223B_CHIME/'
	#listnpzs(prefix)

	files = {
		'20200223B.npz' : {'xos' : [24.52], 'masks' : [91,87,80,78,91]},
		'20200702C.npz' : {'xos' : [24.54], 'masks':[19,58,117,137]}, # good
		'20200821B.npz' : {'xos' : [24.43], 'masks' : [47,51,53,58], 'downfactors': (2,1)},
		'20210115C.npz' : {'xos' : [35.89], 'masks' : [128,125,121,120,63,61,53, 54, 64,84,75,77,110,107]}, # pos slope/might be real but bad resolution
		'20210117F.npz' : {'xos' : [24.69]}, # good
	}

	save = datasets['20200223B_CHIME'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20200223B_CHIME/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20200223B_CHIME/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20190804E_CHIME'].run: ## '20190804E_CHIME'
	prefix = datasets['20190804E_CHIME'].path
	# listnpzs(prefix)


		# '20200709C.npz' : {'xos' : [24.88, 31.88, 40.66]}, # just noise
	files = {
		'20190804E.npz' : {'xos' : [25.6]}, # good but small bandwidth
		'20200629C.npz' : {'xos' : [24.57], 'masks':[166,205]}, # large uncert
		'20201201B.npz' : {'xos' : [24.87], 'masks' : [66,150,152,151,86,88,87]}, # too noisy
		'20201201B.npz' : {'xos' : [24.87]}, #noisy, large uncert
		'20201225B.npz' : {
			'xos' : [18.57,24.99],
			'cuts':[21.46],
			'masks':[179,181,251],
			'legendloc': 4
		},
		'20201228A.npz' : {
			'xos' : [73.52, 86.63],
			'cuts' : [77.47],
			'legendloc': 4
		},
	}

	save = datasets['20190804E_CHIME'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20190804E_CHIME/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20190804E_CHIME/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20121102_Arecibo_1'].run: ## '20121102_Arecibo_1'
	prefix = datasets['20121102_Arecibo_1'].path
	#listnpzs(prefix)

	arrivaltimes.logdebug = False
	# Many of these are too low snr to measure
	files = {
		'57607.53633860508.npz'  : {'xos':[31.27],'downfactors': (4,4)}, ## Set 1
		'57607.538801057606.npz' : {'downfactors': (1,2)}, # no bandwidth
		'57614.484578745796.npz' : {'xos' : [101.3], 'downfactors': (4,1)},
		'57614.48501819413.npz' : {'xos':[100.39], 'masks':[63], 'downfactors': (2,2)}, # no signal/too noisy
		'57614.48730289059.npz'  : {'xos' : [99.88], 'masks':[49], 'downfactors': (4,2)},
		'57614.49251389116.npz'  : {'xos' : [101.79], 'downfactors': (2,2)},
		'57614.493918268054.npz' : {'xos' : [101.0], 'masks' : [35,36,47]}, #pos but low SNR - remove
		'57614.494855657795.npz' : {'downfactors': (2,1)}, #burst exists but algorithm cannot find it well
		'57614.495491159156.npz' : {},
		'57614.49899176857.npz'  : {},
		'57614.50626858259.npz' : {},
		'57614.506322493544.npz' : {'xos' : [30], 'masks':[20,58,63]},
		'57614.506902116074_multi.npz' : {
			'crop': (640, 1830),
			'xos' : [44.98, 52.91, 34.08],
			'cuts' : [49.25, 42.15],
			'fix_xos':True,
			'downfactors': (4,5)
		},
		'57614.50781531692.npz' : {'xos' : [90.52], 'masks':[16]},
		'57614.50798251813.npz' : {'crop': (2432//2-150, 2432//2+150), 'downfactors': (2,2)},
		'57614.50812278726.npz' : {'xos' : [107.33], 'downfactors': (2,2)},
		'57614.50880120199.npz' : {'xos' : [100.26]}, #pos slope - questionable
		'57614.509636188.npz'   : {'crop': (2432//2-150, 2432//2+150), 'downfactors': (2,1)},
		'57614.51066904624.npz' : {'crop': (2432//2-150, 2432//2+150), 'downfactors': (2,1)},
		'57614.51325915795.npz' : {'downfactors': (4,2), 'masks' : [51,43]},
		'57614.51619408759.npz' : {'xos' : [100.53]},
		'57614.51682558449.npz' : {'xos' : [99.12], 'masks' : [34,60,61,63]},
		'57614.52087190506.npz' : {'xos' : [100.12], 'downfactors': (4,1)},
		'57614.52117559237.npz' : {'crop': (2432//2-150, 2432//2+150), 'masks' : [18,22,63,27], 'downfactors': (2,4)},
		'57614.52327954814.npz' : {'downfactors': (4,1)},
		'57614.52963179791.npz'  : {'downfactors': (2,1)}, ## Set 2
		'57614.53168857092.npz'  : {'downfactors': (2,4)},
		'57614.53218220448.npz'  : {'downfactors': (4,2)},
		'57614.532413645844.npz' : {'masks' : [51]},
		'57614.53561090259.npz'  : {'downfactors': (4,2)},
		'57614.54224020448.npz'  : {},
		'57614.54831887388.npz'  : {'crop': (2432//2-150, 2432//2+150), 'downfactors': (2,2)},
		'57614.55565660309_multi.npz' : {'downfactors': (2,1), 'masks' : [42,37,41,30,55]}, #drop it pos slope
		'57628.44767050888.npz'  : {'downfactors': (2,2)},
		'57628.45051358016.npz'  : {},
		'57628.45168153335_multi.npz' : {'downfactors': (2,2)},
		'57628.45181408471.npz'  : {'downfactors': (4,2)},
		'57628.45219099104.npz' : {}, #no/weak signal, AT found top right corner again
		'57628.45245349504.npz'  : {'downfactors': (2,2)},
		'57628.454155076084.npz' : {'downfactors': (2,1)},
		'57628.456039489785.npz' : {'xos' : [99.08]},
		'57628.45628500656.npz'  : {'masks': [50], 'downfactors': (4,1)},
		'57628.4591500268.npz'   : {'xos' : [99.73]}, # good
		'57628.45938817547_multi.npz' : {'xos' : [97.46], 'masks' : [63,51]}, #drop it - pos slope but limited AT
		'57628.459645745694.npz' : {'masks': [37]},
		'57628.46025654371.npz'  : {'masks':[38], 'downfactors': (2,1)},
		'57628.46039877799.npz'  : {'xos' : [102.38], 'downfactors': (2,1)},
		'57628.46046892633.npz'  : {'xos' : [102.03], 'masks':[44]}, #pos but large uncert
		'57628.46136870898_multi.npz' : {'xos' : [102.77], 'masks' : [55,61,58]},
		'57628.46324827085_multi.npz' : {'downfactors': (4,8)},
		'57628.463982619476.npz' : {'masks' : [51,53]}, ## Set 3
		'57628.4658375361.npz'   : {'masks':[45]},
		'57628.4659859905.npz'   : {'downfactors': (4,8)},
		'57628.466287592964.npz' : {'downfactors': (4,4)},
		'57628.467112906845.npz' : {'downfactors': (4,16)}, # decimated
		'57628.46885696472.npz'  : {'masks' : [46,51,56]},
		'57628.46890062069_multi.npz' : {'downfactors': (4,4)},
		'57628.46939640807.npz'  : {'downfactors': (4,4)},
		'57628.472406218.npz'    : {'masks':[59]},
		'57628.4727236359.npz'   : {'downfactors': (4,2)}, # only two arrival times
		'57628.47275568702.npz'  : {'xos' : [99.77], 'masks' : [45,46,63]},
		'57628.47423159253.npz'  : {'masks':[44,45,47,48]}, #large uncert
		'57628.477259733605.npz' : {'masks':[62,63], 'downfactors': (4,8)}, # decimated
		'57628.485452122906.npz' : {
			'masks':[44],
		},
		'57638.45356902133.npz'  : {'xos' : [115.49], 'masks':[13,14]},
		'57638.4556471342.npz'   : {}, #gaussian doesnt fit well, sigmas hasnt helped me
		'57638.457865365875_multi.npz' : {'downfactors': (2,2)},
		'57638.458342336904.npz' : {'masks':[range(19,35),40]},
		'57638.458683219396.npz' : {'downfactors': (4,2)},
		'57638.45938900987.npz'  : {'downfactors': (4,4)},
		'57638.460835678445.npz' : {'masks' : [13,28]},
		'57638.461242350706.npz' : {'xos' : [107.6], 'downfactors': (2,4)}, # only two points
		'57638.463325785335.npz' : {'xos': [32.43], 'downfactors': (2,2)},
		'57638.46381729593.npz'  : {'xos' : [17.52], 'masks' : [48,49,50,59,58], 'crop':[149,272], 'sigmas':[4.5]}, ## Set 4
		'57638.46659798707.npz'  : {'downfactors': (4,8)}, # two points
		'57638.46848533692.npz'  : {'downfactors': (4,8)},
		'57638.46940560556.npz'  : {'downfactors': (4,4)},
		'57638.46984702841.npz'  : {'downfactors': (4,2)},
		'57638.47173135766.npz'  : {'masks' : [14,16,51,50,32], 'downfactors': (4,8)},
		'57638.472320479865.npz' : {'xos' : [102.6], 'masks' : [48,59,58,56,51,48]},
		'57638.472988924914.npz' : {'xos' : [101.9], 'masks':[62], 'downfactors': (4,8)},
		'57638.4735660094.npz'   : {'xos' : [108.84], 'masks':[35]},
		'57638.4739438971.npz'   : {'xos' : [101.7]},
		'57638.47484585113.npz'  : {'xos' : [103.16], 'downfactors': (2,8)},
		'57638.4755446269.npz'   : {'xos' : [101.64], 'downfactors': (2,8)},
		'57638.479543266534.npz' : {},
		'57638.48076364036.npz'  : {'xos' : [102.36],},
		'57638.48144358999.npz'  : {'xos' : [104.35]},
		'57638.481457729176_multi.npz' : {'downfactors': (4,8)},
		'57638.48336581681.npz'  : {},
		'57638.48597169759.npz'  : {'xos' : [103.41]},
		'57638.486501529616.npz' : {
			'crop': (2432-1536,2432),
			'xos' : [26.07,95.46],
			'downfactors': (2,16)
		}, # ignoring first component, a little cheesy
		'57638.486502332715.npz' : {
			'crop': None,
			'xos' : [101.8, 32.17],
			'downfactors': (2,16),
			'legendloc': 3
		}, # almost decimated, let comp a get filtered out
		'57638.49045925895.npz'  : {'xos' : [100.56], 'downfactors': (4,8)},
		'57638.490931637_multi.npz' : {
			'crop': None,
			'xos' : [101.14, 120.1, 134.04, 155.87],
			'cuts' : [111, 128.94, 148.08],
			'masks':[39],
			'sigmas':[6.8, 5, 1, 1],
			'downfactors': (2,8),
			'submasks': [[],[],[],[range(18,33)]],
			'legendloc': 3
		}, # Drop component d
		'57640.41299751431.npz' : {'xos' : [101.69], 'downfactors': (4,2)},
		'57640.41603291378.npz' : {'xos' : [100.02], 'masks' : [59,51], 'downfactors': (2,2)},
		'57640.4167902918.npz'  : {'xos' : [97.17]},
		'57640.41689405917.npz'  : {'downfactors': (4,4)}, ## Set 5
		'57640.417060147985.npz' : {'masks' : [53,47,46,40]},
		'57640.41756883269.npz'  : {'crop': (897,1574)},
		'57640.41818032859.npz'  : {'xos' : [102.77]},
		'57640.419506019694.npz' : {'xos' : [100.99], 'masks':[38,39]},
		'57640.4212807308.npz'   : {'xos' : [100.83]},
		'57641.43834655583.npz'  : {'xos' : [101.54]},
		'57641.44139532731.npz'  : {'xos' : [103.34]},
		'57641.44162941246.npz'  : {
			'xos' : [100.88],
			'masks': [range(47,51), 57, 58, 56],
			'fix_xos' : [True]
		}, #may be a blend
		'57641.4431976958.npz'   : {'xos' : [103.34], 'downfactors': (2,2)},
		'57641.444802754624.npz' : {'masks' : [19], 'downfactors': (2,1)},
		'57641.445077359516.npz' : {'xos' : [102.22], 'downfactors': (4,4)},
		'57641.446497151774.npz' : {'downfactors': (4,4)},
		'57641.44912147685_multi.npz' : {'xos' : [100.84, 137.13], 'masks' : [51,61,47]},
		'57641.44912264844.npz'  : {'crop': None},
		'57641.45056532508.npz'  : {}, # low snr
		'57641.451054803685.npz' : {'downfactors': (4,8)},
		'57641.451174836424.npz' : {},
		'57641.45492924862.npz'  : {},
		'57641.45522199441.npz' : {}, #no burst just noise
		'57641.45532648767.npz'  : {},
		'57641.45684490249.npz'  : {},
		'57641.458226463474.npz' : {},
		'57641.45888738812.npz'  : {},
		'57641.459208207176.npz' : {},
		'57641.461578441355.npz' : { 'xos' : [100.42]}, ## Set 6 # no burst
		'57641.464701394376.npz' : {'masks' : [33,63]},
		'57641.46672554347.npz'  : {'xos' : [100.28], 'masks' : [52]},
		'57641.4683948626.npz'   : {'xos' : [100.73], 'masks' : [45,46,33]},
		'57641.47117746884.npz'  : {'xos' : [102.55]},
		'57641.47137642756.npz'  : {'xos' : [99.53], 'masks':[30,31]},
		'57641.47607449993.npz'  : {'downfactors': (1,8)},
		'57641.4777171409.npz'   : {'downfactors': (2,8),'masks':[13]},
		'57641.48156199899.npz'  : {'xos' : [97.33], 'masks':[27]},
		'57641.48280269835.npz'  : {'masks' : [49,62,63], 'downfactors': (1,2)}, # cutoff
		'57641.48332707437.npz'  : {'downfactors': (2, 4)},
		'57642.454478619984.npz' : {'xos' : [101.6]},
		'57642.4546280749.npz'   : {'crop': (2432//2-150, 2432//2+150)},
		'57642.455112897405.npz' : {
			'crop': (2432//2-150, 2432//2+150),
			'xos': [10.95, 12.33, 13.45],
			'cuts' : [11.54, 12.86],
			'downfactors': (2,1),
			'masks' : [62,57]
		},
		'57642.46957790253_multi.npz' : {
			'crop': (2432//2-150, 2432//2+150),
			'xos' : [7.0, 11.14, 12.56, 13.43],
			'cuts' : [12.21, 12.94],
			'downfactors': (1,1),
			'masks':[46,63]
		}, # d is covered, let it get filtered out
		'57642.475399913455.npz' : {},
		'57644.40891419798.npz'  : {'crop': None},
		'57644.409679758304_multi.npz' : {}, #not a burst
		'57644.410081666254.npz' : {},
		'57644.411828714954.npz' : {
			'downfactors': (1,4)
		},
		'57644.41224769985_multi.npz' : {},
		'57644.414131227386.npz' : {},
		'57644.41448249031.npz' : {'downfactors': (1,4)},
		'57644.414883123805_multi.npz' : {'masks' : [27,63,62,51,50], 'downfactors': (1,1)},
		'57644.41631880207.npz'  : { 'masks' : [50], 'downfactors': (1,1)},
		'57644.41831784125_multi.npz' : {'downfactors': (2,4)}, ## Set 7
		'57644.42051600411.npz'  : {'downfactors': (2,4)},
		'57644.423024306336.npz' : {'masks' : [33,34,44,45]},
		'57644.423810235174.npz' : {'masks':[31]},
		'57644.42860139265.npz'  : {'masks':[29,47]},
		'57644.43017666884.npz'  : {
			'crop': (830,2360),
			'xos': [31.71, 93.92],
			'downfactors': (1,5)
		},
		'57644.430179356954.npz' : {
			'crop': None,
			'masks':[46],
			'downfactors': (2,8),
			'xos' : [38.15, 100.66, 147.5]
		},
		'57644.43017989982.npz'  : {
			'crop': (0,2048),
			'xos' : [100.22, 52.85],
			'masks':[35],
			'downfactors': (2,8)
		},
		'57644.43129956068.npz'  : {'xos' : [102.23]},
		'57644.432247887045.npz' : {
			'xos' : [99.87],
			'masks' : [51,59],
			'downfactors': (2,8)
		},
		'57644.434927011636.npz' : {
			'xos' : [100.62],
			'masks':[18,24,26]
		},
		'57644.43637373566.npz'  : {
			'xos' : [98.6],
			'masks' : [28,45,46,49]
		},
		'57644.43687737929.npz' : {'xos' : [99.08], 'masks':[61]},
		'57644.43880371059.npz'  : {'xos' : [103.4], 'masks':[24], 'downfactors': (2,8)},
		'57644.43885587147.npz'  : {'masks':[49]},
		'57644.43921963195.npz'  : {'crop': None, 'xos' : [102.87],'masks':[20,35,50]},
		'57644.44069728557.npz'  : { 'xos' : [103.13]}, # but maybe just noise
		'57644.443006599766.npz' : {'downfactors': (2,8)}, # possible blend
		'57644.44360077364.npz'  : {'downfactors': (2,8)},
		'57644.445232446735.npz' : { 'xos' : [101.5], 'masks':[58]},
		'57644.44679589674.npz' : {
			'crop': (1150,1300),
			'xos' : [5.8],
			'sigmas': [1],
			'fix_xos': True,
		}, # non-flat continuum
		'57644.4475721964.npz' : {'xos' : [100.78]},
		'57644.44773450097.npz' : {'downfactors':(2,2)}, # non-flat continuum
		'57644.448765269415_multi.npz' : {
			'crop': None,
			'xos' : [67.11, 76.94,102.47],
			'fix_xos' : [True],
			'cuts':[70.84, 90.91],
			'downfactors': (2,8)
		}, #bad measurement for A,B (needed to get meeasurement on C)
		'57644.4499199187.npz' : {},
		'57644.451610529686.npz' : {'downfactors': (4,2)}, ## Set 8
		'57644.45240050198.npz'  : {'downfactors': (2,4)},
		'57644.45240452441.npz'  : {},
		'57644.45394624798.npz'  : {'masks' : [43,51,62,63]}, # pos, not deleting top point, arrival times masking error
		'57644.454482060166.npz' : {'masks':[40],'downfactors': (1,8)},
		'57644.455958056395.npz' : {  'xos' : [101.37], 'masks':[50,51]},
		'57644.457891984704.npz' : {'downfactors': (2,8)},
		'57644.46476354708_multi.npz' : {'downfactors': (2,8),'crop': None, 'xos' : [182.12 , 102.75], 'masks':[53,55,56]},
		'57644.4647678753.npz'   : {'xos' : [103.78]},
		'57644.465671823804.npz' : {},
		'57644.46623207259.npz'  : {'downfactors': (2,8)},
		'57644.46810617537.npz'  : {'downfactors': (1,4)},
		'57644.47472505804.npz'  : {'downfactors': (2,8)},
		'57644.47708667075.npz'  : {'downfactors': (1,8)},
		'57645.41109535448.npz'  : {}, #great
		'57645.41165707204.npz'  : {'downfactors': (1,8)},
		'57645.41189754545_multi.npz' : {'downfactors': (2,8)},
		'57645.41192596034.npz'  : {'crop': None, 'downfactors': (2,8)},
		'57645.41228792074_multi.npz' : {  'xos' : [99.29]},
		'57645.413651875155.npz' : {'downfactors': (2,8)},
		'57645.41446624773_multi.npz' : {'xos' : [100.98]},
		'57645.41610399539.npz'  : {'xos' : [103.19], 'masks':[13,35,36]},
		'57645.41640526472.npz'  : {}, # low SNR
		'57645.41656920486.npz'  : {'xos' : [98.81], 'masks':[30,31,32]},
		'57645.41747443372.npz'  : {'downfactors': (1,8)},
		'57645.41790601228.npz'  : {'downfactors': (2,8)},
		'57645.41901389838.npz'  : {'xos' : [101.47]}, ## Set 9
		'57645.41921720017.npz'  : {},
		'57645.42027373465.npz'  : {'downfactors': (2,8)},
		'57645.42129362376.npz'  : {},
		'57645.421299425805.npz' : {'downfactors': (2,4)},
		'57645.421618708795.npz' : {'downfactors': (2,8)},
		'57645.42185339937.npz'  : {'downfactors': (2,8)},
		'57645.421883132534.npz' : {'downfactors': (2,8)},
		'57645.42246211864.npz'  : {'downfactors': (2,8)}, #uncert
		'57645.422576167075.npz' : {},
		'57645.42302038949.npz'  : {'downfactors': (1,4)}, # top right error
		'57645.4241557079.npz'   : {'downfactors': (2,8), 'masks':[21]},
		'57645.42462875507.npz'  : {'downfactors': (2,8)},
		'57645.42680142881.npz'  : {'downfactors': (2,8)},
		'57645.426931394715.npz' : {'downfactors': (2,8)},
		'57645.42734335042_multi.npz' : {},
		'57645.42891479492.npz'  : {'downfactors': (1,8)},
		'57645.42990951587.npz'  : {'downfactors': (2,8)},
		'57645.43035480602.npz'  : {'xos' : [99.14], 'downfactors': (2,8), 'masks':[32]},
		'57645.43045329503.npz'  : {'xos' : [102.0], 'masks' : [63, 51]}, #not a burst
		'57645.43063245034_multi.npz' : {
			'xos' : [33.04, 41.66],
			'fix_xos': True,
			'cuts' : [107.0],
			'downfactors': (2,8)
		},
		'57645.43131103212.npz'  : {
			'xos' : [103.19], 'masks' : [51,50],
			'downfactors': (1,8)
		},
		'57645.43148239213.npz'  : {'downfactors': (1,4)},
		'57645.43230284946.npz'  : {'downfactors': (2,8)},
		'57645.43309727386.npz'  : {'downfactors': (2,8)},
		'57645.433642420554.npz' : {'xos' : [101.55]}, ## Set 10
		'57645.434336423714.npz' : {'xos' : [100.91], 'masks':[50]},
		'57645.43621394017.npz'  : {'xos' : [103.52]},
		'57645.43910048236.npz'  : {'xos' : [100.18], 'downfactors': (2,8)},
		'57645.440821828444.npz' : {'downfactors': (1,4)},
		'57645.4421089271.npz'   : {'masks':[31,32]},
		'57645.44263812532.npz'  : {'masks':[24,36,37,40]},
		'57645.44448538029.npz'  : {'downfactors': (2,4)},
		'57645.444928046214.npz' : {'downfactors': (2,8)},
		'57645.445449858424.npz' : {'downfactors': (1,8)}, # meh
		'57645.448813374234.npz' : {'downfactors': (1,4)},
		'57645.44989720593.npz'  : {'xos' : [98.3], 'masks':[48]},
		'57645.44999348996_multi.npz' : {
			'crop': None,
			'xos' : [102.73, 130.16],
			'cuts' : [120.79],
			'masks':[20,57],
			'downfactors': (2,8)
		}, # can maybe be improved
		'57645.45191111291.npz'  : {'masks':[16]},
		'57645.45199643116_multi.npz' : {'downfactors': (2,8), 'masks' : [29,47]}, #uncert
		'57645.453433171155.npz' : {'downfactors': (2,8)},
		'57645.453649376446.npz' : {'downfactors': (2,8)},
		'57645.45365054537.npz'  : {},
		'57645.45426749746.npz'  : {'downfactors': (2,8)},
		'57645.454500379936.npz' : {'xos' : [105.83]},
		'57645.458541565305.npz' : {'downfactors': (2,8)},
		'57645.45910953325.npz'  : {'downfactors': (2,4)},
		'57645.462114523834.npz' : {'downfactors': (2,8)},
		'57645.46405978794.npz'  : {'downfactors': (2,8), 'masks':[34]},
		'57645.464192686144.npz' : {'downfactors': (2,8)},
		'57645.46451208004.npz'  : {'downfactors': (2,8), 'masks':[36,61,62]}, ## Set 11
		'57645.465140150794.npz' : {'masks'  : [51,46,48, 56]},
		'57645.46610734056.npz'  : {},
		'57645.46638838374.npz'  : {},
		'57645.47445794096.npz'  : {'downfactors': (2,8), 'masks':[35]},
		'57646.39621407667.npz'  : {}, #Top right error
		'57646.39690719275.npz'  : {'xos' : [101.11], 'fix_xos' : [True]},
		'57646.39890150471.npz'  : {'xos' : [51.61]},
		'57646.40002766818_multi.npz' : {},
		'57646.40326934251.npz'  : {'xos' : [102.75]},
		'57646.403853712414.npz' : {},
		'57646.404755674586.npz' : {},
		'57646.41203494509.npz'  : {},
		'57646.416386908495.npz' : {},
		'57646.42031717954.npz'  : {},
		'57646.4257407876.npz'   : {'masks':[53]},
		'57646.42684986292.npz'  : {'masks' : [46,47,41]},
		'57646.42737489236.npz'  : {'masks' : [47,18]}, #pos slope but WF looks negative
		'57646.42775131678.npz'  : {'downfactors': (2,4)},
		'57646.42892965055.npz'  : {'downfactors': (2,8)},
		'57646.432152691494.npz' : {'downfactors': (2,8)},
		'57646.43251018116.npz'  : {'downfactors': (2,8)},
		'57646.43326578589.npz'  : {'downfactors': (2,8), 'masks':[34,38]},
		'57646.43559359925.npz'  : {'xos' : [100.42], 'downfactors': (2,8)},
		'57646.4376474379.npz'   : {},
		'57648.39469526673.npz'  : { ## Set 12
			'crop': None,
			'xos' : [100.51, 142.31],
			'downfactors': (2,8),
			'submasks': [[50],[40, 39, 50, 51, 53]]
		}, #B is pos, trying to include masking breaks A, Mohammed look at this one
		'57648.39469572189.npz'  : {'downfactors': (2,4), 'masks' : [63,53,55,55,56,57]},
		'57648.397981691356.npz' : {'xos' : [105.21], 'downfactors': (2,8)},
		'57648.398948127055.npz' : {},
		'57648.39896167492_multi.npz' : {
			'crop': (655, -1),
			'xos': [48.69, 81.85],
			'masks':[22,23,28],
			'downfactors': (4,8)
		},
		'57648.40014795116.npz'  : {'downfactors': (2,8), 'masks' : [63,61, 60]},
		'57648.40511332606.npz'  : {'crop': None},
		'57648.40731097323.npz'  : {'downfactors': (2,8), 'masks' : [44,46,63,51]},
		'57648.40824664603.npz'  : {'downfactors': (4,2)},
		'57648.40871828494.npz'  : {},
		'57648.409510112506.npz' : {'masks':[42,43,44,48]},
		'57648.41602501989.npz'  : {'crop': (993, 993+512)},
		'57648.41651528933.npz'  : {},
		'57648.418151267855.npz' : {'downfactors': (2,4)},
		'57648.42095371969.npz'  : {},
		'57648.425130859985.npz' : {'downfactors': (2,8)},
		'57648.42555741543.npz'  : {'downfactors': (2,8), 'masks':[34]},
		'57648.43682414412.npz'  : {'downfactors': (2,8), 'masks':[52,57,58,59]},
		'57648.437795269885.npz' : {'downfactors': (2,2)},
		'57648.44228408834.npz'  : {},
		'57648.44423835427.npz'  : {'downfactors': (2,8)},
		'57648.44620248289.npz'  : {'downfactors': (2,4), 'masks':[44,45,46]}, #low snr
		'57648.45088133808.npz'  : {'masks':[32], 'downfactors': (2,4)},
		'57648.45185223238.npz'  : {'xos' : [104.1], 'downfactors': (2,8)},
		'57649.4012280106.npz'   : {},
		'57649.40208810527.npz'  : {},
		'57649.40556812261.npz'  : {'downfactors': (2,8), 'xos' : [99.66]}, ## Set 13 #A lot of channels masked out in this sample
		'57649.4060381829.npz'   : {'downfactors': (2,2)},
		'57649.40771095569.npz'  : {'downfactors': (2,8)},
		'57649.41733844846.npz'  : {'downfactors': (2,4), 'masks':[34]}, #heavily masked
		'57649.41868545439.npz'  : {'downfactors': (2,8), 'masks':[14,18]}, #uncert
		'57649.418874180716.npz' : {'downfactors': (2,1), 'xos' : [99.57], 'masks' : [63]}, #extremely masked
		'57649.42165401114.npz'  : {'masks' : [40,14]}, #drop completely
		'57649.42165437655.npz'  : {'xos' : [111.44]},
		'57649.42777693603.npz'  : {'xos' : [99.67]},
		'57649.428169933824.npz' : {'xos' : [100.68]},
		'57649.4287729421.npz'   : {'masks' : [33,40]},
		'57649.42957877228.npz'  : {'xos' : [99.78]}, #uncert
		'57649.430031719865.npz' : {'downfactors': (2,1)}, # too much RFI
		'57649.43447209061.npz'  : {'downfactors': (2,8), 'xos' : [104.96]},
		'57649.43539490939.npz'  : {
			'xos': [102.92],
			'masks': [32,33],
			'downfactors': (2,8)
		},
		'57649.43619098549.npz'  : {'crop':None, 'downfactors': (2,8)}, # low snr
		'57649.44316218098.npz'  : {'downfactors': (1,2)},
		'57649.44331530851.npz'  : {'downfactors': (2,8)},
		'57649.453910864206.npz' : {'downfactors': (2,8)},
		'57649.454126644705.npz' : {},
		'57649.456815125624.npz' : {'downfactors': (2,8)}, # iffy
		'57649.45848196302.npz'  : {'downfactors': (2,8)}, #uncert
		'57649.46056619985_multi.npz' : {'downfactors': (2,2)},
		'57666.348165733056.npz' : {'downfactors': (2,4),},
		'57666.34893850425.npz'  : {},
		'57666.350467149954.npz' : {'downfactors': (2,8)}, ## Set 14
		'57666.350914006485.npz' : {'xos' : [104.73]},
		'57666.35352999023.npz'  : {'downfactors': (2,8), 'masks' : [40,60,55,56,57,49,61,59]},
		'57666.35544198991.npz'  : {'crop': None, 'masks' : [54,48,17]},
		'57666.35891707423.npz'  : {}, #top right error
		'57666.35918939836.npz'  : {'downfactors': (2,4)},
		'57666.360757358336.npz' : {'crop': None,'downfactors': (2,8)},
		'57666.36265378668.npz'  : {'downfactors': (2,8), 'masks':[19,29]},
		'57666.36380141672.npz'  : {'downfactors': (1,8)},
		'57666.36790323902.npz'  : {'downfactors': (2,8), 'masks' : [55,51]},
		'57666.36911246032.npz'  : {'downfactors': (2,8), 'masks' : [63,60,51,43,39]},
		'57666.369922217375.npz' : {'downfactors': (2,8), 'masks':[47,48,49]},
		'57666.375009891046.npz' : {'xos' : [102.4], 'downfactors': (2,4)},
		'57666.376544430364.npz' : {'downfactors': (2,8), 'masks':[26,42,43]},
		'57666.377627586204.npz' : {'downfactors': (2,1)}, # can get a measurement with (2,8) but is low snr and cutoff so i'd rather leave out
		'57666.37912666425_multi.npz' : {}, #noise/top right error
		'57666.38202523098.npz'  : {'masks':[39,40,49]}, #uncert
		'57666.3831163329.npz'   : {'downfactors': (2,8), 'masks' : [61,51,50]},
		'57666.38437438991.npz'  : {'downfactors': (2,8)},
		'57666.388477979366.npz' : {'xos' : [105.22], 'masks' : [53]},
		'57666.38928066704.npz'  : {'downfactors': (2,8)},
		'57666.389763840714.npz' : {'downfactors': (2,8)},
		'57666.38992917074.npz'  : {'downfactors': (2,8)},
		'57666.39400495943.npz'  : {'downfactors': (1,8)},
		'57666.400868763456.npz' : {'downfactors': (4,8)}, ## Set 15
		'57666.40086931443.npz'  : {'crop': None, 'xos' : [53.08, 100.28], 'downfactors': (2,16)},
		'57666.40175640415.npz'  : {'downfactors': (2,8)},
		'57666.40303498249.npz'  : {'downfactors': (2,4)},
		'57666.40350757538_multi.npz' : {}, #noise
		'57666.40436234341.npz'  : {'downfactors': (2,8), 'masks':[57,58]},
		'57666.40571495674.npz'  : {'downfactors': (2,8)},
		'57666.40572689542.npz'  : {'downfactors': (2,4)},
		'57666.40748229513.npz'  : {'downfactors': (2,8)},
		'57666.41159127091.npz'  : {'downfactors': (2,8)}, #low SNR, may discard at your discretion
		'57666.41200345347.npz'  : {'downfactors': (2,8)},
		'57666.41774746808.npz'  : {'downfactors': (2,8), 'masks':[42,46,57,58,59,60]},
	}

	save = datasets['20121102_Arecibo_1'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'

		# Burst-basis cropping
		crop = (2432//2-384, 2432//2+384) # 768 width
		if 'crop' in kwargs:
			crop = kwargs['crop']
			del kwargs['crop']

		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20121102_Arecibo_1/',
			show=not save,
			crop=crop,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20121102_Arecibo_1/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20201130A_CHIME_'].run: ## '20201130A_CHIME_'
	prefix = datasets['20201130A_CHIME_'].path
	# prefix = 'E:/Downloads/FRB/20201130A_CHIME/'

	#some positive bursts but low SNR ratios/bad resolution

	#listnpzs(prefix)

	files = {
		'20201130A.npz' : {'xos' : [24.37], 'masks' : [162,129,26,44,55] },
		'20201225D.npz' : {'xos' : [37.4], 'masks' : [range(60,63),196,197,203]},
		'20210105G.npz' : {'xos' : [23.21]}, # good
		'20210114B.npz' : {'xos' : [26.21]},
		'20210117E.npz' : {'xos' : [37.89], 'fix_xos' : [True]},
		'20210118B.npz' : {'xos' : [37.21]}, # good
		'20210205D.npz' : {'xos' : [37.63], 'fix_xos' : [True], 'masks':[223,243]}, # good
		'20210220E.npz' : {'xos' : [9.77]}, # not a burst
		'20210228B.npz' : {'xos' : [48.41]}, # very noisy
		'20210323E.npz' : {'xos' : [49.98], 'masks':[179]}, # pos slope but very noisy, discard
		'20210327F.npz' : {'downfactors': (2,1), 'xos' : [35.79], 'masks':[218]},
		'20210406A.npz' : {'xos' : [34.89], 'fix_xos' : [True], 'sigmas':[7.41]},
	}

	save = datasets['20201130A_CHIME_'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20201130A_CHIME/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20201130A_CHIME/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
########################################

if datasets['20190915D_CHIME'].run: ## '20190915D_CHIME'
	prefix = datasets['20190915D_CHIME'].path
	# prefix = 'E:/Downloads/FRB/20190915D_CHIME/'

	#listnpzs(prefix)

	files = {
		'20190915D.npz' : {
			'xos' : [57.97],
			'masks':[6,227,239],
			'crop': (0,94),
			'downfactors': (2,2)
		},
		'20200213C.npz' : {'xos' : [61.23], 'masks' : [245,248,247]}, # just noise
		'20200214A.npz' : {
			'xos' : [44.11, 121.99],
			'masks':[207,205,204,203,200,154,138,125,111],
			'downfactors': (4,1)
		},
		'20200214B.npz' : {'downfactors': (4,1), 'xos' : [53.86]},
		'20200312A.npz' : {'xos' : [35.55], 'masks' : [204,202,197]}, # very noisy
		'20200313B.npz' : {
			'downfactors': (4,1),
			'xos' : [86.47, 52.49],
			'masks' : [189,180,186, 124,142,132,120,121]
		},
		'20200313C.npz' : {
			'xos' : [42.37, 73.11],
			'masks':[239,247],
			'downfactors': (2,1)
		},
	}

	save = datasets['20190915D_CHIME'].save
	results = []
	for filename, kwargs in files.items():
		filename = f'{prefix}{filename}'
		burst_results = arrivaltimes.measureburst(
			filename,
			outdir='measurements/victor/20190915D_CHIME/',
			show=not save,
			show_components=False,
			save=save,
			**kwargs
		)
		for row in burst_results:
			results.append(row)

	resultsdf = pd.DataFrame(data=results, columns=arrivaltimes.results_columns).set_index('name')

	fileout = f"measurements/victor/20190915D_CHIME/results_{datetime.now().strftime('%b-%d-%Y')}.csv"
	if save:
		resultsdf.to_csv(fileout)
		print(f"Saved {fileout}.")

########################################
