# CHANfiG, Easier Configuration.
# Copyright (c) 2022-Present, CHANfiG Contributors
# This program is free software: you can redistribute it and/or modify
# it under the terms of the following licenses:
# - The Unlicense
# - GNU Affero General Public License v3.0 or later
# - GNU General Public License v2.0 or later
# - BSD 4-Clause "Original" or "Old" License
# - MIT License
# - Apache License 2.0
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the LICENSE file for more details.

from __future__ import annotations

from argparse import ArgumentParser
from copy import copy, deepcopy
from typing import Dict, List, Optional, Tuple, Union

from pytest import raises

from chanfig import FlatDict, Variable


class Test:
    dict = FlatDict()
    dict[1] = 2
    dict[3] = 4

    def test_dict(self):
        assert self.dict == FlatDict({1: 2, 3: 4})

    def test_list(self):
        assert self.dict == FlatDict([(1, 2), (3, 4)])

    def test_args(self):
        dict = FlatDict([("1", 2), ("3", 4)])
        assert dict["1"] == 2
        assert dict["3"] == 4

    def test_kwargs(self):
        dict = FlatDict(**{"1": 2, "3": 4})
        assert dict["1"] == 2
        assert dict["3"] == 4

    def test_copy(self):
        assert copy(self.dict) == self.dict.copy()
        assert deepcopy(self.dict) == self.dict.deepcopy()


class ConfigDict(FlatDict):
    int_value: int
    str_value: str
    float_value: float
    list_int: List[int]
    tuple_str: Tuple[str]
    dict_float: Dict[str, float]
    int_float: Union[int, float]
    optional_str: Optional[str]

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.a = FlatDict()
        self.b = FlatDict({"a": self.a})
        self.c = Variable(FlatDict({"a": self.a}))
        self.d = FlatDict(a=self.a)


class TestConfigDict:
    dict = ConfigDict()

    def test_affinty(self):
        assert id(self.dict.a) == id(self.dict.b.a) == id(self.dict.c.a) == id(self.dict.d.a)

    def test_validate(self):
        ConfigDict(int_value=1, str_value="1", float_value=1.0)
        with raises(TypeError):
            ConfigDict(int_value="1", str_value="1", float_value=1.0)
        with raises(TypeError):
            self.dict.int_value = "1"
            self.dict.validate()
        ConfigDict(list_int=[1, 2, 3])
        with raises(TypeError):
            ConfigDict(list_int=[1, "2", 3])
        ConfigDict(tuple_str=("1", "2", "3"))
        with raises(TypeError):
            ConfigDict(tuple_str=["1", "2", 3])
        ConfigDict(dict_float={"1": 1.0, "2": 2.0, "3": 3.0})
        with raises(TypeError):
            ConfigDict(dict_float={"1": 1.0, "2": 2.0, "3": "3.0"})
        ConfigDict(int_float=1)
        ConfigDict(int_float=0.5)
        with raises(TypeError):
            ConfigDict(int_float="inf")
        ConfigDict(optional_str="1")
        ConfigDict(optional_str=None)
        with raises(TypeError):
            ConfigDict(optional_str=1)

    def test_construct_file(self):
        d = FlatDict("tests/test.json")
        assert d == FlatDict({"a": 1, "b": 2, "c": 3})

    def test_construct_namespace(self):
        parser = ArgumentParser()
        parser.add_argument("--name", type=str)
        parser.add_argument("--seed", type=int)
        d = FlatDict(parser.parse_args(["--name", "chang", "--seed", "1013"]))
        assert d.name == "chang"
        assert d.seed == 1013

    def test_conflicts(self):
        d = FlatDict(keys=0, values=1, items=2)
        p = {"keys": 0, "values": 1, "items": 2}
        assert d["keys"] == 0
        assert d["values"] == 1
        assert d["items"] == 2
        assert d.keys() == p.keys()
        assert list(d.values()) == list(p.values())  # dict_values can't be compared directly
        assert d.items() == p.items()
