﻿#include <iostream>
#include <fstream>
#include <direct.h>

#include <opencv2\opencv.hpp>

using namespace cv;


int vector_finder(std::vector<int> vec, int number) {
	auto itr = std::find(vec.begin(), vec.end(), number);
	size_t index = std::distance(vec.begin(), itr);

	return (int)index;
	//If not found, use index = vec.size()
}

/*generate cell mask*/
void refine_size(const Mat& src, Mat& dst, const Mat& fluo, const int CELL_MIN, const int CELL_MAX) {
	dst = src.clone();

	Mat LabelImg(src.size(), CV_32S);
	Mat stats;
	Mat centroids;
	int n = connectedComponentsWithStats(src, LabelImg, stats, centroids, 4);

	for (int i = 1; i < n; i++) {
		int* param = stats.ptr<int>(i);
		int left = param[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
		int top = param[cv::ConnectedComponentsTypes::CC_STAT_TOP];
		int height = param[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];
		int width = param[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];

		if (width > CELL_MAX || height > CELL_MAX) {
			for (int x = left; x < left + width; x++) {
				for (int y = top; y < top + height; y++) {
					if (LabelImg.at<int>(y, x) == i) {
						dst.at<unsigned char>(y, x) = 0;
					}
				}
			}
		}
		else if (std::min(width, height) < CELL_MIN) {
			// Keep small bright areas
			int brt_count = 0;
			int count = 0;
			for (int x = left; x < left + width; x++) {
				for (int y = top; y < top + height; y++) {
					if (LabelImg.at<int>(y, x) == i) {
						if (fluo.at<unsigned char>(y, x) >= 200) {
							brt_count++;
						}
						count++;
					}
				}
			}

			if (std::min(width, height) < 8 || brt_count < (double)count * 0.6) {
				for (int x = left; x < left + width; x++) {
					for (int y = top; y < top + height; y++) {
						if (LabelImg.at<int>(y, x) == i) {
							dst.at<unsigned char>(y, x) = 0;
						}
					}
				}
			}
		}
	}
}

void gen_gfp_mask(const Mat& gfp_img, const int GFP_THRE_CELL, Mat& gfp_mask, const int GFP_N_MASK_GEN, const int GFP_CELL_MIN0, const int GFP_CELL_MIN,
	const int FIRST_FRAME, const int t) {
	Mat binary;
	threshold(gfp_img, binary, GFP_THRE_CELL, 255, cv::THRESH_BINARY);

	for (int i = 0; i < GFP_N_MASK_GEN; i++) {
		erode(binary, binary, Mat());
	}
	for (int i = 0; i < GFP_N_MASK_GEN; i++) {
		dilate(binary, binary, Mat());
	}

	gfp_mask = binary.clone();

	if (t == FIRST_FRAME) {
		refine_size(binary, gfp_mask, gfp_img, GFP_CELL_MIN0, std::max(gfp_img.cols, gfp_img.rows));
	}
	refine_size(gfp_mask, gfp_mask, gfp_img, GFP_CELL_MIN, std::max(gfp_img.cols, gfp_img.rows));
}

void gen_inclusion_mask(const Mat& img, const Mat& gfp_mask, Mat& inc, const int THRE_INC) {
	//Detection of intracellular aggregates

	inc = Mat(img.size(), CV_8U, Scalar::all(0));

	Mat LabelImg(img.size(), CV_32S);
	Mat stats;
	Mat centroids;
	int n = connectedComponentsWithStats(gfp_mask, LabelImg, stats, centroids, 4);

	for (int i = 1; i < n; i++) {
		int* param = stats.ptr<int>(i);
		int left = param[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
		int top = param[cv::ConnectedComponentsTypes::CC_STAT_TOP];
		int height = param[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];
		int width = param[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];

		Mat sub_img(img, Rect(left, top, width, height));
		Mat sub_mask(gfp_mask, Rect(left, top, width, height));
		Scalar mean, stddev;
		meanStdDev(sub_img, mean, stddev, sub_mask);

		for (int x = left; x < left + width; x++) {
			for (int y = top; y < top + height; y++) {
				if ((LabelImg.at<int>(y, x) == i) && (img.at<unsigned char>(y, x) > THRE_INC)) {
					inc.at<unsigned char>(y, x) = 255;
				}
			}
		}
	}
}

void gen_rfp_mask(const Mat& rfp_img, const int RFP_THRE_CELL, const bool isFirstFrame, const int RFP_N_MASK_GEN0, const int RFP_N_MASK_GEN, Mat& rfp_mask) {
	Mat binary;
	threshold(rfp_img, binary, RFP_THRE_CELL, 255, THRESH_BINARY);

	if (isFirstFrame) {
		for (int i = 0; i < RFP_N_MASK_GEN0; i++) {
			erode(binary, binary, Mat());
		}
		for (int i = 0; i < RFP_N_MASK_GEN0; i++) {
			dilate(binary, binary, Mat());
		}
	}
	else {
		for (int i = 0; i < RFP_N_MASK_GEN; i++) {
			erode(binary, binary, Mat());
		}
		for (int i = 0; i < RFP_N_MASK_GEN; i++) {
			dilate(binary, binary, Mat());
		}
	}
	rfp_mask = binary.clone();
}

/*tracking*/
void get_area_center(const Mat& cell, std::vector<std::vector<int>>& centers) {
	Mat LabelImg(cell.size(), CV_32S);
	Mat stats;
	Mat centroids;
	int n = connectedComponentsWithStats(cell, LabelImg, stats, centroids, 4);

	centers = std::vector<std::vector<int>>(n - 1, std::vector<int>(0));

	for (int i = 1; i < n; i++) {
		double* param = centroids.ptr<double>(i);
		int center_x = (int)param[0];
		int center_y = (int)param[1];
		centers[i - 1] = { center_x, center_y };

		int* param2 = stats.ptr<int>(i);
		int left = param2[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
		int top = param2[cv::ConnectedComponentsTypes::CC_STAT_TOP];
		int width = param2[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];
		int height = param2[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];

		if (cell.at<unsigned char>(center_y, center_x) == 0) {
			int s = -1, t = -1;
			double min_dist = cell.cols + cell.rows;

			for (int x = left; x < left + width; x++) {
				for (int y = top; y < top + height; y++) {
					if (LabelImg.at<int>(y, x) == i && pow(center_x - x, 2) + pow(center_y - y, 2) < pow(min_dist, 2)) {
						min_dist = sqrt(pow(center_x - x, 2) + pow(center_y - y, 2));
						s = x;
						t = y;
					}
				}
			}
			centers[i - 1] = { s, t };
		}
	}
	return;
}

void match_cell_areas(const Mat& cell0, const Mat& cell1, std::vector<int>& result_x) {
	Mat LabelImg0(cell0.size(), CV_32S);
	Mat stats0;
	Mat centroids0;
	Mat LabelImg1(cell1.size(), CV_32S);
	Mat stats1;
	Mat centroids1;
	int n0 = connectedComponentsWithStats(cell0, LabelImg0, stats0, centroids0, 4);
	connectedComponentsWithStats(cell1, LabelImg1, stats1, centroids1, 4);

	result_x = std::vector<int>(n0 - 1, -1);

	const int SEARCH_SIZE = 10;

	for (int i = 1; i < n0; i++) {
		int* param = stats0.ptr<int>(i);
		int left = param[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
		int top = param[cv::ConnectedComponentsTypes::CC_STAT_TOP];
		int height = param[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];
		int width = param[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];

		double* param_center = centroids0.ptr<double>(i);
		int center_x = (int)param_center[0];
		int center_y = (int)param_center[1];

		std::vector<int> areanum;//Region number within the search area
		std::vector<double> min_dist_set;

		//Find the label closest to (center_x, center_y) among other regions in the search area
		for (int x = std::max(0, left - SEARCH_SIZE); x <= std::min(cell0.cols - 1, left + width + SEARCH_SIZE); x++) {
			for (int y = std::max(0, top - SEARCH_SIZE); y <= std::min(cell0.rows - 1, top + height + SEARCH_SIZE); y++) {
				if (cell1.at<unsigned char>(y, x) > 0) {
					int areanum1 = LabelImg1.at<int>(y, x);
					int ind = vector_finder(areanum, areanum1);

					if (ind == areanum.size()) {
						areanum.push_back(areanum1);
						min_dist_set.push_back(sqrt(pow(x - center_x, 2) + pow(y - center_y, 2)));
					}
					min_dist_set[ind] = std::min(min_dist_set[ind], sqrt(pow(x - center_x, 2) + pow(y - center_y, 2)));
				}
			}
		}

		//Decide on a destination
		int areanum1 = -1;
		double min_dist = cell0.cols + cell0.rows;

		for (int k = 0; k < areanum.size(); k++) {
			if (min_dist_set[k] < min_dist) {
				areanum1 = areanum[k];
				min_dist = min_dist_set[k];
			}
		}
		result_x[i - 1] = areanum1;
	}
}

void gen_result_gfp(const Mat& cell0, const Mat& cell1, const std::vector<int>& result_x,
	const std::vector<std::vector<int>>& result_data0, std::vector<std::vector<int>>& result_data1) {

	//Correspondence between area numbers and centroid coordinates
	Mat LabelImg0(cell0.size(), CV_32SC1);
	Mat stats0;
	Mat centroids0;
	Mat LabelImg1(cell1.size(), CV_32SC1);
	Mat stats1;
	Mat centroids1;
	connectedComponentsWithStats(cell0, LabelImg0, stats0, centroids0, 4);
	int n1 = connectedComponentsWithStats(cell1, LabelImg1, stats1, centroids1, 4);

	result_data1 = std::vector<std::vector<int>>(result_data0.size(), std::vector<int>(1, -1));

	for (int i = 0; i < result_data0.size(); i++) {
		if (result_data0[i].size() == 2) {
			//From coordinates, get region number t0
			int areanum0 = LabelImg0.at<int>(result_data0[i][1], result_data0[i][0]);

			//Get region number t1 from region number t0
			int areanum1 = -1;
			if (areanum0 > 0) {
				areanum1 = result_x[areanum0 - 1];
			}

			//Get coordinates from region number t1
			if (areanum1 > 0) {
				double* param = centroids1.ptr<double>(areanum1);
				int center_x = (int)param[0];
				int center_y = (int)param[1];
				result_data1[i] = { center_x, center_y };

				int* param2 = stats1.ptr<int>(areanum1);
				int left = param2[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
				int top = param2[cv::ConnectedComponentsTypes::CC_STAT_TOP];
				int width = param2[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];
				int height = param2[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];

				if (cell1.at<unsigned char>(center_y, center_x) == 0) {
					int s = -1, t = -1;
					double min_dist = cell1.cols + cell1.rows;

					for (int x = left; x < left + width; x++) {
						for (int y = top; y < top + height; y++) {
							if (LabelImg1.at<int>(y, x) == areanum1 && pow(center_x - x, 2) + pow(center_y - y, 2) < pow(min_dist, 2)) {
								min_dist = sqrt(pow(center_x - x, 2) + pow(center_y - y, 2));
								s = x;
								t = y;
							}
						}
					}
					result_data1[i] = { s, t };
				}
			}
			else {
				result_data1[i] = { -1 };
			}
		}
	}
	//New Appearance
	for (int i = 1; i < n1; i++) {
		if (vector_finder(result_x, i) == result_x.size()) {
			double* param = centroids1.ptr<double>(i);
			int center_x = (int)param[0];
			int center_y = (int)param[1];

			int* param2 = stats1.ptr<int>(i);
			int left = param2[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
			int top = param2[cv::ConnectedComponentsTypes::CC_STAT_TOP];
			int width = param2[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];
			int height = param2[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];

			if (cell1.at<unsigned char>(center_y, center_x) == 0) {
				int s = -1, t = -1;
				double min_dist = cell1.cols + cell1.rows;

				for (int x = left; x < left + width; x++) {
					for (int y = top; y < top + height; y++) {
						if (LabelImg1.at<int>(y, x) == i && pow(center_x - x, 2) + pow(center_y - y, 2) < pow(min_dist, 2)) {
							min_dist = sqrt(pow(center_x - x, 2) + pow(center_y - y, 2));
							s = x;
							t = y;
						}
					}
				}
				result_data1.push_back({ s, t });
			}
			else {
				result_data1.push_back({ center_x, center_y });
			}
		}
	}
}

void gen_result_rfp(const Mat& cell0, const Mat& cell1, const std::vector<int>& result_x,
	const std::vector<std::vector<int>>& result_data0, std::vector<std::vector<int>>& result_data1) {

	//Correspondence between area numbers and centroid coordinates
	Mat LabelImg0(cell0.size(), CV_32SC1);
	Mat stats0;
	Mat centroids0;
	Mat LabelImg1(cell1.size(), CV_32SC1);
	Mat stats1;
	Mat centroids1;
	connectedComponentsWithStats(cell0, LabelImg0, stats0, centroids0, 4);
	connectedComponentsWithStats(cell1, LabelImg1, stats1, centroids1, 4);

	result_data1 = std::vector<std::vector<int>>(result_data0.size(), std::vector<int>(1, -1));

	for (int i = 0; i < result_data0.size(); i++) {
		if (result_data0[i].size() == 2) {
			//Get region number t0 from coordinates
			int areanum0 = LabelImg0.at<int>(result_data0[i][1], result_data0[i][0]);

			//Get region number t1 from region number t0
			int areanum1 = -1;
			if (areanum0 > 0) {
				areanum1 = result_x[areanum0 - 1];
			}

			//Get coordinates from region number t1
			if (areanum1 > 0) {
				double* param = centroids1.ptr<double>(areanum1);
				int center_x = (int)param[0];
				int center_y = (int)param[1];
				result_data1[i] = { center_x, center_y };

				int* param2 = stats1.ptr<int>(areanum1);
				int left = param2[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
				int top = param2[cv::ConnectedComponentsTypes::CC_STAT_TOP];
				int width = param2[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];
				int height = param2[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];

				if (cell1.at<unsigned char>(center_y, center_x) == 0) {
					int s = -1, t = -1;
					double min_dist = cell1.cols + cell1.rows;

					for (int x = left; x < left + width; x++) {
						for (int y = top; y < top + height; y++) {
							if (LabelImg1.at<int>(y, x) == areanum1 && pow(center_x - x, 2) + pow(center_y - y, 2) < pow(min_dist, 2)) {
								min_dist = sqrt(pow(center_x - x, 2) + pow(center_y - y, 2));
								s = x;
								t = y;
							}
						}
					}
					result_data1[i] = { s, t };
				}
			}
			else {
				result_data1[i] = { -1 };
			}
		}
	}
}

void refine_merge(const std::vector<std::vector<std::vector<int>>>& result_data_old, std::vector<std::vector<std::vector<int>>>& result_data_new) {
	result_data_new = result_data_old;

	//When masks are merged, erase them all
	for (int t = 1; t < result_data_old.size(); t++) {
		for (int id = 0; id < result_data_old[t].size(); id++) {
			if (result_data_new[t][id].size() == 2) {
				int x0 = result_data_new[t][id][0];
				int y0 = result_data_new[t][id][1];

				for (int k = id + 1; k < result_data_old[t].size(); k++) {
					if (result_data_new[t][k].size() == 2 && result_data_new[t][k][0] == x0 && result_data_new[t][k][1] == y0) {
						std::cout << "merge t" << t << " id" << id << " id" << k << std::endl;

						for (int s = 0; s < result_data_old.size(); s++) {
							if (id < result_data_old[s].size()) {
								result_data_new[s][id] = { -1 };
							}
							if (k < result_data_old[s].size()) {
								result_data_new[s][k] = { -1 };
							}
						}

					}
				}
			}
		}
	}
}

void delete_short_track(const std::vector<std::vector<std::vector<int>>>& result_data_old, std::vector<std::vector<std::vector<int>>>& result_data_new, const int MIN_LIFETIME) {
	result_data_new = result_data_old;

	int max_id = (int)result_data_old[result_data_old.size() - 1].size();

	for (int id = 0; id < max_id; id++) {
		int count = 0;
		for (int t = 0; t < result_data_old.size(); t++) {
			if (id < result_data_old[t].size() && result_data_old[t][id].size() == 2) {
				count++;
			}
		}
		if (count > 0 && count < MIN_LIFETIME) {
			for (int t = 0; t < result_data_old.size(); t++) {
				if (id < result_data_old[t].size()) {
					result_data_new[t][id] = { -1 };
				}
			}
		}
	}
}

void delete_moving_track(const std::vector<std::vector<std::vector<int>>>& result_data_old, std::vector<std::vector<std::vector<int>>>& result_data_new, const int D_MAX) {
	std::vector<std::vector<std::vector<int>>> v_tmp = result_data_old;

	for (int t = 0; t < result_data_old.size() - 1; t++) {
		for (int id = 0; id < std::min(result_data_old[t].size(), result_data_old[t + 1].size()); id++) {
			if (v_tmp[t][id].size() == 2 && v_tmp[t + 1][id].size() == 2) {
				int x0 = v_tmp[t][id][0];
				int y0 = v_tmp[t][id][1];
				int x1 = v_tmp[t + 1][id][0];
				int y1 = v_tmp[t + 1][id][1];

				if (pow(x0 - x1, 2) + pow(y0 - y1, 2) > pow(D_MAX, 2)) {
					std::cout << "delete moving track: id" << id << " t-first_frame = " << t << std::endl;
					for (int s = 0; s < result_data_old.size(); s++) {
						if (id < result_data_old[s].size()) {
							v_tmp[s][id] = { -1 };
						}
					}
				}
			}
		}
	}
	result_data_new = v_tmp;
}

void delete_dammy(std::vector<std::vector<std::vector<int>>>& result_data_sep) {
	//Keep only those alive at the start frame
	std::vector<int> exist_id;
	int t = 0;
	for (int id = 0; id < result_data_sep[t].size(); id++) {
		if (result_data_sep[t][id].size() == 2 && vector_finder(exist_id, id) == exist_id.size()) {
			exist_id.push_back(id);
		}
	}

	sort(exist_id.begin(), exist_id.end());
	std::vector<std::vector<std::vector<int>>>v_tmp(result_data_sep.size(), std::vector<std::vector<int>>(exist_id.size(), { 0 }));

	for (t = 0; t < result_data_sep.size(); t++) {
		for (int i = 0; i < exist_id.size(); i++) {
			if (exist_id[i] < result_data_sep[t].size()) {
				v_tmp[t][i] = result_data_sep[t][exist_id[i]];
			}
		}
	}
	result_data_sep = v_tmp;
}

void refine_trace_data_gfp(const std::vector<std::vector<std::vector<int>>>& result_data_old, std::vector<std::vector<std::vector<int>>>& result_data_new, const int MIN_LIFETIME, const int D_MAX) {
	std::vector<std::vector<std::vector<int>>> v_tmp = result_data_old;

	delete_dammy(v_tmp);//Keep only those alive at the start frame

	//Merging process: Find and delete merging IDs
	refine_merge(v_tmp, v_tmp);

	//Keep only those that exist for the specified number of frames or more
	delete_short_track(v_tmp, v_tmp, MIN_LIFETIME);

	//Eliminate objects with large movement distances
	delete_moving_track(v_tmp, v_tmp, D_MAX);

	result_data_new = v_tmp;
}

void refine_trace_data_rfp(const std::vector<std::vector<std::vector<int>>>& result_data_old, std::vector<std::vector<std::vector<int>>>& result_data_new, const int MIN_LIFETIME, const int D_MAX) {

	std::vector<std::vector<std::vector<int>>> v_tmp = result_data_old;

	delete_dammy(v_tmp);//Keep only those alive at the start frame

	//Merging process: Find and delete merging IDs
	refine_merge(v_tmp, v_tmp);

	//Keep only those that exist for the specified number of frames or more
	delete_short_track(v_tmp, v_tmp, MIN_LIFETIME);

	//Eliminate objects with large movement distances
	delete_moving_track(v_tmp, v_tmp, D_MAX);

	result_data_new = v_tmp;
}

void delete_fog_id(const std::string SAMPLE_NAME, const int FIRST_FRAME, const std::string gfp_or_rfp,
	const std::vector<std::vector<std::vector<int>>>& result_data_sep, std::vector<std::vector<std::vector<int>>>& result_data_sep_new,
	const int CELL_MAX) {

	result_data_sep_new = result_data_sep;
	std::vector<std::vector<int>> height_total = std::vector<std::vector<int>>(result_data_sep.size(), std::vector<int>(result_data_sep[result_data_sep.size() - 1].size(), 0));
	std::vector<std::vector<int>> width_total = std::vector<std::vector<int>>(result_data_sep.size(), std::vector<int>(result_data_sep[result_data_sep.size() - 1].size(), 0));

	//get data
	for (int t = 0; t < result_data_sep.size(); t++) {
		Mat cell = imread(SAMPLE_NAME + "_cellmask\\" + gfp_or_rfp + "_t" + std::to_string(t + FIRST_FRAME) + ".tif", cv::IMREAD_ANYDEPTH);
		if (cell.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		Mat LabelImg(cell.size(), CV_32S);
		Mat stats;
		Mat centroids;
		connectedComponentsWithStats(cell, LabelImg, stats, centroids);

		for (int id = 0; id < result_data_sep[t].size(); id++) {
			if (result_data_sep[t][id].size() == 2) {
				int x = result_data_sep[t][id][0];
				int y = result_data_sep[t][id][1];

				if (LabelImg.at<int>(y, x) > 0) {
					int* param = stats.ptr<int>(LabelImg.at<int>(y, x));
					int height = param[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];
					int width = param[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];

					height_total[t][id] = height;
					width_total[t][id] = width;
				}
			}
		}
	}

	//If a certain size is greater than CELL MAX, delete it.
	std::vector<int> judge_area(result_data_sep[result_data_sep.size() - 1].size(), 0);
	for (int id = 0; id < judge_area.size(); id++) {
		int  count_greater_than_CELL_MAX = 0;

		for (int t = 0; t < result_data_sep.size(); t++) {
			if (height_total[t][id] > CELL_MAX || width_total[t][id] > CELL_MAX) {
				count_greater_than_CELL_MAX++;
				judge_area[id] = -1;
				break;
			}
		}
	}

	for (int id = 0; id < judge_area.size(); id++) {
		if (judge_area[id] == -1) {
			for (int t = 0; t < result_data_sep.size(); t++) {
				if (id < result_data_sep[t].size()) {
					result_data_sep_new[t][id] = { -1 };
				}
			}
		}
	}
}

int area_shrink(std::vector<std::vector<std::vector<int>>>& track_data_sep, const std::string SAMPLE_NAME, const int FIRST_FRAME, const int LAST_FRAME, const int T_VANISH_AREA) {

	std::vector<std::vector<int>> area_data(track_data_sep.size(), std::vector<int>(track_data_sep[track_data_sep.size() - 1].size(), 0));

	for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {
		Mat cell = imread(SAMPLE_NAME + "_cellmask\\rfp_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		if (cell.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		Mat LabelImg(cell.size(), CV_32S);
		Mat stats;
		Mat centroids;
		connectedComponentsWithStats(cell, LabelImg, stats, centroids);

		for (int id = 0; id < track_data_sep[t - FIRST_FRAME].size(); id++) {
			if (track_data_sep[t - FIRST_FRAME][id].size() == 2) {
				int x = track_data_sep[t - FIRST_FRAME][id][0];
				int y = track_data_sep[t - FIRST_FRAME][id][1];

				if (LabelImg.at<int>(y, x) > 0) {
					int* param = stats.ptr<int>(LabelImg.at<int>(y, x));
					int area = param[cv::ConnectedComponentsTypes::CC_STAT_AREA];
					area_data[t - FIRST_FRAME][id] = area;
				}
			}
		}
	}

	std::vector<int> result_area(track_data_sep[0].size(), -1);
	for (int id = 0; id < area_data[0].size(); id++) {
		int flag = 0;
		for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {
			int area0 = 0;//Average from t-2 to t
			for (int s = std::max(FIRST_FRAME, t - 2); s <= t; s++) {
				area0 += area_data[s - FIRST_FRAME][id];
			}

			area0 = area0 / (t - std::max(FIRST_FRAME, t - 2) + 1);

			int area1 = area_data[std::min((int)area_data.size() - 1, t - FIRST_FRAME + 1)][id];
			int area2 = area_data[std::min((int)area_data.size() - 1, t - FIRST_FRAME + 2)][id];

			if (flag == 0 && area_data[0][id] > 0 && area0 > 0 && area1 > 0 && area0 * T_VANISH_AREA / 100 > area1 && area0 * T_VANISH_AREA / 100 > area2) {
				result_area[id] = t + 1;
				flag = 1;
			}
		}
	}

	for (int i = 0; i < result_area.size(); i++) {
		if (result_area[i] > -1) {
			std::cout << "(" << track_data_sep[0][i][0] << "," << track_data_sep[0][i][1] << ") vanish at t" << result_area[i] << std::endl;
			for (int t = result_area[i]; t <= LAST_FRAME; t++) {
				track_data_sep[t - FIRST_FRAME][i] = { -1 };
			}
		}
	}
	return 0;
}

int delete_area_id(std::vector<std::vector<std::vector<int>>>& track_data_sep, const std::string SAMPLE_NAME, const int FIRST_FRAME, const int LAST_FRAME, const int T_DELETE_AREA,
	const std::string gfp_or_rfp) {

	std::vector<std::vector<int>> area_data(track_data_sep.size(), std::vector<int>(track_data_sep[track_data_sep.size() - 1].size(), 0));

	for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {
		Mat cell = imread(SAMPLE_NAME + "_cellmask\\" + gfp_or_rfp + "_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		if (cell.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		Mat LabelImg(cell.size(), CV_32S);
		Mat stats;
		Mat centroids;
		connectedComponentsWithStats(cell, LabelImg, stats, centroids);

		for (int id = 0; id < track_data_sep[t - FIRST_FRAME].size(); id++) {

			if (track_data_sep[t - FIRST_FRAME][id].size() == 2) {
				int x = track_data_sep[t - FIRST_FRAME][id][0];
				int y = track_data_sep[t - FIRST_FRAME][id][1];

				if (LabelImg.at<int>(y, x) > 0) {
					int* param = stats.ptr<int>(LabelImg.at<int>(y, x));
					int area = param[cv::ConnectedComponentsTypes::CC_STAT_AREA];
					area_data[t - FIRST_FRAME][id] = area;
				}
			}
		}
	}

	std::vector<int> result_area(track_data_sep[0].size(), -1);
	for (int id = 0; id < area_data[0].size(); id++) {
		int flag = 0;
		for (int t = FIRST_FRAME + 1; t <= LAST_FRAME; t++) {//The start frame is detected differently.

			int area0 = area_data[std::min((int)area_data.size() - 1, t - FIRST_FRAME)][id];
			int area1 = area_data[std::min((int)area_data.size() - 1, t - FIRST_FRAME + 1)][id];

			if (flag == 0 && area0 > 0 && area1 > 0 && area0 * T_DELETE_AREA / 100 < area1) {
				result_area[id] = t + 1;
				flag = 1;
			}
		}
	}

	for (int i = 0; i < result_area.size(); i++) {
		if (result_area[i] > -1) {
			std::cout << "(" << track_data_sep[0][i][0] << "," << track_data_sep[0][i][1] << ") large mask at t" << result_area[i] << std::endl;
			for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {
				track_data_sep[t - FIRST_FRAME][i] = { -1 };
			}
		}
	}

	return 0;
}

void cal_brt(std::vector<std::vector<std::vector<int>>>& track_data_sep, const std::string SAMPLE_NAME, const int FIRST_FRAME, const int LAST_FRAME,
	std::vector<std::vector<int>>& brt_data, const std::string FLUO_NAMES_PREFIX, const std::string FLUO_NAMES_SUFFIX) {

	brt_data = std::vector<std::vector<int>>(track_data_sep.size(), std::vector<int>(track_data_sep[track_data_sep.size() - 1].size(), 0));

	for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {
		char numStr[256];
		sprintf_s(numStr, 256, "%03d", t);
		Mat fluo0 = imread(FLUO_NAMES_PREFIX + (std::string)numStr + FLUO_NAMES_SUFFIX, cv::IMREAD_ANYDEPTH);
		Mat cell0 = imread(SAMPLE_NAME + "_cellmask\\gfp_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		if (fluo0.data == NULL || cell0.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		Mat LabelImg(cell0.size(), CV_32S);
		Mat stats;
		Mat centroids;
		connectedComponentsWithStats(cell0, LabelImg, stats, centroids);

		for (int id = 0; id < track_data_sep[t - FIRST_FRAME].size(); id++) {
			if (track_data_sep[t - FIRST_FRAME][id].size() == 2) {
				int x = track_data_sep[t - FIRST_FRAME][id][0];
				int y = track_data_sep[t - FIRST_FRAME][id][1];

				if (LabelImg.at<int>(y, x) > 0) {
					int* param = stats.ptr<int>(LabelImg.at<int>(y, x));
					int left = param[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
					int top = param[cv::ConnectedComponentsTypes::CC_STAT_TOP];
					int height = param[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];
					int width = param[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];
					int area = param[cv::ConnectedComponentsTypes::CC_STAT_AREA];

					int brt_sum = 0;
					for (int xp = left; xp < left + width; xp++) {
						for (int yp = top; yp < top + height; yp++) {
							if (LabelImg.at<int>(yp, xp) == LabelImg.at<int>(y, x)) {
								brt_sum += fluo0.at<unsigned char>(yp, xp);
							}
						}
					}
					brt_data[t - FIRST_FRAME][id] = brt_sum / std::max(1, area);
				}
			}
		}

	}
}

void cut_track_brt(std::vector<std::vector<std::vector<int>>>& track_data_sep, const std::string SAMPLE_NAME, const int START, const int END,
	const std::string FLUO_NAMES_PREFIX, const std::string FLUO_NAMES_SUFFIX, const int T_VANISH_BRT) {
	std::cout << "cut_track_brt" << std::endl;

	//Calculation of cell body average brightness
	std::vector<std::vector<int>> brt_data(END - START + 1, std::vector<int>(0));
	cal_brt(track_data_sep, SAMPLE_NAME, START, END, brt_data, FLUO_NAMES_PREFIX, FLUO_NAMES_SUFFIX);

	//Look for areas where the average brightness drops significantly
	std::vector<int> result_brt(track_data_sep[0].size(), -1);
	for (int id = 0; id < brt_data[0].size(); id++) {
		int flag = 0;
		for (int t = START; t < END - 1; t++) {
			int brt00 = brt_data[std::max(0, t - START - 1)][id];
			int brt0 = brt_data[t - START][id];
			int brt1 = brt_data[t - START + 1][id];

			if (flag == 0 && brt_data[0][id] > 0 && brt0 > 0 && brt1 > 0 && brt0 - brt1 > T_VANISH_BRT && (brt00 == 0 || brt00 - brt1 > T_VANISH_BRT)) {
				result_brt[id] = t + 1;
				flag = 1;
			}
		}
	}
	for (int i = 0; i < result_brt.size(); i++) {
		if (result_brt[i] > -1) {
			std::cout << "(" << track_data_sep[0][i][0] << ", " << track_data_sep[0][i][1] << ") vanish(brt) at t" << result_brt[i] << std::endl;
			for (int t = result_brt[i]; t <= END; t++) {
				track_data_sep[t - START][i] = { -1 };
			}
		}
	}

	return;
}

void cell_tracking_gfp(const std::string SAMPLE_NAME, const std::string CH2_NAMES_PREFIX, const std::string CH2_NAMES_SUFFIX, const int FIRST_FRAME, const int LAST_FRAME,
	const int MIN_LIFETIME, const int D_MAX, const int GFP_CELL_MAX, const int T_DELETE_AREA, const int T_VANISH_BRT,
	std::vector<std::vector<std::vector<int>>>& gfp_result_data_sep) {
	//t0 coordinate setting
	Mat cell_t0 = imread(SAMPLE_NAME + "_cellmask\\gfp_t" + std::to_string(FIRST_FRAME) + ".tif", cv::IMREAD_ANYDEPTH);
	if (cell_t0.data == NULL) {
		std::cerr << "ERROR: NO IMAGE";
		exit(1);
	}

	get_area_center(cell_t0, gfp_result_data_sep[0]);

	//Mapping after t1
	for (int t = FIRST_FRAME; t < LAST_FRAME; t++) {
		std::cout << "frame: " << t << "~" << t + 1 << std::endl;

		/*image read (t, t+1)*/
		Mat cell0 = imread(SAMPLE_NAME + "_cellmask\\gfp_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		Mat cell1 = imread(SAMPLE_NAME + "_cellmask\\gfp_t" + std::to_string(t + 1) + ".tif", cv::IMREAD_ANYDEPTH);
		if (cell0.data == NULL || cell1.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		//Area mapping (not one-to-one)
		std::vector<int> result_x(gfp_result_data_sep[t - FIRST_FRAME].size(), -1);//Destination of the region at time t
		match_cell_areas(cell0, cell1, result_x);

		gen_result_gfp(cell0, cell1, result_x, gfp_result_data_sep[t - FIRST_FRAME], gfp_result_data_sep[t - FIRST_FRAME + 1]);
	}

	//Tracking correction
	std::vector<std::vector<std::vector<int>>> v_tmp = gfp_result_data_sep;
	refine_trace_data_gfp(gfp_result_data_sep, v_tmp, MIN_LIFETIME, D_MAX);
	gfp_result_data_sep = v_tmp;

	//Clear possible haze id
	delete_fog_id(SAMPLE_NAME, FIRST_FRAME, "gfp", gfp_result_data_sep, v_tmp, GFP_CELL_MAX);
	gfp_result_data_sep = v_tmp;

	//Eliminate the rapidly increasing ID
	delete_area_id(gfp_result_data_sep, SAMPLE_NAME, FIRST_FRAME, LAST_FRAME, T_DELETE_AREA, "gfp");

	//Judging disappearance by sudden decrease in brightness
	cut_track_brt(gfp_result_data_sep, SAMPLE_NAME, FIRST_FRAME, LAST_FRAME, CH2_NAMES_PREFIX, CH2_NAMES_SUFFIX, T_VANISH_BRT);

	delete_dammy(gfp_result_data_sep);
}

void cell_tracking_rfp(const std::string SAMPLE_NAME, const std::string CH3_NAMES_PREFIX, const std::string CH3_NAMES_SUFFIX, const int FIRST_FRAME, const int LAST_FRAME,
	const int MIN_LIFETIME, const int D_MAX, const int RFP_CELL_MAX, const int T_VANISH_AREA, const int T_DELETE_AREA,
	std::vector<std::vector<std::vector<int>>>& rfp_result_data_sep) {

	//t0 coordinate setting
	Mat cell_t0 = imread(SAMPLE_NAME + "_cellmask\\rfp_t" + std::to_string(FIRST_FRAME) + ".tif", cv::IMREAD_ANYDEPTH);
	if (cell_t0.data == NULL) {
		std::cerr << "ERROR: NO IMAGE";
		exit(1);
	}

	get_area_center(cell_t0, rfp_result_data_sep[0]);

	//Mapping after t1
	for (int t = FIRST_FRAME; t < LAST_FRAME; t++) {
		std::cout << "frame: " << t << "~" << t + 1 << std::endl;

		/*Loading t, t+1 images*/
		Mat cell0 = imread(SAMPLE_NAME + "_cellmask\\rfp_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		Mat cell1 = imread(SAMPLE_NAME + "_cellmask\\rfp_t" + std::to_string(t + 1) + ".tif", cv::IMREAD_ANYDEPTH);
		if (cell0.data == NULL || cell1.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		//Area mapping (not one-to-one)
		std::vector<int> result_x(rfp_result_data_sep[t - FIRST_FRAME].size(), -1);//Destination of the region at time t
		match_cell_areas(cell0, cell1, result_x);

		gen_result_rfp(cell0, cell1, result_x, rfp_result_data_sep[t - FIRST_FRAME], rfp_result_data_sep[t - FIRST_FRAME + 1]);
	}

	//Tracking correction
	//Rapid reduction in area (track may become shorter)
	area_shrink(rfp_result_data_sep, SAMPLE_NAME, FIRST_FRAME, LAST_FRAME, T_VANISH_AREA);

	std::vector<std::vector<std::vector<int>>> v_tmp = rfp_result_data_sep;
	refine_trace_data_rfp(rfp_result_data_sep, v_tmp, MIN_LIFETIME, D_MAX);
	rfp_result_data_sep = v_tmp;

	//Clear possible haze id
	delete_fog_id(SAMPLE_NAME, FIRST_FRAME, "rfp", rfp_result_data_sep, v_tmp, RFP_CELL_MAX);
	rfp_result_data_sep = v_tmp;

	//Eliminate the rapidly increasing ID
	delete_area_id(rfp_result_data_sep, SAMPLE_NAME, FIRST_FRAME, LAST_FRAME, T_DELETE_AREA, "rfp");

	delete_dammy(rfp_result_data_sep);
}

void choose_tracked_cells(const Mat& cellmask_old, Mat& cellmask_new, const std::vector<std::vector<int>>& track_centers) {

	cellmask_new = cellmask_old.clone();

	Mat LabelImg(cellmask_old.size(), CV_32S);
	Mat stats;
	Mat centroids;
	int n = connectedComponentsWithStats(cellmask_old, LabelImg, stats, centroids, 4);

	std::vector<int> existing_labels;

	for (int id = 0; id < track_centers.size(); id++) {
		if (track_centers[id].size() == 2) {
			int x = track_centers[id][0];
			int y = track_centers[id][1];

			if (LabelImg.at<int>(y, x) > 0 && vector_finder(existing_labels, LabelImg.at<int>(y, x)) == existing_labels.size()) {
				existing_labels.push_back(LabelImg.at<int>(y, x));
			}
		}
	}

	for (int i = 1; i < n; i++) {
		if (vector_finder(existing_labels, i) == existing_labels.size()) {
			int* param = stats.ptr<int>(i);
			int left = param[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
			int top = param[cv::ConnectedComponentsTypes::CC_STAT_TOP];
			int height = param[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];
			int width = param[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];

			for (int x = left; x < left + width; x++) {
				for (int y = top; y < top + height; y++) {
					if (LabelImg.at<int>(y, x) == i) {
						cellmask_new.at<unsigned char>(y, x) = 0;
					}
				}
			}
		}
	}
}

void count_inclusion_each_cell(const std::vector<std::vector<std::vector<int>>>& track_data_sep, const std::string SAMPLE_NAME, const std::string GFP_OR_RFP,
	const int FIRST_FRAME, const int LAST_FRAME, std::vector<std::vector<double>>& inc_each_cell) {

	inc_each_cell = std::vector<std::vector<double>>(LAST_FRAME - FIRST_FRAME + 1, std::vector<double>(0));

	for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {
		//load cellmask
		Mat cellmask = imread(SAMPLE_NAME + "_cellmask\\" + GFP_OR_RFP + "_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		Mat inc = imread(SAMPLE_NAME + "_cellmask\\inc_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		if (cellmask.data == NULL || inc.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}
		choose_tracked_cells(cellmask, cellmask, track_data_sep[t - FIRST_FRAME]);

		Mat LabelImg(cellmask.size(), CV_32S);
		Mat stats;
		Mat centroids;
		connectedComponentsWithStats(cellmask, LabelImg, stats, centroids, 4);

		inc_each_cell[t - FIRST_FRAME] = std::vector<double>(track_data_sep[t - FIRST_FRAME].size(), 0);

		for (int id = 0; id < track_data_sep[t - FIRST_FRAME].size(); id++) {
			if (track_data_sep[t - FIRST_FRAME][id].size() == 2) {
				int color = LabelImg.at<int>(track_data_sep[t - FIRST_FRAME][id][1], track_data_sep[t - FIRST_FRAME][id][0]);

				if (color > 0) {//Since tracking is interpolated, there may be cases where there is no mask (considered to be inclusion 0).
					int* param = stats.ptr<int>(color);
					int left = param[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
					int top = param[cv::ConnectedComponentsTypes::CC_STAT_TOP];
					int height = param[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];
					int width = param[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];
					int area = param[cv::ConnectedComponentsTypes::CC_STAT_AREA];

					int count = 0;
					for (int x = left; x < left + width; x++) {
						for (int y = top; y < top + height; y++) {
							if (LabelImg.at<int>(y, x) == color && inc.at<unsigned char>(y, x) > 0) {
								count++;
							}
						}
					}

					if (count > 0) {
						inc_each_cell[t - FIRST_FRAME][id] = (double)count / area;
					}
				}
			}
		}
	}
}

int integrate_tracking_res(const std::vector<std::vector<std::vector<int>>>& gfp_result_data_sep, const std::vector<std::vector<std::vector<int>>>& rfp_result_data_sep, const double THRE_INC_RATIO,
	const std::string SAMPLE_NAME, const int FIRST_FRAME, const int LAST_FRAME, std::vector<std::vector<std::vector<int>>>& result_data_sep, std::vector<std::vector<double>>& inclusion_data) {

	//Inclusion-(all frames) uses tracking results from RFP, others from GFP
	std::vector<std::vector<double>> gfp_inclusion_data, rfp_inclusion_data;
	count_inclusion_each_cell(gfp_result_data_sep, SAMPLE_NAME, "gfp", FIRST_FRAME, LAST_FRAME, gfp_inclusion_data);
	count_inclusion_each_cell(rfp_result_data_sep, SAMPLE_NAME, "rfp", FIRST_FRAME, LAST_FRAME, rfp_inclusion_data);

	std::vector<int> gfp_inclusion1_ind;//Inclusion+ in a certain frame
	std::vector<int> rfp_inclusion0_ind;//inclusion-(all frames)

	for (int id = 0; id < gfp_result_data_sep[gfp_result_data_sep.size() - 1].size(); id++) {
		bool exist_double_id = false;

		int x = gfp_result_data_sep[0][id][0];
		int y = gfp_result_data_sep[0][id][1];
		for (int id_rfp = 0; id_rfp < rfp_result_data_sep[0].size(); id_rfp++) {
			int x_rfp = rfp_result_data_sep[0][id_rfp][0];
			int y_rfp = rfp_result_data_sep[0][id_rfp][1];
			if (pow(x - x_rfp, 2) + pow(y - y_rfp, 2) < pow(5, 2)) {
				exist_double_id = true;
			}
		}
		//If the same coordinates are found in rfp, rfp takes priority.
		if (!exist_double_id) {
			if (id < gfp_result_data_sep[0].size() && gfp_inclusion_data[0][id] > THRE_INC_RATIO) {
				gfp_inclusion1_ind.push_back(id);
			}
			else {
				int count = 0;
				for (int t = 0; t < gfp_result_data_sep.size(); t++) {
					if (id < gfp_result_data_sep[t].size() && gfp_inclusion_data[t][id] > THRE_INC_RATIO) {
						count++;
						break;
					}
				}
				if (count > 0) {
					gfp_inclusion1_ind.push_back(id);
				}
			}
		}
	}
	for (int id = 0; id < rfp_result_data_sep[rfp_result_data_sep.size() - 1].size(); id++) {
		int count = 0;
		for (int t = 0; t < rfp_result_data_sep.size(); t++) {
			if (id<rfp_result_data_sep[t].size() && rfp_inclusion_data[t][id]>THRE_INC_RATIO) {
				count++;
			}
		}
		if (count == 0) {
			rfp_inclusion0_ind.push_back(id);
		}
	}

	result_data_sep = std::vector<std::vector<std::vector<int>>>(gfp_result_data_sep.size(),
		std::vector<std::vector<int>>(gfp_inclusion1_ind.size() + rfp_inclusion0_ind.size(), std::vector<int>(1, -1)));
	inclusion_data = std::vector<std::vector<double>>(gfp_result_data_sep.size(), std::vector<double>(gfp_inclusion1_ind.size() + rfp_inclusion0_ind.size(), 0));

	for (int t = 0; t < gfp_result_data_sep.size(); t++) {
		for (int i = 0; i < gfp_inclusion1_ind.size(); i++) {
			int ind = gfp_inclusion1_ind[i];
			if (ind < gfp_result_data_sep[t].size()) {
				result_data_sep[t][i] = gfp_result_data_sep[t][ind];
				inclusion_data[t][i] = gfp_inclusion_data[t][ind];
			}
		}
		for (int i = 0; i < rfp_inclusion0_ind.size(); i++) {
			int ind = rfp_inclusion0_ind[i];
			if (ind < rfp_inclusion_data[t].size()) {
				result_data_sep[t][i + gfp_inclusion1_ind.size()] = rfp_result_data_sep[t][ind];
				inclusion_data[t][i + gfp_inclusion1_ind.size()] = rfp_inclusion_data[t][ind];
			}
		}
	}

	return (int)gfp_inclusion1_ind.size();
}

/*Result output*/
void make_trace_csv(const int START, const int END, const std::vector<std::vector<std::vector<int>>>& result_data_sep, const std::string RESULT_NAME) {
	int max_id = (int)result_data_sep[END - START].size();

	FILE* fp;
	errno_t error = fopen_s(&fp, RESULT_NAME.c_str(), "w");
	if (error != 0) {
		std::cerr << "ERROR: could not write csv";
		exit(1);
	}

	std::string s;

	for (int id = 0; id < max_id; id++) {
		s += ",id" + std::to_string(id);
	}
	s += "\n";
	fprintf(fp, s.c_str());

	for (int t = START; t <= END; t++) {
		s = "t" + std::to_string(t) + ",";
		for (int id = 0; id < result_data_sep[t - START].size(); id++) {
			if (result_data_sep[t - START][id].size() == 2 && result_data_sep[t - START][id][0] > 0 && result_data_sep[t - START][id][1] > 0) {
				s += std::to_string(result_data_sep[t - START][id][0]) + "-" + std::to_string(result_data_sep[t - START][id][1]) + ",";
			}
			else if (result_data_sep[t - START][id].size() == 0 || (result_data_sep[t - START][id].size() == 1 && result_data_sep[t - START][id][0] == 0)) {
				s += ",";
			}
			else {
				s += "-1,";
			}
		}
		s += "\n";
		fprintf(fp, s.c_str());
	}

	fclose(fp);
}

void make_color_img(const Mat& fluo, const Mat& trace, const Mat& gfpmask, const Mat& rfpmask, const Mat& inc, Mat& color_img) {

	const std::vector<std::vector<int>> color = { { 255, 0, 0 },{ 0, 255, 0 },{ 0, 0, 255 },{ 0, 255, 255 },{ 255, 0, 255 },{ 255, 255, 0 },
	{ 128, 255, 255 },{ 255, 128, 255 },{ 128, 255, 255 },{ 255, 128, 128 },{ 128, 255, 128 },{ 128, 128, 255 } };

	for (int i = 0; i < fluo.cols; i++) {
		for (int j = 0; j < fluo.rows; j++) {
			if (trace.at<unsigned char>(j, i) > 0) {
				int ind_color = trace.at<unsigned char>(j, i);
				color_img.at<Vec3b>(j, i)[0] = (uchar)color[ind_color % color.size()][0];  //blue
				color_img.at<Vec3b>(j, i)[1] = (uchar)color[ind_color % color.size()][1];  //green
				color_img.at<Vec3b>(j, i)[2] = (uchar)color[ind_color % color.size()][2];  //red
			}
			else if (gfpmask.at<unsigned char>(j, i) > 0 && inc.at<unsigned char>(j, i) > 0) {
				color_img.at<Vec3b>(j, i)[0] = (uchar)(fluo.at<unsigned char>(j, i) * 0.7);  //blue
				color_img.at<Vec3b>(j, i)[1] = (uchar)(fluo.at<unsigned char>(j, i) * 0.7 + 255 * 0.3);  //green
				color_img.at<Vec3b>(j, i)[2] = (uchar)(fluo.at<unsigned char>(j, i) * 0.7 + 255 * 0.3);  //red
			}
			else if (fluo.at<unsigned char>(j, i) > 0) {
				color_img.at<Vec3b>(j, i)[0] = fluo.at<unsigned char>(j, i);  //blue
				color_img.at<Vec3b>(j, i)[1] = fluo.at<unsigned char>(j, i);  //green
				color_img.at<Vec3b>(j, i)[2] = fluo.at<unsigned char>(j, i);  //red
			}

			if (gfpmask.at<unsigned char>(j, i) > 0) {
				//Finding cell boundaries
				int count = 0;
				for (int x = std::max(0, i - 1); x <= std::min(gfpmask.cols - 1, i + 1); x++) {
					for (int y = std::max(0, j - 1); y <= std::min(gfpmask.rows - 1, j + 1); y++) {
						if ((x != i || y != j) && gfpmask.at<unsigned char>(y, x) == 0) {
							count++;
						}
					}
				}

				if (count > 0) {
					color_img.at<Vec3b>(j, i)[0] = 117;  //blue
					color_img.at<Vec3b>(j, i)[1] = 175;  //green
					color_img.at<Vec3b>(j, i)[2] = 59;  //red
				}
			}

			if (rfpmask.at<unsigned char>(j, i) > 0) {
				//Finding cell boundaries
				int count = 0;
				for (int x = std::max(0, i - 1); x <= std::min(rfpmask.cols - 1, i + 1); x++) {
					for (int y = std::max(0, j - 1); y <= std::min(rfpmask.rows - 1, j + 1); y++) {
						if ((x != i || y != j) && rfpmask.at<unsigned char>(y, x) == 0) {
							count++;
						}
					}
				}

				if (count > 0) {
					color_img.at<Vec3b>(j, i)[0] = 168;  //blue
					color_img.at<Vec3b>(j, i)[1] = 87;  //green
					color_img.at<Vec3b>(j, i)[2] = 167;  //red
				}
			}
		}
	}
}

void make_result_seq(const int FIRST_FRAME, const int LAST_FRAME, const std::string SAMPLE_NAME, const std::string CH2_NAMES_PREFIX, const std::string CH2_NAMES_SUFFIX,
	const std::string CH3_NAMES_PREFIX, const std::string CH3_NAMES_SUFFIX, const std::string PH_NAMES_PREFIX, const std::string PH_NAMES_SUFFIX,
	const std::vector<std::vector<std::vector<int>>>& result_data_sep, const int n_gfp_cell) {

	std::vector<std::vector<std::vector<int>>> result_data_sep_gfp(result_data_sep.size(), std::vector<std::vector<int>>(n_gfp_cell, std::vector<int>(1, -1)));
	std::vector<std::vector<std::vector<int>>> result_data_sep_rfp(result_data_sep.size(),
		std::vector<std::vector<int>>(result_data_sep[result_data_sep.size() - 1].size() - n_gfp_cell, std::vector<int>(1, -1)));
	for (int t = 0; t < result_data_sep.size(); t++) {
		for (int i = 0; i < result_data_sep[t].size(); i++) {
			if (i < n_gfp_cell) {
				result_data_sep_gfp[t][i] = result_data_sep[t][i];
			}
			else {
				result_data_sep_rfp[t][i - n_gfp_cell] = result_data_sep[t][i];
			}
		}
	}

	for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {

		char numStr[256];
		sprintf_s(numStr, 256, "%03d", t);
		Mat ch2 = imread(CH2_NAMES_PREFIX + (std::string)numStr + CH2_NAMES_SUFFIX, cv::IMREAD_ANYDEPTH);
		Mat ch3 = imread(CH3_NAMES_PREFIX + (std::string)numStr + CH3_NAMES_SUFFIX, cv::IMREAD_ANYDEPTH);
		Mat ph = imread(PH_NAMES_PREFIX + (std::string)numStr + PH_NAMES_SUFFIX, cv::IMREAD_ANYDEPTH);
		if (ch2.data == NULL || ch3.data == NULL || ph.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		//Creating a trajectory image
		Mat trace(ch2.size(), CV_8U, Scalar::all(0));

		for (int id = 0; id < result_data_sep[t - FIRST_FRAME].size(); id++) {
			//Add an ID number
			if (result_data_sep[t - FIRST_FRAME][id].size() == 2) {
				int x0 = result_data_sep[t - FIRST_FRAME][id][0];
				int y0 = result_data_sep[t - FIRST_FRAME][id][1];

				int color = id % 255 + 1;
				putText(trace, std::to_string(id), Point(x0 + 20, y0 - 20), FONT_HERSHEY_SIMPLEX, 1, color);
			}
		}

		for (int id = 0; id < result_data_sep[t - FIRST_FRAME].size(); id++) {
			int color = id % 255 + 1;

			for (int s = FIRST_FRAME; s < t; s++) {
				if (id < result_data_sep[s - FIRST_FRAME].size() && id < result_data_sep[s - FIRST_FRAME + 1].size() &&
					result_data_sep[s - FIRST_FRAME][id].size() == 2 && result_data_sep[s - FIRST_FRAME + 1][id].size() == 2 &&
					id < result_data_sep[t - FIRST_FRAME].size() && result_data_sep[t - FIRST_FRAME][id].size() == 2) {

					int x0 = result_data_sep[s - FIRST_FRAME][id][0];
					int y0 = result_data_sep[s - FIRST_FRAME][id][1];
					int x1 = result_data_sep[s - FIRST_FRAME + 1][id][0];
					int y1 = result_data_sep[s - FIRST_FRAME + 1][id][1];

					if (x0 >= 0 && x0 < ch2.cols && y0 >= 0 && y0 < ch2.rows && x1 >= 0 && x1 < ch2.cols && y1 >= 0 && y1 < ch2.rows) {
						line(trace, Point(x0, y0), Point(x1, y1), color, 1, 4);
					}
				}
			}
		}

		//Read cellmask
		Mat gfp_mask = imread(SAMPLE_NAME + "_cellmask\\gfp_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		Mat rfp_mask = imread(SAMPLE_NAME + "_cellmask\\rfp_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		Mat inc = imread(SAMPLE_NAME + "_cellmask\\inc_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		if (gfp_mask.data == NULL || rfp_mask.data == NULL || inc.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		Mat gfp_mask_modified, rfp_mask_modified;
		choose_tracked_cells(gfp_mask, gfp_mask_modified, result_data_sep_gfp[t - FIRST_FRAME]);
		choose_tracked_cells(rfp_mask, rfp_mask_modified, result_data_sep_rfp[t - FIRST_FRAME]);

		Mat_<cv::Vec3b> gfp_res(ch2.rows, ch2.cols, cv::Vec3b(0, 0, 0));
		Mat_<cv::Vec3b> rfp_res(ch2.rows, ch2.cols, cv::Vec3b(0, 0, 0));
		Mat_<cv::Vec3b> ph_res(ch2.rows, ch2.cols, cv::Vec3b(0, 0, 0));

		make_color_img(ch2, trace, gfp_mask_modified, rfp_mask_modified, inc, gfp_res);
		make_color_img(ch3, trace, gfp_mask_modified, rfp_mask_modified, inc, rfp_res);
		make_color_img(ph, trace, gfp_mask_modified, rfp_mask_modified, inc, ph_res);

		imwrite(SAMPLE_NAME + "_result_seq\\gfp" + std::to_string(t) + ".tif", gfp_res);
		imwrite(SAMPLE_NAME + "_result_seq\\rfp" + std::to_string(t) + ".tif", rfp_res);
		imwrite(SAMPLE_NAME + "_result_seq\\ph" + std::to_string(t) + ".tif", ph_res);
	}
}

void make_inclusion_each_cell_csv(const std::vector<std::vector<std::vector<int>>>& result_data_sep, const std::vector<std::vector<double>>& inc_each_cell,
	const int FIRST_FRAME, const int LAST_FRAME, const std::string RESULT_NAME) {

	int max_id = (int)inc_each_cell[LAST_FRAME - FIRST_FRAME].size();

	FILE* fp;
	errno_t error = fopen_s(&fp, RESULT_NAME.c_str(), "w");
	if (error != 0) {
		std::cerr << "ERROR: could not write csv";
		exit(1);
	}

	std::string s;

	for (int id = 0; id < max_id; id++) {
		s += ",id" + std::to_string(id);
	}
	s += "\n";
	fprintf(fp, s.c_str());

	for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {
		s = "t" + std::to_string(t) + ",";
		for (int id = 0; id < inc_each_cell[t - FIRST_FRAME].size(); id++) {
			if (result_data_sep[t - FIRST_FRAME][id].size() == 2) {
				s += std::to_string(inc_each_cell[t - FIRST_FRAME][id]) + ",";
			}
			else if (result_data_sep[t - FIRST_FRAME][id].size() == 1 && result_data_sep[t - FIRST_FRAME][id][0] == -1) {
				s += "-1,";
			}
			else {
				s += ",";
			}
		}
		s += "\n";
		fprintf(fp, s.c_str());
	}
	fclose(fp);
}

void output_result(const int FIRST_FRAME, const int LAST_FRAME, const std::vector<std::vector<std::vector<int>>>& result_data_sep, const std::vector<std::vector<double>>& inclusion_data,
	const int n_gfp_cell, const std::string SAMPLE_NAME, const std::string CH2_NAMES_PREFIX, const std::string CH2_NAMES_SUFFIX, const std::string CH3_NAMES_PREFIX, const std::string CH3_NAMES_SUFFIX,
	const std::string PH_NAMES_PREFIX, const std::string PH_NAMES_SUFFIX) {

	//Create tracking data (csv)
	make_trace_csv(FIRST_FRAME, LAST_FRAME, result_data_sep, SAMPLE_NAME + ".csv");

	make_inclusion_each_cell_csv(result_data_sep, inclusion_data, FIRST_FRAME, LAST_FRAME, SAMPLE_NAME + "_inclusion.csv");

	make_result_seq(FIRST_FRAME, LAST_FRAME, SAMPLE_NAME, CH2_NAMES_PREFIX, CH2_NAMES_SUFFIX, CH3_NAMES_PREFIX, CH3_NAMES_SUFFIX,
		PH_NAMES_PREFIX, PH_NAMES_SUFFIX, result_data_sep, n_gfp_cell);

	return;
}

void cal_mean_int_of_cell(const std::string SAMPLE_NAME, const int FIRST_FRAME, const int LAST_FRAME,
	const std::vector<std::vector<std::vector<int>>>& result_data_sep, const int n_gfp_cell, const std::string CH2_NAMES_PREFIX, const std::string CH2_NAMES_SUFFIX) {

	std::vector<std::vector<double>> mean_int(LAST_FRAME - FIRST_FRAME + 1);

	for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {
		char numStr[256];
		sprintf_s(numStr, 256, "%03d", t);
		Mat gfp_img = imread(CH2_NAMES_PREFIX + (std::string)numStr + CH2_NAMES_SUFFIX, cv::IMREAD_ANYDEPTH);
		Mat gfp_mask = imread(SAMPLE_NAME + "_cellmask\\gfp_t" + std::to_string(t) + ".tif", cv::IMREAD_ANYDEPTH);
		if (gfp_img.data == NULL || gfp_mask.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		Mat LabelImg(gfp_mask.size(), CV_32S);
		Mat stats;
		Mat centroids;
		connectedComponentsWithStats(gfp_mask, LabelImg, stats, centroids, 4);

		for (int id = 0; id < result_data_sep[t - FIRST_FRAME].size(); id++) {
			if (result_data_sep[t - FIRST_FRAME][id].size() == 2) {
				int x = result_data_sep[t - FIRST_FRAME][id][0];
				int y = result_data_sep[t - FIRST_FRAME][id][1];
				if (LabelImg.at<int>(y, x) > 0 && id < n_gfp_cell) {
					int* param = stats.ptr<int>(LabelImg.at<int>(y, x));
					int left = param[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
					int top = param[cv::ConnectedComponentsTypes::CC_STAT_TOP];
					int height = param[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];
					int width = param[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];

					Mat sub_img(gfp_img, Rect(left, top, width, height));
					Mat sub_mask(gfp_mask, Rect(left, top, width, height));
					Scalar mean, stddev;
					meanStdDev(sub_img, mean, stddev, sub_mask);

					mean_int[t - FIRST_FRAME].push_back(mean[0]);
				}
				else {
					mean_int[t - FIRST_FRAME].push_back(0);
				}
			}
			else {
				mean_int[t - FIRST_FRAME].push_back(-1);
			}
		}
	}

	//CSV output
	int max_id = (int)result_data_sep[LAST_FRAME - FIRST_FRAME].size();

	FILE* fp;
	errno_t error = fopen_s(&fp, (SAMPLE_NAME + "_mean_intens.csv").c_str(), "w");

	if (error != 0) {
		std::cout << "ERROR: could not write csv";
		exit(1);
	}

	std::string s;

	for (int id = 0; id < max_id; id++) {
		s += ",id" + std::to_string(id);
	}
	s += "\n";
	fprintf(fp, s.c_str());

	for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {

		s = "t" + std::to_string(t) + ",";
		for (int id = 0; id < mean_int[t - FIRST_FRAME].size(); id++) {
			if (result_data_sep[t - FIRST_FRAME][id].size() == 2) {
				s += std::to_string(mean_int[t - FIRST_FRAME][id]) + ",";
			}
			else if (result_data_sep[t - FIRST_FRAME][id].size() == 1 && result_data_sep[t - FIRST_FRAME][id][0] == -1) {
				s += "-1,";
			}
			else {
				s += ",";
			}
		}
		s += "\n";
		fprintf(fp, s.c_str());
	}
	fclose(fp);
}

int main(int argc, char* argv[]){
	if (argc != 2) {
		std::cerr << "usage: " << argv[0] << " Image Directory";
		exit(1);
	}
	const std::string inputDir = argv[1];

	const std::string SAMPLE_NAME = inputDir;
	const std::string CH2_NAMES_PREFIX = inputDir + '\\';
	const std::string CH2_NAMES_SUFFIX= ".Ch2.tif";
	const std::string CH3_NAMES_PREFIX = inputDir + '\\';
	const std::string CH3_NAMES_SUFFIX= ".Ch3.tif";
	const std::string PH_NAMES_PREFIX = inputDir + '\\';
	const std::string PH_NAMES_SUFFIX= ".Ph.tif";
	const int FIRST_FRAME = 10;
	const int LAST_FRAME = 39;

	/*GFP-ch cell body detection parameters*/
	const int GFP_THRE_CELL = 60;
	const int GFP_N_MASK_GEN = 2;

	const int GFP_CELL_MIN0 = 10;
	const int GFP_CELL_MIN = 6;//Vertical and horizontal size
	const int GFP_CELL_MAX = 40;

	const int T_VANISH_BRT = 40;

	const int THRE_INC = 150;

	/*RFP-ch cell body detection parameters*/
	const int RFP_THRE_CELL = 250;
	const int RFP_N_MASK_GEN0 = 4;
	const int RFP_N_MASK_GEN = 3;
	const int RFP_CELL_MAX = 30;
	const int T_VANISH_AREA = 50;

	/*Tracking common parameters*/
	const int T_DELETE_AREA = 400;//If it's 140, delete it for 40% more.
	const int D_MAX = 15;//If it has moved beyond D_MAX, delete it.
	const int MIN_LIFETIME = 5;

	/*Aggregate fraction threshold*/
	const double THRE_INC_RATIO = 0.01;

	//Input parameters saved
	std::string input_parameters;
	input_parameters += "input parameters\n";
	input_parameters += "cellbody threshold(RFP) " + std::to_string(RFP_THRE_CELL) + "\n";
	input_parameters += "cellbody threshold(GFP) " + std::to_string(GFP_THRE_CELL) + "\n";
	input_parameters += "area shurink(first frame) " + std::to_string(RFP_N_MASK_GEN0) + "\n";
	input_parameters += "area shurink " + std::to_string(RFP_N_MASK_GEN) + "\n";
	input_parameters += "min size(first frame, GFP) " + std::to_string(GFP_CELL_MIN0) + "\n";
	input_parameters += "min size(GFP) " + std::to_string(GFP_CELL_MIN) + "\n";
	input_parameters += "max size(GFP) " + std::to_string(GFP_CELL_MAX) + "\n";
	input_parameters += "area threshold(Cell death) " + std::to_string(T_VANISH_AREA) + "\n";
	input_parameters += "brightness threshold(Cell death) " + std::to_string(T_VANISH_BRT) + "\n";
	input_parameters += "threshold area fluctuation " + std::to_string(T_DELETE_AREA) + "\n";
	input_parameters += "inclusion threshold " + std::to_string(THRE_INC) + "\n";
	input_parameters += "inclusion ratio threshold " + std::to_string(THRE_INC_RATIO) + "\n";
	input_parameters += "minimum lifetime " + std::to_string(MIN_LIFETIME) + "\n";

	std::ofstream outputfile(SAMPLE_NAME + "_parameters.txt");
	outputfile << input_parameters;
	outputfile.close();

	if (_mkdir((SAMPLE_NAME + "_cellmask").c_str()) == 0) {
		std::cout << "make dir: " << SAMPLE_NAME << "_cellmask" << std::endl;
	}
	else {
		std::cerr << "ERROR make dir: " << SAMPLE_NAME + "_cellmask";
		exit(1);
	}
	if (_mkdir((SAMPLE_NAME + "_result_seq").c_str()) == 0) {
		std::cout << "make dir: " << SAMPLE_NAME << "_result_seq" << std::endl;
	}
	else {
		std::cerr << "ERROR make dir: " << SAMPLE_NAME + "_result_seq";
		exit(1);
	}

	//Cell body detection
	for (int t = FIRST_FRAME; t <= LAST_FRAME; t++) {
		std::cout << t << " ";

		char numStr[256];
		sprintf_s(numStr, 256, "%03d", t);
		Mat gfp_img = imread(CH2_NAMES_PREFIX + (std::string)numStr + CH2_NAMES_SUFFIX, cv::IMREAD_ANYDEPTH);
		Mat rfp_img = imread(CH3_NAMES_PREFIX + (std::string)numStr + CH3_NAMES_SUFFIX, cv::IMREAD_ANYDEPTH);
		if (gfp_img.data == NULL || rfp_img.data == NULL) {
			std::cerr << "ERROR: NO IMAGE";
			exit(1);
		}

		/*GFP ch*/
		Mat gfp_mask;
		gen_gfp_mask(gfp_img, GFP_THRE_CELL, gfp_mask, GFP_N_MASK_GEN, GFP_CELL_MIN0, GFP_CELL_MIN, FIRST_FRAME, t);

		Mat inc;
		gen_inclusion_mask(gfp_img, gfp_mask, inc, THRE_INC);

		/*RFP ch*/
		Mat rfp_mask;
		gen_rfp_mask(rfp_img, RFP_THRE_CELL, t == FIRST_FRAME, RFP_N_MASK_GEN0, RFP_N_MASK_GEN, rfp_mask);

		//Result image creation
		imwrite(SAMPLE_NAME + "_cellmask\\gfp_t" + std::to_string(t) + ".tif", gfp_mask);
		imwrite(SAMPLE_NAME + "_cellmask\\inc_t" + std::to_string(t) + ".tif", inc);
		imwrite(SAMPLE_NAME + "_cellmask\\rfp_t" + std::to_string(t) + ".tif", rfp_mask);
	}

	//tracking
	std::cout << "cell tracking (GFP-ch)" << std::endl;
	std::vector<std::vector<std::vector<int>>> gfp_result_data_sep(LAST_FRAME - FIRST_FRAME + 1, std::vector<std::vector<int>>(0));

	cell_tracking_gfp(SAMPLE_NAME, CH2_NAMES_PREFIX, CH2_NAMES_SUFFIX, FIRST_FRAME, LAST_FRAME,
		MIN_LIFETIME, D_MAX, GFP_CELL_MAX, T_DELETE_AREA, T_VANISH_BRT, gfp_result_data_sep);


	std::cout << "cell tracking (RFP-ch)" << std::endl;
	std::vector<std::vector<std::vector<int>>> rfp_result_data_sep(LAST_FRAME - FIRST_FRAME + 1, std::vector<std::vector<int>>(0));

	cell_tracking_rfp(SAMPLE_NAME, CH3_NAMES_PREFIX, CH3_NAMES_SUFFIX, FIRST_FRAME, LAST_FRAME,
		MIN_LIFETIME, D_MAX, RFP_CELL_MAX, T_VANISH_AREA, T_DELETE_AREA, rfp_result_data_sep);

	std::vector<std::vector<std::vector<int>>> result_data_sep;
	std::vector<std::vector<double>> inclusion_data;
	int n_gfp_cell = integrate_tracking_res(gfp_result_data_sep, rfp_result_data_sep, THRE_INC_RATIO, SAMPLE_NAME, FIRST_FRAME, LAST_FRAME,
		result_data_sep, inclusion_data);
	std::cout << "n_gfp_cell " << n_gfp_cell << std::endl;

	output_result(FIRST_FRAME, LAST_FRAME, result_data_sep, inclusion_data, n_gfp_cell, SAMPLE_NAME,
		CH2_NAMES_PREFIX, CH2_NAMES_SUFFIX, CH3_NAMES_PREFIX, CH3_NAMES_SUFFIX, PH_NAMES_PREFIX, PH_NAMES_SUFFIX);

	//Cell average luminance output
	cal_mean_int_of_cell(SAMPLE_NAME, FIRST_FRAME, LAST_FRAME, result_data_sep, n_gfp_cell, CH2_NAMES_PREFIX, CH2_NAMES_SUFFIX);

	return 0;
}
