r"""
The Sage Notebook Twisted Web Server
"""

#############################################################################
#       Copyright (C) 2007 William Stein <wstein@gmail.com>
#  Distributed under the terms of the GNU General Public License (GPL)
#  The full text of the GPL is available at:
#                  http://www.gnu.org/licenses/
#############################################################################

############################################################
# WARNING: Potential source of confusion!!
#
# The following three global variables get set on
# startup by the script that is generated by run_notebook.py

notebook   = None
OPEN_MODE  = None
SID_COOKIE = None

############################################################

import os, shutil, time

import bz2

from twisted.web2 import server, http, resource, channel
from twisted.web2 import static, http_headers, responsecode

import css, js, keyboards

import notebook as _notebook

HISTORY_MAX_OUTPUT = 92*5
HISTORY_NCOLS = 90

from sage.misc.misc import SAGE_EXTCODE, SAGE_LOCAL, walltime, tmp_filename, tmp_dir
from sage.misc.remote_file import get_remote_file

p = os.path.join
css_path        = p(SAGE_EXTCODE, "notebook/css")
image_path      = p(SAGE_EXTCODE, "notebook/images")
javascript_path = p(SAGE_EXTCODE, "notebook/javascript")
slider_path     = p(SAGE_EXTCODE, "notebook/slider")
java_path       = p(SAGE_LOCAL, "java")

# the list of users waiting to register
waiting = {}

# the user database
from user_db import UserDatabase
users = UserDatabase()

_cols = None
def word_wrap_cols():
    global _cols
    if _cols is None:
        _cols = notebook.conf()['word_wrap_cols']
    return _cols

############################
# Encoding data to go between the server and client
############################
SEP = '___S_A_G_E___'

def encode_list(v):
    return SEP.join([str(x) for x in v])



############################
# Notebook autosave.
############################
# save if make a change to notebook and at least some seconds have elapsed since last save.
def init_updates():
    global save_interval, idle_interval, last_save_time, last_idle_time

    save_interval = notebook.conf()['save_interval']
    idle_interval = notebook.conf()['idle_check_interval']
    last_save_time = walltime()
    last_idle_time = walltime()

def notebook_save_check():
    global last_save_time
    t = walltime()
    if t > last_save_time + save_interval:
        notebook.save()
        last_save_time = t

def notebook_idle_check():
    notebook.quit_idle_worksheet_processes()
    global last_idle_time
    t = walltime()
    if t > last_idle_time + idle_interval:
        notebook.save()
        last_idle_time = t

def notebook_updates():
    notebook_save_check()
    notebook_idle_check()

######################################################################################
# RESOURCES
######################################################################################


############################
# An error message
############################
def message(msg, cont=None):
    s = notebook.html_banner()
    s += '<br><a class="usercontrol" href="/">Home</a>'
    s += '<hr class="usercontrol">'
    s += '<br>'*2
    s += msg
    s += '<br>'*2
    if cont:
        s += '<center><a class="boldusercontrol" href="%s"><font size=+1>Continue</font></a></center>'%cont
    return """
        <html>
        <head>
           <link rel=stylesheet href="/css/main.css">
           <title>The Sage Notebook</title>
        </head>
        <body>
        %s
        </body>
    """%s

############################
# Create a Sage worksheet from a latex2html'd file
############################
from docHTMLProcessor import DocHTMLProcessor

doc_worksheet_number = 0
def doc_worksheet():
    global doc_worksheet_number
    wnames = notebook.worksheet_names()
    name = 'doc_browser_%s'%doc_worksheet_number
    doc_worksheet_number = doc_worksheet_number % notebook.conf()['doc_pool_size']
    if name in wnames:
        W = notebook.get_worksheet_with_name(name)
        W.clear()
    else:
        W = notebook.create_new_worksheet(name, '_sage_', docbrowser=True)
    W.set_is_doc_worksheet(True)
    return W


class WorksheetFile(resource.Resource):
    addSlash = False

    def __init__(self, path, username):
        self.docpath = path
        self.username = username

    def render(self, ctx=None):
        # Create a live Sage worksheet out of self.path and render it.
        doc_page_html = open(self.docpath).read()
        directory = os.path.split(self.docpath)[0]
        doc_page, css_href = DocHTMLProcessor().process_doc_html(DOC,
                               directory, doc_page_html)
        # The empty hideall cell below is specifically because for some mysterious
        # reason if we don't put it there, then the first cell -- when evaluated,
        # causes a jump to the bottom of the worksheet.  With the hideall there,
        # verything works perfectly.
        doc_page = extract_title(doc_page_html) + '\nsystem:sage\n{{{%hideall\n}}}\n' + doc_page
        if css_href:
            css_href = DOC + directory + css_href
        W = doc_worksheet()
        W.edit_save(doc_page)
        s = notebook.html(worksheet_filename = W.filename(),
                          username = self.username)
        return http.Response(stream=s)

    def childFactory(self, request, name):
        path = self.docpath + '/' + name
        if name.endswith('.html'):
            return WorksheetFile(path, self.username)
        else:
            return static.File(path)

############################
# The documentation browsers
############################

DOC = os.path.abspath(os.environ['SAGE_ROOT'] + '/doc/')

class DocStatic(resource.Resource):
    addSlash = True
    def render(self, ctx):
        return static.File('%s/index.html'%DOC)

    def childFactory(self, request, name):
        return static.File('%s/%s'%(DOC,name))

class DocLive(resource.Resource):
    addSlash = True

    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        return http.Response(stream=message('nothing to see.'))

    def childFactory(self, request, name):
        return WorksheetFile('%s/%s'%(DOC,name), username = self.username)

class Doc(resource.Resource):
    addSlash = True
    child_static = DocStatic()

    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        s = notebook.html_doc(username = self.username)
        return http.Response(stream=s)

    def childFactory(self, request, name):
        if name == "live":
            return DocLive(username = self.username)


############################
# SageTex browser
############################
SAGETEX_PATH = ""


class SageTex(resource.Resource):
    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        s = notebook.html_doc(username = self.username)
        return http.Response(stream=s)

    def childFactory(self, request, name):
        return WorksheetFile('%s/%s'%(SAGETEX_PATH,name),
                             username = self.username)


############################
# The source code browser
############################

SRC = os.path.abspath(os.environ['SAGE_ROOT'] + '/devel/sage/sage/')

class SourceBrowser(resource.Resource):
    addSlash = True

    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        return static.File(SRC)

    def childFactory(self, request, name):
        return Source('%s/%s'%(SRC,name), self.username)

class Source(resource.Resource):
    addSlash = True

    def __init__(self, path, username):
        self.path = path
        self.username = username

    def render(self, ctx):
        if os.path.isfile(self.path):
            return http.Response(stream = notebook.html_src(self.path, self.username))
        else:
            return static.File(self.path)

    def childFactory(self, request, name):
        return Source(self.path + '/' + name, self.username)


############################
# A New Worksheet
############################
class NewWorksheet(resource.Resource):
    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        W = notebook.create_new_worksheet("Untitled", self.username)
        return http.RedirectResponse('/home/'+W.filename())


############################
# Uploading a saved worksheet file
############################

def redirect(url):
    return '<html><head><meta http-equiv="REFRESH" content="0; URL=%s"></head></html>'%url

class Upload(resource.Resource):
    def render(self, ctx):
        return http.Response(stream = notebook.upload_window())

class UploadWorksheet(resource.PostableResource):
    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        url = ctx.args['urlField'][0].strip()
        dir = ''  # we will delete the directory below if it is used
        if url != '':
            # downloading a file from the internet
            filename = get_remote_file(url, verbose=True)
        else:
            # uploading a file from the user's computer
            dir = tmp_dir()
            filename = ctx.files['fileField'][0][0]
            # Make tmp file in SAGE temp directory
            filename = '%s/%s'%(dir, filename)
            f = file(filename,'wb')
            # Then download to that file.
            f.write(ctx.files['fileField'][0][2].read())
            # TODO: Server blocking issues (?!)
            f.close()

        try:
            try:
                W = notebook.import_worksheet(filename, self.username)
            except IOError, msg:
                print msg
                raise ValueError, "Unfortunately, there was an error uploading the worksheet.  It could be an old unsupported format or worse.  If you desparately need its contents contact the Google group sage-support and post a link to your worksheet.  Alternatively, an sws file is just a bzip2'd tarball; take a look inside!"
            finally:
                # Clean up the temporarily uploaded filename.
                os.unlink(filename)
                # if a temp directory was created, we delete it now.
                if dir:
                    shutil.rmtree(dir)

        except ValueError, msg:
            s = "Error uploading worksheet '%s'."%msg
            return http.Response(stream = message(s, '/'))

        # If the user requested in the form a specific title for
        # the worksheet set it.
        if ctx.args.has_key('nameField'):
            new_name = ctx.args['nameField'][0].strip()
            if new_name:
                W.set_name(new_name)

        return http.RedirectResponse('/home/'+W.filename())



############################
# A resource attached to a given worksheet.
#
# This has the name of the worksheet and the
# worksheet object itself set as attributes.
# It's much better to do it once-for-all here
# instead of doing it in the derived classes
# over and over again.
############################
class WorksheetResource:
    def __init__(self, name, username):
        self.name = name
        self.username = username
        self.worksheet = notebook.get_worksheet_with_filename(name)
        if not self.worksheet.is_published():
            self.worksheet.set_active(username)
        owner = self.worksheet.owner()
        if owner != '_sage_' and username != owner:
            if not self.worksheet.is_published():
                if not username in self.worksheet.collaborators() and user_type(username) != 'admin':
                    raise RuntimeError, "illegal worksheet access"

    def id(self, ctx):
        return int(ctx.args['id'][0])


###############################################
# Worksheet data -- a file that
# is associated with a cell in some worksheet.
# The file is stored on the filesystem.
#      /home/worksheet_name/data/cell_number/filename
##############################################
class Worksheet_savedatafile(WorksheetResource, resource.PostableResource):
    def render(self, ctx):
        if ctx.args.has_key('button_save'):
            E = ctx.args['textfield'][0]
            filename = ctx.args['filename'][0]
            dest = '%s/%s'%(self.worksheet.data_directory(), filename)
            open(dest,'w').write(E)
        return http.RedirectResponse('/home/'+self.worksheet.filename())

class Worksheet_link_datafile(WorksheetResource, resource.Resource):
    def render(self, ctx):
        target_worksheet_filename = ctx.args['target'][0]
        data_filename = ctx.args['filename'][0]
        src = os.path.abspath(self.worksheet.data_directory() + '/' +data_filename)
        target_ws =  notebook.get_worksheet_with_filename(target_worksheet_filename)
        target = os.path.abspath(target_ws.data_directory() + '/' + data_filename)
        if target_ws.owner() != self.username and not target_ws.user_is_collaborator(self.username):
            return http.Response(stream=message("illegal link attempt!"))
        os.system('ln "%s" "%s"'%(src, target))
        return http.RedirectResponse('/home/'+target_ws.filename() + '/datafile?name=%s'%data_filename)


class Worksheet_upload_data(WorksheetResource, resource.Resource):
    def render(self, ctx):
        return http.Response(stream = notebook.html_upload_data_window(self.worksheet, self.username))

class Worksheet_do_upload_data(WorksheetResource, resource.PostableResource):
    def render(self, ctx):
        name = ''
        if ctx.args.has_key('newField'):
            newfield = ctx.args['newField'][0].strip()
        else:
            newfield = None

        if ctx.args.has_key('nameField'):
            name = ctx.args['nameField'][0].strip()

        url = ctx.args['urlField'][0].strip()

        if not name:
            name = ctx.files['fileField'][0][0]

        if not name:
            name = newfield

        if url and not name:
            name = os.path.split(url)[-1]

        dest = '%s/%s'%(self.worksheet.data_directory(), name)

        if url != '':
            tmp = get_remote_file(url, verbose=True)
            shutil.move(tmp, dest)
        elif newfield:
            open(dest,'w').close()
        else:
            f = file(dest,'wb')
            f.write(ctx.files['fileField'][0][2].read())
            f.close()
        return http.RedirectResponse('/home/'+self.worksheet.filename() + '/datafile?name=%s'%name)


##############################################
# Download or delete a data file
##############################################

class Worksheet_datafile(WorksheetResource, resource.Resource):
    def render(self, ctx):
        dir = os.path.abspath(self.worksheet.data_directory())
        filename = ctx.args['name'][0]
        if ctx.args.has_key('action'):
            if ctx.args['action'][0] == 'delete':
                path = '%s/%s'%(self.worksheet.data_directory(), filename)
                os.unlink(path)
                return http.Response(stream = message("Successfully deleted '%s'"%filename,
                                                      '/home/' + self.worksheet.filename()))
        s = notebook.html_download_or_delete_datafile(self.worksheet, self.username, filename)
        return http.Response(stream=s)

##############################################
# Returns an object in the datafile directory
##############################################
class Worksheet_data(WorksheetResource, resource.Resource):
    addSlash = True

    def render(self, ctx):
        dir = os.path.abspath(self.worksheet.data_directory())
        if os.path.exists(dir):
            return static.File(dir)
        else:
            return http.Response(stream = message("No data files",'..'))

    def childFactory(self, request, name):
        dir = os.path.abspath(self.worksheet.data_directory())
        return static.File('%s/%s'%(dir, name))


class CellData(resource.Resource):
    def __init__(self, worksheet, number):
        self.worksheet = worksheet
        self.number = number

    def render(self, ctx):
        return http.Response(stream = message("No data file (%s)"%self.number,'..'))

    def childFactory(self, request, name):
        dir = self.worksheet.directory()
        path = '%s/cells/%s/%s'%(dir, self.number, name)
        return static.File(path)


class Worksheet_cells(WorksheetResource, resource.Resource):
    addSlash = True

    def render(self, ctx):
        return static.File(self.worksheet.cells_directory())

    def childFactory(self, request, segment):
        return static.File(self.worksheet.cells_directory() + segment)
    #return CellData(self.worksheet, segment)




########################################################
# Use this to wrap a worksheet operation in a confirmation
# request.  See WorksheetDelete and WorksheetAdd for
# examples.
########################################################
## class FastRedirect(resource.Resource):
##     def __init__(self, dest):
##         self.dest = dest
##     def render(self, ctx):
##         return http.RedirectResponse(self.dest)
## class YesNo(resource.Resource):
##     addSlash = True

##     def __init__(self, mesg, yes, no):
##         self.mesg = mesg
##         self.yes = yes
##         self.no  = no

##     def render(self, ctx):
##         from sage.server.notebook.template import yes_no_template
##         lt = yes_no_template(mesg=self.mesg)
##         return http.Response(stream = lt)

##         s = '%s<br>'%self.mesg
##         s += '<a href="yes">Yes</a> or <a href="no">No</a>'
##         return http.Response(stream = message(s))

##     def childFactory(self, request, op):
##         if op == 'yes':
##             return FastRedirect(self.yes())
##         elif op == 'no':
##             return FastRedirect(self.no())


########################################################
# keep alive
########################################################

class Worksheet_alive(WorksheetResource, resource.Resource):
    def render(self, ctx):
        #self.worksheet.ping(self.username)
        self.worksheet.ping(username=None)  # None so doesn't save a revision
        return http.Response(stream = '')

########################################################
# Worksheet configuration.
########################################################
class Worksheet_conf(WorksheetResource, resource.Resource):
    def render(self, ctx):
        conf = self.worksheet.conf()
        s = str(conf)
        # TODO: This should be a form that allows for configuring all options
        # of a given worksheet, saves the result,
        return http.Response(stream = s)

class TrivialResource(resource.Resource):
    def render(self, ctx):
        return http.Response(stream="success")

class Worksheet_system(WorksheetResource, resource.Resource):
    def childFactory(self, request, system):
        self.worksheet.set_system(system)
        return TrivialResource()

class Worksheet_pretty_print(WorksheetResource, resource.Resource):
    def childFactory(self, request, enable):
        self.worksheet.set_pretty_print(enable)
        return TrivialResource()



########################################################
# Cell introspection
########################################################
class Worksheet_introspect(WorksheetResource, resource.PostableResource):
    """
    Cell introspection.  This is called when the user presses the tab
    key in the browser in order to introspect.
    """
    def render(self, ctx):
        try:
            id = int(ctx.args['id'][0])
        except (KeyError,TypeError):
            return http.Response(stream = 'Error in introspection -- invalid cell id.')
        try:
            before_cursor = ctx.args['before_cursor'][0]
        except KeyError:
            before_cursor = ''
        try:
            after_cursor = ctx.args['after_cursor'][0]
        except KeyError:
            after_cursor = ''
        C = self.worksheet.get_cell_with_id(id)
        C.evaluate(introspect=[before_cursor, after_cursor])
        return http.Response(stream = encode_list([C.next_id(),'introspect',id]))

########################################################
# Edit the entire worksheet
########################################################
class Worksheet_edit(WorksheetResource, resource.Resource):
    """
    Return a window that allows the user to edit the text of the
    worksheet with the given filename.
    """
    def render(self, ctx):
        self.worksheet.save_snapshot(self.username)
        s = notebook.html_edit_window(self.worksheet, self.username)
        return http.Response(stream = s)

########################################################
# Plain text log view of worksheet
########################################################
class Worksheet_text(WorksheetResource, resource.Resource):
    """
    Return a window that allows the user to edit the text of the
    worksheet with the given filename.
    """
    def render(self, ctx):
        s = notebook.html_plain_text_window(self.worksheet, self.username)
        return http.Response(stream = s)

########################################################
# Copy a worksheet
########################################################
class Worksheet_copy(WorksheetResource, resource.PostableResource):
    def render(self, ctx):
        W = notebook.copy_worksheet(self.worksheet, self.username)
        return http.RedirectResponse('/home/' + W.filename())

########################################################
# Get a copy of a published worksheet and start editing it
########################################################
class Worksheet_edit_published_page(WorksheetResource, resource.Resource):
    def render(self, ctx):
        if user_type(self.username) == 'guest':
            return http.Response(stream = message(
                'You must <a href="/">login first</a> in order to edit this worksheet.'))
        ws = self.worksheet.worksheet_that_was_published()
        if ws.owner() == self.username:
            W = ws
        else:
            W = notebook.copy_worksheet(self.worksheet, self.username)
            W.set_name(self.worksheet.name())
        return http.RedirectResponse('/home/' + W.filename())

########################################################
# Save a worksheet
########################################################
class Worksheet_save(WorksheetResource, resource.PostableResource):
    """
    Save the contents of a worksheet after editing it in plain-text edit mode.
    """
    def render(self, ctx):
        if ctx.args.has_key('button_save'):
            E = ctx.args['textfield'][0]
            self.worksheet.edit_save(E)
            self.worksheet.record_edit(self.username)
            self.worksheet.save_snapshot(self.username, E)
        return http.RedirectResponse('/home/'+self.worksheet.filename())


class Worksheet_save_snapshot(WorksheetResource, resource.PostableResource):
    """
    Save a snapshot of a worksheet.
    """
    def render(self, ctx):
        self.worksheet.save_snapshot(self.username)
        return http.Response(stream="saved")

class Worksheet_save_and_quit(WorksheetResource, resource.PostableResource):
    """
    Save a snapshot of a worksheet and quit.
    """
    def render(self, ctx):
        self.worksheet.save_snapshot(self.username)
        self.worksheet.quit()
        return http.Response(stream="saved")

class Worksheet_discard_and_quit(WorksheetResource, resource.PostableResource):
    """
    Save a snapshot of a worksheet and quit.
    """
    def render(self, ctx):
        self.worksheet.revert_to_last_saved_state()
        self.worksheet.quit()
        return http.Response(stream="saved")

class Worksheet_revert_to_last_saved_state(WorksheetResource, resource.PostableResource):
    def render(self, ctx):
        self.worksheet.revert_to_last_saved_state()
        return http.Response(stream="reverted")

class Worksheet_save_and_close(WorksheetResource, resource.PostableResource):
    """
    Save a snapshot of a worksheet then quit it.
    """
    def render(self, ctx):
        self.worksheet.save_snapshot(self.username)
        self.worksheet.quit()
        return http.Response(stream="saved")

########################################################
# Collaborate with others
########################################################
class Worksheet_share(WorksheetResource, resource.Resource):
    def render(self, ctx):
        s = notebook.html_share(self.worksheet, self.username)
        return http.Response(stream = s)

class Worksheet_invite_collab(WorksheetResource, resource.PostableResource):
    def render(self, ctx):
        if not ctx.args.has_key('collaborators'):
            v = []
        else:
            collab = ctx.args['collaborators'][0]
            v = [x.strip() for x in collab.split(',')]
        self.worksheet.set_collaborators(v)
        return http.RedirectResponse('.')


#################################
# Revisions
#################################

class PublishWorksheetRevision(resource.Resource):
    def __init__(self, worksheet, rev):
        self.worksheet = worksheet
        self.rev = rev

    def render(self, ctx):
        W = notebook.publish_worksheet(self.worksheet, self.username)
        txt = open(self.worksheet.get_snapshot_text_filename(self.rev)).read()
        W.delete_cells_directory()
        W.edit_save(txt)
        return http.RedirectResponse('/home/'+W.filename())

class RevertToWorksheetRevision(resource.Resource):
    def __init__(self, worksheet, rev):
        self.worksheet = worksheet
        self.rev = rev

    def render(self, ctx):
        self.worksheet.save_snapshot(self.username)
        txt = open(self.worksheet.get_snapshot_text_filename(self.rev)).read()
        self.worksheet.delete_cells_directory()
        self.worksheet.edit_save(txt)
        return http.RedirectResponse('/home/'+self.worksheet.filename())

def worksheet_revision_publish(worksheet, rev, username):
    W = notebook.publish_worksheet(worksheet, username)
    txt = bz2.decompress(open(worksheet.get_snapshot_text_filename(rev)).read())
    W.delete_cells_directory()
    W.edit_save(txt)
    return http.RedirectResponse('/home/'+W.filename())

def worksheet_revision_revert(worksheet, rev, username):
    worksheet.save_snapshot(username)
    txt = bz2.decompress(open(worksheet.get_snapshot_text_filename(rev)).read())
    worksheet.delete_cells_directory()
    worksheet.edit_save(txt)
    return http.RedirectResponse('/home/'+worksheet.filename())


class Worksheet_revisions(WorksheetResource, resource.PostableResource):
    """
    Show a list of revisions of this worksheet.
    """
    def render(self, ctx):
        if not ctx.args.has_key('action'):
            if ctx.args.has_key('rev'):
                rev = ctx.args['rev'][0]
                s = notebook.html_specific_revision(self.username, self.worksheet, rev)
            else:
                s = notebook.html_worksheet_revision_list(self.username, self.worksheet)
        else:
            rev = ctx.args['rev'][0]
            action = ctx.args['action'][0]
            if action == 'revert':
                return worksheet_revision_revert(self.worksheet, rev, self.username)
            elif action == 'publish':
                return worksheet_revision_publish(self.worksheet, rev, self.username)
            else:
                s = message('Error')
        return http.Response(stream = s)


########################################################
# Worksheet/User/Notebooks settings and configuration
########################################################

class Worksheet_input_settings(WorksheetResource, resource.PostableResource):
    def render(self, ctx):
        if ctx.args.has_key('button_cancel'):
            return http.RedirectResponse('/home/'+self.worksheet.filename())
        if user_type(self.username) == 'admin' or \
               self.worksheet.owner() == self.username \
               or self.worksheet.user_is_collaborator(self.username):
            system = ctx.args['system'][0].strip().lower()
            self.worksheet.set_system(system)
            if system != 'sage':
                post = ' (%s)'%system
                n = self.worksheet.name()
                i = n.rfind('(')
                if i != -1:
                    j = n.rfind(')')
                    if j != -1:
                        n = n[:i]
                n = n.strip() + post
                self.worksheet.set_name(n)
            return http.RedirectResponse('/home/'+ self.worksheet.filename())
        else:
            s = 'You must be the owner of this worksheet to configure it.'
            return http.Response(stream = message(s))

class Worksheet_settings(WorksheetResource, resource.Resource):
    def render(self, ctx):
        if self.worksheet.owner() != self.username:
            s = message('You must be the owner of this worksheet to configure it.')
        else:
            s = notebook.html_worksheet_settings(self.worksheet, self.username)
        return http.Response(stream = s)

class ProcessUserSettings(resource.PostableResource):
    def render(self, ctx):
        pass

#class UserSettings(resource.Resource):
#    child_process = ProcessUserSettings()
#
#    def __init__(self, username):
#        self.username = username
#
#    def render(self, ctx):
#        s = notebook.html_user_settings(self.username)
#        return http.Response(stream = s)

class ProcessNotebookSettings(resource.PostableResource):
    def render(self, ctx):
        pass

class NotebookSettings(resource.Resource):
    child_process = ProcessNotebookSettings()

    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        if user_type(self.username) != 'admin':
            s = message('You must an admin to configure the notebook.')
        else:
            s = notebook.html_notebook_settings()
        return http.Response(stream = s)

class SettingsPage(resource.PostableResource):
    def __init__(self, username):
        self.username = username

    def render(self, request):
        error = None
        if 'Oldpass' in request.args or 'Newpass' in request.args or 'RetypePass' in request.args:
            if not 'Oldpass' in request.args:
                error = 'Old password not given'
            elif not notebook.user(self.username).password_is(request.args['Oldpass'][0]):
                error = 'Incorrect password given'
            elif not 'Newpass' in request.args:
                error = 'New password not given'
            elif not 'RetypePass' in request.args:
                error = 'Please type in new password again.'
            elif request.args['Newpass'][0] != request.args['RetypePass'][0]:
                error = 'The passwords you entered do not match.'

            if error:
                return http.Response(stream=message(error, '/settings'))

            notebook.change_password(self.username, request.args['Newpass'][0])
            return http.RedirectResponse('/logout')

        if 'Newemail' in request.args:
            notebook.user(self.username).set_email(request.args['Newemail'][0])
            return http.RedirectResponse('/settings')

        else:
            s = """<html><title>Account Settings</title><h1 align=center>Account Settings</h1>
            <br>
            <hr>
            <br>
            <form method="POST" action="/settings">
            <br><br>
            <table align=center><tr>
            <td colspan=2><h2>Change Password</h2></td></tr><tr>
            <td align=right>Old password:</td><td><input type="password" name="Oldpass" size="15" /></td></tr>
            <tr><td align=right>New password:</td><td>
                <input type="password" name="Newpass" size="15" />
                </td></tr>
            <tr><td align=right>Retype new password:</td><td>
                <input type="password" name="RetypePass" size="15" />
                </td></tr>
          <tr><td></td><td></td></tr>
            <tr><td></td><td align=left><input type="submit" value="Change password" /></td></tr>
            <tr style="height:20px"><td colspan=2></td></tr>
            <tr>
            <td colspan=2><h2>Change E-mail Address</h2></td></tr><tr><tr>
            <td align=right>Current e-mail:</td><td>%s</td></tr>
            <tr><td align=right>New e-mail:</td><td>
                <input type="text" name="Newemail" size="30" />
                </td></tr><tr><td></td><td align=left><input type="submit" value="Change e-mail" /></td></tr>
            </table> </form>
            <br><br>
            <div align=center><a href="../">Cancel</a></div>
            <br>

            </html>""" % notebook.user(self.username)._User__email
        return http.Response(stream=s)

########################################################
# Set output type of a cell
########################################################
class Worksheet_set_cell_output_type(WorksheetResource, resource.PostableResource):
    """
    Set the output type of the cell.

    This enables the type of output cell, such as to allowing wrapping
    for output that is very long.
    """
    def render(self, ctx):
        id = self.id(ctx)
        typ = ctx.args['type'][0]
        W = self.worksheet
        W.get_cell_with_id(id).set_cell_output_type(typ)
        return http.Response(stream = '')

########################################################
# The new cell command: /home/worksheet/new_cell?id=number
########################################################
class Worksheet_new_cell_before(WorksheetResource, resource.PostableResource):
    """
    Adds a new cell before a given cell.
    """
    def render(self, ctx):
        id = self.id(ctx)
        if not ctx.args.has_key('input'):
            input = ''
        else:
            input = ctx.args['input'][0]
        cell = self.worksheet.new_cell_before(id, input=input)
        s = encode_list([cell.id(), cell.html(div_wrap=False), id])
        return http.Response(stream = s)

class Worksheet_new_cell_after(WorksheetResource, resource.PostableResource):
    """
    Adds a new cell after a given cell.
    """
    def render(self, ctx):
        id = self.id(ctx)
        if not ctx.args.has_key('input'):
            input = ''
        else:
            input = ctx.args['input'][0]
        cell = self.worksheet.new_cell_after(id, input=input)
        s = encode_list([cell.id(), cell.html(div_wrap=False), id])
        return http.Response(stream = s)


########################################################
# The delete cell command: /home/worksheet/delete_cell?id=number
########################################################
class Worksheet_delete_cell(WorksheetResource, resource.PostableResource):
    """
    Deletes a notebook cell.

    If there is only one cell left in a given worksheet, the request
    to delete that cell is ignored because there must be a least one
    cell at all times in a worksheet.  (This requirement exists so
    other functions that evaluate relative to existing cells will
    still work, and so one can add new cells.)
    """
    def render(self, ctx):
        id = self.id(ctx)
        W = self.worksheet
        if len(W) <= 1:
            s = 'ignore'
        else:
            prev_id = W.delete_cell_with_id(id)
            s = encode_list(['delete', id, prev_id, W.cell_id_list()])
        return http.Response(stream = s)


############################
# Get the latest update on output appearing
# in a given output cell.
############################
class Worksheet_cell_update(WorksheetResource, resource.PostableResource):
    def render(self, ctx):
        id = self.id(ctx)

        worksheet = self.worksheet

        # update the computation one "step".
        worksheet.check_comp()

        # now get latest status on our cell
        status, cell = worksheet.check_cell(id)

        if status == 'd':
            new_input = cell.changed_input_text()
            out_html = cell.output_html()
            H = "Worksheet '%s' (%s)\n"%(worksheet.name(), time.strftime("%Y-%m-%d at %H:%M",time.localtime(time.time())))
            H += cell.edit_text(ncols=HISTORY_NCOLS, prompts=False,
                                max_out=HISTORY_MAX_OUTPUT)
            notebook.add_to_user_history(H, self.username)
        else:
            new_input = ''
            out_html = ''

        if cell.interrupted():
            inter = 'true'
        else:
            inter = 'false'

        raw = cell.output_text(raw=True).split("\n")
        if "Unhandled SIGSEGV" in raw:
            inter = 'restart'
            print "Segmentation fault detected in output!"

        msg = '%s%s %s'%(status, cell.id(),
                       encode_list([cell.output_text(html=True),
                                    cell.output_text(word_wrap_cols(), html=True),
                                    out_html,
                                    new_input,
                                    inter,
                                    cell.introspect_html()]))

        # There may be more computations left to do, so start one if there is one.
        worksheet.start_next_comp()

        return http.Response(stream=msg)


class Worksheet_eval(WorksheetResource, resource.PostableResource):
    """
    Evaluate a worksheet cell.

    If the request is not authorized (the requester did not enter the
    correct password for the given worksheet), then the request to
    evaluate or introspect the cell is ignored.

    If the cell contains either 1 or 2 question marks at the end (not
    on a comment line), then this is interpreted as a request for
    either introspection to the documentation of the function, or the
    documentation of the function and the source code of the function
    respectively.
    """
    def render(self, ctx):
        id = self.id(ctx)
        if not ctx.args.has_key('input'):
            input_text = ''
        else:
            input_text = ctx.args['input'][0]
            input_text = input_text.replace('\r\n', '\n')   # DOS

        W = self.worksheet
        owner = W.owner()
        if owner != '_sage_':
            if W.owner() != self.username and not (self.username in W.collaborators()):
               return InvalidPage(msg = "can't evaluate worksheet cells", username = self.username)
        cell = W.get_cell_with_id(id)

        cell.set_input_text(input_text)

        if ctx.args.has_key('save_only') and ctx.args['save_only'][0] == '1':
            notebook_updates()
            return http.Response(stream='')
        else:
            newcell = int(ctx.args['newcell'][0])  # whether to insert a new cell or not

        cell.evaluate(username = self.username)

        if cell.is_last():
            new_cell = W.append_new_cell()
            s = encode_list([new_cell.id(), 'append_new_cell', new_cell.html(div_wrap=False)])
        elif newcell:
            new_cell = W.new_cell_after(id)
            s = encode_list([new_cell.id(), 'insert_cell', new_cell.html(div_wrap=False), str(id)])
        else:
            s = encode_list([cell.next_id(), 'no_new_cell', str(id)])

        notebook_updates()
        return http.Response(stream=s)


########################################################
# Publication and rating of a worksheet
########################################################

class Worksheet_publish(WorksheetResource, resource.Resource):
    def render(self, ctx):
        W = notebook.publish_worksheet(self.worksheet, self.username)
        addr = '/home/' + W.filename()
        return http.RedirectResponse(addr)


class Worksheet_rating_info(WorksheetResource, resource.Resource):
    def render(self, ctx):
        s = self.worksheet.html_ratings_info()
        return http.Response(stream=message("""
        <h2 align=center>Ratings for %s</h2>
        <h3 align=center><a href='/home/%s'>Go to the worksheet.</a>
        <br><br>
        <table width=70%%align=center border=1 cellpadding=10 cellspacing=0>
        <tr bgcolor="#7799bb"><td width=30em>User</td><td width=10em align=center>Rating</td><td width=10em align=center width=60em>Comment</td></tr>
        %s
        </table>
        <br><br>
        """%(self.worksheet.name(), self.worksheet.filename(), s)))


class Worksheet_rate(WorksheetResource, resource.Resource):
    def render(self, ctx):
        ret = '/home/' + self.worksheet.filename()
        #if self.worksheet.is_rater(self.username):
        #    return http.Response(stream=message("You have already rated the worksheet <i><b>%s</b></i>."%self.worksheet.name(), ret))
        if user_type(self.username) == "guest":
            return http.Response(stream = message(
                'You must <a href="/">login first</a> in order to rate this worksheet.', ret))

        rating = int(ctx.args['rating'][0])
        if rating < 0 or rating >= 5:
            return http.Response(stream = message(
                "Gees -- You can't fool the rating system that easily!", ret))
        comment = ctx.args['comment'][0]
        self.worksheet.rate(rating, comment, self.username)
        return http.Response(stream=message("""
        Thank you for rating the worksheet <b><i>%s</i></b>!
        You can <a href="rating_info">see all ratings of this worksheet.</a>
        """%self.worksheet.name(), '/pub/'))


########################################################
# Downloading, moving around, renaming, etc.
########################################################


class Worksheet_download(WorksheetResource, resource.Resource):
    def childFactory(self, request, name):
        worksheet_name = self.name
        filename = tmp_filename() + '.sws'
        try:
            notebook.export_worksheet(worksheet_name, filename)
        except KeyError:
            return http.Response(stream=message('No such worksheet.'))
        r = open(filename, 'rb').read()
        os.unlink(filename)
        return static.Data(r, 'application/sage')
        #return static.File(filename)

class Worksheet_rename(WorksheetResource, resource.PostableResource):
    def render(self, ctx):
        self.worksheet.set_name(ctx.args['name'][0])
        return http.Response(stream='done')

class Worksheet_restart_sage(WorksheetResource, resource.Resource):
    def render(self, ctx):
        # TODO -- this must not block long (!)
        self.worksheet.restart_sage()
        return http.Response(stream='done')

class Worksheet_quit_sage(WorksheetResource, resource.Resource):
    def render(self, ctx):
        # TODO -- this must not block long (!)
        self.worksheet.quit()
        return http.Response(stream='done')

class Worksheet_interrupt(WorksheetResource, resource.Resource):
    def render(self, ctx):
        # TODO -- this must not block long (!)
        s = self.worksheet.interrupt()
        return http.Response(stream='ok' if s else 'failed')

class Worksheet_plain(WorksheetResource, resource.Resource):
    def render(self, ctx):
        s = notebook.plain_text_worksheet_html(self.name)
        return http.Response(stream=s)

class Worksheet_hide_all(WorksheetResource, resource.Resource):
    def render(self, ctx):
        self.worksheet.hide_all()
        return http.Response(stream='success')

class Worksheet_show_all(WorksheetResource, resource.Resource):
    def render(self, ctx):
        self.worksheet.show_all()
        return http.Response(stream='success')


# Delete all the output of cells in a worksheet.
class Worksheet_delete_all_output(WorksheetResource, resource.Resource):
    def render(self, ctx):
        try:
            self.worksheet.delete_all_output(self.username)
        except ValueError:
            return http.Response(stream='fail')
        return http.Response(stream='success')

class Worksheet_print(WorksheetResource, resource.Resource):
    def render(self, ctx):
        s = notebook.worksheet_html(self.name, do_print=True)
        return http.Response(stream=s)


class NotImplementedWorksheetOp(resource.Resource):
    def __init__(self, op, ws):
        self.op = op
        self.ws = ws

    def render(self, ctx):
        return http.Response(stream = message(
            'The worksheet operation "%s" is not defined.'%self.op,
            '/home/'+self.ws.filename()))


class Worksheet(WorksheetResource, resource.Resource):
    addSlash = True

    def render(self, ctx):
        s = notebook.html(worksheet_filename = self.name,  username=self.username)
        self.worksheet.sage()
        return http.Response(stream=s)

    def childFactory(self, request, op):
        notebook_updates()
        try:
            # Rather than a bunch of if-else statements, we wrap
            # any Worksheet_... class as a subresource of a worksheet
            # using the following  statement:
            R = globals()['Worksheet_%s'%op]
            return R(self.name, username = self.username)
        except KeyError:
            file = self.worksheet.data_directory() + '/' + op
            if os.path.exists(file):
                return static.File(file)
            dir = self.worksheet.cells_directory()
            for F in os.listdir(dir):
                h = '%s/%s/%s'%(dir,F,op)
                if os.path.exists(h):
                    return static.File(h)
            return NotImplementedWorksheetOp(op, self.worksheet)

def render_worksheet_list(args, pub, username):
    if args.has_key('typ'):
        typ = args['typ'][0]
    else:
        typ = 'active'
    if args.has_key('search'):
        search = args['search'][0]
    else:
        search = None
    if not args.has_key('sort'):
        sort = 'last_edited'
    else:
        sort = args['sort'][0]
    if args.has_key('reverse'):
        reverse = (args['reverse'][0] == 'True')
    else:
        reverse = False

    if pub:
        if username is None or username == tuple([]):
            user = 'pub'
        else:
            user = username
        s = notebook.html_worksheet_list_public(
            user, sort=sort, reverse=reverse, search=search)
    else:
        s = notebook.html_worksheet_list_for_user(
            username, typ=typ, sort=sort, reverse=reverse, search=search)
    return s


class WorksheetsByUser(resource.Resource):
    addSlash = True

    def __init__(self, user, username):
        # user -- who we're requesting the worksheets of
        # username -- who is doing the requesting
        self.user = user
        self.username = username

    def render_list(self, ctx):
        s = render_worksheet_list(ctx.args, pub=False, username=self.username)
        return http.Response(stream = s)

    def render(self, ctx):
        if self.user == self.username:
            return self.render_list(ctx)
        else:
            s = message("User '%s' does not have permission to view the home page of '%s'."%(self.username, self.user))
            return http.Response(stream = s)

    def childFactory(self, request, name):
        if name == "trash":
            return TrashCan(self.user)

        filename = self.user + '/' + name
        try:
            return Worksheet(filename, self.username)
        except KeyError:
            s = "The user '%s' has no worksheet '%s'."%(self.user, name)
            return InvalidPage(msg = s, username = self.username)
        except RuntimeError:
            s = "You are not logged in or do not have access to the worksheet '%s'."%name
            return InvalidPage(msg = s, username = self.username)



############################
# Trash can, archive and active
############################
class EmptyTrash(resource.Resource):
    def __init__(self, username):
        """
        This twisted resource empties the trash of the current user when it
        is rendered.

        EXAMPLES:
        We create an instance of this resource.
            sage: E = sage.server.notebook.twist.EmptyTrash('sage'); E
            <sage.server.notebook.twist.EmptyTrash object at ...>
        """
        self.username = username

    def render(self, ctx):
        """
        Rendering this resource (1) empties the trash, and (2) returns
        a message.

        EXAMPLES:
        We create a notebook with a worksheet, put it in the trash,
        then empty the trash by creating and rendering this worksheet.
            sage: n = sage.server.notebook.notebook.Notebook('notebook-test')
            sage: n.add_user('sage','sage','sage@sagemath.org',force=True)
            sage: W = n.new_worksheet_with_title_from_text('Sage', owner='sage')
            sage: W.move_to_trash('sage')
            sage: n.worksheet_names()
            ['sage/0']
            sage: sage.server.notebook.twist.notebook = n
            sage: E = sage.server.notebook.twist.EmptyTrash('sage'); E
            <sage.server.notebook.twist.EmptyTrash object at ...>
            sage: E.render(None)
            <twisted.web2.http.Response code=200, streamlen=603>

        Finally we verify that the trashed worksheet is gone:
            sage: n.worksheet_names()
            []
            sage: n.delete()
        """
        notebook.empty_trash(self.username)
        return http.Response(stream = message("Trash emptied."))

class SendWorksheetToFolder(resource.PostableResource):
    def __init__(self, username):
        self.username = username

    def action(self, W):
        raise NotImplementedError

    def render(self, ctx):
        X = notebook.user(self.username)
        if user_type(self.username) == 'guest':
            return http.Response(stream = message("You are not authorized to move '%s'"%W.name()))

        def send_worksheet_to_folder(filename):
            W = notebook.get_worksheet_with_filename(filename)
            self.action(W)

        if ctx.args.has_key('filename'):
            filenames = [ctx.args['filename'][0]]
        elif ctx.args.has_key('filenames'):
            sep = ctx.args['sep'][0]
            filenames = [x for x in ctx.args['filenames'][0].split(sep) if len(x.strip()) > 0]

        else:

            filenames = []

        for F in filenames:
            send_worksheet_to_folder(F)

        return http.Response(stream = '')

class SendWorksheetToTrash(SendWorksheetToFolder):
    def action(self, W):
        W.move_to_trash(self.username)

class SendWorksheetToArchive(SendWorksheetToFolder):
    def action(self, W):
        W.move_to_archive(self.username)

class SendWorksheetToActive(SendWorksheetToFolder):
    def action(self, W):
        W.set_active(self.username)

# Using SendWorksheet does feel somewhat hackish.  It however is
# exactly the right thing to actually do, and minimizes code
# duplication.
class SendWorksheetToStop(SendWorksheetToFolder):
    """
    Saves and quits each selected worksheet.
    """
    def action(self, W):
        W.save_snapshot(self.username)
        W.quit()

############################
# Publically Available Worksheets
############################
class PublicWorksheets(resource.Resource):
    addSlash = True

    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        s = render_worksheet_list(ctx.args, pub=True, username=self.username)
        return http.Response(stream = s)

    def childFactory(self, request, name):
        return Worksheet('pub/' + name, username=self.username)

class PublicWorksheetsHome(resource.Resource):
    addSlash = True

    def __init__(self, username):
        self.username = username

    def childFactor(self, request, name):
        if name == 'pub':
            return PublicWorksheets(self.username)

############################
# Resource that gives access to worksheets
############################

class Worksheets(resource.Resource):
    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        return http.Response(stream = message("Please request a specific worksheet"))

    def childFactory(self, request, name):
        return WorksheetsByUser(name, username=self.username)


class WorksheetsByUserAdmin(WorksheetsByUser):
    def render(self, ctx):
        return self.render_list(ctx)

class WorksheetsAdmin(Worksheets):
    def childFactory(self, request, name):
        return WorksheetsByUserAdmin(name, username=self.username)

############################
# Notebook configuration
############################

class NotebookConf(Worksheets):
    def render(self, ctx):
        s = '<html>' + notebook.conf().html_conf_form('submit') + '</html>'
        return http.Response(stream = s)



############################
# Adding a new worksheet
############################

class AddWorksheet(resource.Resource):
    def render(self, ctx):
        name = ctx.args['name'][0]
        W = notebook.create_new_worksheet(name)
        v = notebook.worksheet_list_html(W.name())
        return http.Response(stream = encode_list([v, W.name()]))


############################

class Help(resource.Resource):
    addSlash = True
    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        s = notebook.html_notebook_help_window(self.username)
        return http.Response(stream=s)


############################

############################

class History(resource.Resource):
    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        s = notebook.user_history_html(self.username)
        return http.Response(stream=s)

class LiveHistory(resource.Resource):
    def __init__(self, username):
        self.username = username

    def render(self, ctx):
        W = notebook.create_new_worksheet_from_history('Log', self.username, 100)
        return http.RedirectResponse('/home/'+W.filename())


############################

class Main_css(resource.Resource):
    def render(self, ctx):
        s = css.css()
        return http.Response(stream=s)

class CSS(resource.Resource):
    addSlash = True

    def render(self, ctx):
        return static.File(css_path)

    def childFactory(self, request, name):
        return static.File(css_path + "/" + name)

setattr(CSS, 'child_main.css', Main_css())

############################


############################
# Javascript resources
############################

class Main_js(resource.Resource):
    def render(self, ctx):
        s = js.javascript()
        return http.Response(stream=s)

class Keyboard_js_specific(resource.Resource):
    def __init__(self, browser_os):
        self.s = keyboards.get_keyboard(browser_os)

    def render(self, ctx):
        return http.Response(stream = self.s)


class Keyboard_js(resource.Resource):
    def childFactory(self, request, browser_os):
        return Keyboard_js_specific(browser_os)

class Javascript(resource.Resource):
    addSlash = True
    child_keyboard = Keyboard_js()

    def render(self, ctx):
        return static.File(javascript_path)

    def childFactory(self, request, name):
        return static.File(javascript_path + "/" + name)

setattr(Javascript, 'child_main.js', Main_js())


############################
# Java resources
############################

class Java(resource.Resource):
    addSlash = True

    def render(self, ctx):
        return static.File(java_path)

    def childFactory(self, request, name):
        return static.File(java_path + "/" + name)

############################
# Slide resources
############################

class Slider(resource.Resource):
    addSlash = True

    def render(self, ctx):
        return static.File(slider_path)

    def childFactory(self, request, name):
        return static.File(slider_path + "/" + name)


############################
# Logout
############################
class Logout(resource.Resource):
    def render(self, ctx):
        # TODO -- actually log out.
        notebook.save()
        s = message("<br>Thank you for using Sage.<br><br><a href='/'>Please login and use Sage again soon.</a><br>")
        return http.Response(stream=s)

############################
# Image resource
############################

class Images(resource.Resource):
    addSlash = True

    def render(self, ctx):
        return static.File(image_path)

    def childFactory(self, request, name):
        return static.File(image_path + "/" + name)

#####################################
# Confirmation of registration
####################################
class RegConfirmation(resource.Resource):
    def render(self, request):
        key = request.args['key'][0]
        global notebook
        invalid_confirm_key = """\
<h1>Invalid confirmation key</h1>
<p>You are reporting a confirmation key that has not been assigned by this
server. Please <a href="/register">register</a> with the server.</p>
"""
        key = int(key)
        global waiting
        try:
            username = waiting[key]
        except KeyError:
            return http.Response(stream=message(invalid_confirm_key, '/register'))
        success = """<h1>Hello, %s. Thank you for registering!</h1>""" % username
        del waiting[key]
        return http.Response(stream=message(success))

############################
# Registration page
############################
import re
re_valid_username = re.compile('[a-z|A-Z|0-9|_|\-|.]*')
def is_valid_username(username):
    r"""
    Return True if and only if \var{username} is valid, i.e., contains
    only alphabetic characters, numbers, and underscores.

    EXAMPLES:
        sage: from sage.server.notebook.twist import is_valid_username
        sage: is_valid_username('sage-devel')
        True
        sage: is_valid_username('sage_devel7')
        True
        sage: is_valid_username('a@b.c')
        False
        sage: is_valid_username('ab c')
        False
    """
    m = re_valid_username.match(username)
    return len(username) > 0 and m.start() == 0 and m.end() == len(username)

class RegistrationPage(resource.PostableResource):
    def __init__(self, userdb):
        self.userdb = userdb

    def render(self, request):
        if request.args.has_key('email'):
            if request.args['email'][0] is not None:

                s = ''
                try:
                    username = request.args['username'][0]
                    if not is_valid_username(username):
                        s += "Usernames can only contain letters, numbers, dashes, periods and underscores."
                except KeyError:
                    s += "You must specify a username."
                try:
                    passwd  = request.args['password'][0]
                except KeyError:
                    s += "  You must specify a password."
                else:
                    if len(passwd) == 0:
                        s = "  Password must be nonempty."
                if s:
                    return http.Response(stream=message(s, '/register'))


                destaddr = """%s""" % request.args['email'][0]
                from sage.server.notebook.smtpsend import send_mail
                from sage.server.notebook.register import make_key, build_msg
                # TODO: make this come from the server settings
                key = make_key()
                listenaddr = notebook.address
                port = notebook.port
                fromaddr = 'no-reply@%s' % listenaddr
                body = build_msg(key, username, listenaddr, port,
                                 notebook.secure)

                # Send a confirmation message to the user.
                try:
                    send_mail(self, fromaddr, destaddr, "Sage Notebook Registration",body)
                except ValueError:
                    # the email address is invalid
                    s = message("Registration failed -- the email address '%s' is invalid."%destaddr,
                                 '/register')
                    return http.Response(stream=s)

                # Store in memory that we are waiting for the user to respond
                # to their invitation to join the Sage notebook.
                waiting[key] = username

            # Add the user to passwords.txt
            try:
                self.userdb.add_user(username, passwd, destaddr)
                # now say that the user has been registered.
                s = """
                <html>
                <h1>Registration information received</h1>
                <p>Thank you for registering with the Sage notebook. A
                confirmation message will be sent to %s.</p>
                <br>
                <p><a href="/">Click here to login with your new account.</a></p>
                </html>
                """%destaddr
            except ValueError:
                s = """
                <html>
                <h1>Username is already taken, please choose another one.</h1>
                </html>
                """
        else:
            s = """<html><h1 align=center>Sign up for the Sage Notebook.</h1>
            <br>
            <hr>
            <br>
            <form method="POST" action="/register">
            <br><br>
            <table align=center><tr>
            <td align=right>Username:</td><td><input type="text" name="username" size="15" /></td></tr>
            <tr><td align=right>Password:</td><td>
                <input type="password" name="password" size="15" />
                </td></tr>
            <tr><td align=right>Email
                Address:</td> <td><input type="text" name="email" size="15" />
                </td></tr>
          <tr><td></td><td></td></tr>
            <tr><td></td><td align=left><input type="submit" value="Register Now" /></td></tr>
            </table> </form>
            <br><br>
            <div align=center><a href="/">Cancel and return to the login page</a></div>
            <br>

            </html>"""
        return http.Response(stream=s)

class InvalidPage(resource.Resource):
    addSlash = True

    def __init__(self, msg, username):
        self.msg = msg
        self.username = username

    def render(self, ctx):
        if self.msg:
            s = self.msg
        else:
            s = "This is an invalid page."
            if self.username == 'guest':
                s += ' You might have to login to view this page.'
        return http.Response(stream = message(s, '/'))

    def childFactory(self, request, name):
        return InvalidPage(msg = self.msg, username = self.username)


class RedirectLogin(resource.PostableResource):
    def render(self, ctx):
        return http.RedirectResponse('/')
    def childFactory(self, request, name):
        return RedirectLogin()

import sage.server.simple.twist

class Toplevel(resource.PostableResource):
    child_logout = Logout()
    child_login = RedirectLogin()
    child_simple = sage.server.simple.twist.SimpleServer()

    def __init__(self, cookie, username):
        self.cookie = cookie
        self.username = username if username else 'guest'

    def render(self, ctx):
        return http.Response(stream =  login_page_template(notebook.get_accounts(), notebook.default_user()))

    def userchildFactory(self, request, name):
        return InvalidPage(msg = "unauthorized request", username = self.username)

    def childFactory(self, request, name):
        return self.userchildFactory(request, name)

setattr(Toplevel, 'child_favicon.ico', static.File(image_path + '/favicon.ico'))



from sage.server.notebook.template import login_page_template

from sage.server.notebook.template import failed_login_template

class LoginResourceClass(resource.Resource):
    def render(self, ctx):
        return http.Response(stream =  login_page_template(notebook.get_accounts(), notebook.default_user()))

    def childFactory(self, request, name):
        return LoginResource

LoginResource = LoginResourceClass()

class AnonymousToplevel(Toplevel):
    from sage.server.notebook.avatars import PasswordChecker
    addSlash = True
    child_register = RegistrationPage(PasswordChecker())
    child_confirm = RegConfirmation()

    child_images = Images()
    child_css = CSS()
    child_javascript = Javascript()
    child_java = Java()
    child_slider = Slider()

    def userchildFactory(self, request, name):
        # This is called from Toplevel above
        try:
            return AnonymousToplevel.__dict__['userchild_%s'%name](username = self.username)
        except KeyError:
            pass

    userchild_home = Worksheets
    userchild_pub = PublicWorksheets
    userchild_src = SourceBrowser

    #child_login = LoginResource

    def render(self, ctx):
        return http.Response(stream =  login_page_template(notebook.get_accounts(), notebook.default_user()))

class FailedToplevel(Toplevel):
    def __init__(self, info, problem):
        self.info = info
        self.problem= problem

    def render(self, ctx):
        # Since public access is allowed, which lists usernames in the published
        # worksheets and ratings, this gives no new information way.
        # If published pages were disabled, then this should be disabled too.
        if self.problem == 'username':
            notebook.valid_login_names().sort()
            valid_login_names = "<strong>Valid login names:</strong><br />" + ', <br />'.join(notebook.valid_login_names())
        else:
            valid_login_names = ''
        return http.Response(stream = failed_login_template(problem=self.problem,
                                                            logins = valid_login_names))


class UserToplevel(Toplevel):
    addSlash = True

    child_images = Images()
    child_css = CSS()
    child_javascript = Javascript()
    child_java = Java()
    child_slider = Slider()

    child_upload = Upload()
    child_logout = Logout()



    #child_login = RedirectLogin()

    # userchild_* is like Twisted's child_, etc. except:
    #  (1) it also sets the username in the __init__ method, and
    #  (2) it calls the constructor for the object, i.e., it is
    #      a class rather than an object.
    # NOTE: If you overload childFactory in any derived class, you
    # better call userchildFactory it in the base class (Toplevel)!
    def userchildFactory(self, request, name):
        try:
            return UserToplevel.__dict__['userchild_%s'%name](username = self.username)
        except KeyError:
            pass

    userchild_doc = Doc
    userchild_sagetex = SageTex
    userchild_help = Help
    userchild_history = History
    userchild_home = Worksheets
    userchild_live_history = LiveHistory
    userchild_new_worksheet = NewWorksheet
    userchild_notebook_settings = NotebookSettings
    userchild_settings = SettingsPage
    userchild_pub = PublicWorksheets

    userchild_send_to_trash = SendWorksheetToTrash
    userchild_send_to_archive = SendWorksheetToArchive
    userchild_send_to_active = SendWorksheetToActive
    userchild_send_to_stop = SendWorksheetToStop

    userchild_src = SourceBrowser
    userchild_upload_worksheet = UploadWorksheet
    userchild_emptytrash = EmptyTrash

    def render(self, request):
        # This resource always does a redirect to the user's home directory
        # so that after login (which is a POST operation), the postdata will not remain
        # in the browser on return.  This method is sometimes
        # referred to as the post-redirect-get method.
        response = http.RedirectResponse("/home/%s" % self.username)
        # This allows a Notebook user to select a "remember me" checkbox and not have to
        # sign back in when she restarts her web browser
        # This works by setting an expiration date because without one the browser forgets the cookie.
        if 'remember' in request.args:
            response.headers.setHeader("set-cookie", [http_headers.Cookie('nb_session', self.cookie, expires=(time.time() + 60 * 60 * 24 * 14))])
        else:
            response.headers.setHeader("set-cookie", [http_headers.Cookie('nb_session', self.cookie)])
        return response


class AdminToplevel(UserToplevel):
    userchild_home = WorksheetsAdmin
    userchild_conf = NotebookConf

    def render(self, ctx):
        s = render_worksheet_list(ctx.args, pub=False, username=self.username)
        return http.Response(responsecode.OK,
                             {'set-cookie':set_cookie(self.cookie)},
                             stream=s)



def set_cookie(cookie):
    return [http_headers.Cookie(SID_COOKIE, cookie)]

def user_type(username):
    # one of admin, guest, user
    try:
        U = notebook.user(username)
    except KeyError:
        return 'guest'
    return U.account_type()


def extract_title(html_page):
    h = html_page.lower()
    i = h.find('<title>')
    if i == -1:
        return "Untitled"
    j = h.find('</title>')
    return h[i + len('<title>') : j]

