r"""
Field $\Q$ of Rational Numbers.

The class \class{RationalField} represents the field $\Q$ of
(arbitrary precision) rational numbers.  Each rational number is an
instance of the class \class{Rational}.

Interactively, an instance of \class{RationalField} is available as
\code{QQ}.

   sage: QQ
   Rational Field

Values of various types can be converted to rational numbers by
using the \code{__call__} method of \class{RationalField} (that is,
by treating \code{QQ} as a function).

   sage: RealField(9).pi()
   3.1
   sage: QQ(RealField(9).pi())
   22/7
   sage: QQ(RealField().pi())
   245850922/78256779
   sage: QQ(35)
   35
   sage: QQ('12/347')
   12/347
   sage: QQ(exp(pi*I))
   -1
   sage: x = polygen(ZZ)
   sage: QQ((3*x)/(4*x))
   3/4

TEST:
   sage: Q = RationalField()
   sage: Q == loads(dumps(Q))
   True
   sage: RationalField() is RationalField()
   True

"""

import random
import field
import ring
import sage.rings.rational
import sage.structure.factorization
import infinity
ZZ = None

from sage.structure.parent_gens import ParentWithGens
import sage.rings.number_field.number_field_base as number_field_base


_obj = {}
class _uniq(object):
    def __new__(cls):
        if _obj.has_key(0):
            return _obj[0]
        O = number_field_base.NumberField.__new__(cls)
        _obj[0] = O
        return O

class RationalField(_uniq, number_field_base.NumberField):
    r"""
    The class \class{RationalField} represents the field $\Q$ of
    rational numbers.
    """
    def __init__(self):
        """
        We create the rational numbers $\\Q$, and call a few functions:

            sage: Q = RationalField(); Q
            Rational Field
            sage: Q.characteristic()
            0
            sage: Q.is_field()
            True
            sage: Q.zeta()
            -1

        We next illustrate arithmetic in $\\Q$.

            sage: Q('49/7')
            7
            sage: type(Q('49/7'))
            <type 'sage.rings.rational.Rational'>
            sage: a = Q('19/374'); b = Q('17/371'); print a, b
            19/374 17/371
            sage: a + b
            13407/138754
            sage: b + a
            13407/138754
            sage: a * b
            19/8162
            sage: b * a
            19/8162
            sage: a - b
            691/138754
            sage: b - a
            -691/138754
            sage: a / b
            7049/6358
            sage: b / a
            6358/7049
            sage: b < a
            True
            sage: a < b
            False

        Next finally illustrate arithmetic with automatic coercion.
        The types that coerce into the rational field include
        \\code{str, int, long, Integer}.

            sage: a + Q('17/371')
            13407/138754
            sage: a * 374
            19
            sage: 374 * a
            19
            sage: a/19
            1/374
            sage: a + 1
            393/374

        TESTS:
            sage: QQ.variable_name()
            'x'
            sage: QQ.variable_names()
            ('x',)
        """
        ParentWithGens.__init__(self, self)
        self._assign_names(('x',),normalize=False)

    def __hash__(self):
        return -11115808

    def _repr_(self):
        return "Rational Field"

    def _latex_(self):
        return "\\mathbf{Q}"

    def __len__(self):
        raise TypeError, 'len() of unsized object'

    def __call__(self, x, base=0):
        """
        Coerce $x$ into the field of rational numbers.

        EXAMPLES:
            sage: a = long(901824309821093821093812093810928309183091832091)
            sage: b = QQ(a); b
            901824309821093821093812093810928309183091832091
            sage: QQ(b)
            901824309821093821093812093810928309183091832091
            sage: QQ(int(93820984323))
            93820984323
            sage: QQ(ZZ(901824309821093821093812093810928309183091832091))
            901824309821093821093812093810928309183091832091
            sage: QQ('-930482/9320842317')
            -930482/9320842317
            sage: QQ((-930482, 9320842317))
            -930482/9320842317
            sage: QQ([9320842317])
            9320842317
            sage: QQ(pari(39029384023840928309482842098430284398243982394))
            39029384023840928309482842098430284398243982394
            sage: QQ('sage')
            Traceback (most recent call last):
            ...
            TypeError: unable to convert sage to a rational

        Coercion from the reals to the rational is done by default
        using continued fractions.

            sage: QQ(RR(3929329/32))
            3929329/32
            sage: QQ(-RR(3929329/32))
            -3929329/32
            sage: QQ(RR(1/7)) - 1/7
            0

        If you specify an optional second base argument, then
        the string representation of the float is used.
            sage: QQ(23.2, 2)
            6530219459687219/281474976710656
            sage: 6530219459687219.0/281474976710656
            23.199999999999999
            sage: a = 23.2; a
            23.2000000000000
            sage: QQ(a, 10)
            116/5

        Here's a nice example involving elliptic curves:
            sage: E = EllipticCurve('11a')
            sage: L = E.lseries().at1(300)[0]; L
            0.253841860855911
            sage: O = E.period_lattice().omega(); O
            1.269209304279553421688794616754547305219492241830608667967136921230408338613     # 32-bit
            1.26920930427955342168879461675454730521949224183060866796713692123040833861277772269036230592151260731164529627832128743728170032847684397649271401057075        # 64-bit
            sage: t = L/O; t
            0.200000000000000
            sage: QQ(RealField(45)(t))
            1/5

        Elements from the extended rational field can be coerced
        back into the rational field.
            sage: E = ExtendedRationalField
            sage: QQ(E(2))
            2
            sage: type(_)
            <type 'sage.rings.rational.Rational'>
        """
        if x.__class__ is sage.rings.rational.Rational:
            return x

        return sage.rings.rational.Rational(x, base)

    def construction(self):
        from sage.categories.pushout import FractionField
        import integer_ring
        return FractionField(), integer_ring.ZZ

    def completion(self, p, prec, extras = {}):
        if p == infinity.Infinity:
            from sage.rings.real_mpfr import create_RealField
            return create_RealField(prec, **extras)
        else:
            from sage.rings.padics.factory import Qp
            return Qp(p, prec, **extras)

    def _coerce_impl(self, x):
        if isinstance(x, (int, long, sage.rings.integer.Integer,
                          sage.rings.rational.Rational)):
            return self(x)
        raise TypeError, 'no implicit coercion of element to the rational numbers'

    def coerce_map_from_impl(self, S):
        """
        EXAMPLES:
            sage: f = QQ.coerce_map_from(ZZ); f
            Natural morphism:
              From: Integer Ring
              To:   Rational Field
            sage: f(3)
            3
            sage: f(3^99) - 3^99
            0
            sage: f = QQ.coerce_map_from(int); f
            Native morphism:
              From: Set of Python objects of type 'int'
              To:   Rational Field
            sage: f(44)
            44
        """
        global ZZ
        if ZZ is None:
            import integer_ring
            ZZ = integer_ring.ZZ
        if S is ZZ:
            return sage.rings.rational.Z_to_Q()
        elif S is int:
            return sage.rings.rational.int_to_Q()
        else:
            return field.Field.coerce_map_from_impl(self, S)

    def _is_valid_homomorphism_(self, codomain, im_gens):
        try:
            return im_gens[0] == codomain._coerce_(self.gen(0))
        except TypeError:
            return False

    def __iter__(self):
        r"""
        Creates an iterator that generates the rational numbers without
        repetition. It uses the sequence defined by $a_0=0$ and
        $a_{n+1}=\frac{1}{2\lfloor a_n\rfloor+1-a_n}$ and generates the
        sequence $$a_0,a_1,-a_1,a_2,-a_2,\ldots$$

        EXAMPLES:

        This example creates a list consisting of the first 10 terms
        generated by this function.

            sage: import itertools
            sage: [a for a in itertools.islice(Rationals(),10)]
            [0, 1, -1, 1/2, -1/2, 2, -2, 1/3, -1/3, 3/2]

        NOTES:
            A proof of the correctness of this formula is attributed to
            Sam Vandervelde and Don Zagier [A002487], but a better
            reference for the origin of this formula would be welcome.

            REFERENCES:
                 [A002487] Sloane's OLEIS,
                 http://www.research.att.com/~njas/sequences/A002487

        AUTHORS:
            - Nils Bruin (2007-02-20)
        """

        from sage.rings.arith import integer_floor as floor

        n=self(0)
        yield n
        while True:
          n=1/(2*floor(n)+1-n)
          yield n
          yield -n

    def discriminant(self):
        """
        Return the discriminant of the field of rational numbers,
        which is 1.

        EXAMPLES:
            sage: QQ.discriminant()
            1
        """
        return sage.rings.integer.Integer(1)

    def signature(self):
        """
        Return the signature of the rational field, which is (1,0),
        since there are 1 real and no complex embeddings.

        EXAMPLES:
            sage: QQ.signature()
            (1, 0)
        """
        return (sage.rings.integer.Integer(1), sage.rings.integer.Integer(0))


    def discriminant(self):
        """
        Return the discriminant of the field of rational numbers,
        which is 1.

        EXAMPLES:
            sage: QQ.discriminant()
            1
        """
        return sage.rings.integer.Integer(1)

    def signature(self):
        """
        Return the signature of the rational field, which is (1,0),
        since there are 1 real and no complex embeddings.

        EXAMPLES:
            sage: QQ.signature()
            (1, 0)
        """
        return (sage.rings.integer.Integer(1), sage.rings.integer.Integer(0))


    def embeddings(self, K):
        """
        Return list of the one embedding of $\Q$ into $K$, if it exists.

        EXAMPLES:
            sage: QQ.embeddings(QQ)
            [Ring Coercion endomorphism of Rational Field]
            sage: QQ.embeddings(CyclotomicField(5))
            [Ring Coercion morphism:
              From: Rational Field
              To:   Cyclotomic Field of order 5 and degree 4]

        $K$ must have characteristic 0:
            sage: QQ.embeddings(GF(3))
            Traceback (most recent call last):
            ...
            ValueError: no embeddings of the rational field into K.
        """
        if K.characteristic() != 0:
            raise ValueError, "no embeddings of the rational field into K."
        return [self.hom(K)]

    def complex_embedding(self, prec=53):
        """
        Return embedding of the rational numbers into the complex numbers.

        EXAMPLES:
            sage: QQ.complex_embedding()
            Ring morphism:
              From: Rational Field
              To:   Complex Field with 53 bits of precision
              Defn: 1 |--> 1.00000000000000
            sage: QQ.complex_embedding(20)
            Ring morphism:
              From: Rational Field
              To:   Complex Field with 20 bits of precision
              Defn: 1 |--> 1.0000
        """
        import complex_field
        CC = complex_field.ComplexField(prec)
        return self.hom([CC(1)])

    def gens(self):
        """
        EXAMPLES:
            sage: QQ.gens()
            (1,)
        """
        return (self(1), )

    def gen(self, n=0):
        """
        EXAMPLES:
            sage: QQ.gen()
            1
        """
        if n == 0:
            return self(1)
        else:
            raise IndexError, "n must be 0"

    def degree(self):
        """
        EXAMPLES:
            sage: QQ.degree()
            1
        """
        return sage.rings.integer.Integer(1)

    def absolute_degree(self):
        """
        EXAMPLES:
            sage: QQ.absolute_degree()
            1
        """
        return sage.rings.integer.Integer(1)

    def ngens(self):
        """
        EXAMPLES:
            sage: QQ.ngens()
            1
        """
        return sage.rings.integer.Integer(1)

    def is_absolute(self):
        """
        $\Q$ is an absolute extension of $\Q$.

        EXAMPLES:
            sage: QQ.is_absolute()
            True
        """
        return True

    def is_subring(self, K):
        """
        Return \code{True} if $\Q$ is a subring of $K$.

        We are only able to determine this in some cases, e.g., when $K$
        is a field or of positive characteristic.

        EXAMPLES:
            sage: QQ.is_subring(QQ)
            True
            sage: QQ.is_subring(QQ['x'])
            True
            sage: QQ.is_subring(GF(7))
            False
            sage: QQ.is_subring(CyclotomicField(7))
            True
            sage: QQ.is_subring(ZZ)
            False
            sage: QQ.is_subring(Frac(ZZ))
            True
        """
        if K.is_field():
            return K.characteristic() == 0
        if K.characteristic() != 0:
            return False
        try:
            self.embeddings(K)
        except (TypeError, ValueError):
            return False
        return True

    def is_field(self):
        """
        Return \code{True}, since the rational field is a field.

        EXAMPLES:
            sage: QQ.is_field()
            True
        """
        return True

    def is_finite(self):
        """
        Return \code{False}, since the rational field is not finite.

        EXAMPLES:
            sage: QQ.is_finite()
            False
        """
        return False

    def is_prime_field(self):
        """
        Return \code{True}, since $\Q$ is a prime field.

        EXAMPLES:
            sage: QQ.is_prime_field()
            True
        """
        return True

    def is_atomic_repr(self):
        return True

    def characteristic(self):
        """
        Return 0, since the rational field has characteristic 0.

        EXAMPLES:
            sage: c = QQ.characteristic(); c
            0
            sage: parent(c)
            Integer Ring
        """
        return sage.rings.integer.Integer(0)

    def maximal_order(self):
        """
        Return the maximal order of the rational numbers,
        i.e., the ring $\Z$ of integers.

        EXAMPLES:
            sage: QQ.maximal_order()
            Integer Ring
            sage: QQ.ring_of_integers ()
            Integer Ring
        """
        from integer_ring import ZZ
        return ZZ

    def number_field(self):
        """
        Return the number field associated to $\Q$.  Since $\Q$
        is a number field, this just returns $\Q$ again.

        EXAMPLES:
            sage: QQ.number_field() is QQ
            True
        """
        return self

    def extension(self, poly, names, check=True):
        """
        EXAMPLES:
        We make a single absolute extension:
            sage: K.<a> = QQ.extension(x^3 + 5); K
            Number Field in a with defining polynomial x^3 + 5

        We make an extension generated by roots of two polynomials:
            sage: K.<a,b> = QQ.extension([x^3 + 5, x^2 + 3]); K
            Number Field in a with defining polynomial x^3 + 5 over its base field
            sage: b^2
            -3
            sage: a^3
            -5
        """
        from sage.rings.number_field.all import NumberField
        return NumberField(poly, names=names, check=check)

    def order(self):
        """
        EXAMPLES:
            sage: QQ.order()
            +Infinity
        """
        return infinity.infinity

    def _an_element_impl(self):
        return sage.rings.rational.Rational((1,2))

    def random_element(self, num_bound=None, den_bound=None, distribution=None):
        """
        EXAMPLES:
            sage: QQ.random_element(10,10) # random output
            -5/3
        """
        global ZZ
        if ZZ is None:
            import integer_ring
            ZZ = integer_ring.ZZ
        if num_bound == None:
            num = ZZ.random_element(distribution=distribution)
            den = ZZ.random_element(distribution=distribution)
            while den == 0: den = ZZ.random_element(distribution=distribution)
            return self((num, den))
        else:
            if num_bound == 0:
                num_bound = 2
            if den_bound is None:
                den_bound = num_bound
                if den_bound < 1:
                    den_bound = 2
            return self((ZZ.random_element(-num_bound, num_bound+1, distribution=distribution),
                         ZZ.random_element(1, den_bound+1, distribution=distribution)))
    def zeta(self, n=2):
        """
        Return a root of unity in \code{self}.

        INPUT:
            n -- integer (default: 2) order of the root of unity

        EXAMPLES:
            sage: QQ.zeta()
            -1
            sage: QQ.zeta(2)
            -1
            sage: QQ.zeta(1)
            1
            sage: QQ.zeta(3)
            Traceback (most recent call last):
            ...
            ValueError: no n-th root of unity in rational field
        """
        if n == 1:
            return sage.rings.rational.Rational(1)
        elif n == 2:
            return sage.rings.rational.Rational(-1)
        else:
            raise ValueError, "no n-th root of unity in rational field"

    #################################
    ## Coercions to interfaces
    #################################
    def _gap_init_(self):
        """
        EXAMPLES:
            sage: gap(QQ)
            Rationals
        """
        return 'Rationals'

    def _magma_init_(self):
        """
        EXAMPLES:
            sage: magma(QQ)                       # optional
            Rational Field
        """
        return 'RationalField()'


QQ = RationalField()
Q = QQ

def is_RationalField(x):
    return isinstance(x, RationalField)

def frac(n,d):
    return sage.rings.rational.Rational(n)/sage.rings.rational.Rational(d)

def factor(x):
    assert isinstance(x, sage.rings.rational.Rational)
    return x.numerator().factor() * \
           sage.structure.factorization.Factorization([(p,-e) for p, e in x.denominator().factor()])

