"""nodoctest
Find generators for the ring of modular forms of given level.

AUTHORS:
    -- William Stein (20070824): first version
"""

import random

from sage.rings.all        import Integer, QQ, infinity
from sage.misc.mrange      import cartesian_product_iterator
from sage.misc.misc        import prod, verbose

from sage.modular.congroup import Gamma0
from constructor           import ModularForms

def span_of_series(v, prec=None, basis=False):
    r"""
    Return the free module spanned by the given list of power series
    or objects with a padded_list method.  If prec is not given, the
    precision used is the minimum of the precisions of the elements in
    the list (as determined by a prec method).

    INPUT:
        v    -- a list of power series
        prec -- optional; if given then the series do not have to be
                of finite precision, and will be considered to have
                precision prec.
        basis -- (default: False) if True the input is assumed to
                determine linearly independent vectors, and
                the resulting free module has that as basis.


    OUTPUT:
          A free module of rank prec over the base ring of the forms
          (actually, of the first form in the list).  If the list is
          empty, the free module is over QQ.

    EXAMPLES:
    An example involving modular forms:
        sage: v = ModularForms(11,2, prec=5).basis(); v
        [
        q - 2*q^2 - q^3 + 2*q^4 + O(q^5),
        1 + 12/5*q + 36/5*q^2 + 48/5*q^3 + 84/5*q^4 + O(q^5)
        ]
        sage: span_of_series(v)
        Vector space of degree 5 and dimension 2 over Rational Field
        Basis matrix:
        [ 1  0 12 12 12]
        [ 0  1 -2 -1  2]

    Next we make sure the vector give a basis:
        sage: span_of_series(v,basis=True)
        Vector space of degree 5 and dimension 2 over Rational Field
        User basis matrix:
        [   0    1   -2   -1    2]
        [   1 12/5 36/5 48/5 84/5]

    An example involving power series.
        sage: R.<x> = PowerSeriesRing(QQ, default_prec=5)
        sage: v = [1/(1-x), 1/(1+x), 2/(1+x), 2/(1-x)]; v
        [1 + x + x^2 + x^3 + x^4 + O(x^5),
         1 - x + x^2 - x^3 + x^4 + O(x^5),
         2 - 2*x + 2*x^2 - 2*x^3 + 2*x^4 + O(x^5),
         2 + 2*x + 2*x^2 + 2*x^3 + 2*x^4 + O(x^5)]
        sage: span_of_series(v)
        Vector space of degree 5 and dimension 2 over Rational Field
        Basis matrix:
        [1 0 1 0 1]
        [0 1 0 1 0]
        sage: span_of_series(v,10)
        Vector space of degree 10 and dimension 2 over Rational Field
        Basis matrix:
        [1 0 1 0 1 0 0 0 0 0]
        [0 1 0 1 0 0 0 0 0 0]

    An example involving polynomials.
        sage: x = polygen(QQ)
        sage: span_of_series([x^3, 2*x^2 + 17*x^3, x^2])
        Traceback (most recent call last):
        ...
        ValueError: please specify a precision
        sage: span_of_series([x^3, 2*x^2 + 17*x^3, x^2],5)
        Vector space of degree 5 and dimension 2 over Rational Field
        Basis matrix:
        [0 0 1 0 0]
        [0 0 0 1 0]
        sage: span_of_series([x^3, 2*x^2 + 17*x^3, x^2],3)
        Vector space of degree 3 and dimension 1 over Rational Field
        Basis matrix:
        [0 0 1]
    """
    verbose('computing span of series')
    if len(v) == 0:
        if not prec:
            prec = 0
        return (QQ**prec).zero_submodule()
    if prec:
        n = Integer(prec)
    else:
        n = min([g.prec() for g in v])
        if n == infinity:
            raise ValueError, "please specify a precision"

    K = v[0].parent().base_ring()
    V = K**n
    B = [V(g.padded_list(n)) for g in v]
    if basis:
        M = V.span_of_basis(B)
    else:
        M = V.span(B)
    return M

def multiply_forms_to_weight(forms, weight, stop_dim=None):
    r"""
    Given a list of pairs $(k,f)$, where $k$ is an integer and $f$ is
    a power series, multiply together all $f$'s such that the sum of
    the corresponding $k$'s is weight.

    INPUT:
        forms -- list of pairs (k, f) with k an integer and f a power series
        weight -- an integer
        stop_dim -- integer (optional): if set to an integer and we find that
                    the series so far span a space of at least this dimension,
                    then stop multiplying more forms together.

    EXAMPLES:
        sage: import sage.modular.modform.modform_generators as f
        sage: forms = [(4, 240*eisenstein_series_qexp(4,5)), (6,504*eisenstein_series_qexp(6,5))]
        sage: f.multiply_forms_to_weight(forms, 12)
        [(12, 1 - 1008*q + 220752*q^2 + 16519104*q^3 + 399517776*q^4 + O(q^5)), (12, 1 + 720*q + 179280*q^2 + 16954560*q^3 + 396974160*q^4 + O(q^5))]
        sage: f.multiply_forms_to_weight(forms, 24)
        [(24, 1 - 2016*q + 1457568*q^2 - 411997824*q^3 + 16227967392*q^4 + O(q^5)), (24, 1 - 288*q - 325728*q^2 + 11700864*q^3 + 35176468896*q^4 + O(q^5)), (24, 1 + 1440*q + 876960*q^2 + 292072320*q^3 + 57349833120*q^4 + O(q^5))]
        sage: dimension_modular_forms(SL2Z,24)
        3
    """
    verbose('multiplying forms up to weight %s'%weight)
    # Algorithm: run through the subsets of forms and for each check
    # whether or not the sum of the weights (with coefficients -- i.e.,
    # account for multiplicities) of the forms equals weight.
    # If so, multiply those together and append them to the output
    # list v

    # The answer list
    v = []

    # The range of values to try for each coefficient
    r = [range(weight//k + 1) for k, _ in forms]

    n = len(forms)

    # Do the iteration
    m = stop_dim
    C = list(cartesian_product_iterator(r))
    random.shuffle(C)
    for c in C:
        if sum(c[i]*forms[i][0] for i in xrange(n) if c[i]) == weight:
            print [(c[i],forms[i][0]) for i in xrange(n)]
            g = prod(forms[i][1]**c[i] for i in xrange(n))
            v.append((weight, g))
            if stop_dim and len(v) >= m:
                z = span_of_series([f for _, f in v]).dimension()
                print len(v), z, stop_dim
                if z >= stop_dim:
                    return v
                m += stop_dim
    return v

def basis_for_modform_space(gens, group, weight):
    """
    Given gens as pairs $(k,f)$ for the ring of all modular forms of a
    given level (as output by modform_generators), return a basis of
    $q$-expansions for all forms of the given weight.
    """
    if len(gens) == 0:
        return []
    d = ModularForms(group, weight).dimension()
    v = multiply_forms_to_weight(gens, weight, stop_dim=d)
    s = span_of_series([f for _, f in v])
    R = gens[0][1].parent()
    prec = s.degree()
    return [R(list(f), prec) for f in s.basis()]

def modform_generators(group, maxweight=20, prec=None, start_gens=[], start_weight=2):
    r"""
    Find modular forms in $M_k(group)$ for $k\leq $ maxweight (with
    all $k$ having the same parity, such that these forms generate --
    as an algebra -- all forms on group of weight up to maxweight,
    where all forms are computed as $q$-expansions to precision prec.

    INPUT:
        group -- a level or a congruence subgroup
        maxweight -- integer
        prec -- integer (default: twice largest dimension)
        start_gens -- list of pairs (k,f) where k is an integer and f is a power seris
                      (default: []); if given, we assume the given pairs (k,f) are
                      q-expansions of modular form of the given weight, and start
                      creating modular forms generators using them.
        start_weight -- an integer (default: 2)

    OUTPUT:
        a list of pairs (k, f), where f is the q-expansion
        of a modular form of weight k.

    EXAMPLES:
        sage: import sage.modular.modform.modform_generators as fg
        sage: forms = [(4, 240*eisenstein_series_qexp(4,5)), (6,504*eisenstein_series_qexp(6,5))]
        sage: fg.multiply_forms_to_weight(forms, 12)
        [(12, 1 - 1008*q + 220752*q^2 + 16519104*q^3 + 399517776*q^4 + O(q^5)), (12, 1 + 720*q + 179280*q^2 + 16954560*q^3 + 396974160*q^4 + O(q^5))]
        sage: fg.multiply_forms_to_weight(forms, 24)
        [(24, 1 - 2016*q + 1457568*q^2 - 411997824*q^3 + 16227967392*q^4 + O(q^5)), (24, 1 - 288*q - 325728*q^2 + 11700864*q^3 + 35176468896*q^4 + O(q^5)), (24, 1 + 1440*q + 876960*q^2 + 292072320*q^3 + 57349833120*q^4 + O(q^5))]
        sage: dimension_modular_forms(SL2Z,24)
        3

        sage: modform_generators(1)
        [(4, 1 + 240*q + 2160*q^2 + 6720*q^3 + O(q^4)), (6, 1 - 504*q - 16632*q^2 - 122976*q^3 + O(q^4))]
        sage: modform_generators(2)
        [(2, 1 + 24*q + 24*q^2 + 96*q^3 + 24*q^4 + 144*q^5 + 96*q^6 + 192*q^7 + 24*q^8 + 312*q^9 + 144*q^10 + 288*q^11 + O(q^12)), (4, 1 + 240*q^2 + 2160*q^4 + 6720*q^6 + 17520*q^8 + 30240*q^10 + O(q^12)), (4, q + 8*q^2 + 28*q^3 + 64*q^4 + 126*q^5 + 224*q^6 + 344*q^7 + 512*q^8 + 757*q^9 + 1008*q^10 + 1332*q^11 + O(q^12))]
        sage: modform_generators(4, 12, 20)
        [(2, 1 + 24*q^2 + 24*q^4 + 96*q^6 + 24*q^8 + 144*q^10 + 96*q^12 + 192*q^14 + 24*q^16 + 312*q^18 + O(q^20)), (2, q + 4*q^3 + 6*q^5 + 8*q^7 + 13*q^9 + 12*q^11 + 14*q^13 + 24*q^15 + 18*q^17 + 20*q^19 + O(q^20))]

    Here we see the taking a basis of forms in weights 2 and 4 is
    enough to generate everything up to weight 12 (and probably
    everything else).
        sage: v = modform_generators(11, 12)
        sage: len(v)
        6
        sage: [k for k, _ in v]
        [2, 2, 4, 4, 4, 4]
        sage: dimension_modular_forms(11,2)
        2
        sage: dimension_modular_forms(11,4)
        4
    """
    if prec is None:
        prec = 2 * ModularForms(group, maxweight).dimension()
    k = start_weight
    if start_gens:
        G = list(start_gens)
    else:
        M = ModularForms(group, weight=k)
        B = M.q_expansion_basis(prec)
        G = [(k, f) for f in B]
        k += 2

    already_reported_indep = False
    while k <= maxweight:
        verbose('k = %s'%k)
        M = ModularForms(group, k)
        # 1. Multiply together all forms in G that give an element
        #    of M.
        F = multiply_forms_to_weight(G, k)
        # 2. If the dimension of the span of the result is equal
        #    to the dimension of M, incremenent k.
        gens = [f for _, f in F]
        S = span_of_series(gens, prec=prec, basis=False)
        if S.dimension() < len(gens):
            if not already_reported_indep:
                verbose("Generators are not indepenent (already at weight %s)"%k)
                already_reported_indep = True
        assert S.dimension() <= M.dimension(), "there is a bug in the code for finding generators of modular forms spaces"
        if S.dimension() == M.dimension():
            verbose("Gens so far do span at weight %s"%k)
            k += 2
            continue
        # 3. If the dimension is less, compute a basis for G, and
        #    add each basis element of G not in M into G.
        t = verbose("Computing more modular forms at weight %s"%k)
        B = M.q_expansion_basis(prec)
        for f in B:
            if not f in S:
                G.append( (k, f) )
        verbose('done computing forms', t)
    return G


class ModularFormsRing:
    """
    EXAMPLES:
        sage: m = ModularFormsRing(4); m
        Ring of modular forms on 4 of weights 0 and at least 2
        sage: m.modular_forms_of_weight(2)
        Modular Forms space of dimension 2 for Congruence Subgroup Gamma0(4) of weight 2 over Rational Field
        sage: m.modular_forms_of_weight(10)
        Modular Forms space of dimension 6 for Congruence Subgroup Gamma0(4) of weight 10 over Rational Field
        sage: m == loads(dumps(m))
        sage: m.generators()
        [(2,
          1 + 24*q^2 + 24*q^4 + 96*q^6 + 24*q^8 + 144*q^10 + 96*q^12 + 192*q^14 + 24*q^16 + 312*q^18 + 144*q^20 + O(q^22)),
         (2,
          q + 4*q^3 + 6*q^5 + 8*q^7 + 13*q^9 + 12*q^11 + 14*q^13 + 24*q^15 + 18*q^17 + 20*q^19 + 32*q^21 + O(q^22))]
        sage: m.q_expansion_basis(2,10)
        [1 + 24*q^2 + 24*q^4 + 96*q^6 + 24*q^8 + O(q^10),
         q + 4*q^3 + 6*q^5 + 8*q^7 + 13*q^9 + O(q^10)]
        sage: m.q_expansion_basis(3,10)
        []
        sage: m.q_expansion_basis(10,10)
        [1 + 10560*q^6 + 3960*q^8 + O(q^10),
         q - 8056*q^7 - 30855*q^9 + O(q^10),
         q^2 - 796*q^6 - 8192*q^8 + O(q^10),
         q^3 + 66*q^7 + 832*q^9 + O(q^10),
         q^4 + 40*q^6 + 528*q^8 + O(q^10),
         q^5 + 20*q^7 + 190*q^9 + O(q^10)]
    """
    def __init__(self, group):
        self.__group = group

    def __repr__(self):
        return "Ring of modular forms on %s of weights 0 and at least 2"%self.__group

    def modular_forms_of_weight(self, weight):
        """
        Return the space of modular forms on this group of the given weight.

        EXAMPLES:
        """
        return ModularForms(self.__group, weight)

    def generators(self, minprec=10, maxweight=20):
        try:
            if self.__genprec > minprec and self.__genmaxeight >= maxweight:
                return [(k, f.add_bigoh(minprec)) for k, f in self.__gens]
            elif self.__genprec == minprec and self.__genmaxeight >= maxweight:
                return list(self.__gens)
        except AttributeError:
            pass
        # Now we either don't know generators, or we know them to
        # too small of a precision.
        d = self.modular_forms_of_weight(maxweight).dimension()
        minprec = max(minprec, int(1.5*d))
        gens = modform_generators(self.__group, prec=minprec, maxweight=maxweight)
        self.__gens = gens
        self.__genprec = minprec
        self.__maxweight = maxweight
        return list(gens)

    def q_expansion_basis(self, weight, prec=None):
        d = self.modular_forms_of_weight(weight).dimension()
        orig_prec = prec
        if not prec or prec <= 1.5*d:
            prec = 2*d
        maxweight = min(4, weight)
        while True:
            gens = self.generators(prec, maxweight)
            V = basis_for_modform_space(gens, self.__group, weight)
            if len(V) == d:
                break
            assert len(V) < d, "Bug in q_expansion_basis: dimension too large."
            prec += d
            maxweight += 4
        if orig_prec:
            return [f.add_bigoh(orig_prec) for f in V]
        return V




