r"""
The set $\PP^1(\Q)$ of cusps

EXAMPLES:
    sage: Cusps
    Set P^1(QQ) of all cusps

    sage: Cusp(oo)
    Infinity
"""

#*****************************************************************************
#       Copyright (C) 2005 William Stein <wstein@gmail.com>
#
#  Distributed under the terms of the GNU General Public License (GPL)
#
#    This code is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
#
#  The full text of the GPL is available at:
#
#                  http://www.gnu.org/licenses/
#*****************************************************************************

from sage.rings.all import unsigned_infinity, is_Infinite, Rational, Integer, ZZ, QQ
from sage.rings.integer_ring import IntegerRing
from sage.rings.rational_field import RationalField
from sage.structure.parent_base import ParentWithBase
from sage.structure.element import Element

import congroup

infinity = unsigned_infinity

class Cusps_class(ParentWithBase):
    """
    The set of cusps.

    EXAMPLES:
        sage: C = Cusps; C
        Set P^1(QQ) of all cusps
        sage: loads(C.dumps()) == C
        True
    """
    def __init__(self):
        ParentWithBase.__init__(self, self)

    def __cmp__(self, right):
        """
        Return equality only if right is the set of cusps.

        EXAMPLES:
            sage: Cusps == Cusps
            True
            sage: Cusps == QQ
            False
        """
        return cmp(type(self), type(right))

    def _repr_(self):
        """
        String representation of the set of cusps.

        EXAMPLES:
            sage: Cusps
            Set P^1(QQ) of all cusps
            sage: Cusps.rename('CUSPS'); Cusps
            CUSPS
            sage: Cusps.rename(); Cusps
            Set P^1(QQ) of all cusps
            sage: Cusps
            Set P^1(QQ) of all cusps
        """
        return "Set P^1(QQ) of all cusps"

    def _latex_(self):
        """
        Return latex representation of self.

        EXAMPLES:
            sage: latex(Cusps)
            \mathbf{P}^1(\mathbf{Q})
        """
        return "\\mathbf{P}^1(\\mathbf{Q})"

    def __call__(self, x):
        """
        Coerce x into the set of cusps.

        EXAMPLES:
            sage: a = Cusps(-4/5); a
            -4/5
            sage: Cusps(a) is a
            False
            sage: Cusps(1.5)
            3/2
            sage: Cusps(oo)
            Infinity
            sage: Cusps(I)
            Traceback (most recent call last):
            ...
            TypeError: Unable to coerce I (<class 'sage.functions.constants.I_class'>) to Rational

        """
        return Cusp(x, parent=self)

    def _coerce_impl(self, x):
        """
        Canonical coercion of x into the set of cusps.

        EXAMPLES:
            sage: Cusps._coerce_(7/13)
            7/13
            sage: Cusps._coerce_(GF(7)(3))
            Traceback (most recent call last):
            ...
            TypeError: no canonical coercion of element into self
            sage: Cusps(GF(7)(3))
            3
        """
        if is_Infinite(x):
            return Cusp(x, parent=self)
        else:
            return self._coerce_try(x, QQ)

Cusps = Cusps_class()


class Cusp(Element):
    """
    A cusp.

    A cusp is either a rational number or infinity, i.e., an element
    of the projective line over Q.  A Cusp is stored as a pair (a,b),
    where gcd(a,b)=1 and a,b are of type Integer.

    EXAMPLES:
        sage: a = Cusp(2/3); b = Cusp(oo)
        sage: a.parent()
        Set P^1(QQ) of all cusps
        sage: a.parent() is b.parent()
        True
    """

    def __init__(self, a, b=ZZ(1), construct=False, parent=None):
        r"""
        Create the cusp a/b in $\PP^1(\Q)$, where if b=0 this is the
        cusp at infinity.

        EXAMPLES:
            sage: Cusp(2,3)
            2/3
            sage: Cusp(3,6)
            1/2
            sage: Cusp(1,0)
            Infinity
            sage: Cusp(infinity)
            Infinity
            sage: Cusp(5)
            5
            sage: Cusp(1/2)             # rational number
            1/2
            sage: Cusp(1.5)
            3/2

            sage: Cusp(sqrt(-1))
            Traceback (most recent call last):
            ...
            TypeError: unable to convert I to a rational

            sage: a = Cusp(2,3)
            sage: loads(a.dumps()) == a
            True
        """
        if parent is None:
            parent = Cusps
        Element.__init__(self, parent)

        if construct:
            self.__a = a; self.__b = b
            return

        if is_Infinite(a):
            self.__a = ZZ(1)
            self.__b = ZZ(0)
            return

        elif isinstance(a, Cusp):
            self.__a = a.__a
            self.__b = a.__b
            return

        elif isinstance(a, Rational):
            a = a/b
            self.__a = a.numer()
            self.__b = a.denom()
            return

        elif isinstance(a, (int, long, Integer)) and \
                 isinstance(b, (int, long, Integer)):
            a = ZZ(a)
            b = ZZ(b)

        elif isinstance(a, Integer) and isinstance(b, Integer):
            pass

        elif isinstance(a, str):
            a = RationalField()(a)/b
            self.__a = a.numer()
            self.__b = a.denom()
            return

        elif isinstance(a, (list, tuple)):
            a, b = a
            a = ZZ(a)
            b = ZZ(b)

        elif b==1:

            self.__a = RationalField()(a)
            self.__b = b
            return

        else:

            raise TypeError, "Unable to coerce %s,%s to a Cusp"%(a,b)


        # Now a, b are both of type ZZ.
        if b < 0:
            b *= -1
            a *= -1
        g = a.gcd(b)
        self.__a = a//g
        self.__b = b//g
        return

    def __cmp__(self, right):
        """
        Compare the cusps self and right.  Comparison is as for
        rational numbers, except with the cusp oo greater than
        everything but itself.

        The ordering in comparison is only really meaningful
        for infinity or elements that coerce to the rationals.

        EXAMPLES:
            sage: Cusp(2/3) == Cusp(oo)
            False

            sage: Cusp(2/3) < Cusp(oo)
            True

            sage: Cusp(2/3)> Cusp(oo)
            False

            sage: Cusp(2/3) > Cusp(5/2)
            False

            sage: Cusp(2/3) < Cusp(5/2)
            True

            sage: Cusp(2/3) == Cusp(5/2)
            False

            sage: Cusp(oo) == Cusp(oo)
            True

            sage: 19/3 < Cusp(oo)
            True

            sage: Cusp(oo) < 19/3
            False

            sage: Cusp(2/3) < Cusp(11/7)
            True

            sage: Cusp(11/7) < Cusp(2/3)
            False

            sage: 2 < Cusp(3)
            True
        """
        if self.__b == 0:
            # self is oo, which is bigger than everything but oo.
            if right.__b == 0:
                return 0
            else:
                return 1
        elif right.__b == 0:
            if self.__b == 0:
                return 0
            else:
                return -1
        return cmp(self._rational_(), right._rational_())

    def is_infinity(self):
        """
        Returns True if this is the cusp infinity.

        EXAMPLES:
            sage: Cusp(3/5).is_infinity()
            False
            sage: Cusp(1,0).is_infinity()
            True
            sage: Cusp(0,1).is_infinity()
            False
        """
        return self.__b == 0

    def numerator(self):
        """
        Return the numerator of the cusp a/b.

        EXAMPLES:
            sage: x=Cusp(6,9); x
            2/3
            sage: x.numerator()
            2
            sage: Cusp(oo).numerator()
            1
            sage: Cusp(-5/10).numerator()
            -1
        """
        return self.__a

    def denominator(self):
        """
        Return the denominator of the cusp a/b.

        EXAMPLES:
            sage: x=Cusp(6,9); x
            2/3
            sage: x.denominator()
            3
            sage: Cusp(oo).denominator()
            0
            sage: Cusp(-5/10).denominator()
            2
        """
        return self.__b

    def _rational_(self):
        """
        Coerce to a rational number.

        EXAMPLES:
            sage: QQ(Cusp(oo))
            Traceback (most recent call last):
            ...
            TypeError: cusp Infinity is not a rational number
            sage: QQ(Cusp(-3,7))
            -3/7
        """
        if self.__b == 0:
            raise TypeError, "cusp %s is not a rational number"%self
        return self.__a / self.__b

    def _integer_(self):
        """
        Coerce to an integer.

        EXAMPLES:
            sage: ZZ(Cusp(-19))
            -19

            sage: ZZ(Cusp(oo))
            Traceback (most recent call last):
            ...
            TypeError: cusp Infinity is not an integer
            sage: ZZ(Cusp(-3,7))
            Traceback (most recent call last):
            ...
            TypeError: cusp -3/7 is not an integer
        """
        if self.__b != 1:
            raise TypeError, "cusp %s is not an integer"%self
        return self.__a

    def _repr_(self):
        """
        String representation of this cusp.

        EXAMPLES:
            sage: a = Cusp(2/3); a
            2/3
            sage: a.rename('2/3(cusp)'); a
            2/3(cusp)
        """
        if self.__b.is_zero():
            return "Infinity"
        if self.__b != 1:
            return "%s/%s"%(self.__a,self.__b)
        else:
            return str(self.__a)

    def _latex_(self):
        r"""
        Latex representation of this cusp.

        EXAMPLES:
            sage: latex(Cusp(-2/7))
            \frac{-2}{7}
            sage: latex(Cusp(oo))
            \infty
        """
        if self.__b.is_zero():
            return "\\infty"
        if self.__b != 1:
            return "\\frac{%s}{%s}"%(self.__a,self.__b)
        else:
            return str(self.__a)

    def __neg__(self):
        """
        The negative of this cusp.

        EXAMPLES:
            sage: -Cusp(2/7)
            -2/7
            sage: -Cusp(oo)
            Infinity
        """
        return Cusp(-self.__a, self.__b)

    def is_gamma0_equiv(self, other, N, transformation = False):
        r"""
        Return whether self and other are equivalent modulo the action
        of $\Gamma_0(N)$ via linear fractional transformations.

        INPUT:
            other -- Cusp
            N -- an integer (specifies the group Gamma_0(N))
            transformation -- bool (default: False), if True, also
                              return upper left entry of a matrix in
                              Gamma_0(N) that sends self to other.

        OUTPUT:
            bool -- True if self and other are equivalent
            integer -- returned only if transformation is True

        EXAMPLES:
            sage: x = Cusp(2,3)
            sage: y = Cusp(4,5)
            sage: x.is_gamma0_equiv(y, 2)
            True
            sage: x.is_gamma0_equiv(y, 2, True)
            (True, 1)
            sage: x.is_gamma0_equiv(y, 3)
            False
            sage: x.is_gamma0_equiv(y, 3, True)
            (False, None)
            sage: Cusp(1,0)
            Infinity
            sage: z = Cusp(1,0)
            sage: x.is_gamma0_equiv(z, 3, True)
            (True, 2)

        ALGORITHM:
            See Proposition 2.2.3 of Cremona's book "Algorithms for Modular
            Elliptic Curves", or Prop 2.27 of Stein's Ph.D. thesis.
        """
        if not isinstance(other, Cusp):
            other = Cusp(other)
        N = ZZ(N)
        u1 = self.__a
        v1 = self.__b
        u2 = other.__a
        v2 = other.__b
        if (u1,v1) != (ZZ(0),ZZ(1)):
            if v1 in [ZZ(0),ZZ(1)]:
                s1 = ZZ(1)
            else:
                s1 = u1.inverse_mod(v1)
        else:
            s1 = 0
        if (u2,v2) != (ZZ(0),ZZ(1)):
            if v2 in [ZZ(0),ZZ(1)]:
                s2 = 1
            else:
                s2 = u2.inverse_mod(v2)
        else:
            s2 = 0
        g = (v1*v2).gcd(N)
        a = s1*v2 - s2*v1
        if a%g != ZZ(0):
            if transformation:
                return False, None
            else:
                return False

        if not transformation:
            return True

        # Now we know the cusps are equivalent.  Use the proof of Prop 2.2.3
        # of Cremona to find a matrix in Gamma_0(N) relating them.
        dum, s2, r2 = u2.xgcd(-v2)
        assert dum.is_one()
        dum, s1, r1 = u1.xgcd(-v1)
        assert dum.is_one()
        a = s1*v2 - s2*v1
        assert (a%g).is_zero()
        # solve x*v1*v2 + a = 0 (mod N).
        d,x0,y0 = (v1*v2).xgcd(N)          # x0*v1*v2 + y0*N = d = g.
        # so x0*v1*v2 - g = 0 (mod N)
        x = -x0 * (a/g)
        # now  x*v1*v2 + a = 0 (mod N)
        s1p = s1+x*v1
        return (True, (u2*s1p-r2*v1)%N)

    def is_gamma1_equiv(self, other, N):
        """
        Return whether self and other are equivalent modulo the action
        of Gamma_1(N) via linear fractional transformations.

        INPUT:
            other -- Cusp
            N -- an integer (specifies the group Gamma_1(N))

        OUTPUT:
            bool -- True if self and other are equivalent
            int -- 0, 1 or -1, gives further information
                   about the equivalence:  If the two cusps
                   are u1/v1 and u2/v2, then they are equivalent
                   if and only if
                        v1 = v2 (mod N) and u1 = u2 (mod gcd(v1,N))
                   or
                        v1 = -v2 (mod N) and u1 = -u2 (mod gcd(v1,N))
                   The sign is +1 for the first and -1 for the second.
                   If the two cusps are not equivalent then 0 is returned.

        EXAMPLES:
            sage: x = Cusp(2,3)
            sage: y = Cusp(4,5)
            sage: x.is_gamma1_equiv(y,2)
            (True, 1)
            sage: x.is_gamma1_equiv(y,3)
            (False, 0)
            sage: z = Cusp(QQ(x) + 10)
            sage: x.is_gamma1_equiv(z,10)
            (True, 1)
            sage: z = Cusp(1,0)
            sage: x.is_gamma1_equiv(z, 3)
            (True, -1)
            sage: Cusp(0).is_gamma1_equiv(oo, 1)
            (True, 1)
            sage: Cusp(0).is_gamma1_equiv(oo, 3)
            (False, 0)
        """
        if not isinstance(other, Cusp):
            other = Cusp(other)
        N = ZZ(N)
        u1 = self.__a
        v1 = self.__b
        u2 = other.__a
        v2 = other.__b
        g = v1.gcd(N)
        if ((v2 - v1) % N == 0 and (u2 - u1)%g== 0):
            return True, 1
        elif ((v2 + v1) % N == 0 and (u2 + u1)%g== 0):
            return True, -1
        return False, 0

    def is_gamma_h_equiv(self, other, G):
        """
        Return a pair (b, t), where b is True or False as self
        and other are equivalent under the action of G, and t
        is 1 or -1, as described below.

        Two cusps $u1/v1$ and $u2/v2$ are equivalent modulo Gamma_H(N)
        if and only if
            $v1 =  h*v2 (mod N)$ and $u1 =  h^(-1)*u2 (mod gcd(v1,N))$
        or
            $v1 = -h*v2 (mod N)$ and $u1 = -h^(-1)*u2 (mod gcd(v1,N))$
        for some $h \in H$. Then t is 1 or -1 as c and c' fall into
        the first or second case, respectively.

        INPUT:
            other -- Cusp
            G -- a congruence subgroup Gamma_H(N)

        OUTPUT:
            bool -- True if self and other are equivalent
            int -- -1, 0, 1; extra info

        EXAMPLES:
            sage: x = Cusp(2,3)
            sage: y = Cusp(4,5)
            sage: x.is_gamma_h_equiv(y,GammaH(13,[2]))
            (True, 1)
            sage: x.is_gamma_h_equiv(y,GammaH(13,[5]))
            (False, 0)
            sage: x.is_gamma_h_equiv(y,GammaH(5,[]))
            (False, 0)
            sage: x.is_gamma_h_equiv(y,GammaH(23,[4]))
            (True, -1)

        Enumerating the cusps for a space of modular symbols uses this
        function.
            sage: G = GammaH(25,[6]) ; M = G.modular_symbols() ; M
            Modular Symbols space of dimension 11 for Congruence Subgroup Gamma_H(25) with H generated by [6] of weight 2 with sign 0 and over Rational Field
            sage: M.cusps()
            [37/75, 1/2, 31/125, 1/4, -2/5, -3/5, -1/5, -9/10, -3/10, -14/15, 7/15, 9/20]
            sage: len(M.cusps())
            12

        This is always one more than the associated space of Eisenstein
        series.
            sage: sage.modular.dims.dimension_eis_H(G,2)
            11
            sage: M.cuspidal_subspace()
            Modular Symbols subspace of dimension 0 of Modular Symbols space of dimension 11 for Congruence Subgroup Gamma_H(25) with H generated by [6] of weight 2 with sign 0 and over Rational Field
            sage: sage.modular.dims.dimension_cusp_forms_H(G,2)
            0
        """
        if not isinstance(other, Cusp):
            other = Cusp(other)
        if not congroup.is_GammaH(G):
            raise TypeError, "G must be a group GammaH(N)."

        H = G._list_of_elements_in_H()
        N = ZZ(G.level())
        u1 = self.__a
        v1 = self.__b
        u2 = other.__a
        v2 = other.__b
        g = v1.gcd(N)

        for h in H:
            v_tmp = (h*v1)%N
            u_tmp = (h*u2)%N
            if (v_tmp - v2)%N == 0 and (u_tmp - u1)%g == 0:
                return True, 1
            if (v_tmp + v2)%N == 0 and (u_tmp + u1)%g == 0:
                return True, -1
        return False, 0

    def apply(self, g):
        """
        Return g(self), where g=[a,b,c,d] is a list of length 4, which
        we view as a linear fractional transformation.

        EXAMPLES:
            Apply the identity matrix:
            sage: Cusp(0).apply([1,0,0,1])
            0
            sage: Cusp(0).apply([0,-1,1,0])
            Infinity
            sage: Cusp(0).apply([1,-3,0,1])
            -3
        """
        return Cusp(g[0]*self.__a + g[1]*self.__b, g[2]*self.__a + g[3]*self.__b)


