r"""
SAGE package management commands

A SAGE package has the extension .spkg.  It is a tarball that is
(usually) bzip2 compressed that contains arbitrary data and an
spkg-install file.  An SAGE package typically has the following
components:
\begin{itemize}
  \item spkg-install -- shell script that is run to install the package
  \item SAGE.txt -- file that describes how the package was made, who maintains it, etc.
  \item sage -- directory with extra patched version of files that needed during the install
\end{itemize}

Use the \code{install_package} command to install a new package, and use
\code{optional_packages} to list all optional packages available on the
central SAGE server.   The \code{upgrade} command upgrades all \emph{standard}
packages -- there is no auto-upgrade command for optional packages.

All package management can also be done via the SAGE command line.
"""

import os

__installed_packages = None

def install_package(package=None, force=False):
    """
    Install a package or return a list of all packages
    that have been installed into this SAGE install.

    You must have an internet connection.  Also, you will have to
    restart SAGE for the changes to take affect.

    INPUT:
        package -- optional; if specified, install the
                   given package.  If not, list all
                   installed packages.

    IMPLEMENTATION: calls 'sage -f'.

    RELATED COMMANDS:
        optional_packages -- list of all optional packages
        upgrade -- upgrade to latest version of core packages
                   (optional packages are not automatically upgraded).
    """
    global __installed_packages
    if os.uname()[0][:6] == 'CYGWIN':
        print "install_package may not work correctly under Microsoft Windows"
        print "since you can't change an opened file.  Quit all"
        print "instances of sage and use 'sage -i' instead or"
        print "use the force option to install_package."
        return
    if package is None:
        if __installed_packages is None:
            X = os.popen('sage -f').read().split('\n')
            i = X.index('Currently installed packages:')
            X = [Y for Y in X[i+1:] if Y != '']
            X.sort()
            __installed_packages = X
        return __installed_packages
    os.system('sage -f "%s"'%package)
    __installed_packages = None


def is_package_installed(package):
    """
    Return true if a package starting with the given string is installed.

    EXAMPLES:
        sage: is_package_installed('sage')
        True
    """
    return any(p.startswith(package) for p in install_package())

def standard_packages():
    """
    Return two lists.  The first contains the installed and the second
    contains the not-installed standard packages that are available
    from the SAGE repository.      You must have an internet connection.

    OUTPUT:
        -- installed standard packages (as a list)
        -- NOT installed standard packages (as a list)

    Use \code{install_package(package_name)} to install or re-install
    a given package.

    RELATED COMMANDS:
        install_package -- list of all standard packages
        upgrade -- upgrade to latest version of core packages
                   (standard packages are not automatically upgraded).
    """
    R = os.popen('sage -standard').read()
    X = R.split('\n')
    try:
        i = X.index('INSTALLED:')
        j = X.index('NOT INSTALLED:')
    except ValueError, msg:
        print R
        print "standard package list (shown above) appears to be currently not available or corrupted (network error?)."
        return [], []

    installed = []
    for k in X[i+1:]:
        if k == '':
            break
        installed.append(k)

    not_installed = []
    for k in X[j+1:]:
        if k == '':
            break
        not_installed.append(k)
    return installed, not_installed

def optional_packages():
    """
    Return two lists.  The first contains the installed and the second
    contains the not-installed optional packages that are available
    from the SAGE repository.      You must have an internet connection.

    OUTPUT:
        -- installed optional packages (as a list)
        -- NOT installed optional packages (as a list)

    Use \code{install_package(package_name)} to install or re-install
    a given package.

    RELATED COMMANDS:
        install_package -- list of all optional packages
        upgrade -- upgrade to latest version of core packages
                   (optional packages are not automatically upgraded).
    """
    R = os.popen('sage -optional').read()
    X = R.split('\n')
    try:
        i = X.index('INSTALLED:')
        j = X.index('NOT INSTALLED:')
    except ValueError, msg:
        print R
        print "Optional package list (shown above) appears to be currently not available or corrupted (network error?)."
        return [], []

    installed = []
    for k in X[i+1:]:
        if k == '':
            break
        installed.append(k)

    not_installed = []
    for k in X[j+1:]:
        if k == '':
            break
        not_installed.append(k)
    return installed, not_installed

def experimental_packages():
    """
    Return two lists.  The first contains the installed and the second
    contains the not-installed experimental packages that are available
    from the SAGE repository.      You must have an internet connection.

    OUTPUT:
        -- installed experimental packages (as a list)
        -- NOT installed experimental packages (as a list)

    Use \code{install_package(package_name)} to install or re-install
    a given package.

    RELATED COMMANDS:
        install_package -- list of all experimental packages
        upgrade -- upgrade to latest version of core packages
                   (experimental packages are not automatically upgraded).
    """
    R = os.popen('sage -experimental').read()
    X = R.split('\n')
    try:
        i = X.index('INSTALLED:')
        j = X.index('NOT INSTALLED:')
    except ValueError, msg:
        print R
        print "experimental package list (shown above) appears to be currently not available or corrupted (network error?)."
        return [], []

    installed = []
    for k in X[i+1:]:
        if k == '':
            break
        installed.append(k)

    not_installed = []
    for k in X[j+1:]:
        if k == '':
            break
        not_installed.append(k)
    return installed, not_installed

#################################################################
# Upgrade to latest version of SAGE.
#################################################################


def upgrade():
    """
    Download and build the latest version of SAGE.

    You must have an internet connection.  Also, you will have to
    restart SAGE for the changes to take affect.

    This upgrades to the latest version of core packages (optional
    packages are not automatically upgraded).

    This will not work on systems that don't have a C compiler.

    RELATED COMMANDS:
        install_package -- list of all optional packages
        optional_packages -- list of all optional packages
    """
    global __installed_packages
    if os.uname()[0][:6] == 'CYGWIN':
        print "Upgrade may not work correctly under Microsoft Windows"
        print "since you can't change an opened file.  Quit all"
        print "instances of sage and use 'sage -upgrade' instead."
        return
    os.system('sage -upgrade')
    __installed_packages = None
    print "You should quit and restart SAGE now."


def package_mesg(package_name):
    mesg  = 'To install the package %s type install_package("%s")\n'%(package_name, package_name)
    mesg += 'at the sage prompt.  Note, the version number might\n'
    mesg += 'change; if so, type optional_packages() to see a list \n'
    mesg += 'of possibilities.   All this requires an internet connection.'
    mesg += 'For more help, type optional_packages?'
    return mesg

