r"""
Interface to Octave

Octave is an open source MATLAB-like program with numerical routines
for integrating, solving systems of equations, special functions, and
solving (numerically) differential equations. Please see
\url{http://octave.sourceforge.net} for more details.

The commands in this section only work if you have the
optional "octave" interpreter installed and available in
your PATH.  It's not necessary to install any special
\sage packages.

EXAMPLES:
    sage: octave.eval('2+2')
    'ans = 4'

    sage: a = octave(10)
    sage: a**10
    1e+10

LOG: - creation  (William Stein)
     - ?         (David Joyner, 2005-12-18)
     - Examples  (David Joyner, 2005-01-03)

\subsection{Computation of Special Functions}
Octave implements computation of the following
special functions (see the maxima and gp interfaces
for even more special functions):

\begin{verbatim}
airy
    Airy functions of the first and second kind, and their derivatives.
    airy(0,x) = Ai(x), airy(1,x) = Ai'(x), airy(2,x) = Bi(x), airy(3,x) = Bi'(x)
besselj
    Bessel functions of the first kind.
bessely
    Bessel functions of the second kind.
besseli
    Modified Bessel functions of the first kind.
besselk
    Modified Bessel functions of the second kind.
besselh
    Compute Hankel functions of the first (k = 1) or second (k = 2) kind.
beta
    The Beta function,
          beta (a, b) = gamma (a) * gamma (b) / gamma (a + b).
betainc
    The incomplete Beta function,
erf
    The error function,
erfinv
    The inverse of the error function.
gamma
    The Gamma function,
gammainc
    The incomplete gamma function,
\end{verbatim}

For example,
    sage: octave("airy(3,2)")
    4.10068
    sage: octave("beta(2,2)")
    0.166667
    sage: octave("betainc(0.2,2,2)")
    0.104
    sage: octave("besselh(0,2)")
    (0.223891,0.510376)
    sage: octave("besselh(0,1)")
    (0.765198,0.088257)
    sage: octave("besseli(1,2)")
    1.59064
    sage: octave("besselj(1,2)")
    0.576725
    sage: octave("besselk(1,2)")
    0.139866
    sage: octave("erf(0)")
    0
    sage: octave("erf(1)")
    0.842701
    sage: octave("erfinv(0.842)")
    0.998315
    sage: octave("gamma(1.5)")
    0.886227
    sage: octave("gammainc(1.5,1)")
    0.77687

The Octave interface reads in even very long input (using files)
in a robust manner:
    sage: t = '"%s"'%10^10000   # ten thousand character string.
    sage: a = octave.eval(t + ';')      # < 1/100th of a second
    sage: a = octave(t)

Note that actually reading a back out takes forever.  This *must* be
fixed ASAP -- see http://trac.sagemath.org/sage_trac/ticket/940/.

\subsection{Tutorial}
EXAMPLES:
    sage: octave('4+10')                     # optional
    14
    sage: octave('date')                    # optional; random output
    18-Oct-2007
    sage: octave('5*10 + 6')                 # optional
    56
    sage: octave('(6+6)/3')                  # optional
    4
    sage: octave('9')^2                      # optional
    81
    sage: a = octave(10); b = octave(20); c = octave(30)    # optional
    sage: avg = (a+b+c)/3                    # optional
    sage: avg                                # optional
    20
    sage: parent(avg)                        # optional
    Octave

    sage: my_scalar = octave('3.1415')       # optional
    sage: my_scalar                          # optional
    3.1415
    sage: my_vector1 = octave('[1,5,7]')     # optional
    sage: my_vector1                         # optional
    1     5     7
    sage: my_vector2 = octave('[1;5;7]')     # optional
    sage: my_vector2                         # optional
    1
    5
    7
    sage: my_vector1 * my_vector2            # optional
    75

"""

#*****************************************************************************
#       Copyright (C) 2005 William Stein <wstein@gmail.com>
#
#  Distributed under the terms of the GNU General Public License (GPL)
#
#    This code is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
#
#  The full text of the GPL is available at:
#
#                  http://www.gnu.org/licenses/
#*****************************************************************************

import os

from expect import Expect, ExpectElement

from sage.misc.misc import verbose

#import sage.matrix.matrix_space

class Octave(Expect):
    """
    Interface to the Octave interpreter.

    EXAMPLES:
        sage: octave.eval("a = [ 1, 1, 2; 3, 5, 8; 13, 21, 33 ]")    # optional
        'a =\n\n 1 1 2\n 3 5 8\n 13 21 33\n\n'
        sage: octave.eval("b = [ 1; 3; 13]")                         # optional
        'b =\n\n 1\n 3\n 13\n\n'
        sage: octave.eval("c=a \\ b") # solves linear equation: a*c = b  # optional random output
        'c =\n\n 1\n 7.21645e-16\n -7.21645e-16\n\n'
        sage: octave.eval("c")                                 # optional random output
        'c =\n\n 1\n 7.21645e-16\n -7.21645e-16\n\n'
    """
    def __init__(self, maxread=100, script_subdirectory="", logfile=None, server=None, server_tmpdir=None):
        Expect.__init__(self,
                        name = 'octave',
                        prompt = '>',
                        command = "octave --no-line-editing --silent",
                        maxread = maxread,
                        server = server,
                        server_tmpdir = server_tmpdir,
                        script_subdirectory = script_subdirectory,
                        restart_on_ctrlc = False,
                        verbose_start = False,
                        logfile = logfile,
                        eval_using_file_cutoff=100)

    def __reduce__(self):
        return reduce_load_Octave, tuple([])

    def _read_in_file_command(self, filename):
        return 'source("%s");'%filename

    def _quit_string(self):
        return 'quit;'

    def _install_hints(self):
        return """
        You must get the program "octave" in order to use Octave
        from SAGE.   You can read all about Octave at
                http://www.gnu.org/software/octave/

        LINUX / WINDOWS (colinux):
           Do apt-get install octave as root on your machine
           (or, in Windows, in the colinux console).

        OS X:
           * This website has links to binaries for OS X PowerPC
             and OS X Intel builds of the latest version of Octave:
                     http://hpc.sourceforge.net/
             Once you get the tarball from there, go to the / directory
             and type
                     tar zxvf octave-intel-bin.tar.gz
             to extract it to usr/local/...   Make sure /usr/local/bin
             is in your PATH.  Then type "octave" and verify that
             octave starts up.
           * Don't bother with fink -- if they have octave at all,
             it's probably out of date.
           * Darwin ports has octave as well.
        """

    def quit(self, verbose=False):
        # Don't bother, since it just hangs in some cases, and it
        # isn't necessary, since octave behaves well with respect
        # to signals.
        if not self._expect is None:
            if verbose:
                print "Exiting spawned %s process."%self
        return

    def _start(self):
        Expect._start(self)
        self.eval("page_screen_output=0;")
        self.eval("format none;")
        #self.eval('save_header_format_string="";')

#    pdehaye/20070819: This is no obsolete, see Expect._get_tmpfile_from_server and Expect._send_tmpfile_to_server

#    def get_via_file(self, var_name):
#        t = self._temp_file(var_name)
#        self.eval('save -text "%s" %s'%(t,var_name))
#        r = open(t).read()
#        os.unlink(t)
#        return r

#    def set_via_file(self, var_name, x):
#        t = self._temp_file(var_name)
#        open(t,'w').write(x)
#        print 'load "%s" %s'%(t, var_name)
#        self.eval('load "%s" %s'%(t, var_name))
#        #os.unlink(t)

    def set(self, var, value):
        """
        Set the variable var to the given value.
        """
        cmd = '%s=%s;'%(var,value)
        out = self.eval(cmd)
        if out.find("error") != -1:
            raise TypeError, "Error executing code in Octave\nCODE:\n\t%s\nOctave ERROR:\n\t%s"%(cmd, out)

    def get(self, var):
        """
        Get the value of the variable var.
        """
        s = self.eval('%s'%var)
        i = s.find('=')
        return s[i+1:]

    #def clear(self, var):
    #    """
    #    Clear the variable named var.
    #    """
    #    self.eval('clear %s'%var)

    def console(self):
        octave_console()

    def version(self):
        """
        Return the version of Octave.

        OUTPUT:
            string

        EXAMPLES:
            sage: octave.version()   # optional and random output depending on version
            '2.1.73'
        """
        return octave_version()

    def solve_linear_system(self, A, b):
        """
        Use octave to compute a solution x to A*x = b, as a list.

        INPUT:
            A -- mxn matrix A with entries in QQ or RR
            b -- m-vector b entries in QQ or RR (resp)

        OUTPUT:
            An list x (if it exists) which solves M*x = b

        EXAMPLES:
            sage: M33 = MatrixSpace(QQ,3,3)
            sage: A   = M33([1,2,3,4,5,6,7,8,0])
            sage: V3  = VectorSpace(QQ,3)
            sage: b   = V3([1,2,3])
            sage: octave.solve_linear_system(A,b)    # requires optional octave (and output is slightly random in low order bits)
            [-0.33333299999999999, 0.66666700000000001, -3.5236600000000002e-18]

        AUTHOR: David Joyner and William Stein
        """
        m = A.nrows()
        n = A.ncols()
        if m != len(b):
            raise ValueError, "dimensions of A and b must be compatible"
        from sage.matrix.all import MatrixSpace
        from sage.rings.all import QQ
        MS = MatrixSpace(QQ,m,1)
        b  = MS(list(b)) # converted b to a "column vector"
        sA = self.sage2octave_matrix_string(A)
        sb = self.sage2octave_matrix_string(b)
        self.eval("a = " + sA )
        self.eval("b = " + sb )
        soln = octave.eval("c = a \\ b")
        soln = soln.replace("\n\n ","[")
        soln = soln.replace("\n\n","]")
        soln = soln.replace("\n",",")
        sol  = soln[3:]
        return eval(sol)


    def sage2octave_matrix_string(self, A):
        """
        Return an octave matrix from a SAGE matrix.

        INPUT:
            A SAGE matrix with entries in the rationals or reals.

        OUTPUT:
            A string that evaluates to an Octave matrix.

        EXAMPLES:
            sage: M33 = MatrixSpace(QQ,3,3)
            sage: A = M33([1,2,3,4,5,6,7,8,0])
            sage: octave.sage2octave_matrix_string(A)   # requires optional octave
            '[1, 2, 3; 4, 5, 6; 7, 8, 0]'

        AUTHOR: David Joyner and William Stein
        """
        return str(A.rows()).replace('), (', '; ').replace('(', '').replace(')','')

    def de_system_plot(self, f, ics, trange):
        r"""
        Plots (using octave's interface to gnuplot) the solution
        to a $2\times 2$ system of differential equations.

        INPUT:
            f -- a pair of strings representing the differential equations;
                 The independent variable must be called x and the dependent
                 variable must be called y.
            ics -- a pair [x0,y0] such that x(t0) = x0, y(t0) = y0
            trange -- a pair [t0,t1]

        OUTPUT:
            a gnuplot window appears

        EXAMPLES:
           sage: octave.de_system_plot(['x+y','x-y'], [1,-1], [0,2])  # not tested -- does this actually work (on OS X it fails for me -- William Stein, 2007-10)

        This should yield the two plots $(t,x(t)), (t,y(t))$ on the same graph
        (the $t$-axis is the horizonal axis) of the system of ODEs
        $$
          x' = x+y, x(0) = 1;\qquad y' = x-y, y(0) = -1,
                    \quad\text{for}\quad 0 < t < 2.
        $$
        """
        eqn1 = f[0].replace('x','x(1)').replace('y','x(2)')
        eqn2 = f[1].replace('x','x(1)').replace('y','x(2)')
        fcn = "function xdot = f(x,t) xdot(1) = %s; xdot(2) = %s; endfunction"%(eqn1, eqn2)
        self.eval(fcn)
        x0_eqn = "x0 = [%s; %s]"%(ics[0], ics[1])
        self.eval(x0_eqn)
        t_eqn = "t = linspace(%s, %s, 200)'"%(trange[0], trange[1])
        self.eval(t_eqn)
        x_eqn = 'x = lsode("f",x0,t);'
        self.eval(x_eqn)
        self.eval("plot(t,x)")

    def _object_class(self):
        return OctaveElement


class OctaveElement(ExpectElement):
    def _matrix_(self, R):
        r"""
        Return \sage matrix from this octave element.

        EXAMPLES:
            sage: A = octave('[1,2;3,4]')       # optional octave package
            sage: matrix(ZZ, A)
            [1 2]
            [3 4]
            sage: A = octave('[1,2;3,4.5]')     # optional octave package
            sage: matrix(RR, A)
            [1.00000000000000 2.00000000000000]
            [3.00000000000000 4.50000000000000]
        """
        from sage.matrix.all import MatrixSpace
        s = str(self).strip()
        v = s.split('\n ')
        nrows = len(v)
        if nrows == 0:
            return MatrixSpace(R,0,0)(0)
        ncols = len(v[0].split())
        M = MatrixSpace(R, nrows, ncols)
        v = sum([[x for x in w.split()] for w in v], [])
        return M(v)


# An instance
octave = Octave(script_subdirectory='user')

def reduce_load_Octave():
    return octave


import os
def octave_console():
    """
    This requires that the optional octave program be installed and in
    your PATH, but no optional \sage packages need be installed.

    EXAMPLES:
        sage: octave_console()         # not tested
        GNU Octave, version 2.1.73 (i386-apple-darwin8.5.3).
        Copyright (C) 2006 John W. Eaton.
        ...
        octave:1> 2+3
        ans = 5
        octave:2> [ctl-d]

    Pressing ctrl-d exits the octave console and returns you to SAGE.
    octave, like SAGE, remembers its history from one session to
    another.
    """
    os.system('octave')


def octave_version():
    """
    Return the version of Octave installed.

    EXAMPLES:
        sage: octave_version()    # optional -- requires octave; and output is random
        '2.9.12'
    """
    return str(octave('version')).strip()

