r"""
Permutation groups

A {\it permutation group} is a finite group G whose elements are permutations
of a given finite set X (i.e., bijections X --> X) and whose group operation is
the composition of permutations. The number of elements of $X$ is called the
{\it degree} of G.

In \sage a permutation is represented as either a string that defines a
permutation using disjoint cycle notation, or a list of tuples, which represent
disjoint cycles.

\begin{verbatim}
(a,...,b)(c,...,d)...(e,...,f)  <--> [(a,...,b), (c,...,d),..., (e,...,f)]
                  () = identity <--> []
\end{verbatim}

You can make the "named" permutation groups (see \code{permgp_named.py})
and use the following constructions:

-- permutation group generated by elements,
-- direct_product_permgroups, which takes a list of permutation
             groups and returns their direct product.

JOKE:
    Q: What's hot, chunky, and acts on a polygon? A: Dihedral soup.
    Renteln, P. and Dundes, A. "Foolproof: A Sampling of Mathematical
    Folk Humor." Notices Amer. Math. Soc. 52, 24--34, 2005.

AUTHOR:
    - David Joyner (2005-10-14): first version
    - David Joyner (2005-11-17)
    - William Stein (2005-11-26): rewrite to better wrap Gap
    - David Joyner (2005-12-21)
    - Stein and Joyner (2006-01-04): added conjugacy_class_representatives
    - David Joyner (2006-03): reorganization into subdirectory perm_gps;
                              added __contains__, has_element; fixed _cmp_;
                              added subgroup class+methods, PGL,PSL,PSp, PSU classes,
    - David Joyner (2006-06): added PGU, functionality to SymmetricGroup, AlternatingGroup,
                                    direct_product_permgroups
    - David Joyner (2006-08): added degree, ramification_module_decomposition_modular_curve
                              and ramification_module_decomposition_hurwitz_curve
                              methods to PSL(2,q), MathieuGroup, is_isomorphic
    - Bobby Moretti (2006)-10): Added KleinFourGroup, fixed bug in DihedralGroup
    - David Joyner (2006-10): added is_subgroup (fixing a bug found by Kiran Kedlaya),
                              is_solvable, normalizer, is_normal_subgroup, Suzuki
    - David Kohel (2007-02): fixed __contains__ to not enumerate group elements,
                              following the convention for __call__
    - David Harvey, Mike Hansen, Nick Alexander, William Stein (2007-02,03,04,05): Various patches
    - Nathan Dunfield (2007-05): added orbits
    - David Joyner (2007-06): added subgroup method (suggested by David Kohel),
                              composition_series, lower_central_series, upper_central_series,
			      cayley_table, quotient_group, sylow_subgroup, is_cyclic,
                              homology, homology_part, cohomology, cohomology_part,
                              poincare_series, molien_series, is_simple, is_monomial,
                              is_supersolvable, is_nilpotent, is_perfect, is_polycyclic,
                              is_elementary_abelian, is_pgroup, gens_small,
			      isomorphism_type_info_simple_group.
                              moved all the "named" groups to a new file.
    - Nick Alexander (2007-07): move is_isomorphic to isomorphism_to, add from_gap_list
    - William Stein (2007-07): put is_isomorphic back (and make it better)
    - David Joyner (2007-08): fixed bugs in composition_series, upper/lower_central_series, derived_series,

REFERENCES:
    Cameron, P., Permutation Groups. New York: Cambridge University Press, 1999.
    Wielandt, H., Finite Permutation Groups. New York: Academic Press, 1964.
    Dixon, J. and Mortimer, B., Permutation Groups, Springer-Verlag, Berlin/New York, 1996.

NOTE:
    Though Suzuki groups are okay, Ree groups should *not* be wrapped as permutation groups - the
    construction is too slow - unless (for small values or the parameter) they are
    made using explicit generators.

"""

#*****************************************************************************
#       Copyright (C) 2006 William Stein <wstein@gmail.com>
#                          David Joyner <wdjoyner@gmail.com>
#
#  Distributed under the terms of the GNU General Public License (GPL)
#                  http://www.gnu.org/licenses/
#*****************************************************************************


import random

from sage.misc.randstate import current_randstate
import sage.structure.element as element
import sage.groups.group as group

from sage.rings.all      import RationalField, Integer
from sage.interfaces.all import gap, is_GapElement, is_ExpectElement
from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
import sage.structure.coerce as coerce
from sage.rings.finite_field import FiniteField as GF
from sage.rings.arith import factor
from sage.groups.abelian_gps.abelian_group import AbelianGroup
from sage.misc.functional import is_even, log
from sage.rings.rational_field import RationalField
from sage.matrix.matrix_space import MatrixSpace
from sage.rings.polynomial.polynomial_ring_constructor import PolynomialRing
from sage.rings.fraction_field import FractionField
from sage.matrix.matrix_space import MatrixSpace

def direct_product_permgroups(P):
    """
    Takes the direct product of the permutation groups listed in P.

    EXAMPLES:
        sage: G1 = AlternatingGroup([1,2,4,5])
        sage: G2 = AlternatingGroup([3,4,6,7])
        sage: D = direct_product_permgroups([G1,G2,G1])
        sage: D.order()
        1728
        sage: D = direct_product_permgroups([G1])
        sage: D==G1
        True
        sage: direct_product_permgroups([])
        Symmetric group of order 1! as a permutation group
    """
    from permgroup_named import SymmetricGroup
    n = len(P)
    if n == 0:
        return SymmetricGroup(1)
    if n == 1:
        return P[0]
    from sage.groups.perm_gps.permgroup_morphism import PermutationGroupMorphism_from_gap
    G = [H._gap_init_() for H in P]
    Glist = ""
    for H in G:
        Glist = Glist + H + ","
    cmd = "G:=DirectProduct([" + Glist[:-1] + "])"
    gap.eval(cmd)
    return PermutationGroup(gap.eval("G"), from_group = True)

def gap_format(x):
    """
    Put a permutation in Gap format, as a string.
    """
    if isinstance(x,str): return x
    x = str(tuple(x)).replace(' ','')
    return x.replace('),(',')(').replace(',)',')')

def from_gap_list(G, src):
    r"""Convert a string giving a list of GAP permutations into a list of elements of G.

    EXAMPLES:
        sage: from sage.groups.perm_gps.permgroup import from_gap_list
        sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
        sage: L = from_gap_list(G, "[(1,2,3)(4,5), (3,4)]"); L
        [(1,2,3)(4,5), (3,4)]
        sage: L[0].parent() is G
        True
        sage: L[1].parent() is G
        True
    """
    # trim away the list constructs
    src = src.replace("[", "").replace("]", "")
    # cut out the individual elements
    srcs = src.split("),")
    for i in range(len(srcs[:-1])):
        srcs[i] = srcs[i] + ")"
    srcs = map(G, srcs)
    return srcs

def PermutationGroup(x, from_group=False, check=True):
    """
    Return the permutation group associated to $x$ (typically a list
    of generators).

    EXAMPLES:
        sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
        sage: G
        Permutation Group with generators [(1,2,3)(4,5), (3,4)]

    We can also make permutation groups from PARI groups:
        sage: H = pari('x^4 - 2*x^3 - 2*x + 1').polgalois()
        sage: G = PariGroup(H, 4); G
        PARI group [8, -1, 3, "D(4)"] of degree 4
        sage: H = PermutationGroup(G); H          # requires optional database_gap
        Transitive group number 3 of degree 4
        sage: H.gens()                            # requires optional database_gap
        ((1,2,3,4), (1,3))

    We can also create permutation groups whose generators are
    Gap permutation objects.
        sage: p = gap('(1,2)(3,7)(4,6)(5,8)'); p
        (1,2)(3,7)(4,6)(5,8)
        sage: PermutationGroup([p])
        Permutation Group with generators [(1,2)(3,7)(4,6)(5,8)]

    EXAMPLES:
    There is an underlying gap object that implements each permutation group.

        sage: G = PermutationGroup([[(1,2,3,4)]])
        sage: G._gap_()
        Group( [ (1,2,3,4) ] )
        sage: gap(G)
        Group( [ (1,2,3,4) ] )
        sage: gap(G) is G._gap_()
        True
        sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
        sage: current_randstate().set_seed_gap()
        sage: G._gap_().DerivedSeries()
        [ Group( [ (1,2,3)(4,5), (3,4) ] ), Group( [ (1,5)(3,4), (1,5)(2,3), (1,5,4) ] ) ]
    """
    if not is_ExpectElement(x) and hasattr(x, '_permgroup_'):
        return x._permgroup_()
    return PermutationGroup_generic(x, from_group, check)


class PermutationGroup_generic(group.FiniteGroup):
    """
    EXAMPLES:
        sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
        sage: G
        Permutation Group with generators [(1,2,3)(4,5), (3,4)]
        sage: G.center()
        Permutation Group with generators [()]
        sage: G.group_id()          # requires optional database_gap
        [120, 34]
        sage: n = G.order(); n
        120
        sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
        sage: loads(G.dumps()) == G
        True
    """
    def __init__(self, gens, from_group = False, check=True):
        r"""
        Initializes instance of self, a \code{PermutationGroup_generic} object.
        The flag from_group determines whether or not self should be
        initialized as a Gap \code{Group} object and sent to the Gap
        interpreter. If gens are Gap objects, then self is initialized as a Gap
        \code{Group}.

        EXAMPLES:
        We explicitly construct the alternating group on four elements.
            sage: A4 = PermutationGroup([[(1,2,3)],[(2,3,4)]]); A4
            Permutation Group with generators [(1,2,3), (2,3,4)]
            sage: A4.__init__([[(1,2,3)],[(2,3,4)]]); A4
            Permutation Group with generators [(1,2,3), (2,3,4)]
            sage: A4.center()
            Permutation Group with generators [()]
            sage: loads(A4.dumps()) == A4
            True
        """
        if from_group and isinstance(gens, str):
            self.__gap = gens
            self.gens()  # so will check that group can be defined in GAP (e.g., no missing packages, etc.)
            return
        if is_GapElement(gens):
            if from_group:
                # the variable gens is actually a Gap group or string
                # representation of one.
                self.__gap = str(gens)
            else:
                # gens is a Gap object that represents a list of generators for a group
                if gens.Length()>0:
                    # COMMENT: I'm suspicious.  Maybe
                    # gens.name() would be better here? -- William Stein
                    self.__gap = 'Group(%s)'%gens
                else:
                    self.__gap = 'Group(())'
            self.gens() # so will check that group can be defined in GAP (e.g., no missing packages, etc.)
            return

        if check:
            if isinstance(gens, tuple):
                gens = list(gens)
            elif not isinstance(gens, list):
                raise TypeError, "gens must be a tuple or list"
            new_gens = []
            for x in gens:
                if is_GapElement(x):
                    new_gens.append( str(x) )
                else:
                    new_gens.append(gap_format(x))
            gens = new_gens

        cmd = 'Group(%s)'%gens
        cmd = cmd.replace("'","")  # get rid of quotes
        self.__gap = cmd
        self.gens()

    def _gap_init_(self):
        r"""
        Returns a string showing how to declare / initialize self in Gap.
        Stored in the \code{self.__gap} attribute.

        EXAMPLES:
        The \code{_gap_init_} method shows how you would define the Sage
        \code{PermutationGroup_generic} object in Gap:
            sage: A4 = PermutationGroup([[(1,2,3)],[(2,3,4)]]); A4
            Permutation Group with generators [(1,2,3), (2,3,4)]
            sage: A4._gap_init_()
            'Group([((1,2,3)), ((2,3,4))])'
        """
        return self.__gap

    def _magma_init_(self):
        r"""
        Returns a string showing how to declare / intialize self in Magma.

        EXAMPLES:
        We explicitly construct the alternating group on four elements. In
        Magma, one would type the string below to construct the group.
            sage: A4 = PermutationGroup([[(1,2,3)],[(2,3,4)]]); A4
            Permutation Group with generators [(1,2,3), (2,3,4)]
            sage: A4._magma_init_()
            'PermutationGroup<4 | (1,2,3), (2,3,4)>'
        """
        g = str(self.gens())[1:-1]
        return 'PermutationGroup<%s | %s>'%(self.degree(), g)

    def __cmp__(self, right):
        """
        Compare self and right.

        The ordering is whatever it is in Gap.

        EXAMPLES:
            sage: G1 = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
            sage: G2 = PermutationGroup([[(1,2,3),(4,5)]])
            sage: G1 > G2
            True
            sage: G1 < G2
            False
        """
        if not isinstance(right, PermutationGroup_generic):
            return -1
        return right._gap_().__cmp__(self._gap_())


    def __call__(self, x, check=True):
        """
        Coerce x into this permutation group.

        The input can be either a string that defines a permutation in
        cycle notation, a permutation group element, a list of
        integers that gives the permutation as a mapping, a list of
        tuples, or the integer 1.

        EXAMPLES:
        We illustrate each way to make a permutation in S4:
            sage: G = SymmetricGroup(4)
            sage: G((1,2,3,4))
            (1,2,3,4)
            sage: G([(1,2),(3,4)])
            (1,2)(3,4)
            sage: G('(1,2)(3,4)')
            (1,2)(3,4)
            sage: G('(1,2)(3)(4)')
            (1,2)
            sage: G(((1,2,3),(4,)))
            (1,2,3)
            sage: G(((1,2,3,4),))
            (1,2,3,4)
            sage: G([1,2,4,3])
            (3,4)
            sage: G([2,3,4,1])
            (1,2,3,4)
            sage: G(G((1,2,3,4)))
            (1,2,3,4)
            sage: G(1)
            ()

        Some more examples:
            sage: G = PermutationGroup([(1,2,3,4)])
            sage: G([(1,3), (2,4)])
            (1,3)(2,4)
            sage: G(G.0^3)
            (1,4,3,2)
            sage: G(1)
            ()
            sage: G((1,4,3,2))
            (1,4,3,2)
            sage: G([(1,2)])
            Traceback (most recent call last):
            ...
            TypeError: permutation (1,2) not in Permutation Group with generators [(1,2,3,4)]
        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        if isinstance(x, PermutationGroupElement):
            if x.parent() is self:
                return x
            else:
                return PermutationGroupElement(x._gap_(), self, check = check)
        elif isinstance(x, (list, str)):
            return PermutationGroupElement(x, self, check = check)
        elif isinstance(x, tuple):
            return PermutationGroupElement([x], self, check = check)
        elif isinstance(x, (int, long, Integer)) and x == 1:
            return self.identity()
        else:
            raise TypeError, "unable to coerce %s to permutation in %s"%(x, self)

    def _coerce_impl(self, x):
        r"""
        Implicit coercion of x into self.

        EXAMPLES:
        We illustrate some arithmetic that involves implicit coercion
        of elements in different permutation groups.

            sage: g1 = PermutationGroupElement([(1,2),(3,4,5)])
            sage: g1.parent()
            Symmetric group of order 5! as a permutation group
            sage: g2 = PermutationGroupElement([(1,2)])
            sage: g2.parent()
            Symmetric group of order 2! as a permutation group
            sage: g1*g2
            (3,4,5)
            sage: g2*g2
            ()
            sage: g2*g1
            (3,4,5)

        We try to implicitly coerce in a non-permutation, which raises
        a TypeError:

           sage: G = PermutationGroup([[(1,2,3,4)], [(1,2)]])
           sage: G._coerce_impl(1)
           Traceback (most recent call last):
           ...
           TypeError: no implicit coercion of element into permutation group
        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        if isinstance(x, PermutationGroupElement):
            if x.parent() is self:
                return x
            elif x.parent().degree() <= self.degree() and x._gap_() in self._gap_():
                return PermutationGroupElement(x._gap_(), self, check = False)
        raise TypeError, "no implicit coercion of element into permutation group"

    def list(self):
        """
        Return list of all elements of this group.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3,4)], [(1,2)]])
            sage: G.list()
            [(), (3,4), (2,3), (2,3,4), (2,4,3), (2,4), (1,2), (1,2)(3,4), (1,2,3), (1,2,3,4), (1,2,4,3), (1,2,4), (1,3,2), (1,3,4,2), (1,3), (1,3,4), (1,3)(2,4), (1,3,2,4), (1,4,3,2), (1,4,2), (1,4,3), (1,4), (1,4,2,3), (1,4)(2,3)]
        """
        try:
            return list(self.__list)
        except AttributeError:
            pass
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        X = self._gap_().Elements()
        n = X.Length()
        L = [PermutationGroupElement(X[i], self, check = False)
                            for i in range(1,n+1)]
        self.__list = L
        return list(L)

    def __contains__(self, item):
        """
        Returns boolean value of "item in self"

        EXAMPLES:
            sage: G = SymmetricGroup(16)
	    sage: g = G.gen(0)
	    sage: h = G.gen(1)
	    sage: g^7*h*g*h in G
	    True
	    sage: G = SymmetricGroup(4)
	    sage: g = G((1,2,3,4))
	    sage: h = G((1,2))
            sage: H = PermutationGroup([[(1,2,3,4)], [(1,2),(3,4)]])
	    sage: g in H
	    True
	    sage: h in H
	    False
        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        if isinstance(item, PermutationGroupElement):
            if not item.parent() is self:
                try:
		    item = PermutationGroupElement(item._gap_(), self, check = True)
		except:
		    return False
 	    return True
        elif isinstance(item, (list, str)):
            if isinstance(item, list) and len(item) > 0 and not isinstance(item[0], tuple):
                item = gap.eval('PermList(%s)'%item)
            try:
	        item = PermutationGroupElement(item, self, check = True)
	    except:
	        return False
            return True
        elif isinstance(item, tuple):
            if isinstance(item, list) and len(item) > 0 and not isinstance(item[0], tuple):
                item = gap.eval('PermList(%s)'%item)
            try:
	        item = PermutationGroupElement(item, self, check = True)
	    except:
	        return False
            return True
        elif isinstance(item, (int, long, Integer)) and item == 1:
            return True
        return False

    def has_element(self,item):
        """
        Returns boolean value of "item in self" -- however *ignores* parentage.

        EXAMPLES:
            sage: G = CyclicPermutationGroup(4)
	    sage: gens = G.gens()
	    sage: H = DihedralGroup(4)
	    sage: g = G([(1,2,3,4)]); g
            (1,2,3,4)
	    sage: G.has_element(g)
            True
	    sage: h = H([(1,2),(3,4)]); h
            (1,2)(3,4)
	    sage: G.has_element(h)
            False

        """
        L = [str(x) for x in self.list()]
        return (str(item) in L)

    def __iter__(self):
        """
        Return an iterator over the elements of this group.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3)], [(1,2)]])
            sage: [a for a in G]
            [(), (2,3), (1,2), (1,2,3), (1,3,2), (1,3)]
        """
        for g in self.list():
            yield g

    def gens(self):
        """
        Return tuple of generators of this group.  These need not be
        minimal, as they are the generators used in defining this
        group.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3)], [(1,2)]])
            sage: G.gens()
            ((1,2,3), (1,2))

        Note that the generators need not be minimal.
            sage: G = PermutationGroup([[(1,2)], [(1,2)]])
            sage: G.gens()
            ((1,2), (1,2))

            sage: G = PermutationGroup([[(1,2,3,4), (5,6)], [(1,2)]])
            sage: g = G.gens()
            sage: g[0]
            (1,2,3,4)(5,6)
            sage: g[1]
            (1,2)
        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        try:
            return self.__gens
        except AttributeError:
            try:
                gens = self._gap_().GeneratorsOfGroup()
            except TypeError, s:
                raise RuntimeError, "(It might be necessary to install the database_gap optional SAGE package, if you haven't already.)\n%s"%s
            self.__gens = tuple([PermutationGroupElement(gens[n],
                                    self, check = False) for n in \
                                 range(1, int(gens.Length())+1)])
            return self.__gens

    def gens_small(self):
        """
        Returns a generating set of G which has few elements. As neither
        irredundancy, nor minimal length is proven, it is fast.

        EXAMPLES:
            sage: R = "(25,27,32,30)(26,29,31,28)( 3,38,43,19)( 5,36,45,21)( 8,33,48,24)" ## R = right
            sage: U = "( 1, 3, 8, 6)( 2, 5, 7, 4)( 9,33,25,17)(10,34,26,18)(11,35,27,19)" ## U = top
            sage: L = "( 9,11,16,14)(10,13,15,12)( 1,17,41,40)( 4,20,44,37)( 6,22,46,35)" ## L = left
            sage: F = "(17,19,24,22)(18,21,23,20)( 6,25,43,16)( 7,28,42,13)( 8,30,41,11)" ## F = front
            sage: B = "(33,35,40,38)(34,37,39,36)( 3, 9,46,32)( 2,12,47,29)( 1,14,48,27)" ## B = back or rear
            sage: D = "(41,43,48,46)(42,45,47,44)(14,22,30,38)(15,23,31,39)(16,24,32,40)" ## D = down or bottom
            sage: G = PermutationGroup([R,L,U,F,B,D])
            sage: len(G.gens_small())
            2
        """
        return self._gap_().SmallGeneratingSet()

    def gen(self, i):
        r"""
        Returns the ith generator of self; that is, the ith element of the
        list \code{self.gens()}.

        EXAMPLES:
        We explicitly construct the alternating group on four elements:
            sage: A4 = PermutationGroup([[(1,2,3)],[(2,3,4)]]); A4
            Permutation Group with generators [(1,2,3), (2,3,4)]
            sage: A4.gens()
            ((1,2,3), (2,3,4))
            sage: A4.gen(0)
            (1,2,3)
            sage: A4.gen(1)
            (2,3,4)
            sage: A4.gens()[0]; A4.gens()[1]
            (1,2,3)
            (2,3,4)
        """
        return self.gens()[Integer(i)]

    def cayley_table(self, names="x"):
        """
        Returns the multiplication table, or Cayley table, of the
        finite group G in the form of a matrix with symbolic coefficients.
        This function is useful for learning, teaching, and exploring
        elementary group theory. Of course, G must be a group of low order.

        As the last line below illustrates, the ordering used here in the first
        row is the same as in G.list().

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)', '(2,3)'])
            sage: G.cayley_table()
            [x0 x1 x2 x3 x4 x5]
            [x1 x0 x3 x2 x5 x4]
            [x2 x4 x0 x5 x1 x3]
            [x3 x5 x1 x4 x0 x2]
            [x4 x2 x5 x0 x3 x1]
            [x5 x3 x4 x1 x2 x0]
            sage: G.list()[3]*G.list()[3] == G.list()[4]
            True
            sage: G.cayley_table("y")
            [y0 y1 y2 y3 y4 y5]
            [y1 y0 y3 y2 y5 y4]
            [y2 y4 y0 y5 y1 y3]
            [y3 y5 y1 y4 y0 y2]
            [y4 y2 y5 y0 y3 y1]
            [y5 y3 y4 y1 y2 y0]
            sage: G.cayley_table(names="abcdef")
            [a b c d e f]
            [b a d c f e]
            [c e a f b d]
            [d f b e a c]
            [e c f a d b]
            [f d e b c a]

        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        G = self
        n = G.order()
        phi = G._gap_().RegularActionHomomorphism()
        gens = phi.Image().GeneratorsOfGroup()
        N = Integer(gens.Length())
        nm = gens.name()
        gens = [PermutationGroupElement(gap.eval("%s[%s];"%(nm, i))) for i in range(1,N+1)]
        H = PermutationGroup(gens)
        nH = H.order()
        R,vars = PolynomialRing(RationalField(),names,nH).objgens()
        multiplication_table_G = sum([(H.list()[i]).matrix()*vars[i] for i in range(nH)])
        MT = multiplication_table_G.rows()
        MT.sort()
        MT.reverse()
        MS = MatrixSpace(R,n,n)
        return MS(MT)

    multiplication_table = cayley_table

    def identity(self):
        """
        Return the identity element of this group.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3),(4,5)]])
            sage: e = G.identity()
            sage: e
            ()
            sage: g = G.gen(0)
            sage: g*e
            (1,2,3)(4,5)
            sage: e*g
            (1,2,3)(4,5)
        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        return PermutationGroupElement('()', self, check=True)

    def degree(self):
        """
        Synonym for largest_moved_point().

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
            sage: G.degree()
            5
        """
        try:
            return self._deg
        except AttributeError:
            self._deg = self.largest_moved_point()
            return self._deg

    def exponent(self):
        """
         Computes the exponent of the group. The exponent $e$ of a group $G$ is the lcm of
         the orders of its elements, that is, $e$ is the smallest integer such that $g^e=1$ for all
         $g \in G$.

         EXAMPLES:
             sage: G = AlternatingGroup(4)
             sage: G.exponent()
             6
        """
        return Integer(self._gap_().Exponent())

    def largest_moved_point(self):
        """
        Return the largest point moved by a permutation in this group.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4)]])
            sage: G.largest_moved_point()
            4
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4,10)]])
            sage: G.largest_moved_point()
            10
        """
        try:
            return self.__largest_moved_point
        except AttributeError:
            n = Integer(self._gap_().LargestMovedPoint())
        self.__largest_moved_point = n
        return n

    def smallest_moved_point(self):
        """
        Return the smallest point moved by a permutation in this group.

        EXAMPLES:
            sage: G = PermutationGroup([[(3,4)], [(2,3,4)]])
            sage: G.smallest_moved_point()
            2
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4,10)]])
            sage: G.smallest_moved_point()
            1
        """
        try:
            return self.__smallest_moved_point
        except AttributeError:
            n = Integer(self._gap_().SmallestMovedPoint())
        self.__smallest_moved_point = n
        return n

    def orbits(self):
        """
        Returns the orbits of [1,2,...,degree] under the group action.

        EXAMPLES:
           sage: G = PermutationGroup([ [(3,4)], [(1,3)] ])
           sage: G.orbits()
           [[1, 3, 4], [2]]
           sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4,10)]])
           sage: G.orbits()
           [[1, 2, 3, 4, 10], [5], [6], [7], [8], [9]]

        The answer is cached:
           sage: G.orbits() is G.orbits()
           True

        AUTHOR:
             -- Nathan Dunfield
        """
        try:
            return self.__orbits
        except AttributeError:
            O = self._gap_().Orbits("[1..%d]" % self.largest_moved_point()).sage()
            self.__orbits = O
            return O

    def _repr_(self):
        r"""
        Returns a string describing self.

        EXAMPLES:
        We explicitly construct the alternating group on four elements. Note
        that
        the \code{AlternatingGroup} class has its own representation string:
            sage: A4 = PermutationGroup([[(1,2,3)],[(2,3,4)]]); A4
            Permutation Group with generators [(1,2,3), (2,3,4)]
            sage: A4._repr_()
            'Permutation Group with generators [(1,2,3), (2,3,4)]'
            sage: AlternatingGroup(4)._repr_()
            'Alternating group of order 4!/2 as a permutation group'
        """
        G = self.gens()
        return "Permutation Group with generators %s"%list(self.gens())

    def _latex_(self):
        r"""
        Method for describing self in LaTeX. Encapsulates \code{self.gens()}
        in angle brackets to denote that self in generated by these elements.
        Called by the \code{latex()} function.

        EXAMPLES:
        We explicitly construct the alternating group on four elements.
            sage: A4 = PermutationGroup([[(1,2,3)],[(2,3,4)]]); A4
            Permutation Group with generators [(1,2,3), (2,3,4)]
            sage: latex(A4)
            \langle (1,2,3), (2,3,4) \rangle
            sage: A4._latex_()
            '\\langle (1,2,3), (2,3,4) \\rangle'
        """
        return '\\langle ' + \
               ', '.join([x._latex_() for x in self.gens()]) + ' \\rangle'

    def order(self):
        """
        Return the number of elements of this group.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3),(4,5)], [(1,2)]])
            sage: G.order()
            12
        """
        return Integer(self._gap_().Size())

    def random_element(self):
        """
        Return a random element of this group.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3),(4,5)], [(1,2)]])
            sage: G.random_element()
            (1,2)(4,5)
        """
        current_randstate().set_seed_gap()

        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        return PermutationGroupElement(self._gap_().Random(),
                                       self, check=False)

    def group_id(self):
        """
        Return the ID code of this group, which is a list of two
        integers. Requires "optional" database_gap-4.4.x package.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3),(4,5)], [(1,2)]])
            sage: G.group_id()    # requires optional database_gap-4.4.6 package
            [12, 4]
        """
        return [Integer(n) for n in eval(str(self._gap_().IdGroup()))]

    def id(self):
        """
        (Same as self.group_id().) Return the ID code of this group, which is
        a list of two integers. Requires "optional" database_gap-4.4.x package.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3),(4,5)], [(1,2)]])
            sage: G.group_id()    # requires optional database_gap-4.4.6 package
            [12, 4]
        """
        return self.group_id()

    def center(self):
        """
        Return the subgroup of elements of that commute with
        every element of this group.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2,3,4)]])
            sage: G.center()
            Permutation Group with generators [(1,2,3,4)]
            sage: G = PermutationGroup([[(1,2,3,4)], [(1,2)]])
            sage: G.center()
            Permutation Group with generators [()]
        """
        C = self._gap_().Center()
        return PermutationGroup(C, from_group = True)

    def direct_product(self,other,maps=True):
        """
        Wraps GAP's DirectProduct, Embedding, and Projection.

        SAGE calls GAP's DirectProduct, which chooses an efficient representation
        for the direct product. The direct product of permutation groups will
        be a permutation group again. For a direct product D, the GAP
        operation Embedding(D,i) returns the homomorphism embedding the
        i-th factor into D. The GAP operation Projection(D,i) gives the projection
        of D onto the i-th factor.

        INPUT:
            self, other -- permutation groups

        This method returns a 5-tuple - a permutation groups and 4 morphisms.

        OUTPUT:
            D     -- a direct product of the inputs, returned as a permutation group as well
            iota1 -- an embedding of self into D
            iota2 -- an embedding of other into D
            pr1   -- the projection of D onto self  (giving a splitting 1 -> other -> D ->> self -> 1)
            pr2   -- the projection of D onto other (giving a splitting 1 -> self -> D ->> other -> 1)

        EXAMPLES:
            sage: G = CyclicPermutationGroup(4)
            sage: D = G.direct_product(G,False)
            sage: D
            Permutation Group with generators [(1,2,3,4), (5,6,7,8)]
            sage: D,iota1,iota2,pr1,pr2 = G.direct_product(G)
            sage: D; iota1; iota2; pr1; pr2
            Permutation Group with generators [(1,2,3,4), (5,6,7,8)]
            Homomorphism : Cyclic group of order 4 as a permutation group --> Permutation Group with generators [(1,2,3,4), (5,6,7,8)]
            Homomorphism : Cyclic group of order 4 as a permutation group --> Permutation Group with generators [(1,2,3,4), (5,6,7,8)]
            Homomorphism : Permutation Group with generators [(1,2,3,4), (5,6,7,8)] --> Cyclic group of order 4 as a permutation group
            Homomorphism : Permutation Group with generators [(1,2,3,4), (5,6,7,8)] --> Cyclic group of order 4 as a permutation group

            sage: g=D([(1,3),(2,4)]); g
            (1,3)(2,4)
            sage: d=D([(1,4,3,2),(5,7),(6,8)]); d
            (1,4,3,2)(5,7)(6,8)
            sage: iota1(g); iota2(g); pr1(d); pr2(d)
            (1,3)(2,4)
            (5,7)(6,8)
            (1,4,3,2)
            (1,3)(2,4)

        """
        from sage.groups.perm_gps.permgroup_morphism import PermutationGroupMorphism_from_gap
        G1 = self._gap_init_()
        G2 = other._gap_init_()
        cmd1 = "G:=DirectProduct("+G1+","+G2+")"
        cmd2 = "iota1:=Embedding(G,1)"
        cmd3 = "iota2:=Embedding(G,2)"
        cmd4 = "pr1:=Projection(G,1)"
        cmd5 = "pr2:=Projection(G,2)"
        if not(maps):
            return PermutationGroup(gap.eval(cmd1), from_group = True)
        else:
            D = PermutationGroup_generic(gap.eval(cmd1), from_group = True)
            iota1 = PermutationGroupMorphism_from_gap(self,D, cmd2, "iota1")
            iota2 = PermutationGroupMorphism_from_gap(other,D, cmd3, "iota2")
            pr1 = PermutationGroupMorphism_from_gap(D,self, cmd4, "pr1")
            pr2 = PermutationGroupMorphism_from_gap(D,other, cmd5, "pr2")
            return D,iota1,iota2,pr1,pr2

    def subgroup(self,gens):
        """
        Wraps the PermutationGroup_subgroup constructor. The argument gens is
        a list of elements of self.

        EXAMPLES:
            sage: G = PermutationGroup([(1,2,3),(3,4,5)])
            sage: g = G((1,2,3))
            sage: G.subgroup([g])
            Subgroup of Permutation Group with generators [(1,2,3), (3,4,5)] generated by [(1,2,3)]

        """
        return PermutationGroup_subgroup(self,gens)

    def sylow_subgroup(self, p):
        """
        Returns a Sylow p-subgroups of the finite group G, where p is
        a prime. This is a p-subgroup of G whose index in G is coprime to p.
        Wraps the GAP function SylowSubgroup.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)', '(2,3)'])
            sage: G.sylow_subgroup(2)
            Permutation Group with generators [(2,3)]
            sage: G.sylow_subgroup(5)
            Permutation Group with generators [()]

        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        G = self
        gap.eval("G := %s"%G._gap_init_())
        gap.eval("Ssgp := SylowSubgroup(G, %s);"%p)
        gap.eval("gens := GeneratorsOfGroup( Ssgp );")
        N = Integer(gap.eval("N := Length(gens);"))
        if N>0:
            gens = [PermutationGroupElement(gap.eval("gens[%s];"%j)) for j in range(1,N+1)]
            H = PermutationGroup(gens)
        else:
            H = PermutationGroup([()])
        return H

    def quotient_group(self, N):
        """
        Returns the quotient group permgp/N, where N is a normal subgroup. Wraps the
        GAP operator "/".

        EXAMPLES:
            sage: G = PermutationGroup([(1,2,3), (2,3)])
            sage: N = PermutationGroup([(1,2,3)])
            sage: G.quotient_group(N)
            Permutation Group with generators [(1,2)]

        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        G = self
        gap.eval("G := %s"%G._gap_().name())
        gap.eval("N := %s"%N._gap_().name())
        gap.eval("Q := G/N;")
        gap.eval("phi := RegularActionHomomorphism( Q );")
        gap.eval("gens := GeneratorsOfGroup( Image( phi ));")
        N = Integer(gap.eval("N := Length(gens);"))
        if N>0:
            gens = [PermutationGroupElement(gap.eval("gens[%s];"%i)) for i in range(1,N+1)]
            Q = PermutationGroup(gens)
            return Q
        else:
            return PermutationGroup([()])

    def cohomology(self, n, p = 0):
        r"""
        Computes the group cohomology H_n(G, F), where F = Z if p=0
        and F = Z/pZ if p >0 is a prime. Wraps HAP's GroupHomology
        function, written by Graham Ellis.

        REQUIRES:
            GAP package HAP (in gap_packages-*.spkg).

        EXAMPLES:
            sage: G = SymmetricGroup(3)
            sage: G.cohomology(5)                              # requires optional gap_packages
            Trivial Abelian Group
            sage: G.cohomology(5,2)                            # requires optional gap_packages
            Multiplicative Abelian Group isomorphic to C2
            sage: G.homology(5,3)                              # requires optional gap_packages
            Trivial Abelian Group
            sage: G.homology(5,4)                              # requires optional gap_packages
            Traceback (most recent call last):
            ...
            ValueError: p must be 0 or prime

        This computes $H^4(S_3,ZZ)$, $H^4(S_3,ZZ/2ZZ)$, resp.

        AUTHORS:
            David Joyner and Graham Ellis

        REFERENCES:
            G. Ellis, "Computing group resolutions", J. Symbolic Computation. Vol.38, (2004)1077--1118
            (Available at \code{http://hamilton.nuigalway.ie/}.
            D. Joyner, "A primer on computational group homology and cohomology",
            \code{http://front.math.ucdavis.edu/0706.0549}

        """
        gap.eval('RequirePackage("HAP")')
        from sage.rings.arith import is_prime
        if not (p == 0 or is_prime(p)):
            raise ValueError, "p must be 0 or prime"
        G = self
        GG = G._gap_init_()
        if p == 0:
            L = eval(gap.eval("GroupCohomology(%s,%s)"%(GG,n)))
        else:
	    L = eval(gap.eval("GroupCohomology(%s,%s,%s)"%(GG,n,p)))
        return AbelianGroup(len(L),L)

    def cohomology_part(self, n, p = 0):
        """
        Computes the p-part of the group cohomology $H^n(G, F)$, where $F = Z$ if $p=0$
        and $F = Z/pZ$ if $p >0$ is a prime. Wraps HAP's Homology function, written by
        Graham Ellis, applied to the $p$-Sylow subgroup of $G$.

        REQUIRES:
            GAP package HAP (in gap_packages-*.spkg).

        EXAMPLES:
            sage: G = SymmetricGroup(5)
            sage: G.cohomology_part(7,2)                   # requires optional gap_packages
            Multiplicative Abelian Group isomorphic to C2 x C2 x C2
            sage: G = SymmetricGroup(3)
            sage: G.cohomology_part(2,3)
            Multiplicative Abelian Group isomorphic to C3

        AUTHORS:
            David Joyner and Graham Ellis
        """
        gap.eval('RequirePackage("HAP")')
        from sage.rings.arith import is_prime
        if not (p == 0 or is_prime(p)):
            raise ValueError, "p must be 0 or prime"
        G = self
        GG = G._gap_init_()
        if p == 0:
            H = AbelianGroup(1,[1])
        else:
            gap.eval("S := SylowSubgroup(%s,%s)"%(GG,p))
            gap.eval("R:=ResolutionFiniteGroup(S,%s)"%(n+1))
            gap.eval("HR:=HomToIntegers(R)")
	    L = eval(gap.eval("Cohomology(HR,%s)"%n))
        return AbelianGroup(len(L),L)

    def homology(self, n, p = 0):
        r"""
        Computes the group homology $H_n(G, F)$, where $F = Z$ if $p=0$
        and $F = Z/pZ$ if $p >0$ is a prime. Wraps HAP's GroupHomology
        function, written by Graham Ellis.

        REQUIRES:
            GAP package HAP (in gap_packages-*.spkg).

        AUTHORS:
            David Joyner and Graham Ellis

        The example below computes $H_7(S_5,ZZ)$, $H_7(S_5,ZZ/2ZZ)$, $H_7(S_5,ZZ/3ZZ)$, and $H_7(S_5,ZZ/5ZZ)$,
        resp. To compute the $2$-part of $H_7(S_5,ZZ)$, use the \code{homology_part} function.

        EXAMPLES:
            sage: G = SymmetricGroup(5)
            sage: G.homology(7)                              # requires optional gap_packages
            Multiplicative Abelian Group isomorphic to C2 x C2 x C4 x C3 x C5
            sage: G.homology(7,2)                              # requires optional gap_packages
            Multiplicative Abelian Group isomorphic to C2 x C2 x C2 x C2 x C2
            sage: G.homology(7,3)                              # requires optional gap_packages
            Multiplicative Abelian Group isomorphic to C3
            sage: G.homology(7,5)                              # requires optional gap_packages
            Multiplicative Abelian Group isomorphic to C5

        REFERENCES:
            G. Ellis, "Computing group resolutions", J. Symbolic Computation. Vol.38, (2004)1077--1118
            (Available at \code{http://hamilton.nuigalway.ie/}.
            D. Joyner, "A primer on computational group homology and cohomology",
            \code{http://front.math.ucdavis.edu/0706.0549}

        """
        gap.eval('RequirePackage("HAP")')
        from sage.rings.arith import is_prime
        if not (p == 0 or is_prime(p)):
            raise ValueError, "p must be 0 or prime"
        G = self
        GG = G._gap_init_()
        if p == 0:
            L = eval(gap.eval("GroupHomology(%s,%s)"%(GG,n)))
        else:
	    L = eval(gap.eval("GroupHomology(%s,%s,%s)"%(GG,n,p)))
        return AbelianGroup(len(L),L)

    def homology_part(self, n, p = 0):
        r"""
        Computes the $p$-part of the group homology $H_n(G, F)$, where $F = Z$ if $p=0$
        and $F = Z/pZ$ if $p >0$ is a prime. Wraps HAP's Homology function, written by
        Graham Ellis, applied to the $p$-Sylow subgroup of $G$.

        REQUIRES:
            GAP package HAP (in gap_packages-*.spkg).

        EXAMPLES:
            sage: G = SymmetricGroup(5)
            sage: G.homology_part(7,2)                              # requires optional gap_packages
            Multiplicative Abelian Group isomorphic to C2 x C2 x C2 x C2 x C4

        AUTHORS:
            David Joyner and Graham Ellis
        """
        gap.eval('RequirePackage("HAP")')
        from sage.rings.arith import is_prime
        if not (p == 0 or is_prime(p)):
            raise ValueError, "p must be 0 or prime"
        G = self
        GG = G._gap_init_()
        if p == 0:
            H = AbelianGroup(1,[1])
        else:
            gap.eval("S := SylowSubgroup(%s,%s)"%(GG,p))
            gap.eval("R:=ResolutionFiniteGroup(S,%s)"%(n+1))
            gap.eval("TR:=TensorWithIntegers(R);")
	    L = eval(gap.eval("Homology(TR,%s)"%n))
        return AbelianGroup(len(L),L)

    def character_table(self):
        r"""
        Returns the matrix of values of the irreducible characters of
        a permutation group $G$ at the conjugacy classes of $G$. The
        columns represent the the conjugacy classes of $G$ and the
        rows represent the different irreducible characters in the
        ordering given by GAP.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3)]])
            sage: G.order()
            12
            sage: G.character_table()
            [         1          1          1          1]
            [         1          1 -zeta3 - 1      zeta3]
            [         1          1      zeta3 -zeta3 - 1]
            [         3         -1          0          0]
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3)]])
            sage: CT = gap(G).CharacterTable()

        Type print gap.eval("Display(%s)"%CT.name()) to display this nicely.

            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4)]])
            sage: G.order()
            8
            sage: G.character_table()
            [ 1  1  1  1  1]
            [ 1 -1 -1  1  1]
            [ 1 -1  1 -1  1]
            [ 1  1 -1 -1  1]
            [ 2  0  0  0 -2]
            sage: CT = gap(G).CharacterTable()

        Again, type print gap.eval("Display(%s)"%CT.name()) to display this.

            sage: SymmetricGroup(2).character_table()
            [ 1 -1]
            [ 1  1]
            sage: SymmetricGroup(3).character_table()
            [ 1 -1  1]
            [ 2  0 -1]
            [ 1  1  1]
            sage: SymmetricGroup(5).character_table()
            [ 1 -1  1  1 -1 -1  1]
            [ 4 -2  0  1  1  0 -1]
            [ 5 -1  1 -1 -1  1  0]
            [ 6  0 -2  0  0  0  1]
            [ 5  1  1 -1  1 -1  0]
            [ 4  2  0  1 -1  0 -1]
            [ 1  1  1  1  1  1  1]
            sage: list(AlternatingGroup(6).character_table())
            [(1, 1, 1, 1, 1, 1, 1), (5, 1, 2, -1, -1, 0, 0), (5, 1, -1, 2, -1, 0, 0), (8, 0, -1, -1, 0, zeta5^3 + zeta5^2 + 1, -zeta5^3 - zeta5^2), (8, 0, -1, -1, 0, -zeta5^3 - zeta5^2, zeta5^3 + zeta5^2 + 1), (9, 1, 0, 0, 1, -1, -1), (10, -2, 1, 1, 0, 0, 0)]

        Suppose that you have a class function $f(g)$ on $G$ and you
        know the values $v_1, ..., v_n$ on the conjugacy class
        elements in \code{conjugacy_classes_representatives(G)} =
        $[g_1, \ldots, g_n]$.  Since the irreducible characters
        $\rho_1, \ldots, \rho_n$ of $G$ form an $E$-basis of the space
        of all class functions ($E$ a ``sufficiently large''
        cyclotomic field), such a class function is a linear
        combination of these basis elements, $f = c_1\rho_1 + \cdots +
        c_n\rho_n$. To find the coefficients $c_i$, you simply solve
        the linear system \code{character_table_values(G)}*$[v_1, ...,
        v_n] = [c_1, ..., c_n]$,
        where $[v_1, ...,v_n]$ = \code{character_table_values(G)}$^{-1}[c_1, ...,c_n]$.

        AUTHORS:
            - David Joyner and William Stein (2006-01-04)
        """
        G    = self._gap_()
        cl   = G.ConjugacyClasses()
        n    = Integer(cl.Length())
        irrG = G.Irr()
        ct   = [[irrG[i+1,j+1] for j in range(n)] for i in range(n)]

        from sage.rings.all import CyclotomicField
        e = irrG.Flat().Conductor()
        K = CyclotomicField(e)
        ct = [[K(x) for x in v] for v in ct]

        # Finally return the result as a matrix.
        from sage.matrix.all import MatrixSpace
        MS = MatrixSpace(K,n)
        return MS(ct)

    def conjugacy_classes_representatives(self):
        """
        Returns a complete list of representatives of conjugacy classes
        in a permutation group G. The ordering is that given by GAP.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4)]])
            sage: cl = G.conjugacy_classes_representatives(); cl
            [(), (2,4), (1,2)(3,4), (1,2,3,4), (1,3)(2,4)]
            sage: cl[3] in G
            True

            sage: G = SymmetricGroup(5)
            sage: G.conjugacy_classes_representatives ()
            [(), (1,2), (1,2)(3,4), (1,2,3), (1,2,3)(4,5), (1,2,3,4), (1,2,3,4,5)]


        AUTHOR: David Joyner and William Stein (2006-01-04)
        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        cl = self._gap_().ConjugacyClasses()
        n = Integer(cl.Length())
        L = gap("List([1..Length(%s)], i->Representative(%s[i]))"%(
            cl.name(),  cl.name()))
        return [PermutationGroupElement(L[i], self, check=False) \
                for i in range(1,n+1)]

    def conjugacy_classes_subgroups(self):
        """
        Returns a complete list of representatives of conjugacy classes
        of subgroups in a permutation group G. The ordering is that given by GAP.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4)]])
            sage: cl = G.conjugacy_classes_subgroups()
            sage: cl
            [Permutation Group with generators [()],
             Permutation Group with generators [(1,2)(3,4)],
             Permutation Group with generators [(1,3)(2,4)],
             Permutation Group with generators [(2,4)],
             Permutation Group with generators [(1,4)(2,3), (1,2)(3,4)],
             Permutation Group with generators [(1,3)(2,4), (2,4)],
             Permutation Group with generators [(1,3)(2,4), (1,2,3,4)],
             Permutation Group with generators [(1,3)(2,4), (1,2)(3,4), (1,2,3,4)]]

            sage: G = SymmetricGroup(3)
            sage: G.conjugacy_classes_subgroups()
            [Permutation Group with generators [()],
             Permutation Group with generators [(2,3)],
             Permutation Group with generators [(1,2,3)],
             Permutation Group with generators [(1,3,2), (1,2)]]

        AUTHOR: David Joyner (2006-10)
        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        G = self._gap_()
        cl = G.ConjugacyClassesSubgroups()
        n = Integer(cl.Length())
        L = gap("List([1..Length(%s)], i->Representative(%s[i]))"%(
            cl.name(),  cl.name()))
        return [PermutationGroup(L[i], from_group=True, check=False) \
                for i in range(1,n+1)]

    def normal_subgroups(self):
        """
        Return the normal subgroups of this group as a (sorted in increasing
        order) list of permutation groups.

        The normal subgroups of $H = PSL(2,7)xPSL(2,7)$ are $1$, two copies
        of $PSL(2,7)$ and $H$ itself, as the following example shows.
        EXAMPLES:
            sage: G = PSL(2,7)
            sage: D = G.direct_product(G)
            sage: H = D[0]
            sage: NH = H.normal_subgroups()
            sage: len(NH)
            4
            sage: NH[1].is_isomorphic(G)
            True
            sage: NH[2].is_isomorphic(G)
            True

        """
        ans = []
        NS = self._gap_().NormalSubgroups()
        n = NS.Length()
        for i in range(1,n+1):
            ans.append(PermutationGroup(NS[i].GeneratorsOfGroup()))
        return ans

    def normalizer(self,g):
        """
        Returns the normalizer of g in self.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4)]])
            sage: g = G([(1,3)])
            sage: G.normalizer(g)
            Permutation Group with generators [(1,3), (2,4)]
            sage: g = G([(1,2,3,4)])
            sage: G.normalizer(g)
            Permutation Group with generators [(1,2,3,4), (1,3)(2,4), (2,4)]
            sage: H = G.subgroup([G([(1,2,3,4)])])
            sage: G.normalizer(H)
            Permutation Group with generators [(1,2,3,4), (1,3)(2,4), (2,4)]

        """
        if g in self:
            N = self._gap_().Normalizer(str(g))
        else:
            N = self._gap_().Normalizer(gap(g).name())
        return PermutationGroup(N.GeneratorsOfGroup())

    def isomorphism_type_info_simple_group(self):
        """
        Is the group is simple, then this returns the name of the group.

        EXAMPLES:
            sage: G = CyclicPermutationGroup(5)
            sage: G.isomorphism_type_info_simple_group()
            rec( series := "Z", parameter := 5, name := "Z(5)" )

        """
        if self.is_simple():
            info = self._gap_().IsomorphismTypeInfoFiniteSimpleGroup()
            return info
        else:
            return TypeError, "Group must be simple."

    ######################  Boolean tests #####################

    def is_abelian(self):
        """
        Return True if this group is abelian.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)', '(1,2,3,4,5)'])
            sage: G.is_abelian()
            False
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_abelian()
            True
        """
        t = self._gap_().IsAbelian()
        return t.bool()

    def is_commutative(self):
        """
        Return True if this group is commutative.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)', '(1,2,3,4,5)'])
            sage: G.is_commutative()
            False
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_commutative()
            True
        """
        return self.is_abelian()

    def is_cyclic(self):
        """
        Return True if this group is cyclic.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)', '(1,2,3,4,5)'])
            sage: G.is_cyclic()
            False
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_cyclic()
            True
        """
        t = self._gap_().IsCyclic()
        return t.bool()

    def is_elementary_abelian(self):
        """
        Return True if this group is elementary abelian. An elementary abelian
        group is a finite Abelian group, where every nontrivial element has order p,
        where p is a prime.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)', '(1,2,3,4,5)'])
            sage: G.is_elementary_abelian()
            False
            sage: G = PermutationGroup(['(1,2,3)','(4,5,6)'])
            sage: G.is_elementary_abelian()
            True
        """
        t = self._gap_().IsElementaryAbelian()
        return t.bool()

    def isomorphism_to(self,right):
        """
        Return an isomorphism self to right if the groups are isomorphic, otherwise None.

        INPUT:
            self -- this group
            right -- a permutation group

        OUTPUT:
            None or a morphism of permutation groups.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)', '(1,2,3,4,5)'])
            sage: H = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.isomorphism_to(H) is None
            True
            sage: G = PermutationGroup([(1,2,3), (2,3)])
            sage: H = PermutationGroup([(1,2,4), (1,4)])
            sage: G.isomorphism_to(H)
            Homomorphism : Permutation Group with generators [(1,2,3), (2,3)] --> Permutation Group with generators [(1,2,4), (1,4)]
        """
        current_randstate().set_seed_gap()

        if not isinstance(right, PermutationGroup_generic):
            raise TypeError, "right must be a permutation group"
        G = self._gap_init_()
        H = right._gap_init_()
        gap.eval("x:=IsomorphismGroups( %s, %s )"%(G,H))
        s = gap.eval("x")
        if s == "fail":
            return None
        # slice and dice the GAP return to build a SAGE group homomorphism
        src, dst = s.split("->")
        # we eval to get things as lists
        srcs = from_gap_list(self, src)
        dsts = from_gap_list(right, dst)
        from permgroup_morphism import PermutationGroupMorphism_im_gens
        return PermutationGroupMorphism_im_gens(self, right, srcs, dsts)

    def is_isomorphic(self, right):
        """
        Return True if the groups are isomorphic. If mode="verbose" then
        an isomorphism is printed.

        INPUT:
            self -- this group
            right -- a permutation group
        OUTPUT:
            bool

        EXAMPLES:
            sage: v = ['(1,2,3)(4,5)', '(1,2,3,4,5)']
            sage: G = PermutationGroup(v)
            sage: H = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_isomorphic(H)
            False
            sage: G.is_isomorphic(G)
            True
            sage: G.is_isomorphic(PermutationGroup(list(reversed(v))))
            True
        """
        if not isinstance(right, PermutationGroup_generic):
            raise TypeError, "right must be a permutation group"
        G = self._gap_()
        H = right._gap_()
        return gap.eval("IsomorphismGroups( %s, %s )"%(G.name(),H.name())) != "fail"

    def is_monomial(self):
        """
        Returns True if the group is monomial. A finite group is monomial if every irreducible
        complex character is induced from a linear character of a subgroup.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_monomial()
            True
        """
        ans = self._gap_().IsMonomialGroup()
        return ans.bool()

    def is_nilpotent(self):
        """
        Return True if this group is nilpotent.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)', '(1,2,3,4,5)'])
            sage: G.is_nilpotent()
            False
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_nilpotent()
            True
        """
        t = self._gap_().IsNilpotent()
        return t.bool()

    def is_normal(self,other):
        """
        Return True if the group self is a normal subgroup of other.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: H = PermutationGroup(['(1,2,3)(4,5)', '(1,2,3,4,5)'])
            sage: G.is_normal(H)
            True
        """
        t = self._gap_().IsNormal(other._gap_())
        return t.bool()

    def is_perfect(self):
        """
        Return True if this group is perfect. A group is perfect if it equals its derived subgroup.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)', '(1,2,3,4,5)'])
            sage: G.is_perfect()
            False
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_perfect()
            False
        """
        t = self._gap_().IsPerfectGroup()
        return t.bool()

    def is_pgroup(self):
        """
        Returns True if the group is a p-group. A finite group is a p-group if
        its order is of the form $p^n$ for a prime integer p and a nonnegative integer n.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3,4,5)'])
            sage: G.is_pgroup()
            True
        """
        ans = self._gap_().IsPGroup()
        return ans.bool()

    def is_polycyclic(self):
        r"""
        Return True if this group is polycyclic. A group is polycyclic if it has a subnormal series
        with cyclic factors. (For finite groups this is the same as if the group is solvable - see
        \code{is_solvable})].)

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)', '(1,2,3,4,5)'])
            sage: G.is_polycyclic()
            False
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_polycyclic()
            True
        """
        t = self._gap_().IsPolycyclicGroup()
        return t.bool()

    def is_simple(self):
        """
        Returns True if the group is simple. A group is simple if it has no proper normal
        subgroups.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_simple()
            False
        """
        ans = self._gap_().IsSimpleGroup()
        return ans.bool()

    def is_solvable(self):
        """
        Returns True if the group is solvable.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_solvable()
            True
        """
        ans = self._gap_().IsSolvableGroup()
        return ans.bool()

    def is_subgroup(self,other):
        """
        Returns true if self is a subgroup of other.

        EXAMPLES:
            sage: G = AlternatingGroup(5)
            sage: H = SymmetricGroup(5)
            sage: G.is_subgroup(H)
            True
        """
        G = other
        gens = self.gens()
        for i in range(len(gens)):
            x = gens[i]
            if not (G.has_element(x)):
                return False
        return True

    def is_supersolvable(self):
        """
        Returns True if the group is supersolvable. A finite group is supersolvable if it has a
        normal series with cyclic factors.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)(4,5)'])
            sage: G.is_supersolvable()
            True

        """
        ans = self._gap_().IsSupersolvableGroup()
        return ans.bool()

    def is_transitive(self):
        """
        Return True if self is a transitive group, i.e. if the action of
        self on self.set() is transitive.

        EXAMPLES:
            sage: G = SymmetricGroup(5)
            sage: G.is_transitive()
            True
            sage: G = PermutationGroup(['(1,2)(3,4)(5,6)'])
            sage: G.is_transitive()
            False
        """
        ans = self._gap_().IsTransitive()
        return ans.bool()

    ############## Series ######################

    def composition_series(self):
        """
        Return the composition series of this group as a list of
        permutation groups.

        EXAMPLES:
        These computations use pseudo-random numbers, so we set the
        seed for reproducible testing.
            sage: set_random_seed(0)
            sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
            sage: G.composition_series()  # random output
            [Permutation Group with generators [(1,2,3)(4,5), (3,4)], Permutation Group with generators [(1,5)(3,4), (1,5)(2,3), (1,5,4)], Permutation Group with generators [()]]

        """
        current_randstate().set_seed_gap()
        ans = []
        DS = self._gap_().CompositionSeries()
        n = DS.Length()
        for i in range(1,n+1):
            ans.append(PermutationGroup(DS[i].GeneratorsOfGroup()))
        return ans

    def derived_series(self):
        """
        Return the derived series of this group as a list of
        permutation groups.

        EXAMPLES:
        These computations use pseudo-random numbers, so we set the
        seed for reproducible testing.
            sage: set_random_seed(0)
            sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
            sage: G.derived_series()  # random output
            [Permutation Group with generators [(1,2,3)(4,5), (3,4)], Permutation Group with generators [(1,5)(3,4), (1,5)(2,4), (2,4)(3,5)]]
        """
        current_randstate().set_seed_gap()
        ans = []
        DS = self._gap_().DerivedSeries()
        n = DS.Length()
        for i in range(1,n+1):
            ans.append(PermutationGroup(DS[i].GeneratorsOfGroup()))
        return ans

    def lower_central_series(self):
        """
        Return the lower central series of this group as a list of
        permutation groups.

        EXAMPLES:
        These computations use pseudo-random numbers, so we set the
        seed for reproducible testing.
            sage: set_random_seed(0)
            sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
            sage: G.lower_central_series()  # random output
            [Permutation Group with generators [(1,2,3)(4,5), (3,4)], Permutation Group with generators [(1,5)(3,4), (1,5)(2,3), (1,3)(2,4)]]

        """
        current_randstate().set_seed_gap()
        ans = []
        DS = self._gap_().LowerCentralSeriesOfGroup()
        n = DS.Length()
        for i in range(1,n+1):
            ans.append(PermutationGroup(DS[i].GeneratorsOfGroup()))
        return ans

    def molien_series(self):
        r"""
        Returns the Molien series of a transtive permutation group.
        The function
        $$
        M(x) = (1/|G|)\sum_{g\in G} det(1-x*g)^(-1)
        $$
        is sometimes called the "Molien series" of G.
        GAP's \code{MolienSeries} is associated to a character of a group G.
        How are these related? A group G, given as a permutation
        group on n points, has a "natural" representation of
        dimension n, given by permutation matrices. The Molien series
        of G is the one associated to that permutation representation of
        G using the above formula. Character values then count fixed
        points of the corresponding permutations.

        EXAMPLES:
            sage: G = SymmetricGroup(5)
            sage: G.molien_series()                              # requires optional gap_packages
            1/(-x^15 + x^14 + x^13 - x^10 - x^9 - x^8 + x^7 + x^6 + x^5 - x^2 - x + 1)
            sage: G = SymmetricGroup(3)
            sage: G.molien_series()                              # requires optional gap_packages
            1/(-x^6 + x^5 + x^4 - x^2 - x + 1)

        """
        G = self
        GG = G._gap_init_()
        gap.eval("pi := NaturalCharacter( %s )"%GG)
        gap.eval("cc := ConstituentsOfCharacter( pi )")
        M = gap.eval("M := MolienSeries(Sum(cc))")
        R = PolynomialRing(RationalField(),"x")
	x = R.gen()
        nn = gap.eval("NumeratorOfRationalFunction(M)")
        dd = gap.eval("DenominatorOfRationalFunction(M)")
        FF = FractionField(R)
        return FF(nn.replace("_1",""))/FF(dd.replace("_1",""))

    def normal_subgroups(self):
        """
        Return the normal subgroups of this group as a (sorted in increasing
        order) list of permutation groups.

        The normal subgroups of $H = PSL(2,7)xPSL(2,7)$ are $1$, two copies
        of $PSL(2,7)$ and $H$ itself, as the following example shows.
        EXAMPLES:
            sage: G = PSL(2,7)
            sage: D = G.direct_product(G)
            sage: H = D[0]
            sage: NH = H.normal_subgroups()
            sage: len(NH)
            4
            sage: NH[1].is_isomorphic(G)
            True
            sage: NH[2].is_isomorphic(G)
            True

        """
        ans = []
        NS = self._gap_().NormalSubgroups()
        n = NS.Length()
        for i in range(1,n+1):
            ans.append(PermutationGroup(NS[i].GeneratorsOfGroup()))
        return ans

    def normalizer(self,g):
        """
        Returns the normalizer of g in self.

        EXAMPLES:
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4)]])
            sage: g = G([(1,3)])
            sage: G.normalizer(g)
            Permutation Group with generators [(1,3), (2,4)]
            sage: g = G([(1,2,3,4)])
            sage: G.normalizer(g)
            Permutation Group with generators [(1,2,3,4), (1,3)(2,4), (2,4)]
            sage: H = G.subgroup([G([(1,2,3,4)])])
            sage: G.normalizer(H)
            Permutation Group with generators [(1,2,3,4), (1,3)(2,4), (2,4)]

        """
        if g in self:
            N = self._gap_().Normalizer(str(g))
        else:
            N = self._gap_().Normalizer(gap(g).name())
        return PermutationGroup(N.GeneratorsOfGroup())

    def poincare_series(self, p=2, n=10):
        """
        Returns the Poincare series of G mod p (p must be a prime), for n>1
        large. In other words, if you input a finite group G, a prime p,
        and a positive integer n, it returns a quotient of polynomials
        f(x)=P(x)/Q(x) whose coefficient of $x^k$ equals the rank of the
        vector space $H_k(G,ZZ/pZZ)$, for all k in the range $1\leq k \leq n$.

        REQUIRES:
            GAP package HAP (in gap_packages-*.spkg).

        EXAMPLES:
            sage: G = SymmetricGroup(5)
            sage: G.poincare_series(2,10)                              # requires optional gap_packages
            (x^2 + 1)/(x^4 - x^3 - x + 1)
            sage: G = SymmetricGroup(3)
            sage: G.poincare_series(2,10)                              # requires optional gap_packages
            1/(-x + 1)

        AUTHORS:
            David Joyner and Graham Ellis
        """
        gap.eval('RequirePackage("HAP")')
        from sage.rings.arith import is_prime
        if not (p == 0 or is_prime(p)):
            raise ValueError, "p must be 0 or prime"
        G = self
        GG = G._gap_init_()
        ff = gap.eval("ff := PoincareSeriesPrimePart(%s,%s,%s)"%(GG,p,n))
        R = PolynomialRing(RationalField(),"x")
	x = R.gen()
        nn = gap.eval("NumeratorOfRationalFunction(ff)")
        dd = gap.eval("DenominatorOfRationalFunction(ff)")
        FF = FractionField(R)
        return FF(nn)/FF(dd)

    def smallest_moved_point(self):
        """
        Return the smallest point moved by a permutation in this group.

        EXAMPLES:
            sage: G = PermutationGroup([[(3,4)], [(2,3,4)]])
            sage: G.smallest_moved_point()
            2
            sage: G = PermutationGroup([[(1,2),(3,4)], [(1,2,3,4,10)]])
            sage: G.smallest_moved_point()
            1
        """
        try:
            return self.__smallest_moved_point
        except AttributeError:
            n = Integer(self._gap_().SmallestMovedPoint())
        self.__smallest_moved_point = n
        return n

    def sylow_subgroup(self, p):
        """
        Returns a Sylow p-subgroups of the finite group G, where p is
        a prime. This is a p-subgroup of G whose index in G is coprime to p.
        Wraps the GAP function SylowSubgroup.

        EXAMPLES:
            sage: G = PermutationGroup(['(1,2,3)', '(2,3)'])
            sage: G.sylow_subgroup(2)
            Permutation Group with generators [(2,3)]
            sage: G.sylow_subgroup(5)
            Permutation Group with generators [()]

        """
        from sage.groups.perm_gps.permgroup_element import PermutationGroupElement
        G = self
        gap.eval("G := %s"%G._gap_init_())
        gap.eval("Ssgp := SylowSubgroup(G, %s);"%p)
        gap.eval("gens := GeneratorsOfGroup( Ssgp );")
        N = Integer(gap.eval("N := Length(gens);"))
        if N>0:
            gens = [PermutationGroupElement(gap.eval("gens[%s];"%j)) for j in range(1,N+1)]
            H = PermutationGroup(gens)
        else:
            H = PermutationGroup([()])
        return H


    def upper_central_series(self):
        """
        Return the upper central series of this group as a list of
        permutation groups.

        EXAMPLES:
        These computations use pseudo-random numbers, so we set the
        seed for reproducible testing.
            sage: set_random_seed(0)
            sage: G = PermutationGroup([[(1,2,3),(4,5)],[(3,4)]])
            sage: G.upper_central_series()
            [Permutation Group with generators [()]]
        """
        current_randstate().set_seed_gap()
        ans = []
        DS = self._gap_().UpperCentralSeriesOfGroup()
        n = DS.Length()
        for i in range(1,n+1):
            ans.append(PermutationGroup(DS[i].GeneratorsOfGroup()))
        return ans

class PermutationGroup_subgroup(PermutationGroup_generic):
    """
    Subgroup subclass of PermutationGroup_generic, so instance methods are
    inherited.

    EXAMPLES:
        sage: G = CyclicPermutationGroup(4)
        sage: gens = G.gens()
        sage: H = DihedralGroup(4)
        sage: PermutationGroup_subgroup(H,list(gens))
        Subgroup of Dihedral group of order 8 as a permutation group generated by [(1,2,3,4)]
        sage: K=PermutationGroup_subgroup(H,list(gens))
        sage: K.list()
        [(), (1,2,3,4), (1,3)(2,4), (1,4,3,2)]
        sage: K.ambient_group()
        Dihedral group of order 8 as a permutation group
        sage: K.gens()
        [(1,2,3,4)]
    """
    def __init__(self, ambient, gens, from_group = False,
                 check=True):
        r"""
        Initialization method for the \code{PermutationGroup_subgroup} class.

        INPUTS:
            ambient -- the ambient group from which to construct this subgroup
            gens -- the generators of the subgroup
            from_group -- True: subroup is generated from a Gap string representation of the generators
            check-- True: checks if gens are indeed elements of the ambient group

        EXAMPLES:
        An example involving the dihedral group on four elements. $D_8$
        contains a cyclic subgroup or order four:
            sage: G = DihedralGroup(4)
            sage: H = CyclicPermutationGroup(4)
            sage: gens = H.gens(); gens
            ((1,2,3,4),)
            sage: S = PermutationGroup_subgroup(G,list(gens))
            sage: S
            Subgroup of Dihedral group of order 8 as a permutation group generated by [(1,2,3,4)]
            sage: S.list()
            [(), (1,2,3,4), (1,3)(2,4), (1,4,3,2)]
            sage: S.ambient_group()
            Dihedral group of order 8 as a permutation group

        However, $D_8$ does not contain a cyclic subgroup of order three:
            sage: G = DihedralGroup(4)
            sage: H = CyclicPermutationGroup(3)
            sage: gens = H.gens()
            sage: S = PermutationGroup_subgroup(G,list(gens))
            Traceback (most recent call last):
            ...
            TypeError: each generator must be in the ambient group
        """
        if not isinstance(ambient, PermutationGroup_generic):
            raise TypeError, "ambient (=%s) must be perm group."%ambient
        if not isinstance(gens, list):
            raise TypeError, "gens (=%s) must be a list"%gens
        if check:
            pass

        self.__ambient_group = ambient
        self.__gens = gens
        cmd = 'Group(%s)'%gens
        cmd = cmd.replace("'","")  # get rid of quotes
        self.__gap = cmd

        G = ambient
        if check:
            for g in gens:
                if g not in G:
                    raise TypeError, "each generator must be in the ambient group"
        self.__ambient_group = G

        PermutationGroup_generic.__init__(self, gens, from_group, check)

    def __cmp__(self, other):
        r"""
        Compare self and other.  If self and other are in a common ambient group,
        then self <= other precisely if self is contained in other.

        EXAMPLES:
            sage: G = CyclicPermutationGroup(4)
	    sage: gens = G.gens()
	    sage: H = DihedralGroup(4)
 	    sage: PermutationGroup_subgroup(H,list(gens))
            Subgroup of Dihedral group of order 8 as a permutation group generated by [(1,2,3,4)]
	    sage: K=PermutationGroup_subgroup(H,list(gens))
            sage: G<K
            False
            sage: G>K
            False
        """
        if self is other:
            return 0
        if not isinstance(other, PermutationGroup_generic):
            return -1
        c = cmp(self.ambient_group(), other.ambient_group())
        if c: return c
        if self.is_subgroup(other):
            return -1
        else:
            return 1

    def _repr_(self):
        r"""
        Returns a string representation / description of the permutation
        subgroup.

        EXAMPLES:
        An example involving the dihedral group on four elements, $D_8$:
            sage: G = DihedralGroup(4)
            sage: H = CyclicPermutationGroup(4)
            sage: gens = H.gens()
            sage: S = PermutationGroup_subgroup(G, list(gens))
            sage: S
            Subgroup of Dihedral group of order 8 as a permutation group generated by [(1,2,3,4)]
            sage: S._repr_()
            'Subgroup of Dihedral group of order 8 as a permutation group generated by [(1,2,3,4)]'
        """
        s = "Subgroup of %s generated by %s"%(self.ambient_group(), self.gens())
        return s

    def _latex_(self):
        r"""
        Return latex representation of this group.

        EXAMPLES:
        An example involving the dihedral group on four elements, $D_8$:
            sage: G = DihedralGroup(4)
            sage: H = CyclicPermutationGroup(4)
            sage: gens = H.gens()
            sage: S = PermutationGroup_subgroup(G, list(gens))
            sage: latex(S)
            Subgroup of Dihedral group of order 8 as a permutation group generated by [(1,2,3,4)]
            sage: S._latex_()
            'Subgroup of Dihedral group of order 8 as a permutation group generated by [(1,2,3,4)]'
        """
        return self._repr_()


    def ambient_group(self):
        """
        Return the ambient group related to self.

        EXAMPLES:
        An example involving the dihedral group on four elements, $D_8$:
            sage: G = DihedralGroup(4)
            sage: H = CyclicPermutationGroup(4)
            sage: gens = H.gens()
            sage: S = PermutationGroup_subgroup(G, list(gens))
            sage: S.ambient_group()
            Dihedral group of order 8 as a permutation group
            sage: S.ambient_group() == G
            True
        """
        return self.__ambient_group

    def gens(self):
        """
        Return the generators for this subgroup.

        EXAMPLES:
        An example involving the dihedral group on four elements, $D_8$:
            sage: G = DihedralGroup(4)
            sage: H = CyclicPermutationGroup(4)
            sage: gens = H.gens()
            sage: S = PermutationGroup_subgroup(G, list(gens))
            sage: S.gens()
            [(1,2,3,4)]
            sage: S.gens() == list(H.gens())
            True
        """
        return self.__gens

