##############################################################################
#
#  DSAGE: Distributed SAGE
#
#       Copyright (C) 2006, 2007 Yi Qiang <yqiang@gmail.com>
#
#  Distributed under the terms of the GNU General Public License (GPL)
#
#    This code is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
#
#  The full text of the GPL is available at:
#
#                  http://www.gnu.org/licenses/
#
##############################################################################

from twisted.python import log
import sqlite3

def optimize_sqlite(con):
    """
    Sets some pragma settings that are supposed to optimize SQLite.
    Settings taken from:
    http://web.utk.edu/~jplyon/sqlite/SQLite_optimization_FAQ.html

    """

    cur = con.cursor()
    cur.execute("pragma cache_size=4000") # Use double the default cache_size
    cur.execute("pragma synchronous=off") # do not wait for disk writes
    cur.execute("pragma temp_store=2") # store temporary results in memory

    return con

def table_exists(con, tablename):
    """
    Check if a given table exists.
    If the below query is not None, then the table exists

    """

    query = """SELECT name FROM sqlite_master
    WHERE type = 'table' AND name = ?;
    """

    cur = con.cursor()
    cur.execute(query, (tablename,))
    result = cur.fetchone()

    return result

def create_table(con, tablename, query):
    """
    Creates a table given the connection.

    """

    log.msg('Creating table %s...' % tablename)
    con.execute(query)

def fields(cursor):
    """
    Given a DB API 2.0 cursor object that has been executed, returns
    a dictionary that maps each field name to a column index, 0 and up.

    """

    results = {}
    for column, desc in enumerate(cursor.description):
        results[desc[0]] = column

    return results

def drop_table(con, table):
    """
    Drops a table, use with caution!

    """

    cur = con.cursor()
    query = "DROP TABLE ?"
    cur.execute(query, (table,))

def add_trigger(con, trigger):
    con.execute(trigger)

def update_value(con, table, uniq_name, uniq_id, key, value):
    """
    Sets the appropriate value for a job in the database.

    """

    cur = con.cursor()
    query = """UPDATE %s
               SET %s=?
               WHERE %s=?
            """ % (table, key, uniq_name)
    if table == 'jobs':
        if key == 'data' or key == 'result': # Binary objects
            if value != None:
                cur.execute(query, (sqlite3.Binary(value), uniq_id))
        else:
            cur.execute(query, (value, uniq_id))
    else:
        cur.execute(query, (value, uniq_id))
    con.commit()