////////////////////////////////////////////////////////////////////////////////
//
// Measure the time like using a stop watch
//
// Burkhard Militzer                                     Livermore, 5-10-01
//       
////////////////////////////////////////////////////////////////////////////////

#include <sys/times.h>
#include <sys/time.h>
#include <ostream>

#ifndef _TIMER_
#define _TIMER_

class Timer {
 public:
  enum mode {user, system, total, off};
 private:
  struct tms buf;
  mode m;
  const double f;
  double time0,runTime;
  bool running;
  double runTimeLimit;
  bool outOfTimeReported;

  struct timeval val;
  struct timezone zone;

 public:
  // one might want to change this back later so that Array1 get initialized with Timer::user
  //  Timer(const mode & ma=user, const bool flag=true);
  Timer(const mode & ma=off, const bool flag=true):
    m(ma),
    f(1.0/(double)(GetHertz())),
    runTime(0.0),
    running(flag),
    runTimeLimit(0.0),
    outOfTimeReported(false) {
    zone.tz_minuteswest=0;
    if (flag) Start();
  }
    
  void Start() {
    time0   = GetTime();
    running = true;
  }

  double Read() {
    return (running) ? (GetTime()-time0) : runTime;
  }

  double ReadAndRestart() {
    double t =Read();
    Start();
    return t;
  }

  double Stop() {
    if (!running) error("Timer::Stop: Timer already stopped.");
    double t =GetTime();
    runTime = t-time0;
    running = false;
    return runTime;
  }

  void Continue() {
    if (running) 
      error("Cannot continue a timer before having stopped it.");
    double t = GetTime();
    time0    = t-runTime;
    running  = true;
    runTime  = 0.0;
  }

  void Reset() {
    running = false;
    runTime = 0.0;
  }

  mode GetMode() const {
    return m;
  }
  void SetMode(const mode ma) {
    m = ma;
  }

  double Fraction(const double t) {
    if (t==0.0) return 0.0;
    return Read()/t;
  }
  double Fraction(Timer & t) {
    return Fraction(t.Read());
  }
  
  bool OutOfTime() {
    if (runTimeLimit==0.0) return false; // no time limit set
    return Read()>runTimeLimit;
  }
  double GetTimeLimit() const {
    return runTimeLimit;
  }
  void SetTimeLimit(const double maxTime) {
    runTimeLimit = maxTime;
  }
  bool GetOutOfTimeReportFlag() const {
    return outOfTimeReported;
  }
  void SetOutOfTimeReportFlag() {
    outOfTimeReported=true;
  }
  void UnsetOutOfTimeReportFlag() {
    outOfTimeReported=false;
  }

  friend ostream& operator<<(ostream &os, Timer & t );

 private:

  long GetHertz() const;

  double GetUserTime() {
    times(&buf);
    return (double) (buf.tms_utime)*f;
  }
  double GetSystemTime() {
    times(&buf);
    return (double) (buf.tms_stime)*f;
  }
  double GetTotalTime() {
    gettimeofday(&val,&zone);
    return (double)val.tv_sec + 1e-6*(double) val.tv_usec;
  }
  
  double GetTime() {
    switch (m) {
    case user:
      return GetUserTime();
    case system:
      return GetSystemTime();
    case total:
      return GetTotalTime();
    default: // case "off"
      return 0.0;
    }
  }

};

#endif // _TIMER_
