/////////////////////////////////////////////////////////////////////////////////
//                                                                             //
// Interface to different RANDOM NUMBER GENERATORS                             //
// Make generators interchangeable                                             //                 
//                                                                             //
// Burkhard Militzer                                    Urbana 1998            //
//                                                                             //
/////////////////////////////////////////////////////////////////////////////////

#ifndef _RANDOM_
#define _RANDOM_

#include "Standard.h"

#ifdef USE_SPRNG

// remember, the MPI version is SPRNG must be linked if mpi.h is included here.
// otherwise strange link error
#ifdef USE_MPI
#include <mpi.h>
#endif

// #define SIMPLE_SPRNG // use the simpler SPRNG interface (one stream only)
#include "sprng.h"          

#ifndef SIMPLE_SPRNG
extern int* sprngStream;
#endif 

inline double Random() { 
#ifdef SIMPLE_SPRNG
  return sprng();
#else
  return sprng(sprngStream);
#endif
}

#ifdef SIMPLE_SPRNG
inline void InitRandom(const bool randomSeedFlag) {
  if (randomSeedFlag) {
    int seed = make_sprng_seed();   
    //    init_sprng(seed,SPRNG_DEFAULT);                      // SPRNG version 1.0
    init_sprng(DEFAULT_RNG_TYPE,seed,SPRNG_DEFAULT);  // SPRNG version 2.0
  } else {
    warning("SIMPLE_SPRNG: Check if running without init_sprng() is ok!!!");
  }
}

#else

inline void InitRandom(const bool randomSeedFlag, const int stream=0) {
  int seed;
  if (randomSeedFlag) {
    seed = make_sprng_seed(); // make new seed depending of date and time
  } else {
    //    seed = 985456376;  // Ashok's default seed
    seed = 0;                // Still will reproduce the "SIMPLE_SPRING" results without explicite initialization
  }
  int gType = 0;
  sprngStream = init_sprng(gType,stream,stream+1,seed,SPRNG_DEFAULT); // initialize stream 
}
#endif

#else // USE_SPRNG 

#ifdef USE_CRANF32
#include "cranf32.h"
inline double Random() { 
  return getranf();
}
inline void InitRandom(const bool randomSeedFlag, const int stream=0){
  if (randomSeedFlag || stream>0) error("No implemented for CRANF32");
  int seed[]={12345,54321,-1};
  setseed(seed);
};

#else // USE_CRANF32

#include <stdlib.h>
#include <unistd.h> 
#include <sys/times.h>
#include <sys/time.h>
inline void InitRandom(const bool randomSeedFlag, const int stream=0) {
  if (stream>0) error("No implemented for DRAND48");

  // added to run properly on G5 using gcc 3.3
  unsigned short seed16v[3]={0,0,0};

  if (randomSeedFlag) {
    struct timeval val;
    struct timezone zone;
    gettimeofday(&val,&zone);
    long t = val.tv_sec*1000000 + val.tv_usec; // 'long' should have 8 bytes, 'short' should have 2 bytes
    seed16v[0] = (unsigned short)((t >> 32) & 0xFFFF) ;
    seed16v[1] = (unsigned short)((t >> 16) & 0xFFFF) ;
    seed16v[2] = (unsigned short)( t        & 0xFFFF) ;
    /*
    unsigned short* tt = (unsigned short*) &t;
    seed16v[0] = tt[0];
    seed16v[1] = tt[1];
    seed16v[2] = tt[2];
    */
    //    srand (time (NULL));
  }

  //  unsigned short *j;
  //  j = seed48(seed16v);
  seed48(seed16v);
}

inline double Random() { 
  return drand48();
}

#endif // USE_CRANF32
#endif // USE_SPRNG 

inline double RandomRange(const double d) { // return -d ... +d
  return (2.0*Random()-1.0)*d;
}
inline double RandomRange(const double a, const double b) { // return a ... b
  return (b-a)*Random()+a;
}
inline double RandomRangeLogScale(const double a, const double b) {
  return a*exp(log(b/a)*Random());
}

inline int RandomInt(const int n) { // values from 0 to n-1. 'n' is excluded!
  return int(double(n)*Random());
}

inline int RandomInt(const int n1, const int n2) { // values from n1 to n2-1. 'n2' is excluded!
  return n1+int(double(n2-n1)*Random());
}

inline int RandomIntInclusive(const int n1, const int n2) { // values from n1 to n2. 'n2' is included!
  return n1+int(double(n2-n1+1)*Random());
}

inline void RandomTwoDifferentInt(const int n, int & n1, int & n2) { // values from 0 to n-1
  n1 = RandomInt(n);   // pick first integer
  n2 = RandomInt(n-1); // pick second integer from remaining numbers. n1 is not in list
  if (n2>=n1) n2++;    // must add +1 if n2>n1 because 'n1' is not in list
}

inline double BoxMuller() {
  double x1 = Random();
  double x2 = Random();
  double r = sqrt(-2.0*log(x1)) * cos(2.0*pi*x2);
  return r;
}

inline void BoxMuller(double & r1, double & r2) {
  double x1 = Random();
  double x2 = Random();
  double f  = sqrt(-2.0*log(x1));
  double a  = 2.0*pi*x2;
  r1 = f * cos(a);
  r2 = f * sin(a);
}

#endif // _RANDOM_
