//////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  CMS Method                                                             //
//                                                                         //
//  B. Militzer                                     Berkeley 05-18-19      //
//                                                                         //
//  based on William Hubbard's methodology and                             //
//  based on Sean Wahl's tidal CMS code                                    //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "Standard.h"
#include "Timer.h"
#include "Random.h"
#include "ParseCommandLineArguments.h"
#include "Switch.h"
#include "FindRoot.h"
#include "CMS.h"
#include "Parser.h"

// this routine is NOT used in ReadBackInLastWindState() below
int ParseArray1(Parser & parser, const string & name, Array1 <double> & x, const bool fixed=false, const double compareSizes=false) {
  int n = parser.GetNumberOfDoublesFollowingString(name);
  if (!fixed) x.CopyResize(n);
  if (compareSizes && n!=x.Size()) error("Array size in file not as expected",name,n,x.Size());
  for(int i=0; i<min(n,x.Size()); i++) {
    x[i] = parser.GetDoubleFollowingString(name,i); // i is an offset
  }
  return n;
}

void ParseArray1AllowResizing(Parser & parser, const string & name, Array1 <double> & x) {
  if (x.Size()==0) error("ParseArray1AllowResizing problem");

  Array1 <double> xInFile;
  int n = ParseArray1(parser,name,xInFile,false,false);

  if (n==1) {
    x = xInFile[0];
    return;
  }
  if (n==x.Size()) {
    x = xInFile;
    return;
  }

  Spline s(n);
  for(int i=0; i<n; i++) {
    double t = double(i)/double(n-1);
    s.PushBack(t,xInFile[i]);
  }
  s.Init();
  for(int i=0; i<x.Size(); i++) {
    double t = double(i)/double(x.Size()-1);
    x[i] = s.Interpolate(t);
  }

}

/////////////////////////////////////////////////////////////////////////////////////////////

int main(int argc, char **argv) {

  Array1 <string> arg = CopyCommandLineArgumentsToArray(argc,argv);
  cout.precision(10);

  if (ParseCommandLineFlagWithOrWithoutDash(arg,"noRandom")) {
    //  InitRandom(false); warning("Using standard random numbers");
    InitRandom(false);
    cout << "Using standard random numbers stream: ";
  } else {
    InitRandom(true);
    cout << "Using randomized random numbers stream: ";
  }
  Write(Random());

  bool uranusFlag = true;
  //  bool uranusFlag = false;
  if (ParseCommandLineFlagWithOrWithoutDash(arg,"uranus") || ParseCommandLineFlagWithOrWithoutDash(arg,"U")) {
    uranusFlag = true;
  }
  if (ParseCommandLineFlagWithOrWithoutDash(arg,"neptune") || ParseCommandLineFlagWithOrWithoutDash(arg,"N")) {
    uranusFlag = false;
  }

  cout << endl;
  if (uranusFlag) {
    cout << "Starting a CMS calculation for Uranus." << endl;
  } else {
    cout << "Starting a CMS calculation for Neptune." << endl;
  }
  cout << endl;
  cout << "Process ID: " << getpid() << endl;
  cout << endl;

  ////////////////////////////////////////////////////////////////////////////////////////

  int modelIndex = 3;
  ParseCommandLineArgumentNoSpace(arg,"m=",modelIndex);
  Write(modelIndex);
  string modelFileName = string("log") + ((uranusFlag) ? "U" : "N") + IntToString(modelIndex) + "_";

  ////////////////////////////////////////////////////////////////////////////////////////

  CMS cms(uranusFlag);

  //  int nl=256;
  int nl=512;
  //  int nl=1024;
  //  int nl=2048; // made default on 1/1/22
  ParseCommandLineArgumentNoSpace(arg,"nl=",nl);

  int np=16;
  ParseCommandLineArgumentNoSpace(arg,"np=",np);

  int nInt=16;
  //  int nInt=32;
  ParseCommandLineArgumentNoSpace(arg,"nInt=",nInt);

  if (nl % nInt == 0) {
    cout << " Increasing nl from " << nl << " to " << nl+1 << " to allow acceleration." << endl;
    nl++;
  }

  cms.Init(nl,np,nInt); // (#Layers,#poly) // for accelerated CMS - add +1 for acceleration
  cout << endl;

  //    cms.SetLambdasWeighted();
  cms.SetLambdasSmoothlyWeighted(); 

  cms.SetInitialDensitiesToPolytropeEOS();

  cms.Precompute(); // sets xq(#nnq),wq(#nq), pn_mu(#nq,#np), pn_0(#np)

  cout << endl;

  ////////////////////////////////////////////////////////////////////////////////////////

  MCParameters p;
  p.RandomStartingPoint();

  if (uranusFlag) {
    //  :::::: MC_parameters: r1= 0.8156260000 r2/r1= 0.6004452240 r3/r1= 0.1804199522 H1= 1.9230528947 H2= 0.5015141091 H3= 0.2053039011 chi2= 2.637583787e-08
    p.r1 = 0.815626;
    p.r2 = 0.6004452240; // = la2/la1
    p.r3 = 0.1804199522; // = la3/la1
    p.H1 = 1.9230528947;
    p.H2 = 0.5015141091;
    p.H3 = 0.2053039011;
  } else {
    // MC_parameters: r1= 0.8681910000 r2/r1= 0.5905829194 r3/r1= 0.2437657370 H1= 2.2450339941 H2= 0.4418363789 H3= 0.1055113790 chi2= 1.224015256e-13
    p.r1 = 0.8681910000;
    p.r2 = 0.5905829194; // = la2/la1
    p.r3 = 0.2437657370; // = la3/la1
    p.H1 = 2.2450339941;
    p.H2 = 0.4418363789;
    p.H3 = 0.1055113790;
  }

  cout << " Parameters: " << p << endl; 

  ////////////////////////////////////////////////////////////////////////////////////////

  Parser parser(modelFileName+"state.txt");
  int iModelInFile=0;
  while (parser.ReadLine()) {
    iModelInFile++;
    cout << endl;
    cout << " ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" << endl;
    cout << " ^^^^^ Parsing file " << modelFileName+"state.txt" << " model: " << iModelInFile << " ^^^^^^" << endl;
    cout << " ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" << endl;
    cout << endl;

    //    string logFileName         = modelFileName+"_"+IntToString(iModelInFile);
    //    double chi2DataFile        = parser.GetDoubleFollowingString("chi2Data=");
    //    double chi2ModFile         = parser.GetDoubleFollowingString("chi2Mod=");
    //    double chi2CloudsFile      = parser.GetDoubleFollowingString("chi2Clouds=");
    //    double chi2OmegaCurvature  = parser.GetDoubleFollowingString("chi2OmegaCurvature=");
    //    double chi2HHeImmiscibility= parser.GetDoubleFollowingString("chi2HHeImmiscibility=");
    //    double chi2TotalFile       = parser.GetDoubleFollowingString("chi2Total=");
    p.r1 = parser.GetDoubleFollowingString("r1=");
    p.r2 = parser.GetDoubleFollowingString("r2/r1=");
    p.r3 = parser.GetDoubleFollowingString("r3/r1=");
    p.H1 = parser.GetDoubleFollowingString("H1=");
    p.H2 = parser.GetDoubleFollowingString("H2=");
    p.H3 = parser.GetDoubleFollowingString("H3=");

    cout << " Parameters: " << p << endl; 

    cout << endl;
    cout << " ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" << endl;
  }


  ////////////////////////////////////////////////////////////////////////////////////////

  //  cms.LoadStateFromFiles("logU2_");
  //  cms.LoadStateFromFiles("logN2_");
  cms.ReadFilesAndPerformCMSCalculations(modelFileName);
 
  cout << endl;
  cout << " Volume integration check: total mass      = " << cms.CalculateVolumeIntegral(0) << endl;
  cout << " Moment of inertia                         = " << cms.CalculateVolumeIntegral(1) << endl;
  cout << " Moment of inertia (Hubbard's Eq.5, no DR) = " << cms.CalculateMomentOfInertia() << endl;
  cout << " Angular momentum                          = " << cms.CalculateVolumeIntegral(2) << endl;
  cout << endl;
  cms.PrintRotationPeriod();
  cout << endl;
  cout << " ================================================================== " << endl;

  ////////////////////////////////////////////////////////////////////////////////////////

  cout << endl << "Program finished properly." << endl;
  return 0;

}
