/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MC Parameters                                                          //
//                                                                         //
//  B. Militzer                                  San Antonio, TX, 05-25-17 //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#ifndef _MCPARAMETERS_
#define _MCPARAMETERS_

// needed inside "CloudProfile.h" and "CMS.h"
// #define JUPITER 

// needed here and inside "CMS.h"
// #define DIFF_ROT
// #define TWE

#include "Array.h"
#include "Grid.h"
#include "Physics.h"
#include "Random.h"

class MCParameters {
 public:
  int np; // number of adjustable parameters
  //  double beta,dBeta; 

  //  double S1,dS1;           // do not change, for now
  //  double YXY1,dYXY1;       // change only for Saturn
  //  double Z1,dZ1;

  double r1,dr1; //  r1 = la1               with la1 being the upper radius of H2O+H layer
  double r2,dr2; //  r2 = la2/la1 < 1       with la2 being the upper radius of CNH layer
  double r3,dr3; //  r3 = la3/la1 < la2/la1 with la3 being the upper radius of the core
  double H1,dH1;
  double H2,dH2;
  double H3,dH3;
  
  /////////////////////////////////////////////////////////////////////////////////////////////////////
  
  MCParameters() {
    UpdateNumberOfParameters();
  }

  double Parameter(int i) const {
    if (i<0 || i>np) error("Parameter() i problem",i,np);
    if (i--== 0) return r1;
    if (i--== 0) return r2; 
    if (i--== 0) return r3; 
    if (i--== 0) return H1; 
    if (i--== 0) return H2; 
    return H3; 
  }

  double & Parameter(int i) {
    if (i<0 || i>np) error("Parameter() i problem",i,np);
    if (i--== 0) return r1;
    if (i--== 0) return r2; 
    if (i--== 0) return r3; 
    if (i--== 0) return H1; 
    if (i--== 0) return H2; 
    return H3; 
 }

 double operator[](const int i) const { // was not implemented for a long time to prevent unintentional use of p[i]
    return Parameter(i);
  }
  double & operator[](const int i) { // was not implemented for a long time to prevent unintentional use of p[i]
    return Parameter(i);
  }

  double ParameterStep(int i) const {
    if (i<0 || i>np) error("Parameter() i problem",i,np);
    if (i--== 0) return dr1;
    if (i--== 0) return dr2; 
    if (i--== 0) return dr3; 
    if (i--== 0) return dH1; 
    if (i--== 0) return dH2; 
    return dH3; 
  }

  double & ParameterStep(int i) {
    if (i<0 || i>np) error("Parameter() i problem",i,np);
    if (i--== 0) return dr1;
    if (i--== 0) return dr2; 
    if (i--== 0) return dr3; 
    if (i--== 0) return dH1; 
    if (i--== 0) return dH2; 
    return dH3; 
  }

  private:

  int CalculateNumberOfParameters() const {
    return 6;
  }

  void UpdateNumberOfParameters() {
    np = CalculateNumberOfParameters();
  }

 public:

  /////////////////////////////////////////////////////////////////////////////////////////////////////

  void PrintParametersAndStepSizes() const {
    for(int i=0; i<np; i++) {
      cout << "MCParameter " << IntToStringMaxNumber(i+1,np) << " = " << Parameter(i) << "    step= " << ParameterStep(i) << endl;
    }
  }

  bool Valid(const bool print=false) const {
    if (r1<0.0 || r1>1.0) { if (print) { cout << *this << " but r1<0 or r1>1" << endl; } return false;}
    if (r2<0.0 || r2>1.0) { if (print) { cout << *this << " but r2<0 or r2>1" << endl; } return false;}
    if (r3<0.0 || r3>1.0) { if (print) { cout << *this << " but r3<0 or r3>1" << endl; } return false;}
    //    if (r1<r2)  { if (print) { cout << *this << " but r1<r2" << endl; } return false;}
    if (r2>0.9) { if (print) { cout << *this << " but r2>0.9" << endl; } return false;} // added on 1/18/24
    if (r3>r2)  { if (print) { cout << *this << " but r3<r2"  << endl; } return false;}

    //    if (H1<0.0 || H1>1.0) { if (print) { cout << *this << " but H1<0 or H1>1" << endl; } return false;}
    //    if (H2<0.0 || H2>1.0) { if (print) { cout << *this << " but H2<0 or H2>1" << endl; } return false;}
    //    if (H3<0.0 || H3>1.0) { if (print) { cout << *this << " but H3<0 or H3>1" << endl; } return false;}
    if (H1<0.0) { if (print) { cout << *this << " but H1<0" << endl; } return false;}
    if (H2<0.0) { if (print) { cout << *this << " but H2<0" << endl; } return false;}
    if (H3<0.0) { if (print) { cout << *this << " but H3<0" << endl; } return false;}
    if (H2<0.0 || H2>1.0) { if (print) { cout << *this << " but H2<0 or H2>1" << endl; } return false;}
    if (H3<0.0 || H3>1.0) { if (print) { cout << *this << " but H3<0 or H3>1" << endl; } return false;}
    if (H3>H2)            { if (print) { cout << *this << " but H3>H2"        << endl; } return false;}

    return true;
  }

 public:
  double DScale() const {
    return 1.0;
    //    return 0.1;
    //    return 0.01;
    //    return 0.001;
    //    return 0.0001;
  }
  
  void SetStepSizes() {
    //    dP1 = 1.0 * PC::GPaToAU * DScale();
    //    dP2 = 1.0 * PC::GPaToAU * DScale();
    //    dP3 = 1.0 * PC::GPaToAU * DScale();
    dr1 =  0.01 * DScale();
    dr2 =  0.01 * DScale();
    dr3 =  0.01 * DScale();
    dH1 = 0.02 * DScale();
    dH2 = 0.02 * DScale();
    dH3 = 0.02 * DScale();
  }

  int GetNumberOfParameters() const {
    return np;
  }

  void RandomStartingPoint() { // UpdateOmegaSpline() not called here.
    r1 = RandomRange(0.75,0.85);
    r2 = RandomRange(0.4,0.6)   / r1;
    r3 = RandomRange(0.10,0.20) / r1;
    H1 = RandomRange(0.0,   1.0);
    H2 = RandomRange(H1,1.0);
    H3 = RandomRange(0.0,H2);
  }

  //////////////////////////////////////////////////////////////////////////////////////
  
  friend ostream& operator<<(ostream & os, const MCParameters & p) {
    //    const bool lines = true;
    const bool lines = false;

    os << "MC_parameters:";

    //    Form f(fix,8,6);
    Form f(fix,12,10);
    //    f.SetShowPos();
 
    //    os << " beta=  " << f(p.beta);
    if (lines) os << endl;
    os << " r1= " << f(p.r1);
    os << " r2/r1= " << f(p.r2);
    os << " r3/r1= " << f(p.r3);
    os << " H1= " << f(p.H1);
    os << " H2= " << f(p.H2);
    os << " H3= " << f(p.H3);

    return os;
  }

};

#endif
