/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Adaptive Gauss-Kronrod integration                                      //
//                                                                         //
// This C++ version was written by Burkhard Militzer  Livermore 02-20-02   //
// Based on the GNU scientific library                                     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#ifndef _GKINTEGRATION_
#define _GKINTEGRATION_

#include <iterator>
#include <list>
#include "Standard.h"

class GK15 {
 public:
  static const int n=8;
  static const double xgk[n];
  static const double wg[n/2];
  static const double wgk[n];
};

class GK21 {
 public:
  static const int n=11;
  static const double xgk[n];
  static const double wg[n/2];
  static const double wgk[n];
};

class GK31 {
 public:
  static const int n=16;
  static const double xgk[n];
  static const double wg[n/2];
  static const double wgk[n];
};

class GK41 {
 public:
  static const int n=21;
  static const double xgk[n];
  static const double wg[(n+1)/2];
  static const double wgk[n];
};

class GK51 {
 public:
  static const int n=26;
  static const double xgk[n];
  static const double wg[n/2];
  static const double wgk[n];
};

class GK61 {
 public:
  static const int n=31;
  static const double xgk[n];
  static const double wg[n/2];
  static const double wgk[n];
};

////////////////////////////////////////////////////////////////////////////////////////

template <class F, class GKRule> 
class GKIntegration {
 private:

  class IntervalResult {
  public:
    IntervalResult(const double a_, const double b_):a(a_),b(b_),result(0.0),err(0.0){};
    double a,b,result,err;

    double ErrorL() const {
      return (b!=a) ? err/(b-a) : err;
    }

    friend ostream& operator<<(ostream &os, const IntervalResult & ir) {
      os << "[a= " << ir.a
	 << " b= " << ir.b
	 << " result= " << ir.result
	 << " error/L= " << ir.err/(ir.b-ir.a)
	 << " error= " << ir.err
	 << " ]";
      return os;
    }
  };

  list <IntervalResult> ir;
  F & f; // could be not const in case where calling f() actually modifies the object
  bool relativeErrors;   // true if standard accurracy argument refer to relative errors
  int divisionMax;       // set an upper limit for the number interval divisons
                         // do not subdivide if the intervalk length is < (b-a)*2(-divMax)
                         // --> useful to have in place in case of singular integrands
  int maxListLength;     // stop the integration when the list of intervals exceed a certain
                         // size, OPTIONAL way to limit the integration 
  string name;           // keep a number to e.g. identify which integration gets stuck
  bool warningFlag;

 public:
  GKIntegration(F & f_):
    f(f_),
    relativeErrors(false),
    divisionMax(30),
    maxListLength(0),
    warningFlag(true) {}
  GKIntegration(F & f_, const string & name_):
    f(f_),
    relativeErrors(false),
    divisionMax(30),
    maxListLength(0),
    name(name_),
    warningFlag(true) {}
  
  void SetAbsoluteErrorMode() {
    relativeErrors=false;
  }
  void SetRelativeErrorMode() {
    relativeErrors=true;
  }
  void SetMaximumDivision(const int i) {
    if (i<0) error("SetMaximumDivision",i);
    divisionMax=i;
  }
  void SetMaximumListLength(const int i) {
    if (i<0) error("SetListLengthMaximum",i); // i=0 for no limit
    maxListLength=i;
  }
  void SetName(const string s) {
    name = s;
  }
  void NoWarnings() {
    warningFlag = false;
  }

 private:
  // funnel all calls through this function and branch to specfic n knot rule
  void GK(IntervalResult & r) {
    GKGeneral(GKRule::n,GKRule::xgk,GKRule::wg,GKRule::wgk,r);
  }

  // handle all n knot rule with the passed in positions and weights
  void GKGeneral(const int n, 
		 const double xgk[], const double wg[], const double wgk[],
		 IntervalResult & r) {

    const double center     = 0.5 * (r.a + r.b);
    const double halfLength = 0.5 * (r.b - r.a);
    const double fCenter = f(center);

    double resultGauss   = 0;
    double resultKronrod = fCenter * wgk[n - 1];

    if (n % 2 == 0) {
      resultGauss = fCenter * wg[n / 2 - 1];
    }

    for (int j=0; j<(n-1)/2; j++) {
      const int jtw = j*2+1;	// j=1,2,3 jtw=2,4,6
      const double xx    = halfLength * xgk[jtw];
      const double fval1 = f(center - xx);
      const double fval2 = f(center + xx);
      const double fsum = fval1 + fval2;
      resultGauss   += wg[j]    * fsum;
      resultKronrod += wgk[jtw] * fsum;
    }
    
    for (int j=0; j<n/2; j++) {
      int jtwm1 = j*2;
      const double xx = halfLength * xgk[jtwm1];
      const double fval1 = f(center - xx);
      const double fval2 = f(center + xx);
      resultKronrod += wgk[jtwm1] * (fval1 + fval2);
    };
    
    /* scale by the width of the integration region */
    resultGauss   *= halfLength;
    resultKronrod *= halfLength;
    
    r.result = resultKronrod;
    r.err = pow(200.0 * fabs(resultKronrod - resultGauss), 1.5);
    //    Write(r);
  }

  void PrintList() {
    cout << "/------------------------------------------\\" << endl;
    int i=0;
    for(typename list <IntervalResult>::iterator p=ir.begin(); p!=ir.end(); p++) {
      Write2(i,*p);
      i++;
    }
    cout << "\\------------------------------------------/" << endl;
  }

  //Print interval with maxium error per interval length
  //(not with maximum error - this is on top of the list)
  void PrintMax() {
    typename list <IntervalResult>::iterator rMax(ir.begin());

    for(typename list <IntervalResult>::iterator r=ir.begin()++; r!=ir.end(); r++) {
      if ((*r).ErrorL()>(*rMax).ErrorL())
	rMax=r;
    }

    Write(*rMax);
  }

  void CheckList() {
    if (ir.size()==0) return;
    
    for(typename list <IntervalResult>::iterator p=ir.begin(); p!=ir.end(); p++) {
      typename list <IntervalResult>::iterator pn = p;
      pn++;
      if (pn!=ir.end())
	if ( ((*p).err) < ((*pn).err) ) {
	  PrintList();
	  Write2(*pn,*p);
	  ::error("Ordering problem in list");
	}
    }
  }

  void CheckError(const double err) {
    double errorSum=0.0;

    if (ir.size()>0) {
      for(typename list <IntervalResult>::iterator p=ir.begin(); p!=ir.end(); p++) {
	errorSum += (*p).err;
      }
    }
    
    if (errorSum==0.0) {
      if (err!=0.0)
	error("CheckError",errorSum,err);
    } else {
      if (err/errorSum-1.0>1e-8 && fabs(err-errorSum)>1e-14) 
	error("CheckError",errorSum,err,errorSum-err);
    }

    Write4("PassedErrorCheck",errorSum,err,errorSum-err);
  }

  double RecomputeError() {
    double errorSum=0.0;

    if (ir.size()>0) {
      for(typename list <IntervalResult>::iterator p=ir.begin(); p!=ir.end(); p++) {
	errorSum += (*p).err;
      }
    }
    
    return errorSum;
  }

  void Insert(const IntervalResult & r) {
    //    cout << "Inserting.." << endl;
    //    PrintList();

    if (ir.empty()) {
      ir.push_back(r);
      return;
    }

    if (ir.back().err>=r.err) {
      ir.push_back(r);
      return;
    }

    if (r.err>=ir.front().err) {
      ir.push_front(r);
      return;
    }

    // size must be >=2
    typename list <IntervalResult>::iterator p = ir.end();
    p--;

    // p cannot become less the begin() because of check above
    while (r.err > (*p).err)
      p--;

    // go one down because insert put the element before p
    p++;
    ir.insert(p,r);
    //    CheckList();
  }

  string Name() const {
    return "GK \""+name+"\": ";
  }

  double Integrate(const double a, const double b, 
		   const double absError, const bool absErrorFlag, 
		   const double relError, const bool relErrorFlag, 
		   const bool andFlag) {
    // this routine has trouble if b<a because, e.g., then lengthMin<0
    if (b<a) return -Integrate(b,a,absError,absErrorFlag,relError,relErrorFlag,andFlag);

    // #define PRINT_IT
#ifdef PRINT_IT
    cout << "Beginning integration " << name << endl;
#endif

    double errorUnresolved=0.0;
    double lengthMin = (b-a)/pow(2.0,divisionMax);
    
    IntervalResult r0(a,b);
    GK(r0);
    double result =r0.result;
    double err    =r0.err;
    double errLast=err;

    ir.push_back(r0);

    bool quitFlag;
    do {
      // PrintList();

      if (maxListLength>0) {
	if ( static_cast<int>(ir.size()) >= maxListLength ) {
	  if (warningFlag) warning(Name()+"List length reached specified limit. Integration stopped",ir.size(),result,err);
	  break;
	}
      }

      // Test if the interval with the biggest error has already been subdivided
      // the maximum number of times. If this is the case throw it out and print add
      // this contribution to the 'unresolved' errors to be printed at the end
      while (ir.size()>0) {
	IntervalResult & rTest (ir.front());
	double lengthTest = rTest.b-rTest.a;
	if (lengthTest<lengthMin) {
	  if (warningFlag) warning(Name()+"Interval was divided too many times",divisionMax,
				   rTest.a,rTest.b,rTest.err,ir.size());
	  //	  warning("GK:current result=",result,"error=",err);
	  if (absErrorFlag && warningFlag) warning(Name()+"Absolute accuracy = ",absError);
	  if (relErrorFlag && warningFlag) warning(Name()+"Relative accuracy = ",relError," -> absolute accuracy=",relError*fabs(result));
	  // this means there is a problem with the integrand->you could exit here
	  //	  exit(1);
	  errorUnresolved += rTest.err;
	  //	  PrintList();
	  ir.pop_front();
	} else break;
      }
      // do you want to exit with a warning after the first unresolved sub-interval occured?
      if (ir.size()==0 || errorUnresolved>0.0) break;
      // or print as many as occur
      //      if (ir.size()==0) break;

      IntervalResult & r (ir.front());

      double center = 0.5*(r.a+r.b);
      IntervalResult r1(r.a,   center);
      IntervalResult r2(center,r.b   );

      GK(r1);
      GK(r2);

      // must not use r after popping
      result += r1.result+r2.result - r.result;
      err    += r1.err   +r2.err    - r.err;

#ifdef PRINT_IT
      cout.setf(ios::scientific);
      cout << "Refined [ " << r.a << " " << r.b << " ] " 
	//	   << r.b-r.a << "/" << b-a
	   << " #div= " << int(0.5-log((r.b-r.a)/(b-a))/log(2.0))
	//	   << " l/lMin= " << (r.b-r.a)/lengthMin 
	   << " result= " << result
	   << " err/L=" << (r1.err +r2.err)/(r.b-r.a)
      	   << " error=" << err << endl;
#endif

      // must remove old element first because new ones could move to top
      ir.pop_front();

      Insert(r1);
      Insert(r2);

      // In rare events, the error decreases over many (>10) orders of magnitude
      // during the refinement. Rounding errors from the beginning can prevent
      // err from becoming small enough. Recompute err after a substantial decrease.
      if (err<1e-6*errLast) {
	err     = RecomputeError();
	errLast = err;
      }
 
      //      CheckError(err);
      //       PrintList();
      
      const bool relOk = (err < relError*fabs(result) || result==0.0);
      const bool absOk = (err < absError);

      if (absErrorFlag && relErrorFlag) {
	quitFlag = andFlag ? (relOk && absOk) : (relOk || absOk);
      } else {
	quitFlag = absErrorFlag ? absOk : relOk;
      }

    } while (!quitFlag);

#ifdef PRINT_IT
    PrintMax();
#endif

    if (errorUnresolved>0.0 && warningFlag) {
      warning(Name()+"Unresolved error sum= ",errorUnresolved," for integration interval",a,b);
      warning(Name()+"--> Result=",result,"total error=",err,"rel. error=",
	      ((result!=0.0) ? err/fabs(result) : 0.0));
      //      if (absErrorFlag) warning("Absolute accuracy = ",absError);
      //      if (relErrorFlag) warning("Relative accuracy = ",relError,
      //				"->absolute accuracy=",relError*fabs(result));
    }

    //    Write4(name,ir.size(),result,err);
    //    CheckList();
#ifdef PRINT_IT
    cout << Name()+"End integration" << name << endl;
#endif
    return result;
  }

 public:

  double Integrate(const double a, const double b, 
		   const double acc) {
    return Integrate(a,b,acc,!relativeErrors,acc,relativeErrors,false);
  }
  double Integrate(const double a, const double b, 
		   const double accAbs, const double accRel,
		   const bool andFlag) {
    return Integrate(a,b,accAbs,true,accRel,true,andFlag);
  }

};

#endif // _GKINTEGRATION_
