/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  CMS Method                                                             //
//                                                                         //
//  B. Militzer                                     Berkeley 04-18-17      //
//                                                                         //
//  based on William Hubbard methodology and                               //
//  based on Sean Wahl tidal CMS code                                      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "CMS.h"
#include "LegendrePolynomials.h"
#include "Quadrature.h"
#include "Form.h"
#include "MCParameters.h"
#include "Timer.h"
#include "Grid.h"
#include "MatrixAlgebra.h"
#include "GKIntegration.h"
#include "Physics.h"
#include "Spline.h"
#include "FindMinimum.h"
#include "ReadInTable.h"

double MaxAbsDifference(const Array1 <double> & a, const Array1 <double> & b) {
  const int n = min(a.Size(),b.Size());
  double d = 0.0;
  for(int i=0; i<n; i++) {
    double di = fabs(a[i]-b[i]);
    if (di>d) d=di;
  }
  return d;
}

double MaxAbsDifference(const Array2 <double> & a, const Array2 <double> & b) {
  const int n1 = min(a.dim[0],b.dim[0]);
  const int n2 = min(a.dim[1],b.dim[1]);
  double d = 0.0;
  for(int i=0; i<n1; i++) {
    for(int j=0; j<n2; j++) {
      double di = fabs(a(i,j)-b(i,j));
      if (di>d) d=di;
    }
  }
  return d;
}

double CMS::ZetaDifference(const Array2 <double> & z1, const Array2 <double> & z2) const {
  double d2 = 0.0;
  for(int j=0; j<nl; j++) { // iterate of all layers
    for(int imu=0; imu<nq/2; imu++) { // iterate over all theta/mu grid points
      double d = z1(j,imu)-z2(j,imu);
      d2 += d*d;
    }
  }
  return d2;
}

void CMS::CheckZetaLimits() const {
  for(int j=0; j<nl; j++) { // iterate of all layers
    for(int imu=0; imu<nq/2; imu++) { // iterate over all theta/mu grid points
      if (zeta(j,imu)==ZetaMin()) error("Zeta value right at the permitted minimum found. It should have converged to a different value: ",j,imu,zeta(j,imu));
      if (zeta(j,imu)==ZetaMax()) error("Zeta value right at the permitted maximum found. It should have converged to a different value: ",j,imu,zeta(j,imu));
      //      if (zeta(j,imu)==ZetaMin()) throw("Zeta value right at the permitted minimum found. It should have converged to a different value. l= "+IntToString(j));
      //      if (zeta(j,imu)==ZetaMax()) throw("Zeta value right at the permitted maximum found. It should have converged to a different value. l= "+IntToString(j));
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

// sets xq(#nq),wq(#nq), Pn_mu(#np,#nq), Pn_0(#np)
void CMS::Precompute() {
  // see F90 module "grid" subroutine "setupGrid"

  // see subroutine legQuad(order,x,w) in Quadature
  GaussLegendreQuadraturePoints(+1.0,-1.0,nq,xq,wq); // Sean used decreasing mu values [+1...-1] but then one gets negative weights
  wq *= -1.0;
  //  Write2(xq,wq);
  //  Quit("XX");

  // precompute legendre polynomials
  // see grid.f90: subroutine preCompute()
  Array1 <double> legendre(np+1);
  for(int i=0; i<nq; i++) {
    double mu = xq[i];
    LegendrePolynomials(np,mu,legendre);

    for(int n=0; n<=np; n++) {
      Pn_mu(n,i) = legendre[n];
    }
  }
  //  Write(xq[0]);
  //  for(int n=0; n<=np; n++) {
  //    Write2(n,pn_mu(5,n));
  //  }
  
  LegendrePolynomials(np,0.0,legendre);
  for(int n=0; n<=np; n++) {
    Pn_0[n] = legendre[n];
  }
  //  Write(pn_0);
  //  Quit("XX");
}

////////////////////////////////////////////////////////////////////////////////

void CMS::PerformRegulaFalsiStep(const int iter, const Array2 <double> & zetaPrev, Array2 <double> & zeta, 
				const Array2 <double> & equiPotPrev, const Array2 <double> & equiPot) const {
  if (iter>=100) cout << " Regula falsi step performed in iteration: " << iter << endl;
  for(int j=0; j<nl; j++) { // iterate of all layers
    for(int imu=0; imu<nq/2; imu++) { // iterate over all theta/mu grid points

      double zetaNew = ( equiPotPrev(j,imu)*zeta(j,imu) - equiPot(j,imu)*zetaPrev(j,imu) ) / ( equiPotPrev(j,imu) - equiPot(j,imu) );

      double q = ( zetaNew - zetaPrev(j,imu) ) / ( zeta(j,imu) - zetaPrev(j,imu) );
      if (q<0.0 || q>1.0) zetaNew =  0.5 * ( zeta(j,imu) + zetaPrev(j,imu) );

      zeta(j,imu) = zeta(j,nq-1-imu) = zetaNew;

    }
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// normally r[j+1] < r[j]
inline bool CMS::CheckForSpheroidIntersections(const Array1 <double> & r) const {
  for(int i=0; i<nl-1; i++) { // iterate of all layers
    if (r[i+1] >= r[i]) {
      //      Write4(i,r[i],r[i+1],r[i+1]-r[i]);
      return true;
    }
  }
  return false;
}

int CMS::MaxSpheroidIntersections(const Array1 <double> & r) const {
  int iMax = -1;
  double dMax = 0.0;
  for(int i=0; i<nl-1; i++) { // iterate of all layers
    if (r[i+1] >= r[i]) {
      double d = (r[i+1]-r[i]) / ( (r[i+1]+r[i])/2.0 );
      if (d>dMax) {
	dMax = d;
	iMax = i;
      }
    }
  }
  return iMax;
}

bool CMS::CheckAndRepairSpheroidIntersections(Array2 <double> & zeta, const Array2 <double> & zetaPrev) const {
  bool repaired = false;
  Array1 <double> r(nl);
  for(int imu=0; imu<nq/2; imu++) { // iterate over all theta/mu grid points

    for(int j=0; j<nl; j++) { // iterate over all layers
      r[j] = zeta(j,imu)*lambda[j];
    }

    bool flag = CheckForSpheroidIntersections(r);
    if (flag) {
      int iMax = MaxSpheroidIntersections(r);

      //      Write3(imu,r,lambda); Quit("CheckForSpheroidIntersections");
      //      problem = true;
      double x1=0.0;
      double x2=1.0;
      double xGood = x1; // assume there is no problem with previous zeta values
      // for(int i=0; i<5; i++) {
      for(int i=0; i<10; i++) {
	const double xx = 0.5*(x1+x2);
	const double yy = 1.0-xx;
	for(int j=0; j<nl; j++) { // iterate over all layers
	  r[j] = (xx*zeta(j,imu) + yy*zetaPrev(j,imu)) * lambda[j];
	}
	flag = CheckForSpheroidIntersections(r); // true means there is a problem
	if (flag) { // xx is too large
	  x2 = xx;  // now search in interval x1 ... xx
	} else {
	  xGood = xx;
	  x1    = xx; // now search in interval xx ... x2
	}
	Write7(iMax,i,flag,x1,xx,x2,xGood);
      }
      //      RepairSpheroidIntersections(r);
      for(int j=0; j<nl; j++) { // iterate over all layers
	//	double zetaNew = r[j] / lambda[j];
	double zetaNew = xGood*zeta(j,imu) + (1.0-xGood)*zetaPrev(j,imu);
	if (zetaNew<ZetaMin()) zetaNew=ZetaMin();
	if (zetaNew>ZetaMax()) zetaNew=ZetaMax();
	zeta(j,     imu) = zetaNew;
	zeta(j,nq-1-imu) = zetaNew;
	//	Write3(imu,j,zeta(j,imu));
      }
      //      warning("Spheroid crossing was corrected successfully for mu index",imu);
      cout << "Spheroid crossing was corrected successfully for mu index= " << imu << endl;
      //      Quit("CheckForSpheroidIntersections");
      repaired  = true;
    }

  }
  /*
  if (problem) {
    //    warning("Spheroid crossing detected");
    warning("Spheroid crossing was corrected successfully.");
    int imu = 0;
    for(int j=0; j<nl; j++) { // iterate of all layers
      double rj  = zeta(j,  imu)*lambda[j];
      Write3(j,lambda[j],rj);
    }
  }
  */
  return repaired;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// append value to the end of list
// remove value from front (index=0) if size limit 'n' is exceeded.
inline void SupplyValue(Array1 <double> & list, const double x, const int n) {
  if (list.Size()<n) { 
    list.PushBack(x); 
  } else {
    for(int i=1; i<list.Size(); i++) {
      list[i-1] = list[i];
    }
    list.Last() = x;
  }
}

////////////////////////////////////////////////////////////////////////////////

void CMS::CheckTotalPlanetMass(const bool print) {
  double M = CalculateMass();
  if (print) cout << " M= " << M << " M-1.0= " << M-1.0 << endl;
}

void CMS::PrepareAdjustmentOfLambdaGrid(const Array1 <double> & rr) {
  mLayers = rr.Size() + 1; // 'm-1=3' radii imply 'm=4' layers
  lambdaBoundariesOrg.Resize(mLayers+1);
  dlambdaBoundaries_dEta.Resize(mLayers+1);

  lambdaBoundariesOrg[0] = 1.0;
  dlambdaBoundaries_dEta = 0.0; // outer boundary does not change with eta
  for(int i=1; i<mLayers; i++) {
    lambdaBoundariesOrg[i] = rr[i-1];
    dlambdaBoundaries_dEta[i] = lambdaBoundariesOrg[i];
  }
  lambdaBoundariesOrg.Last()    = lambda.Last(); // inner boundary does not change with eta
  dlambdaBoundaries_dEta.Last() = 0.0;

  lambdaBoundaries = lambdaBoundariesOrg; // make a copy. 'lambdaBoundaries' will change as eta deviates dfrom 1. 

  indexBoundaries.Resize(mLayers+1);
  indexBoundaries[0] = 0; // outer layer always begins with index 0

  eta = 1.0;
  dLa_dEta.Resize(nl);
  //  ApplyNewEtaToLambdaBoundaries(1.0);
}


inline int FindClosestValue(const Array1 <double> & x, const double xx, const int step) {
  int iMin = 0;
  double dMin = fabs(x[iMin]-xx);
  for(int i=step; i<x.Size(); i+=step) {
    double d = fabs(x[i]-xx);
    if (d<dMin) { dMin=d; iMin=i; }
  }
  return iMin;
}



void CMS::ScaleLambdaGrid() {
  /*
  const double scalePlanetRadius = 1.0;
  //  Write2(scaleCoreRadius,scaleCoreRadius*rCore);
  for(int i=nlEnv; i<nl; i++) { // scale the innermost 'nlCore' lambda value
    lambda[i] = scaleCoreRadius * lambdaOrg[i];
    //    cout << " Core:     "; Write3(i,lambdaOrg[i],lambda[i]);
  }
  const double rCoreScaled = scaleCoreRadius*rCore;
  for(int i=0; i<nlEnv; i++) {
    // If i==0 --> f=0 use scalePlanetRadius. If i==nl-nlCore-1 --> f=1 use scaleCoreRadius
    //    double f = double(i)/double(nl-nlCore-1); // do not scale with i but with lambdaOrg[i] instead
    //    double f      = (lambdaOrg[0]-lambdaOrg[i]) / (lambdaOrg[0]-lambdaOrg[nlEnv-1]); 
    //    f             = f*f; // 6/21/21
    //    f             = f*f*f; // 6/21/21
    //    f             = f*f*f*f; // 6/21/21
    //    f             = f*f*f*f*f*f; // 6/21/21
    //    f             = pow(f,12.0); // 6/21/21
    //    double f = SmoothFastSwitch(lambdaOrg[i],rCoreScaled,rCoreScaled*2.0,1.0,0.0);
    double f = SmoothFastSwitch(lambdaOrg[i],rCoreScaled,rCoreScaled*1.5,1.0,0.0);
    double factor = (hasCore) ? (1.0-f)*scalePlanetRadius + f*scaleCoreRadius : scalePlanetRadius;
    lambda[i]     = factor * lambdaOrg[i];
    //    cout << " Envelope: "; Write5(i,f,factor,lambda[i],lambdaOrg[i]);
  }
  for(int i=0; i<nl-1; i++) {
    if (lambda[i+1]>lambda[i]) error("Lambda scaling problem:",scaleCoreRadius,scalePlanetRadius,i,lambda[i],lambda[i+1]);
  }
  */
  SetLambdasBasedOnSplinesFromReferenceCalculation(); // now all lambda grids should be the same for given value of 'scaleCoreRadius*rCore'
  CalculateAndSetAllSpheroidVolumes(); 
  CalculateMoments(); // moments depend on lambda @@8072@@
  CalculateSurfacePotentials();
}

////////////////////////////////////////////////////////////////////////////////

void CMS::CheckSurfacePotentials() {
  double dPotMax = 0.0;

  double potEquator = ExternalPotential(1.0,0.0);
  for(int imu=0; imu<nq; imu++) {
    double mu = xq[imu];
    double potI = ExternalPotential(zeta(0,imu),mu);
    double dPot = fabs(potI-potEquator);
    if (dPot>dPotMax) dPotMax=dPot;
    //    Write6(imu,mu,potEquator,potI,dPot,equiPot(0,imu));
  }

  for(int l=1; l<nl; l++) { // loop over layers
    double potEquator = InternalPotential(l,1.0,0.0);
    for(int imu=0; imu<nq; imu++) {
      double mu = xq[imu];
      double potI = InternalPotential(l,zeta(l,imu),mu);
      double dPot = fabs(potI-potEquator);
      if (dPot>dPotMax) dPotMax=dPot;
      //      Write7(l,imu,mu,potEquator,potI,dPot,equiPot(l,imu));
    }
  }
  
  cout << " Maximum potential deviation remaining = " << dPotMax << endl;

  //  Quit("CheckSurfacePotentials()");
}

////////////////////////////////////////////////////////////////////////////////

double CMS::ExternalPotential(const double zeta0, const double mu) { 
  double V = ExternalV(zeta0,mu);
  double Q = ExternalQ(zeta0,mu);
  double U = V+Q;
  return U;
}

// gravitational term only
double CMS::ExternalV(const double zeta0, const double mu) {
  Array1 <double> pnMu(np+1);
  LegendrePolynomials(np,mu,pnMu);

  double f1 = 0.0;
  for(int l=0; l<nl; l++) { // loop over layers
    for(int n=1; n<=np; n++) {  // loop over zonal wave numbersm starts from 1

      // legendre polynomials
      double pn = pnMu[n];

      // Harmonic expansion term
      f1 += pow(lambda[l]/zeta0,n) * pn * jTilde(l,n);
    }
  }

  double extV = (1.0/zeta0) * ( 1.0 - f1);
  return extV;
}

// rotational term only
double CMS::ExternalQ(const double zeta0, const double mu) {
  //  Array1 <double> pnMu(3);
  //  LegendrePolynomials(2,mu,pnMu);

  double l = zeta0 * sqrt(1.0-mu*mu); // lambda[jLayer=0]=1 by definition
  //  double qRotScaled  = qRot * Omega(l);
  //  double p2 = pnMu[2];
  //  double extQ = qRotScaled / 3.0 * sqr(zeta0) * (1.0 - p2);

  double rot = RotationalPotential(l);
  return rot;
}

double CMS::InternalPotentialSlow(const int jLayer, const double zetaJ, const double mu) {
  // store pn for given mu
  Array1 <double> pnMu(np+1);
  LegendrePolynomials(np,mu,pnMu);

  double f1a = 0.0;
  double f1b = 0.0;
  double f1c = 0.0;
  for(int l=0; l<nl; l++) { // loop over layers

    for(int n=0; n<=np; n++) {  // loop over zonal wave numbers (starts from 0)

      // legendre polynomials
      double pn = pnMu[n];

      if (l>=jLayer) { // layers interior to jlayer
	f1a += pow(lambda[l]/lambda[jLayer]/zetaJ,n) * pn * jTilde(l,n);
      } else { // layers exterior to jlayer
	f1b += pow(lambda[jLayer]/lambda[l]*zetaJ,n+1) * pn * jTildeP(l,n);
      }

    } // loop n

    // layers exterior to jlayer
    if ( l<jLayer ) {
      f1c += jPP[l] * cube(lambda[jLayer] * zetaJ);
    }

  } // loop l

  // rotational term
  double xiJ = zetaJ * lambda[jLayer];
  double l   = xiJ * sqrt(1.0-mu*mu);
  //  double qRotScaled  = qRot * Omega(l);
  //  double f2 = qRotScaled/3.0 * cube(lambda[jLayer]) * sqr(zetaJ) * (1.0 - pnMu[2]);
  //  double f2Org = qRot/3.0 * cube(lambda[jLayer]) * sqr(zetaJ) * (1.0 - pnMu[2]);
  //  double intPot = ( -(1.0/zetaJ) * (f1a + f1b + f1c) + f2 ) / lambda[jLayer];

  double rot = RotationalPotential(l);
  double intPot = rot - (f1a + f1b + f1c) / xiJ;

  //  Write2(rot,f2Org/lambda[jLayer]);

  return intPot;
}

double CMS::InternalPotential(const int jLayer, const double zetaJ, const double mu) {
  // store pn for given mu
  Array1 <double> pnMu(np+1);
  LegendrePolynomials(np,mu,pnMu);

  double f1a = 0.0;
  double f1b = 0.0;
  double f1c = 0.0;
  for(int l=0; l<jLayer; l++) { // loop over layers

    const double FF = lambda[jLayer]/lambda[l]*zetaJ;
    double ll = FF;
    for(int n=0; n<=np; n++) {  // loop over zonal wave numbers (starts from 0)
      f1b += ll * pnMu[n] * jTildeP(l,n);
      ll  *= FF;
    } // loop n

    // layers exterior to jlayer
    f1c += jPP[l] * cube(lambda[jLayer] * zetaJ);

  } // loop l

  for(int l=jLayer; l<nl; l++) { // loop over layers

    const double FF = lambda[l]/lambda[jLayer]/zetaJ;
    double ll = 1.0;
    for(int n=0; n<=np; n++) {  // loop over zonal wave numbers (starts from 0)
      f1a += ll * pnMu[n] * jTilde(l,n);
      ll *= FF;
    } // loop n

  } // loop l

  // rotational term
  double xiJ = zetaJ * lambda[jLayer];
  double l   = xiJ * sqrt(1.0-mu*mu);
  //  double qRotScaled  = qRot * Omega(l);
  //  double f2 = qRotScaled/3.0 * cube(lambda[jLayer]) * sqr(zetaJ) * (1.0 - pnMu[2]);
  //  double f2Org = qRot/3.0 * cube(lambda[jLayer]) * sqr(zetaJ) * (1.0 - pnMu[2]);
  //  double intPot = ( -(1.0/zetaJ) * (f1a + f1b + f1c) + f2 ) / lambda[jLayer];

  double rot = RotationalPotential(l);
  double intPot = rot - (f1a + f1b + f1c) / xiJ;

  //  Write2(rot,f2Org/lambda[jLayer]);

  return intPot;
}

////////////////////////////////////////////////////////////////////////////////

// see eos.f90 subroutine setLambda
void CMS::SetLambdas() {
  lambda[0] = 1.0;

  // if there is a core define radius of the envelope
  double rEnvelope = 1.0;

  // For this version, we add a half-width outer layer with density = 0
  if (nl>=2) {
    // number of layers in the envelope
    int ntmp = nl;

    bool halfLayerOne=false;
    double thickness;
    if (halfLayerOne) {
      // thickness = r_envelope / ( dble(ntmp) - 0.5d0)
      thickness = rEnvelope / (ntmp-1.0);
      lambda[1] =  1.0 - (0.5 * thickness);
    } else {
      thickness = rEnvelope / double(ntmp);
      lambda[1] = 1.0 - thickness;
    }
    
    for(int i=2; i<nl; i++) {
      lambda[i] = lambda[i-1] - thickness;
    }

  } else {

    lambda[1] = 0.5;

  }
  CheckLambdaGrid();
  //  Write(lambda);
  //  Quit("XX");
  lambdaOrg = lambda; // keep a copy 1/12/24
}

// see eos.f90 subroutine setLambda
void CMS::SetLambdasWeighted() {

  // if there is a core define radius of the envelope
  //  double rEnvelope = 1.0;
  //  if (hasCore) rEnvelope = 1.0 - rCore;

  const int sec   = 4;
  const int nGeom = 10;
  int nTmp = nl - nGeom;

  if (nTmp%4 != 0) error("SetLambdasWeighted() number of envelope layers should be divisible by 4",nTmp,nl);

  if (nTmp<=0) error("nl too small",nl,nGeom,nTmp);
  
  int nSec = nTmp / sec;
  //  double aJ = a_cgs * 1e-5;
  double aJ = a_SI * 1e-3; // in km (Sean's routine)

  double thickness = 1.0/aJ;

  lambda[0] = 1.0;
  for(int i=1; i<nGeom; i++) {
    lambda[i] = lambda[i-1] - thickness;
    thickness *= 1.5;
  }

  //  Write(aJ);
  //  Write(lambda);

  double rSec = lambda[nGeom-1];
  double geomFac = pow(2.0,sec) - 1.0;

  thickness = rSec / geomFac / nSec;
  //  Write5(nTmp, nSec,geomFac,rSec, thickness);

  for(int j=0; j<sec; j++) { // j was shifted by -1 compared to F90 code
    //    Write2(nGeom + j*nSec + 1,nGeom + (j+1)*nSec + 1);
    for(int i=nGeom+j*nSec; i<=nGeom+(j+1)*nSec; i++) {// i was shifted by -1 compared for F90 code
      lambda[i] = lambda[i-1] - thickness;
      thickness = rSec / geomFac / nSec * pow(2.0,j);
      //      Write4(j,sec,i,nGeom);
    }
  }

  //  PrintLambdaGrid();
  //  Quit("XX");
  CheckLambdaGrid();
  lambdaOrg = lambda; // keep a copy 1/12/24
}

// rCore must be set
void CMS::SetLambdasSmoothlyWeighted() {
  int nGeom = 10; // set lambda for layers 1 ... nGeom-1
  nGeom--; // changed definition: nGeom now specify the last outer layer

  const double aP = a_SI * 1e-3;    // in km (Sean's routine)
  //  const double a = 1.5;         // Sean's geometric factor = 3/2 for Jupiter
  const double a = 1.1;             // My guess for U+N

  const double thickness0 = 1.0/aP; // thickness of outer most layers
  double thickness  = thickness0;
  lambda[0] = 1.0;
  for(int i=1; i<=nGeom; i++) { // changed from "i<nGeom" to "i<=nGeom" because nGeom -> nGeom-1
    lambda[i] = lambda[i-1] - thickness;
    thickness *= a;
    //    Write3(i,lambda[i],1.0-thickness0*(pow(a,i)-1.0)/ (a-1.0));
    //    Write2(i,lambda[i]);
  }

  double A = 1.0-thickness0*(pow(a,nGeom)-1.0)     / (a-1.0); // A=lambda(i=nGeom)
  double B =    -thickness0* pow(a,nGeom) * log(a) / (a-1.0); // B=slope=dLamdba/di at i=nGeom
  //  Write2(A,B);

  //  int nMax = nl - (nGeom+1);
  //  double rMax = rCore;
  //  if (!hasCore) { nMax++; rMax=0.0; } -- removed on 3/9/20 to run run core-less Jupiter models that still have an rCore defined

  int nMax = nl - nGeom + 1; // nl=2049
  double rMax = 0.0;

  const double C = (rMax - A - B*nMax) / sqr(double(nMax)) ; 
  for(int i=0; i<nMax-1; i++) {
    //  for(int i=0; i<nMax; i++) {
    int ii = nGeom + i;
    lambda[ii] = A + B*i + C*i*i; // rMax = A + B*nMax + C*nMax^2 --> C = (rMax - A - B*nMax) / nMax^2
    //    Write3(i,ii,lambda[ii]);
    if (lambda[ii]<0.0 || lambda[ii]>lambda[ii-1]) error("In lambda grid (bad B)",ii,lambda[ii],lambda[ii-1]);
  }

  CheckLambdaGrid();
  //  if (hasCore) lambda[nl-1] = rCore; // now redundant
  //  Write(lambda);
  //  Quit("XX");
  lambdaOrg = lambda; // keep a copy 1/12/24
  //  Write(lambdaOrg);
}

/* 
// used up to 9/23/20
// rCore must be set
void CMS::SetLambdasBasedOnReferenceCalculation() {
  if (!hasCore) error("Currently SetLambdasBasedOnRefernceCalculation() can only used with calculations that has a core");
  
  Array1 <double> lambdaGridFile, rhoGridFile;
  int nFile;
  //  ReadInTable("layers_EOS_info_reference.txt",4,10,nFile,lambdaGridFile, rhoGridFile);
  ReadInTable("layers_EOS_info_reference.txt",4,8,nFile,lambdaGridFile, rhoGridFile);
 
  int n = nl-1;
  lambda[n] = rCore; // always assume we have a core, for now
  lambda[0] = 1.0;
  
  LSpline s; // lambda as function of density (not log grid needed), allow for linear extrapolation in case the core radius changes
  s.PushBack(rhoGridFile, lambdaGridFile);
  s.Init();

  double rhoMin = rhoGridFile.First();
  double rhoMax = rhoGridFile.Last();
  for(int j=1; j<n; j++) {
    double rho = rhoMin * exp(log(rhoMax/rhoMin)*double(j)/double(n));
    lambda[j] = s.Interpolate(rho);
    //     Write2(j,lambda[j]);
  }

  //  if (hasCore) lambda[nl-1] = rCore; // now redundant
  CheckLambdaGrid();
  CopyLambdaGrid();
  //  PrintLambdaGrid();
  //  PrintDeltaLambda();
  //  Quit("XX");
  lambdaOrg = lambda; // keep a copy 1/12/24
}
*/

void CMS::SetLambdasBasedOnReferenceCalculation() {
  error("Not yet test: SetLambdasBasedOnReferenceCalculation()");

  Array1 <double> lambdaGridFile, rhoGridFile;
  int nFile;
  //  ReadInTable("layers_EOS_info_reference.txt",4,10,nFile,lambdaGridFile, rhoGridFile);
  ReadInTable("layers_EOS_info_reference.txt",4,8,nFile,lambdaGridFile, rhoGridFile);
  if (lambdaGridFile[0] != 1.0) error("Unexpected first lambda value. Should be 1 but is",lambdaGridFile[0]);

  // near the core, we find rho(lambda) = A - B*lambda^2, which means
  // f(x=lambda^2) = A - B * x is a linear function in 'x=lambda^2', 
  // which can be linearly extrapolated towards lambda=0
  for(int i=0; i<lambdaGridFile.Size(); i++) { 
    lambdaGridFile[i] = sqr(lambdaGridFile[i]);
  }
  sRhoLambda2.Reset(); // spline representing lambda^2(rho). Named 's' before. Allows for linear extrapolation towards small r. (No log grid needed.)
  sRhoLambda2.PushBack(rhoGridFile, lambdaGridFile); // lambda^2(rho)
  sRhoLambda2.Init();

  sLambda2Rho.Reset(); // spline representing rho(lambda^2) so that determine rhoMax=rho(lambda^2=rCore^2) if we have a core. Names 'ss' earlier.
  sLambda2Rho.PushBack(lambdaGridFile,rhoGridFile); // rho(lambda^2) works better than rho(lambda) if rCore<lambda_in_file_min
  sLambda2Rho.Init();

  scaleCoreRadius = 1.0;
  SetLambdasBasedOnSplinesFromReferenceCalculation();
  lambdaOrg = lambda; // keep a copy 1/12/24
}

void CMS::SetLambdasBasedOnSplinesFromReferenceCalculation() {
  if (sRhoLambda2.Size()==0 || sRhoLambda2.Size()==0) error("sRhoLambda2.Size()==0 || sRhoLambda2.Size()==0");

  //  double rhoMin = rhoGridFile.First(); // assumed value for j=0 and lambda=1.0
  //  double rhoMax = rhoGridFile.Last();  // assumed value for j=n and lambda=rCore <<< obsolete
  double rhoMin = sRhoLambda2.FirstX();
  double rhoMax = sRhoLambda2.LastX();
  //  Write2(s.SolveForXAtTheEnd(0.0),ss.Interpolate(sqr(rCore)));
  //  Write2(rhoMin,rhoMax);

  lambda[0] = 1.0;
  for(int j=1; j<nl; j++) {
    double rho = rhoMin * exp(log(rhoMax/rhoMin)*double(j)/double(nl));
    double lambda2 = sRhoLambda2.Interpolate(rho);
    if (lambda2<=0.0) error("lambda problem",j);
    lambda[j] = sqrt(lambda2); // before we used lambda[j] = s.Interpolate(rho);
    //    cout << " Env:"; Write4(j,rho,s.Interpolate(rho),lambda[j]);
    if (CheckForNan(lambda[j])) error("NaN produced when setting up lambda grid",j,lambda[j]);
  }

  CheckLambdaGrid();
  //  PrintLambdaGrid();
  //  PrintDeltaLambda();
  //  Quit("XX");
  lambdaOrg = lambda; // keep a copy 1/12/24
}

////////////////////////////////////////////////////////////////////////////////

void CMS::DetermineAllEquiPotentials() {
  double deviation = 0.0;
  //  for(int j=0; j<nl; j++) {
  for(int j=0; j<nl; j+=nInt) {
    double pMin,pMax;
    for(int imu=0; imu<nq; imu++) {
      Array1 <double> PnMu(np+1);
      const double mu = xq[imu];
      double pot = InternalPotential(j,zeta(j,imu),mu);

      if (imu==0 || pot<pMin) pMin=pot;
      if (imu==0 || pot>pMax) pMax=pot;
    }
    double dev = (pMax-pMin) / ( (abs(pMax)+abs(pMin))/2.0 );
    //    Write4(j,pMin,pMax,dev);
    if (j==0 || dev>deviation) deviation=dev;
  }
  cout << " Accuracy of equipotentials= " << deviation << endl;
  //  Write(deviation);
  //  Quit("XX");
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////

// earlier called CMS::Chi2_CMSMinusTarget() 
double CMS::Chi2(const bool print) const {

  double chi2 = 0.0;
  
  double M        = CalculateMass(); // should be 1 for converged results = total mass divided by planet mass
  double diffM    = M - 1.0;
  double dMTarget = 0.0001;
  double chi2Term = sqr( diffM / dMTarget);
  chi2 += chi2Term;
  if (print) Write3(M,diffM,chi2Term);

  double J2     = GetJn(2);
  double diffJ2 = J2 - J2Target;
  chi2Term      = sqr( diffJ2 / dJ2Target);
  chi2         += chi2Term;
  if (print) Write4(J2*1e6,J2Target*1e6,diffJ2*1e6,chi2Term);
  
  double J4  = GetJn(4);
  double diffJ4 = J4 - J4Target;
  chi2Term      = sqr( diffJ4 / dJ4Target);
  chi2         += chi2Term;
  if (print) Write4(J4*1e6,J4Target*1e6,diffJ4*1e6,chi2Term);
  
  if (hydrogenMassFlag) {
    double hydrogenMassDifference = hydrogenMassReleased - hydrogenMassAbsorbed; // should be positive
    double hydrogenMassScale      = 0.05; // **8141**
    chi2Term = (hydrogenMassDifference<0.0) ? sqr(hydrogenMassDifference/hydrogenMassScale) : 0.0;
    chi2    += chi2Term;
    if (print) Write4(hydrogenMassReleased,hydrogenMassAbsorbed,hydrogenMassDifference,chi2Term);
  }

  if (diffRhoFlag) {
    double diffRhoScale = 0.1; // **8142**
    chi2Term = (diffRhoCNHRhoOH<0.0) ? sqr(diffRhoCNHRhoOH/diffRhoScale) : 0.0;
    chi2    += chi2Term;
    if (print) Write2(diffRhoCNHRhoOH,chi2Term);
  }

  if (mCNFlag) {
    double d = fabs(fCN - fCNExpected);
    //    double dScale = 0.05; // used for Neptune
    double dScale = 0.02; // try using it for Uranus
    d -= dScale;
    chi2Term = (d>0.0) ? sqr(d/dScale) : 0.0;
    chi2    += chi2Term;
    if (print) Write3(fCN,fCNExpected,chi2Term);
  }

  if (print) Write(chi2);

  return chi2;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////

class OmegaIntegrand {
  const LSpline & omegaSpline;

public:

  OmegaIntegrand(const LSpline & omegaSpline_):omegaSpline(omegaSpline_){}

  double operator()(const double l) const {
    double w = omegaSpline.Interpolate(l);
    return l*w*w;
  }

};

double OmegaIntegral(const LSpline & omegaSpline, const double l1, const double l2) {
  OmegaIntegrand oi(omegaSpline);

  GKIntegration <OmegaIntegrand,GK15> oii(oi,"Integral l*w(l)^2 dl");
  oii.SetRelativeErrorMode();

  double f = oii.Integrate(l1,l2,1e-12);

  return f;
}

void CMS::WriteMuGridToFile(const string & fileName) const {
  ofstream ofs(fileName.c_str());
  if (!ofs) error("Could not write file",fileName);
  
  for(int i=0; i<nq; i++) {
    double mu = xq[i];
    double theta = acos(mu)*180.0/pi;

    Form f(fix,11,8);
    ofs << "i= " << IntToStringMaxNumber(i+1,nq)
        << " theta[deg]= " << f(theta)
	<< " mu=cos(theta)= " << f(mu)
	<< endl;
  }
  ofs.close();
}

void CMS::WriteXiDataToFile(const string & fileName) const {
  ofstream ofs(fileName.c_str());
  if (!ofs) error("Could not write file",fileName);
  
  for(int i=0; i<nq; i++) {
    double mu = xq[i];
    double theta = acos(mu)*180.0/pi;

    Form f(fix,11,8);
    ofs << "i= " << IntToStringMaxNumber(i+1,nq)
        << " theta[deg]= " << f(theta)
	<< " mu=cos(theta)= " << f(mu);

    for(int j=0; j<nl; j++) {
      ofs << f(xi(j,i));
    }
    ofs << endl;
    
  }
  ofs.close();
}

void CMS::WriteLayerCMSDataToFile(const string & fileName) const {
  ofstream ofs(fileName.c_str());
  if (!ofs) error("Could not write file",fileName);
  
  Form f(fix,12,10);
  Form f2(sci,10);
  for(int j=0; j<nl; j++) {
    ofs << "j= " << IntToStringMaxNumber(j,nl);
    ofs << " la= " << f(lambda[j]);
    ofs << " rho= " << f(rho[j]);
    ofs << " delta= " << f(delta[j]);
    ofs << " P= " << f2(pressure[j]); // pressure in planetary units
    ofs << " pot= " << f(potential[j]);
    ofs << endl;
  }
  ofs.close();

}

void CMS::WriteHarmonicsToMachineReadableFile(const string & fileName) {
  bool file  = (fileName.length()>0);
  ofstream ofs;
  if (file) {
    ofs.open(fileName.c_str());
    if (!ofs) error("Could not write file",fileName);
  }

  Form f(sci,15,8);
  for(int n=2; n<=10; n++) {
    ofs << fix(2,0,n);
    double JnInt   = GetJn(n); // (n%2==0) ? GetJn(n) : 0.0;
    ofs << " " << f(JnInt);
    ofs << endl;
  }
  ofs.close();
}

/*
void CMS::PrintReferenceConditions() const {
  const int j=0;
  double P      = pressure[j]; // in planetary units
  double PGPa   = PressurePUToGPa(P);
  double Pbar   = PressurePUToBar(P);
  double PAU    = PressurePUToAU(P);
  double rhoGCC = DensityPUToGCC(rho[j]);
  double rhoGCC_from_P     = barotrope.Density(PAU); // is different because of discretization error
  double rhoGCC_protosolar = barotrope.DensityProtosolar(PAU); 
  double T = barotrope.Temperature(PAU);
  double X = barotrope.GetX(PAU);
  double Y = barotrope.GetY(PAU);
  double Z = barotrope.GetZ(PAU);
  double S = barotrope.Entropy(PAU);
  Form f(fix,14,12);
  Form f1(fix,8,6);
  Form f2(sci,10);
  Form f3(fix,10,6);
  cout << " Reference conditions:" << endl;
  cout << " P0[GPa]= " << f(PGPa) << endl;
  cout << " P0[bar]= " << f(Pbar) << endl;
  cout << " T0[K]=   " << f3(T*PC::AUToK) << endl;
  cout << " rho0[g/cc]=       " << f(rhoGCC) << endl;
  cout << " rho0(P)[g/cc]=    " << f(rhoGCC_from_P) << endl;
  cout << " rho0_PS(P)[g/cc]= " << f(rhoGCC_protosolar) << endl;
  cout << " S[kb/el]= " << f1(S) << endl;
  cout << " X=        " << f1(X) << endl;
  cout << " Y=        " << f1(Y) << endl;
  cout << " Z=        " << f1(Z) << endl;
  cout << " Y/(X+Y)=  " << f1(Y/(X+Y)) << endl;
}
*/

////////////////////////////////////////////////////////////////////////////////////////////////////

/*
void CMS::WriteBarotropeToFile(const string & fileName) const {
  ofstream ofs(fileName.c_str());
  if (!ofs) error("Could not write file",fileName);
  
  Form f(fix,12,10);
  Form f2(sci,10);
  for(int i=0; i<PRhoSpline.Size(); i++) {
    double log10PCGS   = PRhoSpline.x[i];
    double log10Rho = PRhoSpline.y[i];
    double PCGS = pow10(log10PCGS);
    double rho  = pow10(log10Rho);
    double P    = PCGS * PC::P_CGSToAU;
    ofs << " P[GPa]= " << fix(16,10,P*PC::AUToGPa) << " log10(P[cgs])= " << fix(12,8,log10(P*PC::P_AUToCGS));
    ofs << " rho[g/cc]= " << fix(12,10,rho) << " log10(rho[g/cc])= " << fix(12,8,log10(rho));
    ofs << endl;
  }
  ofs.close();

}
*/

////////////////////////////////////////////////////////////////////////////////////////////////////

void CMS::SetUpPressureCumulativeMassSpline() {
  //  cout << endl << " Calculate log(P) cumulative mass spline (to better determine Y2 as function of Y1)." << endl;
  PCumMassSpline.Reset();
  // first point (index=0) is the outermost spheroid surface with P0=1 bar or 0.1 bar
  // the last point (index=nl-1) is the core-mantle boundary
  PCumMassSpline.Allocate(nl+1); 

  //  int jMax = (hasCore) ? nl-1 : nl; // exclude core 
  //  if (!hasCore) warning("No sure if SetUpPressureCumulativeMassSpline() works correctly without a core.");

  CalculateAndSetAllSpheroidVolumes(); // then we can use the faster routines GetLayerMass(j) and GetSpheroidMass(j) below
  //  double mTotal = 1.0;
  double mTotal = CalculateMass(); // for U+N, no longer assume we have converged onto 1.0.

  for(int j=0; j<nl; j++) {
    //    if (j>0 && pressure[j]<=pressure[j-1]) warning("Pressure not monotononic",j-1,j,PressurePUToGPa(pressure[j-1]),PressurePUToGPa(pressure[j]),DensityPUToGCC(rho[j-1]));
    if (j==0 || pressure[j]>pressure[j-1]) {
      double PAU    = PressurePUToAU(pressure[j]);
      double logP   = log(PAU);
      // double mCum = CalculateSpheroidMass(j); // all mass enclosed in spheroid 'j', mass decreases as 'j' increases
      double mCum    = GetSpheroidMass(j);       // all mass enclosed in spheroid 'j', mass decreases as 'j' increases
      PCumMassSpline.PushBack(logP, mCum/mTotal); // normalization and integral are well defined
      //      Write5(j,logP,mCum,PCumMassSpline.n,PCumMassSpline.LastX());
      //      Write2(j,(PCumMassSpline.x[PCumMassSpline.n-1]>PCumMassSpline.x[PCumMassSpline.n-2]));
      //      Write4(j,PressurePUToGPa(pressure[j]),mCum,DensityPUToGCC(rho[j]));
    } else {
      warning("SetUpPressureCumulativeMassSpline: Pressure not monotononic",j-1,j,PressurePUToGPa(pressure[j-1]),PressurePUToGPa(pressure[j]),DensityPUToGCC(rho[j-1]));
    }
  }
  if (CentralPressure()>pressure[nl-1]) {
    double PAU    = PressurePUToAU(CentralPressure());
    double logP   = log(PAU);
    PCumMassSpline.PushBack(logP,0.0); // added on 1/11/24
  } else {
    warning("Central pressure not greater as expected",PressurePUToGPa(CentralPressure()),PressurePUToGPa(pressure[nl-1]),DensityPUToGCC(rho[nl-1]));
  }

  //  PCumMassSpline.Print();
  //  PCumMassSpline.Sort(); // Normally P increases with 'j', so no sorting needed (but pressure may not have converged properly, or there could be an EOS problem!)
  PCumMassSpline.Init();
  //  cout << endl << " log(P) cumulative mass spline has been set up (to better determine Y2 as function of Y1)." << endl;
}

void CMS::SetUpRadiusCumulativeMassSpline() {
  rCumMassSpline.Reset();
  rCumMassSpline.Allocate(nl+1); 

  CalculateAndSetAllSpheroidVolumes(); // then we can use the faster routines GetLayerMass(j) and GetSpheroidMass(j) below
  //  double mTotal = 1.0;
  double mTotal = CalculateMass(); // for U+N, no longer assume we have converged onto 1.0.

  for(int j=0; j<nl; j++) {
    // double mCum = CalculateSpheroidMass(j); // all mass enclosed in spheroid 'j', mass decreases as 'j' increases
    double mCum    = GetSpheroidMass(j);       // all mass enclosed in spheroid 'j', mass decreases as 'j' increases
    rCumMassSpline.PushBack(lambda[j], mCum/mTotal); // normalization and integral are well defined
  }
  rCumMassSpline.PushBack(0.0,0.0); // added on 1/11/24
  rCumMassSpline.Sort(); // P inseases with 'j', no sorting needed
  rCumMassSpline.Init();
}

void CMS::SetUpPressureRadiusSplines() {
  //  cout << endl << " Calculate log(P) cumulative mass spline (to better determine Y2 as function of Y1)." << endl;
  PVolumetricRadiusSpline.Reset();
  PEquatorialRadiusSpline.Reset();
  EquatorialRadiusPSpline.Reset();
  // first point (index=0) is the outermost spheroid surface with P0=1 bar or 0.1 bar
  // the last point (index=nl-1) is the innermost bundary
  PVolumetricRadiusSpline.Allocate(nl+1); 
  PEquatorialRadiusSpline.Allocate(nl+1); 
  EquatorialRadiusPSpline.Allocate(nl+1); 

  CalculateAndSetAllSpheroidVolumes(); // then we can use the faster routines GetLayerMass(j) and GetSpheroidMass(j) below

  for(int j=0; j<nl; j++) {
    if (j==0 || pressure[j]>pressure[j-1]) {
      double PAU    = PressurePUToAU(pressure[j]);
      PVolumetricRadiusSpline.PushBack(log(PAU), CalculateSphericalizedSpheroidRadius(j)); 
      PEquatorialRadiusSpline.PushBack(log(PAU), lambda[j]                              ); 
      EquatorialRadiusPSpline.PushBack(lambda[j],log(PAU)                               ); 
    }
  }
  if (CentralPressure()>pressure[nl-1]) {
    PVolumetricRadiusSpline.PushBack(log(PressurePUToAU(CentralPressure())),0.0); // added on 1/11/24
    PEquatorialRadiusSpline.PushBack(log(PressurePUToAU(CentralPressure())),0.0); // added on 1/11/24
    EquatorialRadiusPSpline.PushBack(0.0,log(PressurePUToAU(CentralPressure()))); // added on 1/11/24
  }
  PVolumetricRadiusSpline.Init();
  PEquatorialRadiusSpline.Init();
  EquatorialRadiusPSpline.Sort();
  EquatorialRadiusPSpline.Init();
  //  cout << endl << " log(P) cumulative mass spline has been set up (to better determine Y2 as function of Y1)." << endl;
}

////////////////////////////////////////////////////////////////////////////////////////////////////

void CMS::PrintCMSResults() const {
  //  Form f1(fix,10,4);
  //  Form f2(fix,8,6);
  Form f1(fix,14,8);
  Form f2(fix,10,8);
  //  int nDeltaMax          = -1;
  //  double maxAbsDelta     = 0.0;
  //  double sumAbsDelta     = 0.0;
  //  double sumDeltaSquared = 0.0;

  cout << endl;
  cout << " CMS interior model:";
  for(int n=2; n<=10; n+=2) {
    cout << " J" << n << "= " << f1(1e6* GetJn(n) );
  }
  cout << endl;

  cout << " Observations:      ";
  for(int n=2; n<=4; n+=2) {
    cout << " J" << n << "= " << f1(1e6* GetJnTarget(n) );
  }
  cout << endl;

  //  cout << " Juno-(interior+wind):  ";
  cout << " Model-obs:         ";
  for(int n=2; n<=4; n+=2) {
    //    double delta = GetJnJuno(n)-GetJn(n)-GetJnWind(n);
    double delta = GetJn(n) - GetJnTarget(n);
    //    if (fabs(delta)>maxAbsDelta) {
    //      nDeltaMax   = n;
    //      maxAbsDelta = fabs(delta);
    //    }
    //    sumAbsDelta     += fabs(delta);
    //    sumDeltaSquared += sqr(delta);
    cout << " J" << n << "= " << f1(1e6*delta);
  }
  cout << endl;

  //  cout << endl;
  //  cout << " Max_|delta_Jn|= " << f2(1e6*maxAbsDelta) 
  //       << " for n= " << nDeltaMax 
  //       << " Sum_|delta_Jn|= " << f2(1e6*sumAbsDelta) 
  //       << " Sum_|delta_Jn|^2= " << f2(1e12*sumDeltaSquared) 
  //       << " Sqrt= " << f2(1e6*sqrt(sumDeltaSquared)) << endl;
  cout << " Moment of inertia= " << f2(CalculateMomentOfInertia()) << endl; // equivalent to line in Main.C that prints "Moment of inertia (Hubbard's Eq.5, no DR)"
  //  cout << endl;

  cout << " Radius boundaries:";
  for(int i=1; i<mLayers; i++) {
    cout << " la[" << i << "]= " << fix(10,8,lambdaBoundaries[i]);
  }
  cout << endl;

  cout << " Radii ratios:     ";
  for(int i=2; i<mLayers; i++) {
    cout << " la[" << i << "]/la[1]= " << fix(10,8,lambdaBoundaries[i]/lambdaBoundaries[1]);
  }
  cout << endl;

}

////////////////////////////////////////////////////////////////////////////////////////////////////

class IntegrandForFixedMuPoint {
  CMS & cms;
  const int imu;
  const int l;
  const int mode;

  public:
  IntegrandForFixedMuPoint(CMS & cms_, const int imu_, const int l_, const int mode_):cms(cms_),imu(imu_),l(l_),mode(mode_) {}

  double operator()(const double xi) {
    double r2dm = cms.rho[l]*sqr(xi);
    if (mode==0) return r2dm; // dm * r^2 --> integration yields total mass

    double mu = cms.xq[imu];
    double l = xi * sqrt(1.0-mu*mu); 
    double dI = sqr(l) * r2dm;
    if (mode==1) return dI;                                       // dm * r^2 * l^2                    --> integration yields moment of inertia
    //    if (mode==2) return dI*cms.qRot;                              // dm * r^2 * l^2 * omega0           --> integration yields moment of inertia * omega0 <<<<<< was missing sqrt(...) until 9/30/23
    //    if (mode==3) return dI*cms.qRot*cms.Omega(l);                 // dm * r^2 * l^2 * omega(l)_surface --> integration yields angular momentum           <<<<<< was missing sqrt(...) until 9/30/23
    //    if (mode==4) return dI*cms.qRot*cms.OmegaDeepModified(xi,mu); // dm * r^2 * l^2 * omega(l,mu)      --> integration yields angular momentum           <<<<<< was missing sqrt(...) until 9/30/23
    if (mode==2) return dI*sqrt(cms.qRot);                              // dm * r^2 * l^2 * omega0           --> integration yields moment of inertia * omega0
    error("invalid mode",mode);
    return 0.0;
  }

};

double CMS::CalculateVolumeIntegral(const int mode) {
  // Use Gauss quadrature to perform integration over mu, as Bill has consitently used
  double integral = 0.0;

  for(int imu=0; imu<nq/2; imu++) { // use N-S symmetry and introduce factor 2 below
    //    double mu = xq[imu];

    for(int l=0; l<nl; l++) {
      IntegrandForFixedMuPoint integrand(*this,imu,l,mode);
      
      GKIntegration <IntegrandForFixedMuPoint,GK61> gKIntegration(integrand,"Integral over xi at fixed mu ...");
      gKIntegration.SetRelativeErrorMode();
      
      //    double integralSForingleMu = gKIntegration.Integrate(0.0,zeta(0,imu),1e-10);
      //    Write2(xi(11,15),zeta(11,15)*lambda[11]);
      double xiMin = (l<nl-1) ? xi(l+1,imu) : 0.0;
      double integralForSingleMuAndLayer = gKIntegration.Integrate(xiMin,xi(l,imu),1e-16);
      //      Write3(imu,integralForSingleMu,rho[nl-1]*cube(xi(nl-1,imu))/3.0);

      integral += 2.0 * 2.0*pi*wq[imu] * integralForSingleMuAndLayer;
    }
  }

  //  Write(integral);
  //  Write(CalculateSpheroidVolume(50));
  //  Write(CalculateSpheroidVolume(nl-1));
  //  Write(CalculateCoreMass());

  return integral;
}

// J2 must have been set
// assumes there is no DR !!!
// compare with CMS::CalculateMass() 
double CMS::CalculateMomentOfInertia() const { 
  //  Write3(delta.Sum(),lambda.Sum(),zeta.Sum());
  //  Write3(delta.Sum2(),lambda.Sum2(),zeta.Sum2());
  double I = 0.0;
  for(int j=0; j<nl; j++) { // loop over layers

    // note the 2.0 in 2.0/3.0 was probably entered because we use over 
    // double factor = 2.0/3.0 * delta[j] * cube(lambda[j]) * 2.0 * pi;  // was used in mass calculation
    double factor = 2.0/5.0 * 2.0/3.0 * delta[j] * pow(lambda[j],5.0) * 2.0 * pi;

    for(int k=0; k<nq/2; k++) {
      //      M += factor * cube(zeta(j,k)) * wq[k]; // was used in mass calculation
      I += factor * pow(zeta(j,k),5.0) * wq[k];
    }

  }

  I += 2.0/3.0*GetJn(2);
  
  return I; 
}

double CMS::CalculateMomentOfInertiaAgain() const { 
  //  Write3(delta.Sum(),lambda.Sum(),zeta.Sum());
  //  Write3(delta.Sum2(),lambda.Sum2(),zeta.Sum2());
  double I = 0.0;
  for(int j=0; j<nl; j++) { // loop over layers

    // note the 2.0 in 2.0/3.0 was probably entered because we use over 
    // double factor = 2.0/3.0 * delta[j] * cube(lambda[j]) * 2.0 * pi;  // was used in mass calculation
    //    double factor = 2.0/5.0 * 2.0/3.0 * delta[j] * pow(lambda[j],5.0) * 2.0 * pi; // was use above, 2/5*pi moved down
    double factor = delta[j] * pow(lambda[j],5.0) * 2.0; // 2.0 b/c of N-S symmetry so that we only sum k=0...nq/2 

    for(int k=0; k<nq/2; k++) {
      double mu = xq[k];
      I += factor * pow(zeta(j,k),5.0) * (1.0-sqr(mu)) * wq[k];
    }

  }

  I *= 2.0/5.0*pi;
  
  return I; 
}

double CMS::CalculateMomentOfInertiaAgainAndAgain() const { 
  //  Write3(delta.Sum(),lambda.Sum(),zeta.Sum());
  //  Write3(delta.Sum2(),lambda.Sum2(),zeta.Sum2());
  double I = 0.0;
  for(int j=0; j<nl; j++) { // loop over layers

    // note the 2.0 in 2.0/3.0 was probably entered because we use over 
    // double factor = 2.0/3.0 * delta[j] * cube(lambda[j]) * 2.0 * pi;  // was used in mass calculation
    //    double factor = 2.0/5.0 * 2.0/3.0 * delta[j] * pow(lambda[j],5.0) * 2.0 * pi; // was use above, 2/5*pi moved down
    //    double factor = delta[j] * pow(lambda[j],5.0) * 2.0; // 2.0 b/c of N-S symmetry so that we only sum k=0...nq/2 
    double factor = 2.0 * rho[j]; // 2.0 b/c of N-S symmetry so that we only sum k=0...nq/2 

    for(int k=0; k<nq/2; k++) {
      double rjk  = zeta(j,k) * lambda[j];
      //      double rj1k = zeta(j+1,k) * lambda[j+1]; 
      double rj1k = (j==nl-1) ? 0.0 : zeta(j+1,k) * lambda[j+1]; // last spheroid included core with inner radius equal to zero.
      double mu = xq[k];
      I += factor * ( pow(rjk,5.0)-pow(rj1k,5.0) ) * (1.0-sqr(mu)) * wq[k];
    }

  }

  I *= 2.0/5.0*pi;
  
  return I; 
}

void CMS::ReadFilesAndPerformCMSCalculations(const string & prefix) {
  LoadVectorText(prefix+"lambda.txt",lambda,true);
  CheckLambdaGrid();
  lambdaOrg = lambda; // keep a copy 1/12/24
  //  Write(lambda);

  LoadVectorText(prefix+"rho.txt",rho,true);
  CalculateDeltaRhoFromRho();

  LoadMatrixText(prefix+"zeta.txt",zeta,false); 
  CalculateMoments(); // one initial evaluation of Jn

  bool print = true;
  if (print) PrintJnTarget();
  if (print) { cout << " Final: "; PrintEvenJnCompact(); }

  DetermineAllEquiPotentials();
  UpdatePressure();

  Array1 <double> pressureInFile(nl);
  LoadVectorText(prefix+"pressure.txt",pressureInFile,false); 
  DetermineSurfacePressures(pressureInFile);
  
}
