%==========================================================================
%Magma mixing model
% Multicomponent diffusion of 8 melt oxides (SiO2, TiO_2, Al2O_3, FeO,
% MgO, CaO, Na_2O, K_2O) and effective binary diffusion-exsolution of H2O 
% and CO2
%
%Author: Simone Colucci
%Affiliation: Istituto Nazionale di Geofisica e Vulcanologia
%             sezione diPisa
%Date: January 2024
%==========================================================================

function [a,f,s] = reactionDiffusion_pde(x,t,u,dudx,p,T,R,Nb,rhol,Ym_oxides_left,Ym_oxides_right,lambda,P)
% System of differential equations
% 1): Reaction-diffusion CO2
%     u(1)=(mass CO2 in liquid)/(volume liquid+gas)
% 2): Reaction-diffusion H2O
%     u(2)=(mass H2O in liquid)/(volume liquid+gas)
% 3): H2O gas mass conservation
%     u(3)=(mass H2O in gas)/(volume liquid+gas)
% 4): CO2 gas mass conservation
%     u(4)=(mass CO2 in gas)/(volume liquid+gas)
% The standard form expected is
% a * dc/dt = x^(-m) * d( x^m * f )/dx + s

time_str = strcat('time:  ',num2str(t));
disp(time_str)

%% specie mass fraction

yg_H2O = u(3)/(u(4)+u(3));
yg_CO2 = u(4)/(u(4)+u(3));

%% gas density

Rgas = 8.314462618;  % gas constant
Rs = Rgas*(yg_H2O/0.018+yg_CO2/0.044);
rhog = p/(Rs*T);

%% gas volume fraction

alphag = (u(3)+u(4))/rhog;

%% a coefficients

a = [1; 1; 1; 1];

%% flux terms

ym_H2O = u(2)/(rhol*(1-alphag));  % mass fraction of H2O in the melt

D_CO2 = D_CO2_model(ym_H2O,p,T);
D_H2O = D_H2O_model(ym_H2O,T);

f = [D_CO2; D_H2O; 0; 0].* dudx;

%% source/sink terms

rhomix = alphag*rhog + (1-alphag)*rhol;
ytot_CO2 = (u(1) + u(4))./rhomix; % (mass CO2l + mass CO2gas)/(mass liquid + mass gas)
ytot_H2O = (u(2) + u(3))./rhomix; % (mass H2Ol + mass H2Ogas)/(mass liquid + mass gas)

ym_oxides = multicomponentDiffusion(x,t,Ym_oxides_left,Ym_oxides_right,lambda,P);
Composition = zeros(1,10);
Composition(1) = 1 - sum(ym_oxides);     % SiO2
Composition(2:3) = ym_oxides(1:2);       % TiO2, Al2O3
Composition(5) = ym_oxides(3);           % FeO
Composition(7) = ym_oxides(4);           % MgO
Composition(8:10) = ym_oxides(5:7);      % CaO, Na2O, K2O

[ym_H2Oeq,ym_CO2eq] = runSolwcad(Composition, p, T, ytot_H2O, ytot_CO2);
cm_CO2eq = rhol*ym_CO2eq.*(1-alphag);
cm_H2Oeq = rhol*ym_H2Oeq.*(1-alphag);

J_CO2 = D_CO2*(u(1)-cm_CO2eq)/R;              % mass flux CO2
Gamma_CO2 = J_CO2*4*pi*R*R*Nb*(1-alphag);

J_H2O = D_H2O*(u(2)-cm_H2Oeq)/R;              % mass flux H2O
Gamma_H2O = J_H2O*4*pi*R*R*Nb*(1-alphag);

if ( alphag<=0 )
    Gamma_CO2_corr =max(Gamma_CO2, 0);
    Gamma_H2O_corr =max(Gamma_H2O, 0);
elseif (alphag>=1)
    Gamma_CO2_corr = min(Gamma_CO2, 0);
    Gamma_H2O_corr = min(Gamma_H2O, 0);
else
    Gamma_CO2_corr = Gamma_CO2;
    Gamma_H2O_corr = Gamma_H2O;
end

s = [-Gamma_CO2_corr; -Gamma_H2O_corr; Gamma_H2O_corr; Gamma_CO2_corr];

end
