%==========================================================================
%Magma mixing model
% Multicomponent diffusion of 8 melt oxides (SiO2, TiO_2, Al2O_3, FeO,
% MgO, CaO, Na_2O, K_2O) and effective binary diffusion-exsolution of H2O 
% and CO2
%
%Author: Simone Colucci
%Affiliation: Istituto Nazionale di Geofisica e Vulcanologia
%             sezione diPisa
%Date: January 2024
%==========================================================================

function [H2Ol_eq,CO2l_eq] = runSolwcad(Composition, P, T, H2Otot, CO2tot)
% run SOLWCAD model

% references:
% Papale, P., Moretti, R. and Barbato, D. (2006)
% The compositional dependence of the saturation surface of H2O + CO2 
% fluids in silicate melts. 
% Chem. Geol.229(1-3): 78-95.

% input:
% Composition = [SiO2 TiO2 Al2O3 Fe2O3 FeO MnO MgO CaO Na2O K2O] (mass oxide)/(mass melt)
% p = pressure (Pa)
% T = temperature (K)
% H2Otot = (mass H2O in liquid + mass H2O in gas)/(mass liquid + mass gas) 
% CO2tot = (mass CO2 in liquid + mass CO2 in gas)/(mass liquid + mass gas) 

% output:
% H2Ol_eq = mass fraction of dissolved H2O
% CO2l_eq = mass fraction of dissolved CO2

%% Run solwcad

run_dir=pwd;

%path for solwcad folder
mfilename='magmaMixing';
mpath = strrep(which(mfilename),[mfilename '.m'],'');
src_solwcad_fd = strcat(mpath,'solwcad_');
% write thermo.dat
cd(src_solwcad_fd);
fileID = fopen('thermo.dat','w');
formatSpec = '%6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f\n %6.4f %6.4f';
fprintf(fileID,formatSpec,P,T,Composition,H2Otot,CO2tot);
fclose(fileID);
% run solwcad and get output
if (isunix)
    command=strcat(src_solwcad_fd,'/solwcad');        % Unix platforms (tested on Ubuntu 20.04)
elseif (ispc)
    command=strcat(src_solwcad_fd,'/solwcadwin.exe'); % Windows platforms (tested on Windows 10)
elseif (ismac)
    disp('run solwcad on Mac platforms: to be implemented');  % Mac platforms (as for Unix?)
end
[status,cmdout] = system(command); 
% read output file
fileID = fopen('sw.out','r');
formatSpec = '%f %f %f %f %f';
A = textscan(fileID,formatSpec,'HeaderLines',1);
fclose(fileID);
H2Ol_eq = A{1}; 
CO2l_eq = A{2}; 

cd(run_dir)
