%==========================================================================
%Magma mixing model
% Multicomponent diffusion of 8 melt oxides (SiO2, TiO_2, Al2O_3, FeO,
% MgO, CaO, Na_2O, K_2O) and effective binary diffusion-exsolution of H2O 
% and CO2
%
%Author: Simone Colucci
%Affiliation: Istituto Nazionale di Geofisica e Vulcanologia
%             sezione diPisa
%Date: January 2024
%==========================================================================

function sol = magmaMixing(Dict)
%Main
%% Read dictionary

% space domain
Nodes = Dict.Mesh.Nodes;  % number of mesh nodes      
Xh = Dict.Mesh.Xh;        % half length (m)

% time
nt = Dict.Time.nt;        % number of time steps
tf = Dict.Time.tf;        % final time (s)

% initial conditions    
% [TiO_2 Al2O_3 FeO MgO CaO Na_2O K_2O] 
Ym_oxides_left = Dict.InitialConditions.Ym_oxides_left;      % anhydrous melt oxides at left couple side (mass oxide)/(mass melt)
Ym_oxides_right = Dict.InitialConditions.Ym_oxides_right;    % anhydrous melt oxides at right couple side (mass oxide)/(mass melt)
Ym_CO2_left = Dict.InitialConditions.Ym_CO2_left;            % CO2 at left couple side (mass CO2)/(mass melt)
Ym_CO2_right = Dict.InitialConditions.Ym_CO2_right;          % CO2 at right couple side (mass CO2)/(mass melt)
Ym_H2O_left = Dict.InitialConditions.Ym_H2O_left;            % H2O at left couple side (mass H2O)/(mass melt)
Ym_H2O_right = Dict.InitialConditions.Ym_H2O_right;          % H2O at right couple side (mass H2O)/(mass melt)
alphag0 = Dict.InitialConditions.alphag0;                    % gas volume fraction
Yg_CO2_0 = Dict.InitialConditions.Yg_CO2_0;                  % CO2 gas mass fraction (mass CO2 gas)/(mass gas)

% physical parameters
p = Dict.System.p;               % pressure (Pa)
T = Dict.System.T;               % temperature (K)
rhol = Dict.System.rhol;         % melt density (kg/m^3)
R = Dict.Bubble.R;               % bubble radius [m]
Nb = Dict.Bubble.Nb;             % bubble number density [(num. bubbles)/(m^3 liquid)]

% eigenvectors of oxides diffusion matrix
P = Dict.Diffusion.P;
% eigenvalues of oxides diffusion matrix
lambda = Dict.Diffusion.lambda;

xlim_plot = Dict.Plot.xlim*1e6; % x axis limit [micron]

% plot experiments (no=0, yes=1)
plot_exp = Dict.plot.data;

%% Solve equations

% Concentration in the melt (mass CO2(H2O) in liquid)/(volume liquid + volume gas)
C_CO2m_left = Ym_CO2_left*rhol*(1-alphag0);
C_CO2m_right = Ym_CO2_right*rhol*(1-alphag0);
C_H2Om_left = Ym_H2O_left*rhol*(1-alphag0);
C_H2Om_right = Ym_H2O_right*rhol*(1-alphag0);

% Concentration in the gas (mass CO2(H2O) in gas)/(volume liquid + volume gas)
Mc = 0.04401;        % CO2 molar mass [kg/mol]
Mh = 0.018;          % H2O molar mass [kg/mol]
Rgas = 8.314462618;  % gas constant
rhog = p/(Rgas*T*(Yg_CO2_0/Mc + (1-Yg_CO2_0)/Mh));
C_CO2g_0 = Yg_CO2_0*rhog*alphag0;
C_H2Og_0 = (1-Yg_CO2_0)*rhog*alphag0;

% mesh grid
xmax = Xh;
L = logspace(log10(xmax)-2,log10(xmax),round(Nodes/2));
x = [flip(-L), 0, L ]';

% time array
t = linspace(0,tf,nt);

% solve pde
m = 0;     % pde parameter
myFun_pde = @(x,t,u,dudx)reactionDiffusion_pde(x,t,u,dudx,p,T,R,Nb,rhol,Ym_oxides_left,Ym_oxides_right,lambda,P);
myFun_ic = @(x)reactionDiffusion_ic(x,C_CO2m_left,C_CO2m_right,C_H2Om_left,C_H2Om_right,C_CO2g_0,C_H2Og_0);
myFun_bc = @(xl,ul,xr,ur,t)reactionDiffusion_bc(xl,ul,xr,ur,t);
sol = pdepe(m,myFun_pde,myFun_ic,myFun_bc,x,t);
C_CO2m = sol(:,:,1);
C_H2Om = sol(:,:,2);
C_H2Og = sol(:,:,3);
C_CO2g = sol(:,:,4);

%% Plot results

Yg_H2O = C_H2Og./(C_CO2g+C_H2Og);
Yg_CO2 = C_CO2g./(C_CO2g+C_H2Og);
Rs = Rgas*(Yg_H2O/Mh+Yg_CO2/Mc);
rhog = p./(Rs*T);
alphag = (C_CO2g+C_H2Og)./rhog;

Ym_CO2 = (C_CO2m./(rhol*(1-alphag)));
Ym_H2O = (C_H2Om./(rhol*(1-alphag)));

figure(1)

C=multicomponentDiffusion(x,t,Ym_oxides_left,Ym_oxides_right,lambda,P);

% normalize oxides including H2O and CO2
Ym_ox=[1-sum(C);C];
Ym_ox(9,:,:)=Ym_H2O';
Ym_ox(10,:,:)=Ym_CO2';
Ym_ox_v=(1-Ym_ox(9,:,:)-Ym_ox(10,:,:)).*Ym_ox(1:8,:,:);
Ym_ox_v(9,:,:)=Ym_ox(9,:,:);
Ym_ox_v(10,:,:)=Ym_ox(10,:,:);
Ym_ox_norm= Ym_ox_v./sum(Ym_ox);

oxides = ["SiO_2";"TiO_2";"Al2O_3";"FeO";"MgO";"CaO";"Na_2O";"K_2O";"H_2O";"CO_2"];
title_lect = ["a)";"b)";"c)";"d)";"e)";"f)";"g)";"h)";"i)";"j)";"k)";"l)"];
x0=500;
y0=500;
width=600;
height= 800;
set(gcf,'color','w');
set(gcf,'units','points','position',[x0,y0,width,height])
set(gcf,'color','w');

color=jet(length(t))/1.4;
color(1,:)=[0 0 1];
color(end,:)=[1 0 0];

formatSpec = '%.2f';

Lt = length(t);
I_t = [1, Lt/3, Lt/2, 2*Lt/3, Lt]; %1:1:Lt;
I_t = round(I_t);

pl=ones(length(I_t),1);
n_line = 6;
n_col = 2;
for k=1:length(oxides)
    
    a(k)=subplot(n_line,n_col,k);
    title(title_lect(k), 'FontSize', 8);
    
    for n=1:length(I_t)
        hold all
        pl(n)=plot(x*1e6,Ym_ox_norm(k,:,I_t(n))*100,'color',color(I_t(n),:));
    end
    ylabel('wt%')
    xlim([-xlim_plot xlim_plot])
    dim = [.2 .55 .3 .3];
    annotation('textbox',dim,'String',oxides(k),'Position',a(k).Position,'Vert','top','FitBoxToText','on','EdgeColor','none');
    set(gca,'Xticklabel',[])
    
end

a(length(oxides)+1)=subplot(n_line,n_col,length(oxides)+1);
title(title_lect(end-1), 'FontSize', 8);
for n=1:length(I_t)
    hold all
    pl(n)=plot(x*1e6,alphag(I_t(n),:)*100,'color',color(I_t(n),:));
end
ylabel('vol.%')
xlim([-xlim_plot xlim_plot])
dim = [.2 .55 .3 .3];
annotation('textbox',dim,'String',"Gas",'Position',a(length(oxides)+1).Position,'Vert','top','FitBoxToText','on','EdgeColor','none');
xlabel('x (\mu m)')
xlim([-xlim_plot xlim_plot])

a(length(oxides)+2)=subplot(n_line,n_col,length(oxides)+2);
title(title_lect(end), 'FontSize', 8);
for n=1:length(I_t)
    hold all
    pl(n)=plot(x*1e6,C_CO2g(I_t(n),:)*100,'color',color(I_t(n),:));
end
ylabel('wt%')
xlim([-xlim_plot xlim_plot])
dim = [.2 .55 .3 .3];
annotation('textbox',dim,'String',"CO_2^{gas}",'Position',a(length(oxides)+2).Position,'Vert','top','FitBoxToText','on','EdgeColor','none');
xlabel('x (\mu m)')
xlim([-xlim_plot xlim_plot])

leg_str{1}=strcat(num2str(t(I_t(1)),formatSpec),' s');
pls(1)=pl(1);
for i=2:length(I_t)
    leg_str{i}=strcat(num2str(t(I_t(i)),formatSpec),' s');
    pls(i)=pl(i);
end
legend(pls,leg_str,'EdgeColor','none')

if plot_exp
    plot_experiments
end

