import pandas as p
import math
import numpy as np
import itertools as it
from haversine import *
from haversine_script import *
import time
import json
from random import *


def get_clusters_by_index(y,range_th,maximum_points_per_cluster):
    """
    Τhis method scans sequentially the dataset, and returns for each location the 
    first n=maximum_points_per_cluster locations met that are in the range_th from said location

    Parameters
    ----------
    y : list
        The location of the training points
    range_th : int
        The distance in meters, below which two fingerprints are concidered proximal (ProxyFAUG parameter)
    maximum_points_per_cluster : int
        The maximum ammount of fingerprints allowed in a cluster (ProxyFAUG parameter)

    Returns
    -------
    list_of_clusters : list
        A list with one cluster for each of the training set points
    """
    list_of_clusters = []
    for index_i in range(len(y)):
        loc = y[index_i]
        cluster = []
        for index_j in range(index_i,len(y)):  # starting from index_i to avoid selecting the same pairs
            temp = y[index_j]
            if haversine(loc,temp)*1000<range_th:
                cluster.append(index_j)
                if len(cluster) >= maximum_points_per_cluster:
                    break
        if len(cluster)>1:
            list_of_clusters.append(cluster)
    return list_of_clusters


def crossover_and_mutate(f0,f1,p):
    """
    Τhis method performs the crossover and mutate operator of ProxyFAUG. 
    It uses f0,f1 as the parrent fingerprints and p as the mutation propability

    Parameters
    ----------
    f0: list
        The fingerpint vector of the first parrent
    f1 : lists
        The fingerpint vector of the first parrent
    p : float
        The mutation propability (ProxyFAUG parameter)

    Returns
    -------
    result : list
        The resulting augmented fingerprint
    """
    length = len(f1)
    randBinList = [randint(0,1) for b in range(length)]
    result = []
    for i in range(length):
        r = uniform(0, 1)
        if p>r:    # should we mutate?
            w = uniform(0, 1)
            weighted_element = w*f0[i] + (1-w)*f1[i]
            result.append(weighted_element)
        else:      # crossover
            if randBinList[i]==0:
                result.append(f0[i])
            elif randBinList[i]==1:
                result.append(f1[i])
    return result


def get_all_pairs(cluster_x,cluster_y): 
    """
    Returning all posible pairs of fingerprints from a cluster. 
    Two lists of pairs are returned, one for the fingerprint features x and one for the locations y.

    Parameters
    ----------
    cluster_x: numpy.ndarray
        The cluster of fingerpints, in the feature side x of the fingerprints
    cluster_y : numpy.ndarray
        The cluster of fingerpints, in the location side y of the fingerprints

    Returns
    -------
    result_x : list
        The list of all posible pairs, in the feature side x of the fingerprints
    result_y : list
        The list of all posible pairs, in the location side y of the fingerprints
    """
    length = cluster_x.shape[0]
    result_x = []
    result_y = []
    for i in range(length-1):
        for j in range(i+1,length):
            result_x.append([cluster_x[i,:],cluster_x[j,:]])
            result_y.append([cluster_y[i,:],cluster_y[j,:]])
    return result_x, result_y


def make_pair_crossovers(pair_x,pair_y,p,crossovers_per_pair):
    """
    Performing the crossover_and_mutate operator, at the given pair, for crossovers_per_pair times.
    Returns the lists of augmented fingerprints, in seperate lists of features x and locations yTwo lists of pairs are returned, one for the fingerprint features x and one for the locations y.

    Parameters
    ----------
    pair_x: list
        The cluster of fingerpints, in the feature side x of the fingerprints
    pair_y : list
        The cluster of fingerpints, in the location side y of the fingerprints
    p : float
        The mutation propability (ProxyFAUG parameter)
    crossovers_per_pair : int
        The ammount of augmented fingerprints resulting from a pair of parrent fingerprints,
        created by the crossover_and_mutate operator (ProxyFAUG parameter)

    Returns
    -------
    result_x : list
        The list with the feature side x of the augmented fingerprints
    result_y : list
        The list with the location side y of the augmented fingerprints
    """
    x_0 = pair_x[0]
    x_1 = pair_x[1]
    y_0 = pair_y[0]
    y_1 = pair_y[1]
    result_x = []
    result_y = []
    for c in range(crossovers_per_pair):
        result_x.append(crossover_and_mutate(x_0,x_1,p))
        midpoint_y = np.mean(np.array((y_0,y_1)),axis=0)
        result_y.append(midpoint_y) 
    return result_x, result_y


def make_cluster_crossovers(cluster_x,cluster_y,p,crossovers_per_pair):
    """
    This method creates all the augmented fingerprints of a cluster

    Parameters
    ----------
    cluster_x: numpy.ndarray
        The cluster of fingerpints, in the feature side x of the fingerprints
    cluster_y : numpy.ndarray
        The cluster of fingerpints, in the location side y of the fingerprints
    p : float
        The mutation propability (ProxyFAUG parameter)
    crossovers_per_pair : int
        The ammount of augmented fingerprints resulting from a pair of parrent fingerprints,
        created by the crossover_and_mutate operator (ProxyFAUG parameter)

    Returns
    -------
    all_cluster_crossovers_x : numpy.ndarray
        The array containing the feature side x information of all the resulting augmented fingerprints
    all_cluster_crossovers_y : numpy.ndarray
        The array containing the location side y information of all the resulting augmented fingerprints
    """
    pairs_x, pairs_y = get_all_pairs(cluster_x,cluster_y)
    all_cluster_crossovers_x = []
    all_cluster_crossovers_y = []
    for i in range(len(pairs_x)):
        pair_x = pairs_x[i]
        pair_y = pairs_y[i]
        pair_crossovers_x, pair_crossovers_y  = make_pair_crossovers(pair_x,pair_y,p,crossovers_per_pair)
        for j in range(len(pair_crossovers_x)):
            element_x = pair_crossovers_x[j]
            element_y = pair_crossovers_y[j]
            all_cluster_crossovers_x.append(element_x)
            all_cluster_crossovers_y.append(element_y)
    all_cluster_crossovers_x = np.array(all_cluster_crossovers_x)
    all_cluster_crossovers_y = np.array(all_cluster_crossovers_y)
    return all_cluster_crossovers_x, all_cluster_crossovers_y


def get_cluster_centroid(cluster_indeces,y):
    """
    This method returns the centroid location of a cluster of locations

    Parameters
    ----------
    cluster_indeces: list
        The list of indeces, characterizing the elements of a cluster, indexed in the y
    y : numpy.ndarray
        The location information of the training set, with respect to which the indeces were taken

    Returns
    -------
    result : numpy.ndarray
        The centroid location of the cluster
    """
    cluster_locations = []
    for index in cluster_indeces:
        cluster_locations.append(y[index])
    cluster_locations_array = np.array(cluster_locations)
    result = np.mean(cluster_locations_array,axis=0)
    return result


def augment(X,Y,range_th,maximum_points_per_cluster,p,crossovers_per_pair):
    """
    This method creates the location informtion y of the augmented fingerprints.
    To do so, it assigns the centroid location of a cluster of locations,
    to a list of equal size to the ammount of resulting augmented set from the cluster.

    Parameters
    ----------
    X : numpy.ndarray
        The feature information of the fingerprints of the training set
    Y : numpy.ndarray
        The location information of the fingerprints of the training set
    range_th : int
        The distance in meters, below which two fingerprints are concidered proximal (ProxyFAUG parameter)
    maximum_points_per_cluster : int
        The maximum ammount of fingerprints allowed in a cluster (ProxyFAUG parameter)
    p : float
        The mutation propability (ProxyFAUG parameter)
    crossovers_per_pair : int
        The ammount of augmented fingerprints resulting from a pair of parrent fingerprints,
        created by the crossover_and_mutate operator (ProxyFAUG parameter)

    Returns
    -------
    result : list
        The resulting list with the centroid location of the cluster,
        as the ground truth of the augmented finerprints
    """
    seed(0)  # for reproducibility
    list_of_clusters = get_clusters_by_index(Y,range_th,maximum_points_per_cluster)
    x_aug= np.empty([1,X.shape[1]])
    y_aug = np.empty([1,Y.shape[1]])

    for cluster_indeces in list_of_clusters:
        cluster_x = X[cluster_indeces]
        cluster_y = Y[cluster_indeces]
        all_cluster_crossovers_x, all_cluster_crossovers_y = make_cluster_crossovers(cluster_x,cluster_y,p,crossovers_per_pair)
        x_aug = np.vstack((x_aug,all_cluster_crossovers_x))        
        y_aug = np.vstack((y_aug,all_cluster_crossovers_y))
    y_aug = np.delete(y_aug, (0), axis=0)
    x_aug = np.delete(x_aug, (0), axis=0)

    print('done')
    return x_aug, y_aug
