library(Rmisc)
library(tidyverse)
library(cowplot)
library(scales)

# ############# START FIGURE 1 ############# #

# Figure 1A
# Read in data and calculate summary statistics
siD11 <- read_tsv("siD11.txt")
siD11_summary <- summarySE(siD11,measurevar="RLuc", groupvars=c("siRNA","Time","Experiment"))
# Apply relative mean normalisation
siD11_summary <- 
	siD11_summary %>% 
	left_join(group_by(group_by(siD11_summary, Experiment, Time) %>% summarise(mean = mean(RLuc)),Time) %>% mutate(scale = mean[Experiment==1] / mean),by= c("Time" = "Time", "Experiment" = "Experiment")) %>% 
	mutate(RLuc = RLuc * scale)
siD11_summary_all <- summarySE(siD11_summary,measurevar="RLuc", groupvars=c("siRNA","Time"))
siD11_summary_all$Time <- factor(siD11_summary_all$Time, levels = c("6","24","48"))
siD11_summary_all$siRNA <- factor(siD11_summary_all$siRNA, levels = c("siNC","siD11"))
# Generate the plot
siD11.plot <- 
    ggplot(siD11_summary_all, 
           aes(x=`Time`,y=`RLuc`,group=`siRNA`,colour=`siRNA`)) +
    geom_bar(aes(fill = `siRNA`), 
             stat="identity",
             position = position_dodge(width=0.8), 
             width=0.7, 
             colour="black") +
    geom_errorbar(aes(ymin=`RLuc`-`se`, ymax=`RLuc`+`se`), 
                  position = position_dodge(width=0.8),
                  colour="black", 
                  width=0.25) +
    scale_y_log10(name = "Relative Luciferase Units", 
                  expand = c(0,0),
                  limits=c(1,10^7), 
                  breaks=c(10^1,10^2,10^3,10^4,10^5,10^6,10^7),
                  labels = trans_format("log10", math_format(10^.x))) +
    theme(legend.position = "top",
          legend.background = element_rect(fill="white",
                                           size=0.5, 
                                           linetype="solid",
                                           colour ="black"),
          legend.title=element_blank(),
		  legend.margin=margin(c(1,1,2,1)),
		  axis.text.x = element_text(size=8),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=9),
          axis.title.y = element_text(size=9),
		  legend.text = element_text(size=8),
          legend.key.size = unit(0.25, "cm")) +
    scale_x_discrete(name = "Hours Post Transfection") +
    annotation_logticks(sides = "l")  +
    coord_cartesian(ylim=c(5*10^3,3*10^6))
siD11.plot

# Figure 1B
# Read in data and calculate summary statistics
KO_Luc <- read_tsv("KO_Luc.txt")
# Summarise each experiment
KO_Luc_summary <- summarySE(KO_Luc,measurevar="RLuc", groupvars=c("Cell.Line","Time","Experiment"))
# Apply relative mean normalisation
KO_Luc_summary <- 
	KO_Luc_summary %>% 
	left_join(group_by(group_by(KO_Luc_summary, Experiment, Time) %>% summarise(mean = mean(RLuc)),Time) %>% mutate(scale = mean[Experiment==1] / mean),by= c("Time" = "Time", "Experiment" = "Experiment")) %>% 
	mutate(RLuc = RLuc * scale)
# Summarise accross all experiments
KO_Luc_summary_all <- summarySE(KO_Luc_summary,measurevar="RLuc", groupvars=c("Cell.Line","Time"))
KO_Luc_summary_all$Time <- factor(KO_Luc_summary_all$Time, levels = c("6","24","48"))
KO_Luc_summary_all$Cell.Line <- factor(KO_Luc_summary_all$Cell.Line, levels = c("Huh7","D11-KO-8","D11-KO-9"))
# Generate the plot
KO_Luc.plot <- 
    ggplot(KO_Luc_summary_all,aes(x=`Time`,y=`RLuc`,group=`Cell.Line`,colour=`Cell.Line`)) +
    geom_bar(aes(fill = `Cell.Line`),
             stat="identity",
             position = position_dodge(width=0.8),
             width=0.7, 
             colour="black") +
    geom_errorbar(aes(ymin=`RLuc`-`se`, ymax=`RLuc`+`se`),
                  position = position_dodge(width=0.8),
                  colour="black", 
                  width=0.5) +
    scale_y_log10(name = "Relative Luciferase Units", 
                  expand = c(0,0),
                  limits=c(1,10^7), 
                  breaks=c(10^1,10^2,10^3,10^4,10^5,10^6,10^7),
                  labels = trans_format("log10", math_format(10^.x))) +
    theme(legend.position = "top",
          legend.background = element_rect(fill="white",
                                           size=0.5, 
                                           linetype="solid",
                                           colour ="black"), 
                                           legend.title=element_blank(),
		  legend.margin=margin(c(1,1,2,1)),
		  axis.text.x = element_text(size=8),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=9),
          axis.title.y = element_text(size=9),
		  legend.text = element_text(size=8),
          legend.key.size = unit(0.25, "cm")) +
    scale_x_discrete(name = "Hours Post Transfection") +
    annotation_logticks(sides = "l")  +
    coord_cartesian(ylim=c(5*10^3,2*10^6))

# Figure 1E
# A fix for problem with scaling on graph
# See https://stackoverflow.com/questions/22295253/force-bars-to-start-from-a-lower-value-than-0-in-ggplot-geom-bar-in-r
# Credit to user Shadow : https://stackoverflow.com/users/2591234/shadow
mylog_trans <- function(base=exp(1), from=0) 
{
  trans <- function(x) log(x, base)-from
  inv <- function(x) base^(x+from)
  trans_new("mylog", trans, inv, log_breaks(base=base), 
            domain = c(base^from, Inf))
}
# Read in data and calculate summary statistics
Huh7_RNA <- read_tsv("Huh7_RNA.txt")
Huh7_RNA_summary_all <- summarySE(Huh7_RNA,measurevar="RNA", groupvars=c("Cell.Line","Time"))
Huh7_RNA_summary_all$Time <- factor(Huh7_RNA_summary_all$Time, levels = c("24","48","72"))
Huh7_RNA_summary_all$Cell.Line <- factor(Huh7_RNA_summary_all$Cell.Line, levels = c("Huh7","D11-KO-9"))
# Generate the plot
Huh7_RNA.plot <- 
    ggplot(Huh7_RNA_summary_all,aes(x=`Time`,y=`RNA`,group=`Cell.Line`,colour=`Cell.Line`)) +
    geom_bar(aes(fill = `Cell.Line`),
             stat="identity",
             position = position_dodge(width=0.8),
             width=0.7, 
             colour="black") +
    geom_errorbar(aes(ymin=`RNA`-`se`, ymax=`RNA`+`se`),
                  position = position_dodge(width=0.8),
                  colour="black", 
                  width=0.5) +
     scale_y_continuous(name = "Relative HCV RNA", 
                  expand = c(0,0),
                  #limits=c(10^-1,10^3), 
                  breaks=c(1,10^1,10^2,10^3),
                  #labels = trans_format("log10", math_format(10^.x)),
				  trans = mylog_trans(base=10, from=-2))  +
    theme(legend.position = "top",
          legend.background = element_rect(fill="white",
                                           size=0.5, 
                                           linetype="solid",
                                           colour ="black"), 
                                           legend.title=element_blank(),
		  legend.margin=margin(c(1,1,2,1)),
		  axis.text.x = element_text(size=8),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=9),
          axis.title.y = element_text(size=9),
		  legend.text = element_text(size=8),
          legend.key.size = unit(0.25, "cm")) +
    scale_x_discrete(name = "Hours Post Infection") +
    annotation_logticks(sides = "l")  +
    coord_cartesian(ylim=c(10^-1,300))
Huh7_RNA.plot


# Figure 1F
# Read in data and calculate summary statistics
Huh7_TCID50 <- read_tsv("Huh7_TCID50.txt")
Huh7_TCID50_summary_all <- summarySE(Huh7_TCID50,measurevar="TCID50", groupvars=c("Cell.Line","Time"))
Huh7_TCID50_summary_all$Time <- factor(Huh7_TCID50_summary_all$Time, levels = c("48","72"))
Huh7_TCID50_summary_all$Cell.Line <- factor(Huh7_TCID50_summary_all$Cell.Line, levels = c("Huh7","D11-KO-9"))
# Generate the plot
Huh7_TCID50.plot <- 
    ggplot(Huh7_TCID50_summary_all,aes(x=`Time`,y=`TCID50`,group=`Cell.Line`,colour=`Cell.Line`)) +
    geom_bar(aes(fill = `Cell.Line`),
             stat="identity",
             position = position_dodge(width=0.8),
             width=0.7, 
             colour="black") +
    geom_errorbar(aes(ymin=`TCID50`-`se`, ymax=`TCID50`+`se`),
                  position = position_dodge(width=0.8),
                  colour="black", 
                  width=0.5) +
    scale_y_log10(name = "Infectious HCV (TCID50)", 
                  expand = c(0,0),
                  limits=c(1,10^7), 
                  breaks=c(10^1,10^2,10^3,10^4,10^5,10^6,10^7),
                  labels = trans_format("log10", math_format(10^.x))) +
    theme(legend.position = "top",
          legend.background = element_rect(fill="white",
                                           size=0.5, 
                                           linetype="solid",
                                           colour ="black"), 
                                           legend.title=element_blank(),
          legend.margin=margin(c(1,1,2,1)),
		  axis.text.x = element_text(size=8),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=9),
          axis.title.y = element_text(size=9),
		  legend.text = element_text(size=8),
          legend.key.size = unit(0.25, "cm")) +
    scale_x_discrete(name = "Hours Post Infection") +
    annotation_logticks(sides = "l")  +
    coord_cartesian(ylim=c(10^1,10^4))
Huh7_TCID50.plot

# Figure 1G
# Read in data and calculate summary statistics
HepG2_TCID50 <- read_tsv("HepG2_TCID50.txt")
HepG2_TCID50_summary_all <- summarySE(HepG2_TCID50,measurevar="TCID50", groupvars=c("Cell.Line"))
HepG2_TCID50_summary_all$Cell.Line <- factor(HepG2_TCID50_summary_all$Cell.Line, levels = c("HepG2-CD81","-D11-KO","-miR-122","-miR-122-D11-KO"))
# Generate the plot
HepG2_TCID50.plot <- 
    ggplot(HepG2_TCID50_summary_all,aes(x=`Cell.Line`,y=`TCID50`)) +
    geom_bar(aes(fill = `Cell.Line`),
             stat="identity",
             position = position_dodge(width=0.8),
             width=0.7, 
             colour="black") +
    geom_errorbar(aes(ymin=`TCID50`-`se`, ymax=`TCID50`+`se`),
                  position = position_dodge(width=0.8),
                  colour="black", 
                  width=0.5) +
    scale_y_log10(name = "Infectious HCV (TCID50)", 
                  expand = c(0,0),
                  limits=c(1,10^7), 
                  breaks=c(10^0,10^1,10^2,10^3,10^4,10^5,10^6,10^7),
                  labels = trans_format("log10", math_format(10^.x))) +
    #theme(legend.position="none",axis.text.x = element_text(angle = 60, hjust = 1)) +
	theme(legend.position = "top",
          legend.background = element_rect(fill="white",
                                           size=0.25, 
                                           linetype="solid",
                                           colour ="black"), 
                                           legend.title=element_blank(),
			legend.margin=margin(c(1,1,2,1)),
#		   axis.title.x=element_blank(),
           axis.text.x=element_blank(),
           axis.ticks.x=element_blank(),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=9),
          axis.title.y = element_text(size=9),
		  legend.text = element_text(size=8),
          legend.key.size = unit(0.25, "cm")) +
    scale_x_discrete(name = "Cell Line") +
    annotation_logticks(sides = "l")  +
    coord_cartesian(ylim=c(10^0,10^4)) +
	guides(fill=guide_legend(nrow=2,byrow=TRUE))
HepG2_TCID50.plot


# Figure 1H
HCV_REPLICON_AREA <- read_tsv("Colony.txt")
HCV_REPLICON_AREA_summary <- summarySE(HCV_REPLICON_AREA,measurevar="Area", groupvars=c("Cell.Line","Image"))
HCV_REPLICON_AREA_summary$Area <- HCV_REPLICON_AREA_summary$Area / mean(filter(HCV_REPLICON_AREA_summary, `Cell.Line` == "Huh7")$Area)
HCV_REPLICON_AREA_summary_all <- summarySE(HCV_REPLICON_AREA_summary,measurevar="Area", groupvars=c("Cell.Line"))
HCV_REPLICON_AREA_summary_all$Cell.Line <- factor(HCV_REPLICON_AREA_summary_all$Cell.Line, levels = c("Huh7","D11-KO-9"))
# Generate the plot
HCV_REPLICON_AREA.plot <- ggplot(HCV_REPLICON_AREA_summary_all,aes(x=`Cell.Line`,y=`Area`,group=`Cell.Line`,colour=`Cell.Line`)) +
	geom_bar(aes(fill = `Cell.Line`),stat="identity",position = position_dodge(width=0.8),width=0.7, colour="black") +
	geom_errorbar(aes(ymin=`Area`-`se`, ymax=`Area`+`se`),position = position_dodge(width=0.8),colour="black", width=0.5) +
	guides(fill=FALSE) +
	scale_y_continuous(name = "Relative Area Colonized",expand = c(0,0), breaks=c(1,2,3,4),limits=c(0,5)) +
	scale_x_discrete(name = "Cell Line") +
	coord_cartesian(ylim=c(0,4)) +
	theme(axis.text.x = element_text(size=8,angle = 90, hjust = 1),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=9),
          axis.title.y = element_text(size=9),
		  legend.text = element_text(size=8),
          legend.key.size = unit(0.25, "cm"))
	#theme(legend.title=element_blank())

HCV_REPLICON_AREA.plot

# Plot Figure 1
p1 <- plot_grid(NULL, NULL, siD11.plot, KO_Luc.plot, Huh7_TCID50.plot,Huh7_RNA.plot,  HepG2_TCID50.plot, plot_grid(HCV_REPLICON_AREA.plot,NULL,NULL,ncol=3), labels = c("A","C","B","D","E","F","G","H"), ncol=2, align="h", rel_heights=c(0.25,1,1,1))
save_plot("Figure1.svg",p1,ncol=2,nrow=4,base_width=2.5,base_height = 2)
# ############# END FIGURE 1 ############# #


# ############# START FIGURE 2 ############# #

# Figure 2A
# Read in data and calculate summary statistics
KO_Luc_GND <- read_tsv("KO_Luc_GND.txt")
KO_Luc_GND_summary <- summarySE(KO_Luc_GND,measurevar="RLuc", groupvars=c("Cell.Line","Time","Experiment"))
# Apply relative mean normalisation
KO_Luc_GND_summary <- 
	KO_Luc_GND_summary %>% 
	left_join(group_by(group_by(KO_Luc_GND_summary, Experiment, Time) %>% summarise(mean = mean(RLuc)),Time) %>% mutate(scale = mean[Experiment==1] / mean),by= c("Time" = "Time", "Experiment" = "Experiment")) %>% 
	mutate(RLuc = RLuc * scale)
# Summarise across all experiments
KO_Luc_GND_summary_all <- summarySE(KO_Luc_GND_summary,measurevar="RLuc", groupvars=c("Cell.Line","Time"))
# Fix the timepoint order
KO_Luc_GND_summary_all$Time <- factor(KO_Luc_GND_summary_all$Time, levels = c("6","24","48"))
# Fix the cell line order
KO_Luc_GND_summary_all$Cell.Line <- factor(KO_Luc_GND_summary_all$Cell.Line, levels = c("Huh7","D11-KO-8","D11-KO-9"))
# Generate the plot
KO_Luc_GND.plot <- ggplot(KO_Luc_GND_summary_all,aes(x=`Time`,y=`RLuc`,group=`Cell.Line`,colour=`Cell.Line`)) +
 geom_bar(aes(fill = `Cell.Line`),stat="identity",position = position_dodge(width=0.8),width=0.7, colour="black") +
 geom_errorbar(aes(ymin=`RLuc`-`se`, ymax=`RLuc`+`se`),position = position_dodge(width=0.8),colour="black", width=0.5) +
 scale_y_log10(name = "Relative Luciferase Units", expand = c(0,0),limits=c(1,10^6), breaks=c(10^1,10^2,10^3,10^4,10^5,10^6),labels = trans_format("log10", math_format(10^.x))) +
     theme(legend.position = "top",
          legend.background = element_rect(fill="white",
                                           size=0.5, 
                                           linetype="solid",
                                           colour ="black"),
          legend.title=element_blank(),
		  legend.margin=margin(c(1,1,2,1)),
		  axis.text.x = element_text(size=8),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=9),
          axis.title.y = element_text(size=9),
		  legend.text = element_text(size=8),
          legend.key.size = unit(0.25, "cm")) +
    scale_x_discrete(name = "Hours Post Transfection") +
    annotation_logticks(sides = "l")  +
 coord_cartesian(ylim=c(5*10^1,10^5))
KO_Luc_GND.plot

# Figure 2B
# Read in data and calculate summary statistics per experiment
siXRN <- read_tsv("siXRN.txt")
siXRN_summary <- summarySE(siXRN,measurevar="RLuc", groupvars=c("Cell.Line","Time","Experiment","siRNA"))
# Calculate the ratios for each experiment
siXRN_summary_NC <- filter(siXRN_summary,siRNA=="siNC")
siXRN_summary_XRN <- filter(siXRN_summary,siRNA=="siXRN1/2")
siXRN_summary_XRN$RLucNC <- siXRN_summary_NC$RLuc
siXRN_summary <- transform(siXRN_summary_XRN, RLuc = RLuc / RLucNC)
# Calculate the summary statistics for all experiments
siXRN_summary_all <- summarySE(siXRN_summary,measurevar="RLuc", groupvars=c("Cell.Line","Time","siRNA"))
siXRN_summary_all$Time <- factor(siXRN_summary_all$Time, levels = c("6","24","48"))
siXRN_summary_all$Cell.Line <- factor(siXRN_summary_all$Cell.Line, levels = c("Huh7","D11-KO-8","D11-KO-9"))
# Fix legend title (if using)
siXRN_summary_all$"Cell Line:" <- factor(siXRN_summary_all$Cell.Line, levels = c("Huh7","D11-KO-8","D11-KO-9"))
# Generate the plot
siXRN.plot <- ggplot(siXRN_summary_all,aes(x=`Time`,y=`RLuc`,group=`Cell Line:`,colour=`Cell Line:`)) +
	geom_bar(aes(fill = `Cell Line:`),stat="identity",position = position_dodge(width=0.8),width=0.7, colour="black") +
	geom_errorbar(aes(ymin=`RLuc`-`se`, ymax=`RLuc`+`se`),position = position_dodge(width=0.8),colour="black", width=0.5) +
	theme(legend.position = "top",legend.background = element_rect(fill="white",size=0.5, linetype="solid",colour ="black"), legend.title=element_blank()) +
	scale_y_continuous(name = "Relative Luciferase (siXRNs/siNC)",expand = c(0,0), breaks=c(0.0,0.5,1.0,1.5,2.0,2.5,3.0)) +
	theme(legend.position = "top",
          legend.background = element_rect(fill="white",
                                           size=0.5, 
                                           linetype="solid",
                                           colour ="black"),
          legend.title=element_blank(),
		  legend.margin=margin(c(1,1,2,1)),
		  axis.text.x = element_text(size=8),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=9),
          axis.title.y = element_text(size=9),
		  legend.text = element_text(size=8),
          legend.key.size = unit(0.25, "cm")) +
    scale_x_discrete(name = "Hours Post Transfection")
	#theme(legend.title=element_blank())

p2 <- plot_grid(KO_Luc_GND.plot, siXRN.plot, NULL, labels = c("A","C","B"), ncol=2, align="h", rel_heights=c(1,0.5))
save_plot("Figure2.svg",p2,ncol=2,nrow=2,base_width=2.5,base_height = 2)
# ############# END FIGURE 2 ############# #


# ############# START FIGURE 3 ############# #

anti122 <- read_tsv("anti122.txt")
anti122_summary <- summarySE(anti122,measurevar="RLuc", groupvars=c("Cell.Line","Time","Experiment","antimiR"))

# Ratios
anti122_summary_NC <- filter(anti122_summary,antimiR=="antiNC")
anti122_summary_122 <- filter(anti122_summary,antimiR=="anti122")
anti122_summary_122$RLucNC <- anti122_summary_NC$RLuc
anti122_summary <- transform(anti122_summary_122, RLuc = RLuc / RLucNC)

anti122_summary_all <- summarySE(anti122_summary,measurevar="RLuc", groupvars=c("Cell.Line","Time","antimiR"))
anti122_summary_all$Time <- factor(anti122_summary_all$Time, levels = c("6","24","48"))
anti122_summary_all$Cell.Line <- factor(anti122_summary_all$Cell.Line, levels = c("Huh7","D11-KO-8","D11-KO-9"))
# Fix legend title 
anti122_summary_all$"Cell Line:" <- factor(anti122_summary_all$Cell.Line, levels = c("Huh7","D11-KO-8","D11-KO-9"))

anti122.plot <- ggplot(anti122_summary_all,aes(x=`Time`,y=`RLuc`,group=`Cell Line:`,colour=`Cell Line:`)) +
	geom_bar(aes(fill = `Cell Line:`),stat="identity",position = position_dodge(width=0.8),width=0.7, colour="black") +
	geom_errorbar(aes(ymin=`RLuc`-`se`, ymax=`RLuc`+`se`),position = position_dodge(width=0.8),colour="black", width=0.5) +
	theme(legend.position = "top",legend.background = element_rect(fill="white",size=0.5, linetype="solid",colour ="black"), legend.title=element_blank()) +
	scale_y_continuous(name = "Relative Luciferase (anti122/antiNC)",expand = c(0,0), limits = c(0.0,0.8), breaks=c(0.0,0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.8)) +
	theme(legend.position = "top",
          legend.background = element_rect(fill="white",
                                           size=0.5, 
                                           linetype="solid",
                                           colour ="black"),
          legend.title=element_blank(),
		  legend.margin=margin(c(1,1,2,1)),
		  axis.text.x = element_text(size=8),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=8),
          axis.title.y = element_text(size=8),
		  legend.text = element_text(size=8),
          legend.key.size = unit(0.25, "cm")) +
	scale_x_discrete(name = "Hours Post Transfection")
anti122.plot
# Plot Figure 5
p3 <- plot_grid(NULL, anti122.plot, labels = c("A", "B"), ncol = 2)
save_plot("Figure3.svg",p3,ncol=2,nrow=1,base_width=2.5,base_height = 2)

# ############# START FIGURE 5 ############# #

####### Start Figure 5 ##########
TCGA <- read_tsv("TCGA_HCC.txt")
TCGA <- mutate(TCGA, Fraction=Low/(Low+NormalToHigh))
TCGA$Category <- factor(TCGA$Category, levels = c("Non-viral","HBV","HCV"))
# Generate the plot
TCGA.plot <- ggplot(TCGA,aes(x=`Category`,y=`Fraction`)) +
	geom_bar(aes(fill = `Category`),stat="identity",position = position_dodge(width=0.8),width=0.7, colour="black") +
	scale_y_continuous(name = "Fraction with Low DUSP11",expand = c(0,0), breaks=c(0.05,0.10,0.15,0.2,0.25)) +
	theme(axis.text.x = element_text(size=8),
          axis.text.y = element_text(size=8),  
          axis.title.x = element_text(size=9),
          axis.title.y = element_text(size=9)) +
    scale_x_discrete(name = "Clinical Classification") +
	theme(legend.position="none") +
	coord_cartesian(ylim=c(0,0.25))
TCGA.plot
p5 <- plot_grid(TCGA.plot,NULL,labels = c("A"), ncol = 2)
save_plot("Figure5.svg",p5,ncol=2,nrow=1,base_width=2.5,base_height = 2)

# ############# END FIGURE 5 ############# #



























