using Pkg
Pkg.activate(joinpath(@__DIR__, ".."))
Pkg.instantiate()

using Mimi, CSVFiles, DataFrames, VegaLite, MimiGIVE, Random, NBInclude

################################################################################
# Step 1. Setup
################################################################################

our_seed = 24523438

output_dir = joinpath(@__DIR__, "..", "output")
num_trials = 10_000

################################################################################
# Step 2. Define all discounting schemes, specified with:

# label::String - a label to define the set of parameters
# eta::Float64 - eta parameter for Ramsey discounting (pure rate of time preference)
# prtp::Float64 - prtp parameter for Ramsey discounting (elasticity of marginal utility of consumption)
# ew::Symbol - the approach used for distributional weighting as defined by a spatial
#            resolution (country or region) and metric (gdp or consumption)
# ew_norm_region::Symbol - normalization region for distributional weighting
################################################################################

# eta and prtp pairs
prtp_values = sort([0.001, 0.002, 0.005, 0.01, 0.015])
eta_values = [collect(0:0.05:0.5)..., collect(0.6:0.1:2.)...]

discounting_params = [
    (params_name = "prtp_$(prtp)_eta_$eta", 
                    prtp = prtp,
                    eta = eta
    ) for prtp in prtp_values, eta in eta_values
][:]

# country resolution approaches
countries = ["USA"] # Any subset of available 184 countries in GIVE
approaches = [:gdp_country] # options are :consumption_country :gdp_country

discount_rates_country = [
    (
        label = "$(discounting_params[i].params_name) $(countries[k]) $(approaches[j])", 
        prtp = discounting_params[i].prtp, 
        eta = discounting_params[i].eta, 
        ew = approaches[j], 
        ew_norm_region = countries[k]
    ) for i in 1:length(discounting_params) for j in 1:length(approaches) for k in 1:length(countries) 
]

# No distributional weighting
discount_rates_no_distrib_weighting = [
        (
            label = "$(discounting_params[i].params_name)", 
            prtp = discounting_params[i].prtp, 
            eta = discounting_params[i].eta, 
            ew = nothing, 
            ew_norm_region = nothing
        ) for i in 1:length(discounting_params) 
    ]

discount_rates_give_original = [
    (
            label = "GIVE original", 
            prtp = exp(0.001972641) - 1, 
            eta = 1.244458999, 
            ew = nothing, 
            ew_norm_region = nothing
    ),
    (
            label = "GIVE original ew", 
            prtp = exp(0.001972641) - 1, 
            eta = 1.244458999, 
            ew = :gdp_country, 
            ew_norm_region = "USA"
    )
]

# Combine all three sets of discount rates so that the final set includes all required
# permutations of discounting parameters, spatial resolutions (country or region),
# and approaches (consumption_country, consumption_reigion, gdp_country, or 
# gdp_region)

discount_rates = vcat(discount_rates_country, discount_rates_no_distrib_weighting, discount_rates_give_original)

################################################################################
# Step 3. Produce Results
################################################################################

mkpath(output_dir)

m = MimiGIVE.get_model(socioeconomics_source = :RFF)

Random.seed!(our_seed)
results = MimiGIVE.compute_scc(m;
    year = 2020,
    last_year = 2300,
    discount_rates = discount_rates,
    fair_parameter_set = :random,
    rffsp_sampling = :random,
    n = num_trials,
    gas = :CO2,
    save_slr_damages = false,
    save_md = false,
    save_cpc = false,
    compute_sectoral_values = false,
    compute_domestic_values = false,
    CIAM_foresight = :perfect,
    CIAM_GDPcap = true,
    pulse_size = 1e-4,
)

# Save the expected SCC values with standard errors
df = DataFrame()
for (k, v) in results[:scc]
    append!(df, 
        DataFrame(
            :label => k.dr_label,
            :sector => k.sector,
            :scc => v.expected_scc,
            :scc_se => v.se_expected_scc,
            :prtp => k.prtp,
            :eta => k.eta,
            :distrib_approach => isnothing(k.ew) ? :NA : k.ew,
            :norm_region => isnothing(k.ew) ? "NA" : k.ew_norm_region
        )
    )
end
sort!(df, :label) |> save(joinpath(output_dir, "expected_scc.csv"))

@nbinclude("figures.ipynb")
